"use strict";
/* eslint-disable no-undef */

const ModbusRTU = require("../index");
const TestPort = ModbusRTU.TestPort;
const testPort = new TestPort();
const modbusRTU = new ModbusRTU(testPort);

const sinon = require("sinon");
const expect = require("chai").expect;

describe("ModbusRTU", function() {

    describe("Setup", function() {
        describe("#open() - open serial port.", function() {
            it("should open the port without errors", function(done) {
                modbusRTU.open(function(err) {
                    expect(err).to.be.a("null");
                    done();
                });
            });
        });

        describe("#close() - close serial port.", function() {
            it("should close the port without errors", function(done) {
                modbusRTU.close(function(err) {
                    expect(err).to.be.a("null");

                    done();
                });
            });
        });
    });

    describe("FunctionCodes", function() {
        beforeEach(function(done) {
            modbusRTU.open(function() {
                done();
            });
        });

        afterEach(function(done) {
            modbusRTU.close(function() {
                done();
            });
        });

        describe("#writeFC3() - read holding registers.", function() {
            it("should read 3 registers [0xa12b, 0xffff, 0xb21a] without errors", function(done) {
                modbusRTU.writeFC3(1, 8, 3, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("data").with.length(3);
                    expect(data.data.toString()).to.equal([0xa12b, 0xffff, 0xb21a].toString());

                    done();
                });
            });

            it("should consume a message with overlength if allowDeformedMessage is on", function(done) {
                modbusRTU.writeFC3_deformedReadEnabled(7, 8, 3, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("data").with.length(3);
                    expect(data.data.toString()).to.equal([0xa12b, 0xffff, 0xb21a].toString());

                    done();
                }, true);
            });

            it("should read raw buffer \"a12bffffb21a\" without errors", function(done) {
                modbusRTU.writeFC3(1, 8, 3, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("buffer");
                    expect(data.buffer.toString("hex")).to.equal("a12bffffb21a");

                    done();
                });
            });

            it("should fail on short data answer", function(done) {
                modbusRTU.writeFC3(2, 8, 3, function(err) {
                    expect(err.message).to.have.string("Data length error");

                    done();
                });
            });

            it("should fail on CRC error", function(done) {
                modbusRTU.writeFC3(3, 8, 3, function(err) {
                    expect(err.message).to.have.string("CRC error");

                    done();
                });
            });

            it("should fail on unexpected reply", function(done) {
                modbusRTU.writeFC3(4, 8, 3, function(err) {
                    expect(err.message).to.have.string("Unexpected data error");

                    done();
                });
            });

            it("should fail with an exception", function(done) {
                modbusRTU.writeFC3(5, 8, 3, function(err) {
                    expect(err.message).to.have.string("Modbus exception");

                    done();
                });
            });
        });

        describe("#sendCustomFc() - write custom function code", function() {
            it("should return with an error if there are no maps supplied", function(done) {
                modbusRTU._sendCustomFc(1, 0x40,
                    null, null, function(error) {
                        expect(error.name).to.be.equal("MissingArgumentMapsError");
                        expect(error.message).to.be.equal("Tried to call a custom function code without supplying at least one argument map, every custom function code needs at least a request map or a response map.");
                        expect(error.errno).to.be.equal(null);
                        done();
                    });
            });

            it("should return an error if the data type is unknown", function(done) {
                modbusRTU._sendCustomFc(1, 0x43,
                    [
                        { name: "meicode", data: 14, offset: 0, type: "notvalid32be" }
                    ],
                    null,
                    function(error) {
                        expect(error.name).to.be.equal("UnknownDataType");
                        expect(error.message).to.be.equal("Unknown data type in argument map definition.");
                        expect(error.errno).to.be.a("null");
                        done();
                    });
            });

            it("should return an error if the data offset is wrong in the request map", function(done) {
                modbusRTU._sendCustomFc(1, 0x43,
                    [
                        { name: "meicode", data: 14, offset: 5, type: "uint16be" }
                    ],
                    null,
                    function(error) {
                        expect(error.name).to.be.equal("OffsetPositionInvalid");
                        expect(error.message).to.be.equal("The supplied offset for one of the fields in the argument map want's to read at the wrong position in memory");
                        expect(error.errno).to.be.a("null");
                        done();
                    });
            });

            it("should return an error if one of the arguments in the request map is missing it's data field", function(done) {
                modbusRTU._sendCustomFc(1, 0x43,
                    [
                        { name: "meicode", offset: 0, type: "uint16be" }
                    ],
                    null,
                    function(error) {
                        expect(error.name).to.be.equal("ArgumentWithoutData");
                        expect(error.message).to.be.equal("One of the arguments in the Custom Functioncode request is missing it's data field.");
                        expect(error.errno).to.be.a("null");
                        done();
                    });
            });

            it("should return MissingArgumentMapProperty if it's missing name, offset or type properties", function(done) {
                modbusRTU._sendCustomFc(1, 0x43,
                    [
                        { name: "meicode", type: "uint16be" }
                    ],
                    null,
                    function(error) {
                        expect(error.name).to.be.equal("MissingArgumentMapProperty");
                        expect(error.message).to.be.equal("One of the defined data types in your argument map is missing a required field check if all data types have their required fields.");
                        expect(error.errno).to.be.a("null");
                        done();
                    });
            });

            it("can receive fc20 file commands", function(done) {
                modbusRTU._sendCustomFc(1, 20,
                    [
                        {
                            "name": "byteCount",
                            "data": 7,
                            "offset": 0,
                            "type": "uint8be"
                        },
                        {
                            "name": "subReqReferenceType",
                            "data": 6,
                            "offset": 1,
                            "type": "uint8be"
                        },
                        {
                            "name": "subReqFileNumber",
                            "data": 1,
                            "offset": 2,
                            "type": "uint16be"
                        },
                        {
                            "name": "subReqRecordNumber",
                            "data": 5,
                            "offset": 4,
                            "type": "uint16be"
                        },
                        {
                            "name": "subReqRecordLength",
                            "data": 2,
                            "offset": 6,
                            "type": "uint16be"
                        }
                    ],
                    [
                        {
                            "name": "responseDataLength",
                            "data": 0,
                            "offset": 0,
                            "type": "uint8be"
                        },
                        {
                            "name": "subReqFileResponsLength",
                            "data": 0,
                            "offset": 1,
                            "type": "uint8be"
                        },
                        {
                            "name": "subReqReferenceType",
                            "data": 0,
                            "offset": 2,
                            "type": "uint8be"
                        },
                        {
                            "name": "subReqRecordData",
                            "data": 0,
                            "offset": 3,
                            "type": "uint32be"
                        }
                    ],
                    function(error, data) {
                        expect(data).to.have.ownProperty("responseDataLength");
                        expect(data.responseDataLength).to.be.equal(0x06);
                        expect(data).to.have.ownProperty("subReqFileResponsLength");
                        expect(data.subReqFileResponsLength).to.be.equal(0x05);
                        expect(data).to.have.ownProperty("subReqReferenceType");
                        expect(data.subReqReferenceType).to.be.equal(0x06);
                        expect(data).to.have.ownProperty("subReqRecordData");
                        expect(data.subReqRecordData).to.be.equal(0x0DFE0020);
                        done();
                    });
            });

            it("can have a custom response argument map which is used if the response is unknown", function(done) {
                modbusRTU._sendCustomFc(1, 0x40,
                    [
                        { name: "length", data: 0x01, offset: 0, type: "uint8be" },
                        { name: "data", data: 0x01, offset: 1, type: "uint8be" }
                    ],
                    [
                        { name: "length",  offset: 0, type: "uint8be" },
                        { name: "acmx",    offset: 1, type: "uint16be" },
                        { name: "acmn",    offset: 3, type: "uint16be" },
                        { name: "dcmx",    offset: 5, type: "uint16be" },
                        { name: "dcmn",    offset: 7, type: "uint16be" },
                        { name: "fmx",     offset: 9, type: "uint16be" },
                        { name: "fmn",     offset: 11, type: "uint16be" },
                        { name: "ot1",     offset: 13, type: "uint16be" },
                        { name: "ot2",     offset: 15, type: "uint16be" },
                        { name: "ot3",     offset: 17, type: "uint16be" },
                        { name: "ot4",     offset: 19, type: "uint16be" },
                        { name: "ot5",     offset: 21, type: "uint16be" },
                        { name: "com",     offset: 23, type: "uint16be" },
                        { name: "isl",     offset: 25, type: "uint16be" },
                        { name: "res1",    offset: 27, type: "uint16be" },
                        { name: "res2",    offset: 29, type: "uint16be" },
                        { name: "ovrcurr", offset: 31, type: "uint16be" },
                        { name: "res3",    offset: 33, type: "uint16be" },
                        { name: "ovrvolt", offset: 35, type: "uint16be" },
                        { name: "avrg",    offset: 37, type: "uint16be" }
                    ],
                    function(err, data) {
                    // TODO(Kay): Fill me out tomorrow !
                        expect(err).to.be.a("null");
                        expect(data).to.have.ownProperty("length");
                        expect(data.length).to.be.equal(40);
                        expect(data).to.have.ownProperty("acmx");
                        expect(data.acmx).to.be.equal(0xAAAA);
                        expect(data).to.have.ownProperty("acmn");
                        expect(data.acmn).to.be.equal(0xBBBB);
                        expect(data).to.have.ownProperty("dcmx");
                        expect(data.dcmx).to.be.equal(0xCCCC);
                        expect(data).to.have.ownProperty("dcmn");
                        expect(data.dcmn).to.be.equal(0xDDDD);
                        expect(data).to.have.ownProperty("fmx");
                        expect(data.fmx).to.be.equal(0xEEEE);
                        expect(data).to.have.ownProperty("fmn");
                        expect(data.fmn).to.be.equal(0xFFFF);
                        expect(data).to.have.ownProperty("ot1");
                        expect(data.ot1).to.be.equal(0xAAAA);
                        expect(data).to.have.ownProperty("ot2");
                        expect(data.ot2).to.be.equal(0xBBBB);
                        expect(data).to.have.ownProperty("ot3");
                        expect(data.ot3).to.be.equal(0xCCCC);
                        expect(data).to.have.ownProperty("ot4");
                        expect(data.ot4).to.be.equal(0xDDDD);
                        expect(data).to.have.ownProperty("ot5");
                        expect(data.ot5).to.be.equal(0xEEEE);
                        expect(data).to.have.ownProperty("com");
                        expect(data.com).to.be.equal(0xFFFF);
                        expect(data).to.have.ownProperty("isl");
                        expect(data.isl).to.be.equal(0xAAAA);
                        expect(data).to.have.ownProperty("res1");
                        expect(data.res1).to.be.equal(0xBBBB);
                        expect(data).to.have.ownProperty("res2");
                        expect(data.res2).to.be.equal(0xCCCC);
                        expect(data).to.have.ownProperty("ovrcurr");
                        expect(data.ovrcurr).to.be.equal(0xDDDD);
                        expect(data).to.have.ownProperty("res3");
                        expect(data.res3).to.be.equal(0xEEEE);
                        expect(data).to.have.ownProperty("ovrvolt");
                        expect(data.ovrvolt).to.be.equal(0xFFFF);
                        expect(data).to.have.ownProperty("avrg");
                        expect(data.avrg).to.be.equal(0xAAAA);
                        // expect(data.acmx).to.equal(0x0000)
                        // expect(data.acmn).to.equal(0x0000)
                        // ...
                        done();
                    });
            });
        });

        describe("#writeFC4() - read input registers.", function() {
            it("should read 3 registers [8, 9, 10] without errors", function(done) {
                modbusRTU.writeFC4(1, 8, 3, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("data").with.length(3);
                    expect(data.data.toString()).to.equal([8, 9, 10].toString());

                    done();
                });
            });


            it("should consume a message with overlength if allowDeformedMessage is on", function(done) {
                modbusRTU.writeFC4_deformedReadEnabled(7, 8, 3, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("data").with.length(3);
                    expect(data.data.toString()).to.equal([8, 9, 10].toString());

                    done();
                }, true);
            });

            it("should fail on short data answer", function(done) {
                modbusRTU.writeFC4(2, 8, 1, function(err) {
                    expect(err.message).to.have.string("Data length error");

                    done();
                });
            });

            it("should fail on CRC error", function(done) {
                modbusRTU.writeFC4(3, 8, 1, function(err) {
                    expect(err.message).to.have.string("CRC error");

                    done();
                });
            });

            it("should fail on unexpected reply", function(done) {
                modbusRTU.writeFC4(4, 8, 1, function(err) {
                    expect(err.message).to.have.string("Unexpected data error");

                    done();
                });
            });

            it("should fail with an exception", function(done) {
                modbusRTU.writeFC4(5, 8, 3, function(err) {
                    expect(err.message).to.have.string("Modbus exception");

                    done();
                });
            });
        });

        describe("#writeFC6() - write single holding register.", function() {
            it("should write to register 1 42 without errors", function(done) {
                modbusRTU.writeFC6(1, 1, 42, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data.address).to.equal(1);
                    expect(data.value).to.equal(42);

                    done();
                });
            });

            it("should fail on short data answer", function(done) {
                modbusRTU.writeFC6(2, 1, 42, function(err) {
                    expect(err.message).to.have.string("Data length error");

                    done();
                });
            });

            it("should fail on CRC error", function(done) {
                modbusRTU.writeFC6(3, 1, 42, function(err) {
                    expect(err.message).to.have.string("CRC error");

                    done();
                });
            });

            it("should fail on unexpected reply", function(done) {
                modbusRTU.writeFC6(4, 1, 42, function(err) {
                    expect(err.message).to.have.string("Unexpected data error");

                    done();
                });
            });

            it("should fail with an exception", function(done) {
                modbusRTU.writeFC6(5, 1, 42, function(err) {
                    expect(err.message).to.have.string("Modbus exception");

                    done();
                });
            });
        });

        describe("#writeFC15() - force multiple coils.", function() {
            it("should write 3 coils [true, false, true] without errors", function(done) {
                modbusRTU.writeFC15(1, 8, [true, false, true], function(err) {
                    expect(err).to.be.a("null");

                    done();
                });
            });

            it("should fail on short data answer", function(done) {
                modbusRTU.writeFC15(2, 8, [true, false, true], function(err) {
                    expect(err.message).to.have.string("Data length error");

                    done();
                });
            });

            it("should fail on CRC error", function(done) {
                modbusRTU.writeFC15(3, 8, [true, false, true], function(err) {
                    expect(err.message).to.have.string("CRC error");

                    done();
                });
            });

            it("should fail on unexpected reply", function(done) {
                modbusRTU.writeFC15(4, 8, [true, false, true], function(err) {
                    expect(err.message).to.have.string("Unexpected data error");

                    done();
                });
            });

            it("should fail with an exception", function(done) {
                modbusRTU.writeFC15(5, 8, [true, false, true], function(err) {
                    expect(err.message).to.have.string("Modbus exception");

                    done();
                });
            });
        });

        describe("#writeFC1() - read coils after force multiple coils.", function() {
            it("should read coil 8, 9 ,10 to be true, false, true", function(done) {
                modbusRTU.writeFC1(1, 8, 4, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("data");
                    expect(data.data[0]).to.equal(true);
                    expect(data.data[1]).to.equal(false);
                    expect(data.data[2]).to.equal(true);

                    done();
                });
            });
        });


        it("should consume a message with overlength if allowDeformedMessage is on", function(done) {
            modbusRTU.writeFC1_deformedReadEnabled(7, 8, 3, function(err, data) {
                expect(err).to.be.a("null");
                expect(data).to.have.property("data");
                expect(data.data[0]).to.equal(false);
                expect(data.data[1]).to.equal(false);
                expect(data.data[2]).to.equal(false);

                done();
            }, true);
        });

        describe("#writeFC16() - write holding registers.", function() {
            it("should write 3 registers [42, 128, 5] without errors", function(done) {
                modbusRTU.writeFC16(1, 8, [42, 128, 5], function(err) {
                    expect(err).to.be.a("null");

                    done();
                });
            });

            it("should fail on short data answer", function(done) {
                modbusRTU.writeFC16(2, 8, [42, 128, 5], function(err) {
                    expect(err.message).to.have.string("Data length error");

                    done();
                });
            });

            it("should fail on CRC error", function(done) {
                modbusRTU.writeFC16(3, 8, [42, 128, 5], function(err) {
                    expect(err.message).to.have.string("CRC error");

                    done();
                });
            });

            it("should fail on unexpected reply", function(done) {
                modbusRTU.writeFC16(4, 8, [42, 128, 5], function(err) {
                    expect(err.message).to.have.string("Unexpected data error");

                    done();
                });
            });

            it("should fail with an exception", function(done) {
                modbusRTU.writeFC16(5, 8, [42, 128, 5], function(err) {
                    expect(err.message).to.have.string("Modbus exception");

                    done();
                });
            });
        });

        describe("#writeFC3() - read holding registers after write.", function() {
            it("should read 3 registers [42, 128, 5] without errors", function(done) {
                modbusRTU.writeFC3(1, 8, 3, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("data").with.length(3);
                    expect(data.data.toString()).to.equal([42, 128, 5].toString());

                    done();
                });
            });

            it("should include raw payload data in response if debug enabled", function(done) {
                /* This feature is common to _all_ function handlers. */
                modbusRTU.isDebugEnabled = true;

                modbusRTU.writeFC3(1, 8, 3, function(err, data) {
                    modbusRTU.isDebugEnabled = false;

                    expect(err).to.be.a("null");
                    expect(data).to.have.property("request").with.length(8);
                    expect(data.request.toString("hex")).to.equal("0103000800038409");

                    expect(data).to.have.property("responses").with.length(1);
                    expect(data.responses[0]).to.be.instanceof(Buffer).with.length(11);
                    expect(data.responses[0].toString("hex")).to.equal("010306002a00800005f958");

                    done();
                });
            });

            it("should include raw payload data in exception if debug enabled", function(done) {
                /* This feature is common to _all_ function handlers. */
                modbusRTU.isDebugEnabled = true;

                // Pretend Unit 3 sends buggy CRCs apparently.
                modbusRTU.writeFC3(3, 8, 3, function(err, data) {
                    modbusRTU.isDebugEnabled = false;

                    expect(err).to.not.be.a("null");
                    expect(err).to.have.property("modbusRequest").with.length(8);
                    expect(err.modbusRequest.toString("hex")).to.equal("03030008000385eb");

                    expect(err).to.have.property("modbusResponses").with.length(1);
                    expect(err.modbusResponses[0]).to.be.instanceof(Buffer).with.length(11);
                    expect(err.modbusResponses[0].toString("hex")).to.equal("030306002a00800005e138");

                    expect(data).to.be.undefined;

                    done();
                });
            });
        });

        describe("#writeFC5() - force one coil.", function() {
            it("should force coil 3 to be true, without errors", function(done) {
                modbusRTU.writeFC5(1, 3, true, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("state");
                    expect(data.state).to.equal(true);

                    done();
                });
            });
        });

        describe("#writeFC1() - read coils after force coil.", function() {
            it("should read coil 3 to be true, without errors", function(done) {
                modbusRTU.writeFC1(1, 3, 9, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("data");
                    expect(data.data[0]).to.equal(true);
                    expect(data.data[3]).to.equal(false);

                    done();
                });
            });
        });

        describe("#writeFC1() - read inputs.", function() {
            it("should read input 0 to be false, without errors", function(done) {
                modbusRTU.writeFC1(1, 0, 9, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data).to.have.property("data");
                    expect(data.data[0]).to.equal(false);
                    expect(data.data[3]).to.equal(true);

                    done();
                });
            });
        });

        describe("#writeFC43() - read device identification", function() {
            it("should return a device identificationm without errors", function(done) {
                modbusRTU.writeFC43(1, 4, 1, function(err, data) {
                    expect(err).to.be.a("null");
                    expect(data.conformityLevel).to.equal(1);
                    expect(data.data["1"]).to.equal("MyProductCode1234");
                    done();
                });
            });
        });

        describe("Timeout", function() {
            const timeout = 1000;
            let clock;
            beforeEach(function() {
                clock = sinon.useFakeTimers();
            });

            afterEach(function() {
                clock.restore();
            });

            it("should time out", function(done) {
                modbusRTU._timeout = timeout;
                modbusRTU.writeFC3(6, 8, 3, function(err) {
                    expect(err.message).to.have.string("Timed out");
                    done();
                });

                clock.tick(timeout);
            });

            describe("Promise", function() {
                it("should reject with error if timeout is hit", function(done) {
                    modbusRTU.setID(6);
                    modbusRTU.setTimeout(timeout);
                    modbusRTU.readCoils(1, 1)
                        .then(function() {
                            done(new Error("Call should timeout"));
                        })
                        .catch(function(err) {
                            expect(err.message).to.have.string("Timed out");
                            done();
                        });

                    clock.tick(timeout);
                });
            });
        });
    });
});
