/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/


/**
 * Trigger enabled/disabled events when element.prop("disabled",false/true) is
 * called.
 * Used by RED.popover to hide a popover when the trigger element is disabled
 * as a disabled element doesn't emit mouseleave
 */
jQuery.propHooks.disabled = {
    set: function (element, value) {
        if (element.disabled !== value) {
            element.disabled = value;
            if (value) {
                $(element).trigger('disabled');
            } else {
                $(element).trigger('enabled');
            }
        }
    }
};
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
var RED = (function() {


    function loadPluginList() {
        loader.reportProgress(RED._("event.loadPlugins"), 10)
        $.ajax({
            headers: {
                "Accept":"application/json"
            },
            cache: false,
            url: 'plugins',
            success: function(data) {
                RED.plugins.setPluginList(data);
                loader.reportProgress(RED._("event.loadPlugins"), 13)
                RED.i18n.loadPluginCatalogs(function() {
                    loadPlugins(function() {
                        loadNodeList();
                    });
                });
            }
        });
    }
    function loadPlugins(done) {
        loader.reportProgress(RED._("event.loadPlugins",{count:""}), 17)
        var lang = localStorage.getItem("editor-language")||RED.i18n.detectLanguage();

        $.ajax({
            headers: {
                "Accept":"text/html",
                "Accept-Language": lang
            },
            cache: false,
            url: 'plugins',
            success: function(data) {
                var configs = data.trim().split(/(?=<!-- --- \[red-plugin:\S+\] --- -->)/);
                var totalCount = configs.length;
                var stepConfig = function() {
                    // loader.reportProgress(RED._("event.loadNodes",{count:(totalCount-configs.length)+"/"+totalCount}), 30 + ((totalCount-configs.length)/totalCount)*40 )
                    if (configs.length === 0) {
                        done();
                    } else {
                        var config = configs.shift();
                        appendPluginConfig(config,stepConfig);
                    }
                }
                stepConfig();
            }
        });
    }

    function appendConfig(config, moduleIdMatch, targetContainer, done) {
        done = done || function(){};
        var moduleId;
        if (moduleIdMatch) {
            moduleId = moduleIdMatch[1];
            RED._loadingModule = moduleId;
        } else {
            moduleId = "unknown";
        }
        try {
            var hasDeferred = false;
            var nodeConfigEls = $("<div>"+config+"</div>");
            var scripts = nodeConfigEls.find("script");
            var scriptCount = scripts.length;
            scripts.each(function(i,el) {
                var srcUrl = $(el).attr('src');
                if (srcUrl && !/^\s*(https?:|\/|\.)/.test(srcUrl)) {
                    $(el).remove();
                    var newScript = document.createElement("script");
                    newScript.onload = function() {
                        scriptCount--;
                        if (scriptCount === 0) {
                            $(targetContainer).append(nodeConfigEls);
                            delete RED._loadingModule;
                            done()
                        }
                    }
                    if ($(el).attr('type') === "module") {
                        newScript.type = "module";
                    }
                    $(targetContainer).append(newScript);
                    newScript.src = RED.settings.apiRootUrl+srcUrl;
                    hasDeferred = true;
                } else {
                    if (/\/ace.js$/.test(srcUrl) || /\/ext-language_tools.js$/.test(srcUrl)) {
                        // Block any attempts to load ace.js from a CDN - this will
                        // break the version of ace included in the editor.
                        // At the time of commit, the contrib-python nodes did this.
                        // This is a crude fix until the python nodes are fixed.
                        console.warn("Blocked attempt to load",srcUrl,"by",moduleId)
                        $(el).remove();
                    }
                    scriptCount--;
                }
            })
            if (!hasDeferred) {
                $(targetContainer).append(nodeConfigEls);
                delete RED._loadingModule;
                done();
            }
        } catch(err) {
            RED.notify(RED._("notification.errors.failedToAppendNode",{module:moduleId, error:err.toString()}),{
                type: "error",
                timeout: 10000
            });
            console.log("["+moduleId+"] "+err.toString());
            delete RED._loadingModule;
            done();
        }
    }
    function appendPluginConfig(pluginConfig,done) {
        appendConfig(
            pluginConfig,
            /<!-- --- \[red-plugin:(\S+)\] --- -->/.exec(pluginConfig.trim()),
            "#red-ui-editor-plugin-configs",
            done
        );
    }

    function appendNodeConfig(nodeConfig,done) {
        appendConfig(
            nodeConfig,
            /<!-- --- \[red-module:(\S+)\] --- -->/.exec(nodeConfig.trim()),
            "#red-ui-editor-node-configs",
            done
        );
    }

    function loadNodeList() {
        loader.reportProgress(RED._("event.loadPalette"), 20)
        $.ajax({
            headers: {
                "Accept":"application/json"
            },
            cache: false,
            url: 'nodes',
            success: function(data) {
                RED.nodes.setNodeList(data);
                loader.reportProgress(RED._("event.loadNodeCatalogs"), 25)
                RED.i18n.loadNodeCatalogs(function() {
                    loadIconList(loadNodes);
                });
            }
        });
    }

    function loadIconList(done) {
        $.ajax({
            headers: {
                "Accept":"application/json"
            },
            cache: false,
            url: 'icons',
            success: function(data) {
                RED.nodes.setIconSets(data);
                if (done) {
                    done();
                }
            }
        });
    }

    function loadNodes() {
        loader.reportProgress(RED._("event.loadNodes",{count:""}), 30)
        var lang = localStorage.getItem("editor-language")||RED.i18n.detectLanguage();

        $.ajax({
            headers: {
                "Accept":"text/html",
                "Accept-Language": lang
            },
            cache: false,
            url: 'nodes',
            success: function(data) {
                var configs = data.trim().split(/(?=<!-- --- \[red-module:\S+\] --- -->)/);
                var totalCount = configs.length;

                var stepConfig = function() {
                    loader.reportProgress(RED._("event.loadNodes",{count:(totalCount-configs.length)+"/"+totalCount}), 30 + ((totalCount-configs.length)/totalCount)*40 )

                    if (configs.length === 0) {
                        $("#red-ui-editor").i18n();
                        $("#red-ui-palette > .red-ui-palette-spinner").hide();
                        $(".red-ui-palette-scroll").removeClass("hide");
                        $("#red-ui-palette-search").removeClass("hide");
                        if (RED.settings.theme("projects.enabled",false)) {
                            RED.projects.refresh(function(activeProject) {
                                loadFlows(function() {
                                    RED.sidebar.info.refresh()
                                    var showProjectWelcome = false;
                                    if (!activeProject) {
                                        // Projects enabled but no active project
                                        RED.menu.setDisabled('menu-item-projects-open',true);
                                        RED.menu.setDisabled('menu-item-projects-settings',true);
                                        if (activeProject === false) {
                                            // User previously decline the migration to projects.
                                        } else { // null/undefined
                                            showProjectWelcome = true;
                                        }
                                    }
                                    completeLoad(showProjectWelcome);
                                });
                            });
                        } else {
                            loadFlows(function() {
                                // Projects disabled by the user
                                RED.sidebar.info.refresh()
                                completeLoad();
                            });
                        }
                    } else {
                        var config = configs.shift();
                        appendNodeConfig(config,stepConfig);
                    }
                }
                stepConfig();
            }
        });
    }

    function loadFlows(done) {
        loader.reportProgress(RED._("event.loadFlows"),80 )
        $.ajax({
            headers: {
                "Accept":"application/json",
            },
            cache: false,
            url: 'flows',
            success: function(nodes) {
                if (nodes) {
                    var currentHash = window.location.hash;
                    RED.nodes.version(nodes.rev);
                    loader.reportProgress(RED._("event.importFlows"),90 )
                    try {
                        RED.nodes.import(nodes.flows);
                        RED.nodes.dirty(false);
                        RED.view.redraw(true);
                        if (/^#(flow|node|group)\/.+$/.test(currentHash)) {
                            const hashParts = currentHash.split('/')
                            const showEditDialog = hashParts.length > 2 && hashParts[2] === 'edit'
                            if (hashParts[0] === '#flow') {
                                RED.workspaces.show(hashParts[1], true);
                                if (showEditDialog) {
                                    RED.workspaces.edit()
                                }
                            } else if (hashParts[0] === '#node') {
                                const nodeToShow = RED.nodes.node(hashParts[1])
                                if (nodeToShow) {
                                    setTimeout(() => {
                                        RED.view.reveal(nodeToShow.id)
                                        window.location.hash = currentHash
                                        RED.view.select(nodeToShow.id)
                                        if (showEditDialog) {
                                            RED.editor.edit(nodeToShow)
                                        }
                                    }, 50)
                                }
                            } else if (hashParts[0] === '#group') {
                                const nodeToShow = RED.nodes.group(hashParts[1])
                                if (nodeToShow) {
                                    RED.view.reveal(nodeToShow.id)
                                    window.location.hash = currentHash
                                    RED.view.select(nodeToShow.id)
                                    if (showEditDialog) {
                                        RED.editor.editGroup(nodeToShow)
                                    }
                                }
                            }
                        }
                        if (RED.workspaces.count() > 0) {
                            const hiddenTabs = JSON.parse(RED.settings.getLocal("hiddenTabs")||"{}");
                            const workspaces = RED.nodes.getWorkspaceOrder();
                            if (RED.workspaces.active() === 0) {
                                for (let index = 0; index < workspaces.length; index++) {
                                    const ws = workspaces[index];
                                    if (!hiddenTabs[ws]) {
                                        RED.workspaces.show(ws);
                                        break;
                                    }
                                }
                            }
                            if (RED.workspaces.active() === 0) {
                                RED.workspaces.show(workspaces[0]);
                            }
                        }
                        RED.events.emit('flows:loaded')
                    } catch(err) {
                        console.warn(err);
                        RED.notify(
                            RED._("event.importError", {message: err.message}),
                            {
                                fixed: true,
                                type: 'error'
                            }
                        );
                    }
                }
                done();
            }
        });
    }

    function completeLoad(showProjectWelcome) {
        var persistentNotifications = {};
        RED.comms.subscribe("notification/#",function(topic,msg) {
            var parts = topic.split("/");
            var notificationId = parts[1];
            if (notificationId === "runtime-deploy") {
                // handled in ui/deploy.js
                return;
            }
            if (notificationId === "node") {
                // handled below
                return;
            }
            if (notificationId === "flows-run-state") {
                // handled in editor-client/src/js/runtime.js
                return;
            }
            if (notificationId === "project-update") {
                loader.start(RED._("event.loadingProject"), 0);
                RED.nodes.clear();
                RED.history.clear();
                RED.view.redraw(true);
                RED.projects.refresh(function() {
                    loadFlows(function() {
                        var project = RED.projects.getActiveProject();
                        var message = {
                            "change-branch": RED._("notification.project.change-branch", {project: project.git.branches.local}),
                            "merge-abort": RED._("notification.project.merge-abort"),
                            "loaded": RED._("notification.project.loaded", {project: msg.project}),
                            "updated": RED._("notification.project.updated", {project: msg.project}),
                            "pull": RED._("notification.project.pull", {project: msg.project}),
                            "revert": RED._("notification.project.revert", {project: msg.project}),
                            "merge-complete": RED._("notification.project.merge-complete")
                        }[msg.action];
                        loader.end()
                        RED.notify($("<p>").text(message));
                        RED.sidebar.info.refresh()
                        RED.menu.setDisabled('menu-item-projects-open',false);
                        RED.menu.setDisabled('menu-item-projects-settings',false);
                    });
                });
                return;
            }
            if (notificationId === "update-available") {
                 // re-emit as an event to be handled in editor-client/src/js/ui/palette-editor.js
                 RED.events.emit("notification/update-available", msg)
            }
            if (msg.text) {
                msg.default = msg.text;
                var text = RED._(msg.text,msg);
                var options = {
                    type: msg.type,
                    fixed: msg.timeout === undefined,
                    timeout: msg.timeout,
                    id: notificationId
                }
                if (notificationId === "runtime-state") {
                    if (msg.error === "safe-mode") {
                        options.buttons = [
                            {
                                text: RED._("common.label.close"),
                                click: function() {
                                    persistentNotifications[notificationId].hideNotification();
                                }
                            }
                        ]
                    } else if (msg.error === "missing-types") {
                        text+="<ul><li>"+msg.types.map(RED.utils.sanitize).join("</li><li>")+"</li></ul>";
                        if (!!RED.projects.getActiveProject()) {
                            options.buttons = [
                                {
                                    text: RED._("notification.label.manage-project-dep"),
                                    click: function() {
                                        persistentNotifications[notificationId].hideNotification();
                                        RED.projects.settings.show('deps');
                                    }
                                }
                            ]
                            // } else if (RED.settings.get('externalModules.palette.allowInstall', true) !== false) {
                        } else {
                            options.buttons = [
                                {
                                    text: RED._("notification.label.unknownNodesButton"),
                                    class: "pull-left",
                                    click: function() {
                                        RED.actions.invoke("core:search", "type:unknown ");
                                    }
                                },
                                {
                                    class: "primary",
                                    text: RED._("common.label.close"),
                                    click: function() {
                                        persistentNotifications[notificationId].hideNotification();
                                    }
                                }
                            ]
                        }
                    } else if (msg.error === "missing-modules") {
                        text+="<ul><li>"+msg.modules.map(function(m) { return RED.utils.sanitize(m.module)+(m.error?(" - <small>"+RED.utils.sanitize(""+m.error)+"</small>"):"")}).join("</li><li>")+"</li></ul>";
                        options.buttons = [
                            {
                                text: RED._("common.label.close"),
                                click: function() {
                                    persistentNotifications[notificationId].hideNotification();
                                }
                            }
                        ]
                    } else if (msg.error === "credentials_load_failed") {
                        if (RED.settings.theme("projects.enabled",false)) {
                            // projects enabled
                            if (RED.user.hasPermission("projects.write")) {
                                options.buttons = [
                                    {
                                        text: RED._("notification.project.setupCredentials"),
                                        click: function() {
                                            persistentNotifications[notificationId].hideNotification();
                                            RED.projects.showCredentialsPrompt();
                                        }
                                    }
                                ]
                            }
                        } else {
                            options.buttons = [
                                {
                                    text: RED._("common.label.close"),
                                    click: function() {
                                        persistentNotifications[notificationId].hideNotification();
                                    }
                                }
                            ]
                        }
                    } else if (msg.error === "missing_flow_file") {
                        if (RED.user.hasPermission("projects.write")) {
                            options.buttons = [
                                {
                                    text: RED._("notification.project.setupProjectFiles"),
                                    click: function() {
                                        persistentNotifications[notificationId].hideNotification();
                                        RED.projects.showFilesPrompt();
                                    }
                                }
                            ]
                        }
                    } else if (msg.error === "missing_package_file") {
                        if (RED.user.hasPermission("projects.write")) {
                            options.buttons = [
                                {
                                    text: RED._("notification.project.setupProjectFiles"),
                                    click: function() {
                                        persistentNotifications[notificationId].hideNotification();
                                        RED.projects.showFilesPrompt();
                                    }
                                }
                            ]
                        }
                    } else if (msg.error === "project_empty") {
                        if (RED.user.hasPermission("projects.write")) {
                            options.buttons = [
                                {
                                    text: RED._("notification.project.no"),
                                    click: function() {
                                        persistentNotifications[notificationId].hideNotification();
                                    }
                                },
                                {
                                    text: RED._("notification.project.createDefault"),
                                    click: function() {
                                        persistentNotifications[notificationId].hideNotification();
                                        RED.projects.createDefaultFileSet();
                                    }
                                }
                            ]
                        }
                    } else if (msg.error === "git_merge_conflict") {
                        RED.nodes.clear();
                        RED.sidebar.versionControl.refresh(true);
                        if (RED.user.hasPermission("projects.write")) {
                            options.buttons = [
                                {
                                    text: RED._("notification.project.mergeConflict"),
                                    click: function() {
                                        persistentNotifications[notificationId].hideNotification();
                                        RED.sidebar.versionControl.showLocalChanges();
                                    }
                                }
                            ]
                        }
                    }
                } else if (notificationId === 'restart-required') {
                    options.buttons = [
                        {
                            text: RED._("common.label.close"),
                            click: function() {
                                persistentNotifications[notificationId].hideNotification();
                            }
                        }
                    ]
                }
                if (!persistentNotifications.hasOwnProperty(notificationId)) {
                    persistentNotifications[notificationId] = RED.notify(text,options);
                } else {
                    persistentNotifications[notificationId].update(text,options);
                }
            } else if (persistentNotifications.hasOwnProperty(notificationId)) {
                persistentNotifications[notificationId].close();
                delete persistentNotifications[notificationId];
            }
            if (notificationId === 'runtime-state') {
                RED.events.emit("runtime-state",msg);
            }
        });
        RED.comms.subscribe("status/#",function(topic,msg) {
            var parts = topic.split("/");
            var node = RED.nodes.node(parts[1]);
            if (node) {
                if (msg.hasOwnProperty("text") && msg.text !== null && /^[@a-zA-Z]/.test(msg.text)) {
                    msg.text = node._(msg.text.toString(),{defaultValue:msg.text.toString()});
                }
                node.status = msg;
                node.dirtyStatus = true;
                node.dirty = true;
                RED.view.redrawStatus(node);
            }
        });
        RED.comms.subscribe("notification/plugin/#",function(topic,msg) {
            if (topic == "notification/plugin/added") {
                RED.settings.refreshSettings(function(err, data) {
                    let addedPlugins = [];
                    msg.forEach(function(m) {
                        let id = m.id;
                        RED.plugins.addPlugin(m);

                        m.plugins.forEach((p) => {
                            addedPlugins.push(p.id);
                        })

                        RED.i18n.loadNodeCatalog(id, function() {
                            var lang = localStorage.getItem("editor-language")||RED.i18n.detectLanguage();
                            $.ajax({
                                headers: {
                                    "Accept":"text/html",
                                    "Accept-Language": lang
                                },
                                cache: false,
                                url: 'plugins/'+id,
                                success: function(data) {
                                    appendPluginConfig(data);
                                }
                            });
                        });
                    });
                    if (addedPlugins.length) {
                        let pluginList = "<ul><li>"+addedPlugins.map(RED.utils.sanitize).join("</li><li>")+"</li></ul>";
                        // ToDo: Adapt notification (node -> plugin)
                        RED.notify(RED._("palette.event.nodeAdded", {count:addedPlugins.length})+pluginList,"success");
                    }
                })
            }
        });

        let pendingNodeRemovedNotifications = []
        let pendingNodeRemovedTimeout

        RED.comms.subscribe("notification/node/#",function(topic,msg) {
            var i,m;
            var typeList;
            var info;
            if (topic == "notification/node/added") {
                RED.settings.refreshSettings(function(err, data) {
                    var addedTypes = [];
                    msg.forEach(function(m) {
                        var id = m.id;
                        RED.nodes.addNodeSet(m);
                        addedTypes = addedTypes.concat(m.types);
                        RED.i18n.loadNodeCatalog(id, function() {
                            var lang = localStorage.getItem("editor-language")||RED.i18n.detectLanguage();
                            $.ajax({
                                headers: {
                                    "Accept":"text/html",
                                    "Accept-Language": lang
                                },
                                cache: false,
                                url: 'nodes/'+id,
                                success: function(data) {
                                    appendNodeConfig(data);
                                }
                            });
                        });
                    });
                    if (addedTypes.length) {
                        typeList = "<ul><li>"+addedTypes.map(RED.utils.sanitize).join("</li><li>")+"</li></ul>";
                        RED.notify(RED._("palette.event.nodeAdded", {count:addedTypes.length})+typeList,"success");
                    }
                    loadIconList();
                })
            } else if (topic == "notification/node/removed") {
                for (i=0;i<msg.length;i++) {
                    m = msg[i];
                    info = RED.nodes.removeNodeSet(m.id);
                    if (info.added) {
                        pendingNodeRemovedNotifications = pendingNodeRemovedNotifications.concat(m.types.map(RED.utils.sanitize))
                        if (pendingNodeRemovedTimeout) {
                            clearTimeout(pendingNodeRemovedTimeout)
                        }
                        pendingNodeRemovedTimeout = setTimeout(function () {
                            typeList = "<ul><li>"+pendingNodeRemovedNotifications.join("</li><li>")+"</li></ul>";
                            RED.notify(RED._("palette.event.nodeRemoved", {count:pendingNodeRemovedNotifications.length})+typeList,"success");
                            pendingNodeRemovedNotifications = []
                        }, 200)
                    }
                }
                loadIconList();
            } else if (topic == "notification/node/enabled") {
                if (msg.types) {
                    RED.settings.refreshSettings(function(err, data) {
                        info = RED.nodes.getNodeSet(msg.id);
                        if (info.added) {
                            RED.nodes.enableNodeSet(msg.id);
                            typeList = "<ul><li>"+msg.types.map(RED.utils.sanitize).join("</li><li>")+"</li></ul>";
                            RED.notify(RED._("palette.event.nodeEnabled", {count:msg.types.length})+typeList,"success");
                        } else {
                            var lang = localStorage.getItem("editor-language")||RED.i18n.detectLanguage();
                            $.ajax({
                                headers: {
                                    "Accept":"text/html",
                                    "Accept-Language": lang
                                },
                                cache: false,
                                url: 'nodes/'+msg.id,
                                success: function(data) {
                                    appendNodeConfig(data);
                                    typeList = "<ul><li>"+msg.types.map(RED.utils.sanitize).join("</li><li>")+"</li></ul>";
                                    RED.notify(RED._("palette.event.nodeAdded", {count:msg.types.length})+typeList,"success");
                                }
                            });
                        }
                    });
                }
            } else if (topic == "notification/node/disabled") {
                if (msg.types) {
                    RED.nodes.disableNodeSet(msg.id);
                    typeList = "<ul><li>"+msg.types.map(RED.utils.sanitize).join("</li><li>")+"</li></ul>";
                    RED.notify(RED._("palette.event.nodeDisabled", {count:msg.types.length})+typeList,"success");
                }
            } else if (topic == "notification/node/upgraded") {
                RED.notify(RED._("palette.event.nodeUpgraded", {module:msg.module,version:msg.version}),"success");
                RED.nodes.registry.setModulePendingUpdated(msg.module,msg.version);
            }
        });
        RED.comms.subscribe("event-log/#", function(topic,payload) {
            var id = topic.substring(9);
            RED.eventLog.log(id,payload);
        });

        $(".red-ui-header-toolbar").show();

        RED.sidebar.show(":first", true);

        setTimeout(function() {
            loader.end();
            checkTelemetry(function () {
                checkFirstRun(function() {
                    if (showProjectWelcome) {
                        RED.projects.showStartup();
                    }
                });
            })
        },100);
    }

    function checkTelemetry(done) {
        const telemetrySettings = RED.settings.telemetryEnabled;
        // Can only get telemetry permission from a user with permission to modify settings
        if (RED.user.hasPermission("settings.write") && telemetrySettings === undefined) {
            
            const dialog = RED.popover.dialog({
                title: RED._("telemetry.settingsTitle"),
                content: `${RED._("telemetry.settingsDescription")}${RED._("telemetry.settingsDescription2")}`,
                closeButton: false,
                buttons: [
                    {
                        text: RED._("telemetry.enableLabel"),
                        click: () => {
                            RED.settings.set("telemetryEnabled", true)
                            dialog.close()
                            done()
                        }
                    },
                    {
                        text: RED._("telemetry.disableLabel"),
                        click: () => {
                            RED.settings.set("telemetryEnabled", false)
                            dialog.close()
                            done()
                        }
                    }
                ]
            })
        } else {
            done()
        }
    }
    function checkFirstRun(done) {
        if (RED.settings.theme("tours") === false) {
            done();
            return;
        }
        if (!RED.settings.get("editor.view.view-show-welcome-tours", true)) {
            done();
            return;
        }
        RED.actions.invoke("core:show-welcome-tour", RED.settings.get("editor.tours.welcome"), done);
    }

    function buildMainMenu() {
        var menuOptions = [];
        if (RED.settings.theme("projects.enabled",false)) {
            menuOptions.push({id:"menu-item-projects-menu",label:RED._("menu.label.projects"),options:[
                {id:"menu-item-projects-new",label:RED._("menu.label.projects-new"),disabled:false,onselect:"core:new-project"},
                {id:"menu-item-projects-open",label:RED._("menu.label.projects-open"),disabled:false,onselect:"core:open-project"},
                {id:"menu-item-projects-settings",label:RED._("menu.label.projects-settings"),disabled:false,onselect:"core:show-project-settings"}
            ]});
        }
        menuOptions.push({id:"menu-item-edit-menu", label:RED._("menu.label.edit"), options: [
            {id: "menu-item-edit-undo", label:RED._("keyboard.undoChange"), disabled: true, onselect: "core:undo"},
            {id: "menu-item-edit-redo", label:RED._("keyboard.redoChange"), disabled: true, onselect: "core:redo"},
            null,
            {id: "menu-item-edit-cut", label:RED._("keyboard.cutNode"), onselect: "core:cut-selection-to-internal-clipboard"},
            {id: "menu-item-edit-copy", label:RED._("keyboard.copyNode"), onselect: "core:copy-selection-to-internal-clipboard"},
            {id: "menu-item-edit-paste", label:RED._("keyboard.pasteNode"), disabled: true, onselect: "core:paste-from-internal-clipboard"},
            null,
            {id: "menu-item-edit-copy-group-style", label:RED._("keyboard.copyGroupStyle"), onselect: "core:copy-group-style"},
            {id: "menu-item-edit-paste-group-style", label:RED._("keyboard.pasteGroupStyle"), disabled: true, onselect: "core:paste-group-style"},
            null,
            {id: "menu-item-edit-select-all", label:RED._("keyboard.selectAll"), onselect: "core:select-all-nodes"},
            {id: "menu-item-edit-select-connected", label:RED._("keyboard.selectAllConnected"), onselect: "core:select-connected-nodes"},
            {id: "menu-item-edit-select-none", label:RED._("keyboard.selectNone"), onselect: "core:select-none"},
            null,
            {id: "menu-item-edit-split-wire-with-links", label:RED._("keyboard.splitWireWithLinks"), onselect: "core:split-wire-with-link-nodes"},

        ]});

        menuOptions.push({id:"menu-item-view-menu",label:RED._("menu.label.view.view"),options:[
            {id:"menu-item-palette",label:RED._("menu.label.palette.show"),toggle:true,onselect:"core:toggle-palette", selected: true},
            {id:"menu-item-sidebar",label:RED._("menu.label.sidebar.show"),toggle:true,onselect:"core:toggle-sidebar", selected: true},
            {id:"menu-item-event-log",label:RED._("eventLog.title"),onselect:"core:show-event-log"},
            {id:"menu-item-action-list",label:RED._("keyboard.actionList"),onselect:"core:show-action-list"},
            null
        ]});

        menuOptions.push({id:"menu-item-arrange-menu", label:RED._("menu.label.arrange"), options: [
            {id: "menu-item-view-tools-align-left", label:RED._("menu.label.alignLeft"), disabled: true, onselect: "core:align-selection-to-left"},
            {id: "menu-item-view-tools-align-center", label:RED._("menu.label.alignCenter"), disabled: true, onselect: "core:align-selection-to-center"},
            {id: "menu-item-view-tools-align-right", label:RED._("menu.label.alignRight"), disabled: true, onselect: "core:align-selection-to-right"},
            null,
            {id: "menu-item-view-tools-align-top", label:RED._("menu.label.alignTop"), disabled: true, onselect: "core:align-selection-to-top"},
            {id: "menu-item-view-tools-align-middle", label:RED._("menu.label.alignMiddle"), disabled: true, onselect: "core:align-selection-to-middle"},
            {id: "menu-item-view-tools-align-bottom", label:RED._("menu.label.alignBottom"), disabled: true, onselect: "core:align-selection-to-bottom"},
            null,
            {id: "menu-item-view-tools-distribute-horizontally", label:RED._("menu.label.distributeHorizontally"), disabled: true, onselect: "core:distribute-selection-horizontally"},
            {id: "menu-item-view-tools-distribute-veritcally", label:RED._("menu.label.distributeVertically"), disabled: true, onselect: "core:distribute-selection-vertically"},
            null,
            {id: "menu-item-view-tools-move-to-back", label:RED._("menu.label.moveToBack"), disabled: true, onselect: "core:move-selection-to-back"},
            {id: "menu-item-view-tools-move-to-front", label:RED._("menu.label.moveToFront"), disabled: true, onselect: "core:move-selection-to-front"},
            {id: "menu-item-view-tools-move-backwards", label:RED._("menu.label.moveBackwards"), disabled: true, onselect: "core:move-selection-backwards"},
            {id: "menu-item-view-tools-move-forwards", label:RED._("menu.label.moveForwards"), disabled: true, onselect: "core:move-selection-forwards"}
        ]});

        menuOptions.push(null);
        if (RED.settings.theme("menu.menu-item-import-library", true)) {
            menuOptions.push({id: "menu-item-import", label: RED._("menu.label.import"), onselect: "core:show-import-dialog"});
        }
        if (RED.settings.theme("menu.menu-item-export-library", true)) {
            menuOptions.push({id: "menu-item-export", label: RED._("menu.label.export"), onselect: "core:show-export-dialog"});
        }
        menuOptions.push(null);
        menuOptions.push({id:"menu-item-search",label:RED._("menu.label.search"),onselect:"core:search"});
        menuOptions.push(null);
        menuOptions.push({id:"menu-item-config-nodes",label:RED._("menu.label.displayConfig"),onselect:"core:show-config-tab"});
        menuOptions.push({id:"menu-item-workspace",label:RED._("menu.label.flows"),options:[
            {id:"menu-item-workspace-add",label:RED._("menu.label.add"),onselect:"core:add-flow"},
            {id:"menu-item-workspace-edit",label:RED._("menu.label.edit"),onselect:"core:edit-flow"},
            {id:"menu-item-workspace-delete",label:RED._("menu.label.delete"),onselect:"core:remove-flow"}
        ]});
        menuOptions.push({id:"menu-item-subflow",label:RED._("menu.label.subflows"), options: [
            {id:"menu-item-subflow-create",label:RED._("menu.label.createSubflow"),onselect:"core:create-subflow"},
            {id:"menu-item-subflow-convert",label:RED._("menu.label.selectionToSubflow"),disabled:true,onselect:"core:convert-to-subflow"},
        ]});
        menuOptions.push({id:"menu-item-group",label:RED._("menu.label.groups"), options: [
            {id:"menu-item-group-group",label:RED._("menu.label.groupSelection"),disabled:true,onselect:"core:group-selection"},
            {id:"menu-item-group-ungroup",label:RED._("menu.label.ungroupSelection"),disabled:true,onselect:"core:ungroup-selection"},
            null,
            {id:"menu-item-group-merge",label:RED._("menu.label.groupMergeSelection"),disabled:true,onselect:"core:merge-selection-to-group"},
            {id:"menu-item-group-remove",label:RED._("menu.label.groupRemoveSelection"),disabled:true,onselect:"core:remove-selection-from-group"}
        ]});

        menuOptions.push(null);
        if (RED.settings.get('externalModules.palette.allowInstall', true) !== false) {
            menuOptions.push({id:"menu-item-edit-palette",label:RED._("menu.label.editPalette"),onselect:"core:manage-palette"});
            menuOptions.push(null);
        }

        menuOptions.push({id:"menu-item-user-settings",label:RED._("menu.label.settings"),onselect:"core:show-user-settings"});
        menuOptions.push(null);

        if (RED.settings.theme("menu.menu-item-keyboard-shortcuts", true)) {
            menuOptions.push({id: "menu-item-keyboard-shortcuts", label: RED._("menu.label.keyboardShortcuts"), onselect: "core:show-help"});
        }
        menuOptions.push({id:"menu-item-help",
            label: RED.settings.theme("menu.menu-item-help.label",RED._("menu.label.help")),
            href: RED.settings.theme("menu.menu-item-help.url","https://nodered.org/docs")
        });
        menuOptions.push({id:"menu-item-node-red-version", label:"v"+RED.settings.version, onselect: "core:show-about" });


        $('<li><a id="red-ui-header-button-sidemenu" class="button" href="#"><i class="fa fa-bars"></i></a></li>').appendTo(".red-ui-header-toolbar")
        RED.menu.init({id:"red-ui-header-button-sidemenu",options: menuOptions});

    }

    function loadEditor() {
        RED.workspaces.init();
        RED.statusBar.init();
        RED.view.init();
        RED.userSettings.init();
        RED.user.init();
        RED.notifications.init();
        RED.library.init();
        RED.palette.init();
        RED.eventLog.init();

        if (RED.settings.get('externalModules.palette.allowInstall', true) !== false) {
            RED.palette.editor.init();
        } else {
            console.log("Palette editor disabled");
        }

        RED.sidebar.init();

        if (RED.settings.theme("projects.enabled",false)) {
            RED.projects.init();
        } else {
            console.log("Projects disabled");
        }

        RED.subflow.init();
        RED.group.init();
        RED.clipboard.init();
        RED.search.init();
        RED.actionList.init();
        RED.editor.init();
        RED.diagnostics.init();
        RED.diff.init();


        RED.deploy.init(RED.settings.theme("deployButton",null));

        RED.keyboard.init(buildMainMenu);
        RED.envVar.init();

        RED.nodes.init();
        RED.runtime.init()

        if (RED.settings.theme("multiplayer.enabled",false)) {
            RED.multiplayer.init()
        }
        RED.comms.connect();

        $("#red-ui-main-container").show();

        loadPluginList();
    }


    function buildEditor(options) {
        var header = $('<div id="red-ui-header"></div>').appendTo(options.target);
        var logo = $('<span class="red-ui-header-logo"></span>').appendTo(header);
        $('<ul class="red-ui-header-toolbar hide"></ul>').appendTo(header);
        $('<div id="red-ui-header-shade" class="hide"></div>').appendTo(header);
        $('<div id="red-ui-main-container" class="red-ui-sidebar-closed hide">'+
            '<div id="red-ui-workspace"></div>'+
            '<div id="red-ui-editor-stack" tabindex="-1"></div>'+
            '<div id="red-ui-palette"></div>'+
            '<div id="red-ui-sidebar"></div>'+
            '<div id="red-ui-sidebar-separator"></div>'+
        '</div>').appendTo(options.target);
        $('<div id="red-ui-editor-plugin-configs"></div>').appendTo(options.target);
        $('<div id="red-ui-editor-node-configs"></div>').appendTo(options.target);
        $('<div id="red-ui-full-shade" class="hide"></div>').appendTo(options.target);

        loader.init().appendTo("#red-ui-main-container");
        loader.start("...",0);

        $.getJSON(options.apiRootUrl+"theme", function(theme) {
            if (theme.header) {
                if (theme.header.url) {
                    logo = $("<a>",{href:theme.header.url}).appendTo(logo);
                }
                if (theme.header.image) {
                    $('<img>',{src:theme.header.image}).appendTo(logo);
                }
                if (theme.header.title) {
                    $('<span>').html(theme.header.title).appendTo(logo);
                }
            }
            if (theme.themes) {
                knownThemes = theme.themes;
            }
        });
    }
    var knownThemes = null;
    var initialised = false;

    function init(options) {
        if (initialised) {
            throw new Error("RED already initialised");
        }
        initialised = true;
        if(window.ace) { window.ace.require("ace/ext/language_tools"); }
        options = options || {};
        options.apiRootUrl = options.apiRootUrl || "";
        if (options.apiRootUrl && !/\/$/.test(options.apiRootUrl)) {
            options.apiRootUrl = options.apiRootUrl+"/";
        }
        options.target = $("#red-ui-editor");
        options.target.addClass("red-ui-editor");

        buildEditor(options);

        RED.i18n.init(options, function() {
            RED.settings.init(options, function() {
                if (knownThemes) {
                    RED.settings.editorTheme = RED.settings.editorTheme || {};
                    RED.settings.editorTheme.themes = knownThemes;
                }
                loadEditor();
            });
        })
    }

    var loader = {
        init: function() {
            var wrapper = $('<div id="red-ui-loading-progress"></div>').hide();
            var container = $('<div>').appendTo(wrapper);
            var label = $('<div>',{class:"red-ui-loading-bar-label"}).appendTo(container);
            var bar = $('<div>',{class:"red-ui-loading-bar"}).appendTo(container);
            var fill =$('<span>').appendTo(bar);
            return wrapper;
        },
        start: function(text, prcnt) {
            if (text) {
                loader.reportProgress(text,prcnt)
            }
            $("#red-ui-loading-progress").show();
        },
        reportProgress: function(text, prcnt) {
            $(".red-ui-loading-bar-label").text(text);
            $(".red-ui-loading-bar span").width(prcnt+"%")
        },
        end: function() {
            $("#red-ui-loading-progress").hide();
            loader.reportProgress("",0);
        }
    }

    return {
        init: init,
        loader: loader
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

 RED.events = (function() {
     var handlers = {};

     function on(evt,func) {
         handlers[evt] = handlers[evt]||[];
         handlers[evt].push(func);
     }
     function off(evt,func) {
         var handler = handlers[evt];
         if (handler) {
             for (var i=0;i<handler.length;i++) {
                 if (handler[i] === func) {
                     handler.splice(i,1);
                     return;
                 }
             }
         }
     }
     function emit() {
         var evt = arguments[0]
         var args = Array.prototype.slice.call(arguments,1);
         if (RED.events.DEBUG) {
             console.warn(evt,args);
         }
         if (handlers[evt]) {
             let cpyHandlers = [...handlers[evt]];
          
             for (var i=0;i<cpyHandlers.length;i++) {
                 try {
                     cpyHandlers[i].apply(null, args);
                 } catch(err) {
                     console.warn("RED.events.emit error: ["+evt+"] "+(err.toString()));
                     console.warn(err);
                 }
             }
         }
     }
     return {
         on: on,
         off: off,
         emit: emit
     }
 })();
;RED.hooks = (function() {

    var VALID_HOOKS = [

    ]

    var hooks = { }
    var labelledHooks = { }

    function add(hookId, callback) {
        var parts = hookId.split(".");
        var id = parts[0], label = parts[1];

        // if (VALID_HOOKS.indexOf(id) === -1) {
        //     throw new Error("Invalid hook '"+id+"'");
        // }
        if (label && labelledHooks[label] && labelledHooks[label][id]) {
            throw new Error("Hook "+hookId+" already registered")
        }
        var hookItem = {cb:callback, previousHook: null, nextHook: null }

        var tailItem = hooks[id];
        if (tailItem === undefined) {
            hooks[id] = hookItem;
        } else {
            while(tailItem.nextHook !== null) {
                tailItem = tailItem.nextHook
            }
            tailItem.nextHook = hookItem;
            hookItem.previousHook = tailItem;
        }

        if (label) {
            labelledHooks[label] = labelledHooks[label]||{};
            labelledHooks[label][id] = hookItem;
        }
    }
    function remove(hookId) {
        var parts = hookId.split(".");
        var id = parts[0], label = parts[1];
        if ( !label) {
            throw new Error("Cannot remove hook without label: "+hookId)
        }
        if (labelledHooks[label]) {
            if (id === "*") {
                // Remove all hooks for this label
                var hookList = Object.keys(labelledHooks[label]);
                for (var i=0;i<hookList.length;i++) {
                    removeHook(hookList[i],labelledHooks[label][hookList[i]])
                }
                delete labelledHooks[label];
            } else if (labelledHooks[label][id]) {
                removeHook(id,labelledHooks[label][id])
                delete labelledHooks[label][id];
                if (Object.keys(labelledHooks[label]).length === 0){
                    delete labelledHooks[label];
                }
            }
        }
    }

    function removeHook(id,hookItem) {
        var previousHook = hookItem.previousHook;
        var nextHook = hookItem.nextHook;

        if (previousHook) {
            previousHook.nextHook = nextHook;
        } else {
            hooks[id] = nextHook;
        }
        if (nextHook) {
            nextHook.previousHook = previousHook;
        }
        hookItem.removed = true;
        if (!previousHook && !nextHook) {
            delete hooks[id];
        }
    }

    function trigger(hookId, payload, done) {
        var hookItem = hooks[hookId];
        if (!hookItem) {
            if (done) {
                done();
            }
            return;
        }
        function callNextHook(err) {
            if (!hookItem || err) {
                if (done) { done(err) }
                return err;
            }
            if (hookItem.removed) {
                hookItem = hookItem.nextHook;
                return callNextHook();
            }
            var callback = hookItem.cb;
            if (callback.length === 1) {
                try {
                    let result = callback(payload);
                    if (result === false) {
                        // Halting the flow
                        if (done) { done(false) }
                        return result;
                    }
                    hookItem = hookItem.nextHook;
                    return callNextHook();
                } catch(e) {
                    console.warn(e);
                    if (done) { done(e);}
                    return e;
                }
            } else {
                // There is a done callback
                try {
                    callback(payload,function(result) {
                        if (result === undefined) {
                            hookItem = hookItem.nextHook;
                            callNextHook();
                        } else {
                            if (done) { done(result)}
                        }
                    })
                } catch(e) {
                    console.warn(e);
                    if (done) { done(e) }
                    return e;
                }
            }
        }

        return callNextHook();
    }

    function clear() {
        hooks = {}
        labelledHooks = {}
    }

    function has(hookId) {
        var parts = hookId.split(".");
        var id = parts[0], label = parts[1];
        if (label) {
            return !!(labelledHooks[label] && labelledHooks[label][id])
        }
        return !!hooks[id]
    }

    return {
        has: has,
        clear: clear,
        add: add,
        remove: remove,
        trigger: trigger
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

RED.i18n = (function() {

    var apiRootUrl;

    function detectLanguage() {
        return navigator.language
    }

    return {
        init: function(options, done) {
            apiRootUrl = options.apiRootUrl||"";
            var preferredLanguage = localStorage.getItem("editor-language") || detectLanguage();
            var opts = {
                backend: {
                    loadPath: apiRootUrl+'locales/__ns__?lng=__lng__',
                },
                lng: 'en-US',
                // debug: true,
                preload:['en-US'],
                ns: ["editor","node-red","jsonata","infotips"],
                defaultNS: "editor",
                fallbackLng: ['en-US'],
                returnObjects: true,
                keySeparator: ".",
                nsSeparator: ":",
                interpolation: {
                    unescapeSuffix: 'HTML',
                    escapeValue: false,
                    prefix: '__',
                    suffix: '__'
                }
            };
            if (preferredLanguage) {
                opts.lng = preferredLanguage;
            }

            i18next.use(i18nextHttpBackend).init(opts,function() {
                done();
            });
            jqueryI18next.init(i18next, $, { handleName: 'i18n' });


            RED["_"] = function() {
                var v = i18next.t.apply(i18next,arguments);
                if (typeof v === 'string') {
                    return v;
                } else {
                    return arguments[0];
                }
            }
        },
        lang: function() {
            // Gets the active message catalog language. This is based on what
            // locale the editor is using and what languages are available.
            //
            var preferredLangs = [localStorage.getItem("editor-language")|| detectLanguage()].concat(i18next.languages);
            var knownLangs = RED.settings.theme("languages")||["en-US"];
            for (var i=0;i<preferredLangs.length;i++) {
                if (knownLangs.indexOf(preferredLangs[i]) > -1) {
                    return preferredLangs[i]
                }
            }
            return 'en-US'
        },
        loadNodeCatalog: function(namespace,done) {
            var languageList = [localStorage.getItem("editor-language")|| detectLanguage()].concat(i18next.languages);
            var toLoad = languageList.length;
            languageList.forEach(function(lang) {
                $.ajax({
                    headers: {
                        "Accept":"application/json"
                    },
                    cache: false,
                    url: apiRootUrl+'nodes/'+namespace+'/messages?lng='+lang,
                    success: function(data) {
                        i18next.addResourceBundle(lang,namespace,data);
                        toLoad--;
                        if (toLoad === 0) {
                            done();
                        }
                    }
                });
            })

        },

        loadNodeCatalogs: function(done) {
            var languageList = [localStorage.getItem("editor-language")|| detectLanguage()].concat(i18next.languages);
            var toLoad = languageList.length;

            languageList.forEach(function(lang) {
                $.ajax({
                    headers: {
                        "Accept":"application/json"
                    },
                    cache: false,
                    url: apiRootUrl+'nodes/messages?lng='+lang,
                    success: function(data) {
                        var namespaces = Object.keys(data);
                        namespaces.forEach(function(ns) {
                            i18next.addResourceBundle(lang,ns,data[ns]);
                        });
                        toLoad--;
                        if (toLoad === 0) {
                            done();
                        }
                    }
                });
            })
        },

        loadPluginCatalogs: function(done) {
            var languageList = [localStorage.getItem("editor-language")|| detectLanguage()].concat(i18next.languages);
            var toLoad = languageList.length;

            languageList.forEach(function(lang) {
                $.ajax({
                    headers: {
                        "Accept":"application/json"
                    },
                    cache: false,
                    url: apiRootUrl+'plugins/messages?lng='+lang,
                    success: function(data) {
                        var namespaces = Object.keys(data);
                        namespaces.forEach(function(ns) {
                            i18next.addResourceBundle(lang,ns,data[ns]);
                        });
                        toLoad--;
                        if (toLoad === 0) {
                            done();
                        }
                    }
                });
            })
        },
        detectLanguage: detectLanguage
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/


RED.settings = (function () {

    var loadedSettings = {};
    var userSettings = {};
    var pendingSave;

    var hasLocalStorage = function () {
        try {
            return 'localStorage' in window && window['localStorage'] !== null;
        } catch (e) {
            return false;
        }
    };

    var set = function (key, value) {
        if (!hasLocalStorage()) {
            return;
        }
        if (key.startsWith("auth-tokens")) {
            localStorage.setItem(key+this.authTokensSuffix, JSON.stringify(value));
        } else {
            RED.utils.setMessageProperty(userSettings,key,value);
            saveUserSettings();
        }
    };

    /**
     * If the key is not set in the localStorage it returns <i>undefined</i>
     * Else return the JSON parsed value
     * @param key
     * @param defaultIfUndefined
     * @returns {*}
     */
    var get = function (key,defaultIfUndefined) {
        if (!hasLocalStorage()) {
            return undefined;
        }
        if (key.startsWith("auth-tokens")) {
            return JSON.parse(localStorage.getItem(key+this.authTokensSuffix));
        } else {
            var v;
            try { v = RED.utils.getMessageProperty(userSettings,key); } catch(err) {}
            if (v === undefined) {
                try { v = RED.utils.getMessageProperty(RED.settings,key); } catch(err) {}
            }
            if (v === undefined) {
                v = defaultIfUndefined;
            }
            return v;
        }
    };

    var remove = function (key) {
        if (!hasLocalStorage()) {
            return;
        }
        if (key.startsWith("auth-tokens")) {
            localStorage.removeItem(key+this.authTokensSuffix);
        } else {
            delete userSettings[key];
            saveUserSettings();
        }
    };

    var setProperties = function(data) {
        for (var prop in loadedSettings) {
            if (loadedSettings.hasOwnProperty(prop) && RED.settings.hasOwnProperty(prop)) {
                delete RED.settings[prop];
            }
        }
        for (prop in data) {
            if (data.hasOwnProperty(prop)) {
                RED.settings[prop] = data[prop];
            }
        }
        loadedSettings = data;
    };

    var setUserSettings = function(data) {
        userSettings = data;
    }

    var init = function (options, done) {
        var accessTokenMatch = /[?&]access_token=(.*?)(?:$|&)/.exec(window.location.search);
        var path=window.location.pathname.slice(0,-1);
        RED.settings.authTokensSuffix=path.replace(/\//g, '-');
        if (accessTokenMatch) {
            var accessToken = accessTokenMatch[1];
            RED.settings.set("auth-tokens",{access_token: accessToken});
            window.location.search = "";
        }
        RED.settings.apiRootUrl = options.apiRootUrl;

        $.ajaxSetup({
            beforeSend: function(jqXHR,settings) {
                // Only attach auth header for requests to relative paths
                if (!/^\s*(https?:|\/|\.)/.test(settings.url)) {
                    if (options.apiRootUrl) {
                        settings.url = options.apiRootUrl+settings.url;
                    }
                    var auth_tokens = RED.settings.get("auth-tokens");
                    if (auth_tokens) {
                        jqXHR.setRequestHeader("Authorization","Bearer "+auth_tokens.access_token);
                    }
                    jqXHR.setRequestHeader("Node-RED-API-Version","v2");
                }
            }
        });

        load(done);
    }

    var refreshSettings = function(done) {
        $.ajax({
            headers: {
                "Accept": "application/json"
            },
            dataType: "json",
            cache: false,
            url: 'settings',
            success: function (data) {
                setProperties(data);
                done(null, data);
            },
            error: function(jqXHR,textStatus,errorThrown) {
                if (jqXHR.status === 401) {
                    if (/[?&]access_token=(.*?)(?:$|&)/.test(window.location.search)) {
                        window.location.search = "";
                    }
                    RED.user.login(function() { refreshSettings(done); });
                } else {
                    console.log("Unexpected error loading settings:",jqXHR.status,textStatus);
                }
            }
        });
    }
    var load = function(done) {
        refreshSettings(function(err, data) {
            if (!err) {
                if (!RED.settings.user || RED.settings.user.anonymous) {
                    RED.settings.remove("auth-tokens");
                }
                console.log("Node-RED: " + data.version);
                console.groupCollapsed("Versions");
                console.log("jQuery",$().jquery)
                console.log("jQuery UI",$.ui.version);
                if(window.ace) { console.log("ACE",ace.version); }
                if(window.monaco) { console.log("MONACO",monaco.version || "unknown"); }
                console.log("D3",d3.version);
                console.groupEnd();
                loadUserSettings(done);
            }
        })
    };

    function loadUserSettings(done) {
        $.ajax({
            headers: {
                "Accept": "application/json"
            },
            dataType: "json",
            cache: false,
            url: 'settings/user',
            success: function (data) {
                setUserSettings(data);
                done();
            },
            error: function(jqXHR,textStatus,errorThrown) {
                console.log("Unexpected error loading user settings:",jqXHR.status,textStatus);
            }
        });
    }

    function saveUserSettings() {
        if (RED.user.hasPermission("settings.write")) {
            if (pendingSave) {
                clearTimeout(pendingSave);
            }
            pendingSave = setTimeout(function() {
                pendingSave = null;
                $.ajax({
                    method: 'POST',
                    contentType: 'application/json',
                    url: 'settings/user',
                    data: JSON.stringify(userSettings),
                    success: function (data) {
                    },
                    error: function(jqXHR,textStatus,errorThrown) {
                        console.log("Unexpected error saving user settings:",jqXHR.status,textStatus);
                    }
                });
            },300);
        }
    }

    function theme(property,defaultValue) {
        if (!RED.settings.editorTheme) {
            return defaultValue;
        }
        var parts = property.split(".");
        var v = RED.settings.editorTheme;
        try {
            for (var i=0;i<parts.length;i++) {
                v = v[parts[i]];
            }
            if (v === undefined) {
                return defaultValue;
            }
            return v;
        } catch(err) {
            return defaultValue;
        }
    }
    function getLocal(key) {
        return localStorage.getItem(key)
    }
    function setLocal(key, value) {
        localStorage.setItem(key, value);
    }
    function removeLocal(key) {
        localStorage.removeItem(key)
    }


    return {
        init: init,
        load: load,
        loadUserSettings: loadUserSettings,
        refreshSettings: refreshSettings,
        set: set,
        get: get,
        remove: remove,
        theme: theme,
        setLocal: setLocal,
        getLocal: getLocal,
        removeLocal: removeLocal
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
RED.user = (function() {

    function login(opts,done) {
        if (typeof opts == 'function') {
            done = opts;
            opts = {};
        }

        var dialog = $('<div id="node-dialog-login" class="hide" style="display: flex; align-items: flex-end;">'+
                       '<div style="width: 250px; flex-grow: 0;"><img id="node-dialog-login-image" src=""/></div>'+
                       '<div style="flex-grow: 1;">'+
                            '<form id="node-dialog-login-fields" class="form-horizontal" style="margin-bottom: 0px; margin-left:20px;"></form>'+
                       '</div>'+
                       '</div>');

        dialog.dialog({
            autoOpen: false,
            classes: {
                "ui-dialog": "red-ui-editor-dialog",
                "ui-dialog-titlebar-close": "hide",
                "ui-widget-overlay": "red-ui-editor-dialog"
            },
            modal: true,
            closeOnEscape: !!opts.cancelable,
            width: 600,
            resizable: false,
            draggable: false,
            close: function( event, ui ) {
                $("#node-dialog-login").dialog('destroy').remove();
                RED.keyboard.enable()
            }
        });

        $("#node-dialog-login-fields").empty();
        $.ajax({
            dataType: "json",
            url: "auth/login",
            success: function(data) {
                var i=0;

                if (data.type == "credentials") {

                    for (;i<data.prompts.length;i++) {
                        var field = data.prompts[i];
                        var row = $("<div/>",{class:"form-row"});
                        $('<label for="node-dialog-login-'+field.id+'">'+RED._(field.label)+':</label><br/>').appendTo(row);
                        var input = $('<input style="width: 100%" id="node-dialog-login-'+field.id+'" type="'+field.type+'" tabIndex="'+(i+1)+'"/>').appendTo(row);

                        if (i<data.prompts.length-1) {
                            input.keypress(
                                (function() {
                                    var r = row;
                                    return function(event) {
                                        if (event.keyCode == 13) {
                                            r.next("div").find("input").trigger("focus");
                                            event.preventDefault();
                                        }
                                    }
                                })()
                            );
                        }
                        row.appendTo("#node-dialog-login-fields");
                    }
                    $('<div class="form-row" style="text-align: right; margin-top: 10px;"><span id="node-dialog-login-failed" style="line-height: 2em;float:left;color:var(--red-ui-text-color-error);" class="hide">'+RED._("user.loginFailed")+'</span><img src="red/images/spin.svg" style="height: 30px; margin-right: 10px; " class="login-spinner hide"/>'+
                        (opts.cancelable?'<a href="#" id="node-dialog-login-cancel" class="red-ui-button" style="margin-right: 20px;" tabIndex="'+(i+1)+'">'+RED._("common.label.cancel")+'</a>':'')+
                        '<input type="submit" id="node-dialog-login-submit" class="red-ui-button" style="width: auto;" tabIndex="'+(i+2)+'" value="'+RED._("user.login")+'"></div>').appendTo("#node-dialog-login-fields");


                    $("#node-dialog-login-submit").button();
                    $("#node-dialog-login-fields").on("submit", function(event) {
                        $("#node-dialog-login-submit").button("option","disabled",true);
                        $("#node-dialog-login-failed").hide();
                        $(".login-spinner").show();

                        var body = {
                            client_id: "node-red-editor",
                            grant_type: "password",
                            scope:""
                        }
                        for (var i=0;i<data.prompts.length;i++) {
                            var field = data.prompts[i];
                            body[field.id] = $("#node-dialog-login-"+field.id).val();
                        }
                        $.ajax({
                            url:"auth/token",
                            type: "POST",
                            data: body
                        }).done(function(data,textStatus,xhr) {
                            RED.settings.set("auth-tokens",data);
                            if (opts.updateMenu) {
                                updateUserMenu();
                            }
                            $("#node-dialog-login").dialog("close");
                            done();
                        }).fail(function(jqXHR,textStatus,errorThrown) {
                            RED.settings.remove("auth-tokens");
                            $("#node-dialog-login-failed").show();
                        }).always(function() {
                            $("#node-dialog-login-submit").button("option","disabled",false);
                            $(".login-spinner").hide();
                        });
                        event.preventDefault();
                    });

                } else if (data.type == "strategy") {
                    var sessionMessage = /[?&]session_message=(.*?)(?:$|&)/.exec(window.location.search);
                    RED.sessionMessages = RED.sessionMessages || [];
                    if (sessionMessage) {
                        RED.sessionMessages.push(decodeURIComponent(sessionMessage[1]));
                        if (history.pushState) {
                            var newurl = window.location.protocol+"//"+window.location.host+window.location.pathname
                            window.history.replaceState({ path: newurl }, "", newurl);
                        } else {
                            window.location.search = "";
                        }
                    }

                    if (RED.sessionMessages.length === 0 && data.autoLogin) {
                        document.location = data.loginRedirect
                        return
                    }

                    i = 0;
                    for (;i<data.prompts.length;i++) {
                        var field = data.prompts[i];
                        if (RED.sessionMessages) {
                            var sessionMessages = $("<div/>",{class:"form-row",style:"text-align: center"}).appendTo("#node-dialog-login-fields");
                            RED.sessionMessages.forEach(function (msg) {
                                $('<div>').css("color","var(--red-ui-text-color-error)").text(msg).appendTo(sessionMessages);
                            });
                            delete RED.sessionMessages;
                        }
                        var row = $("<div/>",{class:"form-row",style:"text-align: center"}).appendTo("#node-dialog-login-fields");

                        var loginButton = $('<a href="#" class="red-ui-button"></a>',{style: "padding: 10px"}).appendTo(row).on("click", function() {
                            document.location = field.url;
                        });
                        if (field.image) {
                            $("<img>",{src:field.image}).appendTo(loginButton);
                        } else if (field.label) {
                            var label = $('<span></span>').text(field.label);
                            if (field.icon) {
                                $('<i></i>',{class: "fa fa-2x "+field.icon, style:"vertical-align: middle"}).appendTo(loginButton);
                                label.css({
                                    "verticalAlign":"middle",
                                    "marginLeft":"8px"
                                });

                            }
                            label.appendTo(loginButton);
                        }
                        loginButton.button();
                    }


                } else {
                    if (data.prompts) {
                        if (data.loginMessage) {
                            const sessionMessages = $("<div/>",{class:"form-row",style:"text-align: center"}).appendTo("#node-dialog-login-fields");
                            $('<div>').text(data.loginMessage).appendTo(sessionMessages);
                        }

                        i = 0;
                        for (;i<data.prompts.length;i++) {
                            var field = data.prompts[i];
                            var row = $("<div/>",{class:"form-row",style:"text-align: center"}).appendTo("#node-dialog-login-fields");
                            var loginButton = $('<a href="#" class="red-ui-button"></a>',{style: "padding: 10px"}).appendTo(row).on("click", function() {
                                document.location = field.url;
                            });
                            if (field.image) {
                                $("<img>",{src:field.image}).appendTo(loginButton);
                            } else if (field.label) {
                                var label = $('<span></span>').text(field.label);
                                if (field.icon) {
                                    $('<i></i>',{class: "fa fa-2x "+field.icon, style:"vertical-align: middle"}).appendTo(loginButton);
                                    label.css({
                                        "verticalAlign":"middle",
                                        "marginLeft":"8px"
                                    });

                                }
                                label.appendTo(loginButton);
                            }
                            loginButton.button();
                        }
                    }
                }
                if (opts.cancelable) {
                    $("#node-dialog-login-cancel").button().on("click", function( event ) {
                        $("#node-dialog-login").dialog('close');

                    });
                }

                var loginImageSrc = data.image || "red/images/node-red-256.svg";

                $("#node-dialog-login-image").load(function() {
                    dialog.dialog("open");
                }).attr("src",loginImageSrc);
                RED.keyboard.disable();
            }
        });
    }

    function logout() {
        RED.events.emit('logout')
        var tokens = RED.settings.get("auth-tokens");
        var token = tokens?tokens.access_token:"";
        $.ajax({
            url: "auth/revoke",
            type: "POST",
            data: {token:token}
        }).done(function(data,textStatus,xhr) {
            RED.settings.remove("auth-tokens");
            if (data && data.redirect) {
                document.location.href = data.redirect;
            } else {
                document.location.reload(true);
            }
        }).fail(function(jqXHR,textStatus,errorThrown) {
            if (jqXHR.status === 401) {
                document.location.reload(true);
            } else {
                console.log(textStatus);
            }
        })
    }

    function updateUserMenu() {
        $("#red-ui-header-button-user-submenu li").remove();
        const userMenu = $("#red-ui-header-button-user")
        userMenu.empty()
        if (RED.settings.user.anonymous) {
            RED.menu.addItem("red-ui-header-button-user",{
                id:"usermenu-item-login",
                label:RED._("menu.label.login"),
                onselect: function() {
                    RED.user.login({cancelable:true},function() {
                        RED.settings.load(function() {
                            RED.notify(RED._("user.loggedInAs",{name:RED.settings.user.username}),"success");
                            updateUserMenu();
                            RED.events.emit("login",RED.settings.user.username);
                        });
                    });
                }
            });
        } else {
            RED.menu.addItem("red-ui-header-button-user",{
                id:"usermenu-item-username",
                label:"<b>"+RED.settings.user.username+"</b>"
            });
            RED.menu.addItem("red-ui-header-button-user",{
                id:"usermenu-item-logout",
                label:RED._("menu.label.logout"),
                onselect: function() {
                    RED.user.logout();
                }
            });
        }
        const userIcon = generateUserIcon(RED.settings.user)
        userIcon.appendTo(userMenu);
    }

    function init() {
        if (RED.settings.user) {
            if (!RED.settings.editorTheme || !RED.settings.editorTheme.hasOwnProperty("userMenu") || RED.settings.editorTheme.userMenu) {

                var userMenu = $('<li><a id="red-ui-header-button-user" class="button hide" href="#"></a></li>')
                    .prependTo(".red-ui-header-toolbar");
                RED.menu.init({id:"red-ui-header-button-user",
                    options: []
                });
                updateUserMenu();
            }
        }

    }

    var readRE = /^((.+)\.)?read$/
    var writeRE = /^((.+)\.)?write$/

    function hasPermission(permission) {
        if (permission === "") {
            return true;
        }
        if (!RED.settings.user) {
            return true;
        }
        return checkPermission(RED.settings.user.permissions||"",permission);
    }
    function checkPermission(userScope,permission) {
        if (permission === "") {
            return true;
        }
        var i;

        if (Array.isArray(permission)) {
            // Multiple permissions requested - check each one
            for (i=0;i<permission.length;i++) {
                if (!checkPermission(userScope,permission[i])) {
                    return false;
                }
            }
            // All permissions check out
            return true;
        }

        if (Array.isArray(userScope)) {
            if (userScope.length === 0) {
                return false;
            }
            for (i=0;i<userScope.length;i++) {
                if (checkPermission(userScope[i],permission)) {
                    return true;
                }
            }
            return false;
        }

        if (userScope === "*" || userScope === permission) {
            return true;
        }

        if (userScope === "read" || userScope === "*.read") {
            return readRE.test(permission);
        } else if (userScope === "write" || userScope === "*.write") {
            return writeRE.test(permission);
        }
        return false;
    }

    function generateUserIcon(user) {
        const userIcon = $('<span class="red-ui-user-profile"></span>')
        if (user.image) {
            userIcon.addClass('has_profile_image')
            userIcon.css({
                backgroundImage: "url("+user.image+")",
            })
        } else if (user.anonymous || (!user.username && !user.email)) {
            $('<i class="fa fa-user"></i>').appendTo(userIcon);
        } else {
            $('<span>').text((user.username || user.email).substring(0,2)).appendTo(userIcon);
        }
        if (user.profileColor !== undefined) {
            userIcon.addClass('red-ui-user-profile-color-' + user.profileColor)
        }
        return userIcon
    }

    return {
        init: init,
        login: login,
        logout: logout,
        hasPermission: hasPermission,
        generateUserIcon
    }

})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

RED.comms = (function() {

    var errornotification = null;
    var clearErrorTimer = null;
    var connectCountdownTimer = null;
    var connectCountdown = 10;
    var subscriptions = {};
    var ws;
    var pendingAuth = false;
    var reconnectAttempts = 0;
    var active = false;

    RED.events.on('login', function(username) {
        // User has logged in
        // Need to upgrade the connection to be authenticated
        if (ws && ws.readyState == 1) {
            const auth_tokens = RED.settings.get("auth-tokens");
            ws.send(JSON.stringify({auth:auth_tokens.access_token}))
        }
    })

    function connectWS() {
        active = true;
        var wspath;

        if (RED.settings.apiRootUrl) {
            var m = /^(https?):\/\/(.*)$/.exec(RED.settings.apiRootUrl);
            if (m) {
                console.log(m);
                wspath = "ws"+(m[1]==="https"?"s":"")+"://"+m[2]+"comms";
            }
        } else {
            var path = location.hostname;
            var port = location.port;
            if (port.length !== 0) {
                path = path+":"+port;
            }
            path = path+document.location.pathname;
            path = path+(path.slice(-1) == "/"?"":"/")+"comms";
            wspath = "ws"+(document.location.protocol=="https:"?"s":"")+"://"+path;
        }

        var auth_tokens = RED.settings.get("auth-tokens");
        pendingAuth = (auth_tokens!=null);

        function completeConnection() {
            for (var t in subscriptions) {
                if (subscriptions.hasOwnProperty(t)) {
                    ws.send(JSON.stringify({subscribe:t}));
                }
            }
            emit('connect')
        }

        ws = new WebSocket(wspath);
        ws.onopen = function() {
            reconnectAttempts = 0;
            if (errornotification) {
                clearErrorTimer = setTimeout(function() {
                    errornotification.close();
                    errornotification = null;
                },1000);
            }
            if (pendingAuth) {
                ws.send(JSON.stringify({auth:auth_tokens.access_token}));
            } else {
                completeConnection();
            }
        }
        ws.onmessage = function(event) {
            var message = JSON.parse(event.data);
            if (message.auth) {
                if (pendingAuth) {
                    if (message.auth === "ok") {
                        pendingAuth = false;
                        completeConnection();
                    } else if (message.auth === "fail") {
                        // anything else is an error...
                        active = false;
                        RED.user.login({updateMenu:true},function() {
                            connectWS();
                        })
                    }
                } else if (message.auth === "fail") {
                    // Our current session has expired
                    active = false;
                    RED.user.login({updateMenu:true},function() {
                        connectWS();
                    })
                }
            } else {
                // Otherwise, 'message' is an array of actual comms messages
                for (var m = 0; m < message.length; m++) {
                    var msg = message[m];
                    if (msg.topic) {
                        for (var t in subscriptions) {
                            if (subscriptions.hasOwnProperty(t)) {
                                var re = new RegExp("^"+t.replace(/([\[\]\?\(\)\\\\$\^\*\.|])/g,"\\$1").replace(/\+/g,"[^/]+").replace(/\/#$/,"(\/.*)?")+"$");
                                if (re.test(msg.topic)) {
                                    var subscribers = subscriptions[t];
                                    if (subscribers) {
                                        for (var i=0;i<subscribers.length;i++) {
                                            try {
                                                subscribers[i](msg.topic,msg.data);
                                            } catch (error) {
                                                // need to decide what to do with this uncaught error
                                                console.warn('Uncaught error from RED.comms.subscribe: ' + error.toString())
                                                console.warn(error)
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        };
        ws.onclose = function() {
            if (!active) {
                return;
            }
            if (clearErrorTimer) {
                clearTimeout(clearErrorTimer);
                clearErrorTimer = null;
            }
            reconnectAttempts++;
            if (reconnectAttempts < 10) {
                setTimeout(connectWS,1000);
                if (reconnectAttempts > 5 && errornotification == null) {
                    errornotification = RED.notify(RED._("notification.errors.lostConnection"),"error",true);
                }
            } else if (reconnectAttempts < 20) {
                setTimeout(connectWS,2000);
            } else {
                connectCountdown = 60;
                connectCountdownTimer = setInterval(function() {
                    connectCountdown--;
                    if (connectCountdown === 0) {
                        errornotification.update(RED._("notification.errors.lostConnection"));
                        clearInterval(connectCountdownTimer);
                        connectWS();
                    } else {
                        var msg = RED._("notification.errors.lostConnectionReconnect",{time: connectCountdown})+' <a href="#">'+ RED._("notification.errors.lostConnectionTry")+'</a>';
                        errornotification.update(msg,{silent:true});
                        $(errornotification).find("a").on("click", function(e) {
                            e.preventDefault();
                            errornotification.update(RED._("notification.errors.lostConnection"),{silent:true});
                            clearInterval(connectCountdownTimer);
                            connectWS();
                        })
                    }
                },1000);
            }

        }
    }

    function subscribe(topic,callback) {
        if (subscriptions[topic] == null) {
            subscriptions[topic] = [];
        }
        subscriptions[topic].push(callback);
        if (ws && ws.readyState == 1) {
            ws.send(JSON.stringify({subscribe:topic}));
        }
    }

    function unsubscribe(topic,callback) {
        if (subscriptions[topic]) {
            for (var i=0;i<subscriptions[topic].length;i++) {
                if (subscriptions[topic][i] === callback) {
                    subscriptions[topic].splice(i,1);
                    break;
                }
            }
            if (subscriptions[topic].length === 0) {
                delete subscriptions[topic];
            }
        }
    }

    function send(topic, msg) {
        if (ws && ws.readyState == 1) {
            ws.send(JSON.stringify({
                topic,
                data: msg
            }))
        }
    }

    const eventHandlers = {};
    function on(evt,func) {
        eventHandlers[evt] = eventHandlers[evt]||[];
        eventHandlers[evt].push(func);
    }
    function off(evt,func) {
        const handler = eventHandlers[evt];
        if (handler) {
            for (let i=0;i<handler.length;i++) {
                if (handler[i] === func) {
                    handler.splice(i,1);
                    return;
                }
            }
        }
    }
    function emit() {
        const evt = arguments[0]
        const args = Array.prototype.slice.call(arguments,1);
        if (eventHandlers[evt]) {
            let cpyHandlers = [...eventHandlers[evt]];
            for (let i=0;i<cpyHandlers.length;i++) {
                try {
                    cpyHandlers[i].apply(null, args);
                } catch(err) {
                    console.warn("RED.comms.emit error: ["+evt+"] "+(err.toString()));
                    console.warn(err);
                }
            }
        }
    }

    return {
        connect: connectWS,
        subscribe: subscribe,
        unsubscribe:unsubscribe,
        on,
        off,
        send
    }
})();
;RED.runtime = (function() {
    let state = ""
    let settings = { ui: false, enabled: false };
    const STOPPED = "stop"
    const STARTED = "start"
    const SAFE = "safe"

    return {
        init: function() {
            // refresh the current runtime status from server
            settings = Object.assign({}, settings, RED.settings.runtimeState);
            RED.events.on("runtime-state", function(msg) {
                if (msg.state) {
                    const currentState = state
                    state = msg.state
                    $(".red-ui-flow-node-button").toggleClass("red-ui-flow-node-button-stopped", state !== STARTED)
                    if(settings.enabled === true && settings.ui === true) {
                        RED.menu.setVisible("deploymenu-item-runtime-stop", state === STARTED)
                        RED.menu.setVisible("deploymenu-item-runtime-start", state !== STARTED)
                    }
                    // Do not notify the user about this event if:
                    // - This is the very first event we've received after loading the editor (currentState = '')
                    // - The state matches what we already thought was the case (state === currentState)
                    // - The event was triggered by a deploy (msg.deploy === true)
                    // - The event is a safe mode event - that gets notified separately
                    if (currentState !== '' && state !== currentState && !msg.deploy && state !== SAFE) {
                        RED.notify(RED._("notification.state.flows"+(state === STOPPED?'Stopped':'Started'), msg), "success")
                    }
                }
            });
        },
        get started() {
            return state === STARTED
        }
    }
})()
;RED.multiplayer = (function () {

    // activeSessionId - used to identify sessions across websocket reconnects
    let activeSessionId

    let headerWidget
    // Map of session id to { session:'', user:{}, location:{}}
    let sessions = {}
    // Map of username to { user:{}, sessions:[] }
    let users = {}

    function addUserSession (session) {
        if (sessions[session.session]) {
            // This is an existing connection that has been authenticated
            const existingSession = sessions[session.session]
            if (existingSession.user.username !== session.user.username) {
                removeUserHeaderButton(users[existingSession.user.username])
            }
        }
        sessions[session.session] = session
        const user = users[session.user.username] = users[session.user.username] || {
            user: session.user,
            sessions: []
        }
        if (session.user.profileColor === undefined) {
            session.user.profileColor = (1 + Math.floor(Math.random() * 5))
        }
        session.location = session.location || {}
        user.sessions.push(session)

        if (session.session === activeSessionId) {
            // This is the current user session - do not add a extra button for them
        } else {
            if (user.sessions.length === 1) {
                if (user.button) {
                    clearTimeout(user.inactiveTimeout)
                    clearTimeout(user.removeTimeout)
                    user.button.removeClass('inactive')
                } else {
                    addUserHeaderButton(user)
                }
            }
            sessions[session.session].location = session.location
            updateUserLocation(session.session)
        }
    }

    function removeUserSession (sessionId, isDisconnected) {
        removeUserLocation(sessionId)
        const session = sessions[sessionId]
        delete sessions[sessionId]
        const user = users[session.user.username]
        const i = user.sessions.indexOf(session)
        user.sessions.splice(i, 1)
        if (isDisconnected) {
            removeUserHeaderButton(user)
        } else {
            if (user.sessions.length === 0) {
                // Give the user 5s to reconnect before marking inactive
                user.inactiveTimeout = setTimeout(() => {
                    user.button.addClass('inactive')
                    // Give the user further 20 seconds to reconnect before removing them
                    // from the user toolbar entirely
                    user.removeTimeout = setTimeout(() => {
                        removeUserHeaderButton(user)
                    }, 20000)
                }, 5000)
            }
        }
    }

    function addUserHeaderButton (user) {
        user.button = $('<li class="red-ui-multiplayer-user"><button type="button" class="red-ui-multiplayer-user-icon"></button></li>')
            .attr('data-username', user.user.username)
            .prependTo("#red-ui-multiplayer-user-list");
        var button = user.button.find("button")
        RED.popover.tooltip(button, user.user.username)
        button.on('click', function () {
            const location = user.sessions[0].location
            revealUser(location)
        })

        const userProfile = RED.user.generateUserIcon(user.user)
        userProfile.appendTo(button)
    }

    function removeUserHeaderButton (user) {
        user.button.remove()
        delete user.button
    }

    function getLocation () {
        const location = {
            workspace: RED.workspaces.active()
        }
        const editStack = RED.editor.getEditStack()
        for (let i = editStack.length - 1; i >= 0; i--) {
            if (editStack[i].id) {
                location.node = editStack[i].id
                break
            }
        }
        if (isInWorkspace) {
            const chart = $('#red-ui-workspace-chart')
            const chartOffset = chart.offset()
            const scaleFactor = RED.view.scale()
            location.cursor = {
                x: (lastPosition[0] - chartOffset.left + chart.scrollLeft()) / scaleFactor,
                y: (lastPosition[1] - chartOffset.top + chart.scrollTop()) / scaleFactor
            }
        }
        return location
    }

    let publishLocationTimeout
    let lastPosition = [0,0]
    let isInWorkspace = false

    function publishLocation () {
        if (!publishLocationTimeout) {
            publishLocationTimeout = setTimeout(() => {
                const location = getLocation()
                if (location.workspace !== 0) {
                    log('send', 'multiplayer/location', location)
                    RED.comms.send('multiplayer/location', location)
                }
                publishLocationTimeout = null
            }, 100)
        }
    }


    function revealUser(location, skipWorkspace) {
        if (location.node) {
            // Need to check if this is a known node, so we can fall back to revealing
            // the workspace instead
            const node = RED.nodes.node(location.node)
            if (node) {
                RED.view.reveal(location.node)
            } else if (!skipWorkspace && location.workspace) {
                RED.view.reveal(location.workspace)
            }
        } else if (!skipWorkspace && location.workspace) {
            RED.view.reveal(location.workspace)
        }
    }

    const workspaceTrays = {}
    function getWorkspaceTray(workspaceId) {
        // console.log('get tray for',workspaceId)
        if (!workspaceTrays[workspaceId]) {
            const tray = $('<div class="red-ui-multiplayer-users-tray"></div>')
            const users = []
            const userIcons = {}

            const userCountIcon = $(`<div class="red-ui-multiplayer-user-location"><span class="red-ui-user-profile red-ui-multiplayer-user-count"><span></span></span></div>`)
            const userCountSpan = userCountIcon.find('span span')
            userCountIcon.hide()
            userCountSpan.text('')
            userCountIcon.appendTo(tray)
            const userCountTooltip = RED.popover.tooltip(userCountIcon, function () {
                    const content = $('<div>')
                    users.forEach(sessionId => {
                        $('<div>').append($('<a href="#">').text(sessions[sessionId].user.username).on('click', function (evt) {
                            evt.preventDefault()
                            revealUser(sessions[sessionId].location, true)
                            userCountTooltip.close()
                        })).appendTo(content)
                    })
                    return content
                },
                null,
                true
            )

            const updateUserCount = function () {
                const maxShown = 2
                const children = tray.children()
                children.each(function (index, element) {
                    const i = users.length - index
                    if (i > maxShown) {
                        $(this).hide()
                    } else if (i >= 0) {
                        $(this).show()
                    }
                })
                if (users.length < maxShown + 1) { 
                    userCountIcon.hide()
                } else {
                    userCountSpan.text('+'+(users.length - maxShown))
                    userCountIcon.show()
                }
            }
            workspaceTrays[workspaceId] = {
                attached: false,
                tray,
                users,
                userIcons,
                addUser: function (sessionId) {
                    if (users.indexOf(sessionId) === -1) {
                        // console.log(`addUser ws:${workspaceId} session:${sessionId}`)
                        users.push(sessionId)
                        const userLocationId = `red-ui-multiplayer-user-location-${sessionId}`
                        const userLocationIcon = $(`<div class="red-ui-multiplayer-user-location" id="${userLocationId}"></div>`)
                        RED.user.generateUserIcon(sessions[sessionId].user).appendTo(userLocationIcon)
                        userLocationIcon.prependTo(tray)
                        RED.popover.tooltip(userLocationIcon, sessions[sessionId].user.username)
                        userIcons[sessionId] = userLocationIcon
                        updateUserCount()
                    }
                },
                removeUser: function (sessionId) {
                    // console.log(`removeUser ws:${workspaceId} session:${sessionId}`)
                    const userLocationId = `red-ui-multiplayer-user-location-${sessionId}`
                    const index = users.indexOf(sessionId)
                    if (index > -1) {
                        users.splice(index, 1)
                        userIcons[sessionId].remove()
                        delete userIcons[sessionId]
                    }
                    updateUserCount()
                },
                updateUserCount
            }
        }
        const trayDef = workspaceTrays[workspaceId]
        if (!trayDef.attached) {
            const workspaceTab = $(`#red-ui-tab-${workspaceId}`)
            if (workspaceTab.length > 0) {
                trayDef.attached = true
                trayDef.tray.appendTo(workspaceTab)
                trayDef.users.forEach(sessionId => {
                    trayDef.userIcons[sessionId].on('click', function (evt) {
                        revealUser(sessions[sessionId].location, true)
                    })
                })
            }
        }
        return workspaceTrays[workspaceId]
    }
    function attachWorkspaceTrays () {
        let viewTouched = false
        for (let sessionId of Object.keys(sessions)) {
            const location = sessions[sessionId].location
            if (location) {
                if (location.workspace) {
                    getWorkspaceTray(location.workspace).updateUserCount()
                }
                if (location.node) {
                    addUserToNode(sessionId, location.node)
                    viewTouched = true
                }
            }
        }
        if (viewTouched) {
            RED.view.redraw()
        }
    }

    function addUserToNode(sessionId, nodeId) {
        const node = RED.nodes.node(nodeId)
        if (node) {
            if (!node._multiplayer) {
                node._multiplayer = {
                    users: [sessionId]
                }
                node._multiplayer_refresh = true
            } else {
                if (node._multiplayer.users.indexOf(sessionId) === -1) {
                    node._multiplayer.users.push(sessionId)
                    node._multiplayer_refresh = true
                }
            }
        }
    }
    function removeUserFromNode(sessionId, nodeId) {
        const node = RED.nodes.node(nodeId)
        if (node && node._multiplayer) {
            const i = node._multiplayer.users.indexOf(sessionId)
            if (i > -1) {
                node._multiplayer.users.splice(i, 1)
            }
            if (node._multiplayer.users.length === 0) {
                delete node._multiplayer
            } else {
                node._multiplayer_refresh = true
            }
        }

    }

    function removeUserLocation (sessionId) {
        updateUserLocation(sessionId, {})
        removeUserCursor(sessionId)
    }
    function removeUserCursor (sessionId) {
        // return
        if (sessions[sessionId]?.cursor) {
            sessions[sessionId].cursor.parentNode.removeChild(sessions[sessionId].cursor)
            delete sessions[sessionId].cursor
        }
    }

    function updateUserLocation (sessionId, location) {
        let viewTouched = false
        const oldLocation = sessions[sessionId].location
        if (location) {
            if (oldLocation.workspace !== location.workspace) {
                // console.log('removing', sessionId, oldLocation.workspace)
                workspaceTrays[oldLocation.workspace]?.removeUser(sessionId)
            }
            if (oldLocation.node !== location.node) {
                removeUserFromNode(sessionId, oldLocation.node)
                viewTouched = true
            }
            sessions[sessionId].location = location
        } else {
            location = sessions[sessionId].location
        }
        // console.log(`updateUserLocation sessionId:${sessionId} oldWS:${oldLocation?.workspace} newWS:${location.workspace}`)
        if (location.workspace) {
            getWorkspaceTray(location.workspace).addUser(sessionId)
            if (location.cursor && location.workspace === RED.workspaces.active()) {
                if (!sessions[sessionId].cursor) {
                    const user = sessions[sessionId].user
                    const cursorIcon = document.createElementNS("http://www.w3.org/2000/svg","g");
                    cursorIcon.setAttribute("class", "red-ui-multiplayer-annotation")
                    cursorIcon.appendChild(createAnnotationUser(user, true))
                    $(cursorIcon).css({
                        transform: `translate( ${location.cursor.x}px, ${location.cursor.y}px)`,
                        transition: 'transform 0.1s linear'
                    })
                    $("#red-ui-workspace-chart svg").append(cursorIcon)
                    sessions[sessionId].cursor = cursorIcon
                } else {
                    const cursorIcon = sessions[sessionId].cursor
                    $(cursorIcon).css({
                        transform: `translate( ${location.cursor.x}px, ${location.cursor.y}px)`
                    })
    
                }
            } else if (sessions[sessionId].cursor) {
                removeUserCursor(sessionId)
            }
        }
        if (location.node) {
            addUserToNode(sessionId, location.node)
            viewTouched = true
        }
        if (viewTouched) {
            RED.view.redraw()
        }
    }

    // function refreshUserLocations () {
    //     for (const session of Object.keys(sessions)) {
    //         if (session !== activeSessionId) {
    //             updateUserLocation(session)
    //         }
    //     }
    // }


    function createAnnotationUser(user, pointer = false) {
        const radius = 20
        const halfRadius = radius/2
        const group = document.createElementNS("http://www.w3.org/2000/svg","g");
        const badge = document.createElementNS("http://www.w3.org/2000/svg","path");
        let shapePath
        if (!pointer) {
            shapePath = `M 0 ${halfRadius} a ${halfRadius} ${halfRadius} 0 1 1 ${radius} 0 a ${halfRadius} ${halfRadius} 0 1 1 -${radius} 0 z`
        } else {
            shapePath = `M 0 0 h ${halfRadius} a ${halfRadius} ${halfRadius} 0 1 1 -${halfRadius} ${halfRadius} z`
        }
        badge.setAttribute('d', shapePath)
        badge.setAttribute("class", "red-ui-multiplayer-annotation-background")
        group.appendChild(badge)
        if (user && user.profileColor !== undefined) {
            badge.setAttribute("class", "red-ui-multiplayer-annotation-background red-ui-user-profile-color-" + user.profileColor)
        }
        if (user && user.image) {
            const image = document.createElementNS("http://www.w3.org/2000/svg","image");
            image.setAttribute("width", radius)
            image.setAttribute("height", radius)
            image.setAttribute("href", user.image)
            image.setAttribute("clip-path", "circle("+Math.floor(radius/2)+")")
            group.appendChild(image)
        } else if (user && user.anonymous) {
            const anonIconHead = document.createElementNS("http://www.w3.org/2000/svg","circle");
            anonIconHead.setAttribute("cx", radius/2)
            anonIconHead.setAttribute("cy", radius/2 - 2)
            anonIconHead.setAttribute("r", 2.4)
            anonIconHead.setAttribute("class","red-ui-multiplayer-annotation-anon-label");
            group.appendChild(anonIconHead)
            const anonIconBody = document.createElementNS("http://www.w3.org/2000/svg","path");
            anonIconBody.setAttribute("class","red-ui-multiplayer-annotation-anon-label");
            // anonIconBody.setAttribute("d",`M ${radius/2 - 4} ${radius/2 + 1} h 8 v4 h -8 z`);
            anonIconBody.setAttribute("d",`M ${radius/2} ${radius/2 + 5} h -2.5 c -2 1 -2 -5 0.5 -4.5 c 2 1 2 1 4 0 c 2.5 -0.5  2.5 5.5  0 4.5  z`);
            group.appendChild(anonIconBody)
        } else {
            const label = document.createElementNS("http://www.w3.org/2000/svg","text");
            if (user.username || user.email) {
                label.setAttribute("class","red-ui-multiplayer-annotation-label");
                label.textContent = (user.username || user.email).substring(0,2)
            } else {
                label.setAttribute("class","red-ui-multiplayer-annotation-label red-ui-multiplayer-user-count")
                label.textContent = 'nr'
            }
            label.setAttribute("text-anchor", "middle")
            label.setAttribute("x",radius/2);
            label.setAttribute("y",radius/2 + 3);
            group.appendChild(label)
        }
        const border = document.createElementNS("http://www.w3.org/2000/svg","path");
        border.setAttribute('d', shapePath)
        border.setAttribute("class", "red-ui-multiplayer-annotation-border")
        group.appendChild(border)
        return group
    }

    return {
        init: function () {

            
            
            RED.view.annotations.register("red-ui-multiplayer",{
                type: 'badge',
                align: 'left',
                class: "red-ui-multiplayer-annotation",
                show: "_multiplayer",
                refresh: "_multiplayer_refresh",
                element: function(node) {
                    const containerGroup = document.createElementNS("http://www.w3.org/2000/svg","g");
                    containerGroup.setAttribute("transform","translate(0,-4)")
                    if (node._multiplayer) {
                        let y = 0
                        for (let i = Math.min(1, node._multiplayer.users.length - 1); i >= 0; i--) {
                            const user = sessions[node._multiplayer.users[i]].user
                            const group = createAnnotationUser(user)
                            group.setAttribute("transform","translate("+y+",0)")
                            y += 15
                            containerGroup.appendChild(group)
                        }
                        if (node._multiplayer.users.length > 2) {
                            const group = createAnnotationUser('+'+(node._multiplayer.users.length - 2))
                            group.setAttribute("transform","translate("+y+",0)")
                            y += 12
                            containerGroup.appendChild(group)
                        }

                    }
                    return containerGroup;
                },
                tooltip: node => { return node._multiplayer.users.map(u => sessions[u].user.username).join('\n') }
            });


            // activeSessionId = RED.settings.getLocal('multiplayer:sessionId')
            // if (!activeSessionId) {
                activeSessionId = RED.nodes.id()
            //     RED.settings.setLocal('multiplayer:sessionId', activeSessionId)
            //     log('Session ID (new)', activeSessionId)
            // } else {
                log('Session ID', activeSessionId)
            // }
            
            headerWidget = $('<li><ul id="red-ui-multiplayer-user-list"></ul></li>').prependTo('.red-ui-header-toolbar')

            RED.comms.on('connect', () => {
                const location = getLocation()
                const connectInfo = {
                    session: activeSessionId
                }
                if (location.workspace !== 0) {
                    connectInfo.location = location
                }
                RED.comms.send('multiplayer/connect', connectInfo)
            })
            RED.comms.subscribe('multiplayer/#', (topic, msg) => {
                log('recv', topic, msg)
                if (topic === 'multiplayer/init') {
                    // We have just reconnected, runtime has sent state to
                    // initialise the world
                    sessions = {}
                    users = {}
                    $('#red-ui-multiplayer-user-list').empty()

                    msg.sessions.forEach(session => {
                        addUserSession(session)
                    })
                } else if (topic === 'multiplayer/connection-added') {
                    addUserSession(msg)
                } else if (topic === 'multiplayer/connection-removed') {
                    removeUserSession(msg.session, msg.disconnected)
                } else if (topic === 'multiplayer/location') {
                    const session = msg.session
                    delete msg.session
                    updateUserLocation(session, msg)
                }
            })

            RED.events.on('workspace:change', (event) => {
                getWorkspaceTray(event.workspace)
                publishLocation()
            })
            RED.events.on('editor:open', () => {
                publishLocation()
            })
            RED.events.on('editor:close', () => {
                publishLocation()
            })
            RED.events.on('editor:change', () => {
                publishLocation()
            })
            RED.events.on('login', () => {
                publishLocation()
            })
            RED.events.on('flows:loaded', () => {
                attachWorkspaceTrays()
            })
            RED.events.on('workspace:close', (event) => {
                // A subflow tab has been closed. Need to mark its tray as detached
                if (workspaceTrays[event.workspace]) {
                    workspaceTrays[event.workspace].attached = false
                }
            })
            RED.events.on('logout', () => {
                const disconnectInfo = {
                    session: activeSessionId
                }
                RED.comms.send('multiplayer/disconnect', disconnectInfo)
                RED.settings.removeLocal('multiplayer:sessionId')
            })
            
            const chart = $('#red-ui-workspace-chart')
            chart.on('mousemove', function (evt) {
                lastPosition[0] = evt.clientX
                lastPosition[1] = evt.clientY
                publishLocation()
            })
            chart.on('scroll', function (evt) {
                publishLocation()
            })
            chart.on('mouseenter', function () {
                isInWorkspace = true
                publishLocation()
            })
            chart.on('mouseleave', function () {
                isInWorkspace = false
                publishLocation()
            })
        }
    }

    function log() {
        if (RED.multiplayer.DEBUG) {
            console.log('[multiplayer]', ...arguments)
        }
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
RED.text = {};
RED.text.bidi = (function() {
    var textDir = "";
    var LRE = "\u202A",
        RLE = "\u202B",
        PDF = "\u202C";

    function isRTLValue(stringValue) {
        var length = stringValue.length;
        for (var i=0;i<length;i++) {
            if (isBidiChar(stringValue.charCodeAt(i))) {
                return true;
            }
            else if(isLatinChar(stringValue.charCodeAt(i))) {
                return false;
            }
         }
         return false;
    }

    function isBidiChar(c)  {
        return (c >= 0x05d0 && c <= 0x05ff)||
               (c >= 0x0600 && c <= 0x065f)||
               (c >= 0x066a && c <= 0x06ef)||
               (c >= 0x06fa && c <= 0x07ff)||
               (c >= 0xfb1d && c <= 0xfdff)||
               (c >= 0xfe70 && c <= 0xfefc);
    }

    function isLatinChar(c){
        return (c > 64 && c < 91)||(c > 96 && c < 123)
    }

    /**
     * Determines the text direction of a given string.
     * @param value - the string
     */
    function resolveBaseTextDir(value) {
        if (textDir == "auto") {
            if (isRTLValue(value)) {
                return "rtl";
            } else {
                return "ltr";
            }
        }
        else {
            return textDir;
        }
    }

    function onInputChange() {
        $(this).attr("dir", resolveBaseTextDir($(this).val()));
    }

    /**
     * Adds event listeners to the Input to ensure its text-direction attribute
     * is properly set based on its content.
     * @param input - the input field
     */
    function prepareInput(input) {
        input.on("keyup",onInputChange).on("paste",onInputChange).on("cut",onInputChange);
        // Set the initial text direction
        onInputChange.call(input);
    }

    /**
     * Enforces the text direction of a given string by adding
     * UCC (Unicode Control Characters)
     * @param value - the string
     */
    function enforceTextDirectionWithUCC(value) {
        if (value) {
            var dir = resolveBaseTextDir(value);
            if (dir == "ltr") {
               return LRE + value + PDF;
            }
            else if (dir == "rtl") {
               return RLE + value + PDF;
            }
        }
        return value;
    }

    /**
     * Enforces the text direction for all the spans with style red-ui-text-bidi-aware under
     * workspace or sidebar div
     */
    function enforceTextDirectionOnPage() {
        $("#red-ui-workspace").find('span.red-ui-text-bidi-aware').each(function() {
            $(this).attr("dir", resolveBaseTextDir($(this).html()));
        });
        $("#red-ui-sidebar").find('span.red-ui-text-bidi-aware').each(function() {
            $(this).attr("dir", resolveBaseTextDir($(this).text()));
        });
    }

    /**
     * Sets the text direction preference
     * @param dir - the text direction preference
     */
    function setTextDirection(dir) {
        textDir = dir;
        RED.nodes.eachNode(function(n) { n.dirty = true;});
        RED.view.redraw();
        RED.palette.refresh();
        enforceTextDirectionOnPage();
    }

    return {
        setTextDirection: setTextDirection,
        enforceTextDirectionWithUCC: enforceTextDirectionWithUCC,
        resolveBaseTextDir: resolveBaseTextDir,
        prepareInput: prepareInput
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
RED.text.format = (function() {

    var TextSegment = (function() {
        var TextSegment = function (obj) {
            this.content = "";
            this.actual = "";
            this.textDirection = "";
            this.localGui = "";
            this.isVisible = true;
            this.isSeparator = false;
            this.isParsed = false;
            this.keep = false;
            this.inBounds = false;
            this.inPoints = false;
            var prop = "";
            for (prop in obj) {
                if (obj.hasOwnProperty(prop)) {
                    this[prop] = obj[prop];
                }
            }
        };
        return TextSegment;
    })();

    var tools = (function() {
        function initBounds(bounds) {
            if (!bounds) {
                return false;
            }
            if (typeof(bounds.start) === "undefined") {
                bounds.start = "";
            }
            if (typeof(bounds.end) === "undefined") {
                bounds.end = "";
            }
            if (typeof(bounds.startAfter) !== "undefined") {
                bounds.start = bounds.startAfter;
                bounds.after = true;
            } else {
                bounds.after = false;
            }
            if (typeof(bounds.endBefore) !== "undefined") {
                bounds.end = bounds.endBefore;
                bounds.before = true;
            } else {
                bounds.before = false;
            }
            var startPos = parseInt(bounds.startPos, 10);
            if (!isNaN(startPos)) {
                bounds.usePos = true;
            } else {
                bounds.usePos = false;
            }
            var bLength = parseInt(bounds.length, 10);
            if (!isNaN(bLength)) {
                bounds.useLength = true;
            } else {
                bounds.useLength = false;
            }
            bounds.loops = typeof(bounds.loops) !== "undefined" ? !!bounds.loops : true;
            return true;
        }

        function getBounds(segment, src) {
            var bounds = {};
            for (var prop in src) {
                if (src.hasOwnProperty(prop)) {
                    bounds[prop] = src[prop];
                }
            }
            var content = segment.content;
            var usePos = bounds.usePos && bounds.startPos < content.length;
            if (usePos) {
                bounds.start = "";
                bounds.loops = false;
            }
            bounds.bStart = usePos ? bounds.startPos : bounds.start.length > 0 ? content.indexOf(bounds.start) : 0;
            var useLength = bounds.useLength && bounds.length > 0 && bounds.bStart + bounds.length < content.length;
            if (useLength) {
                bounds.end = "";
            }
            bounds.bEnd = useLength ? bounds.bStart + bounds.length : bounds.end.length > 0 ?
                    content.indexOf(bounds.end, bounds.bStart + bounds.start.length) + 1 : content.length;
            if (!bounds.after) {
                bounds.start = "";
            }
            if (!bounds.before) {
                bounds.end = "";
            }
            return bounds;
        }

        return {
            handleSubcontents: function (segments, args, subs, origContent, locale) { // jshint unused: false
                if (!subs.content || typeof(subs.content) !== "string" || subs.content.length === 0) {
                    return segments;
                }
                var sLoops = true;
                if (typeof(subs.loops) !== "undefined") {
                    sLoops = !!subs.loops;
                }
                for (var j = 0; true; j++) {
                    if (j >= segments.length) {
                        break;
                    }
                    if (segments[j].isParsed || segments.keep || segments[j].isSeparator) {
                        continue;
                    }
                    var content = segments[j].content;
                    var start = content.indexOf(subs.content);
                    if (start < 0) {
                        continue;
                    }
                    var end;
                    var length = 0;
                    if (subs.continued) {
                        do {
                            length++;
                            end = content.indexOf(subs.content, start + length * subs.content.length);
                        } while (end === 0);
                    } else {
                        length = 1;
                    }
                    end = start + length * subs.content.length;
                    segments.splice(j, 1);
                    if (start > 0) {
                        segments.splice(j, 0, new TextSegment({
                            content: content.substring(0, start),
                            localGui: args.dir,
                            keep: true
                        }));
                        j++;
                    }
                    segments.splice(j, 0, new TextSegment({
                        content: content.substring(start, end),
                        textDirection: subs.subDir,
                        localGui: args.dir
                    }));
                    if (end < content.length) {
                        segments.splice(j + 1, 0, new TextSegment({
                            content: content.substring(end, content.length),
                            localGui: args.dir,
                            keep: true
                        }));
                    }
                    if (!sLoops) {
                        break;
                    }
                }
            },

            handleBounds: function (segments, args, aBounds, origContent, locale) {
                for (var i = 0; i < aBounds.length; i++) {
                    if (!initBounds(aBounds[i])) {
                        continue;
                    }
                    for (var j = 0; true; j++) {
                        if (j >= segments.length) {
                            break;
                        }
                        if (segments[j].isParsed || segments[j].inBounds || segments.keep || segments[j].isSeparator) {
                            continue;
                        }
                        var bounds = getBounds(segments[j], aBounds[i]);
                        var start = bounds.bStart;
                        var end = bounds.bEnd;
                        if (start < 0 || end < 0) {
                            continue;
                        }
                        var content = segments[j].content;

                        segments.splice(j, 1);
                        if (start > 0) {
                            segments.splice(j, 0, new TextSegment({
                                content: content.substring(0, start),
                                localGui: args.dir,
                                keep: true
                            }));
                            j++;
                        }
                        if (bounds.start) {
                            segments.splice(j, 0, new TextSegment({
                                content: bounds.start,
                                localGui: args.dir,
                                isSeparator: true
                            }));
                            j++;
                        }
                        segments.splice(j, 0, new TextSegment({
                            content: content.substring(start + bounds.start.length, end - bounds.end.length),
                            textDirection: bounds.subDir,
                            localGui: args.dir,
                            inBounds: true
                        }));
                        if (bounds.end) {
                            j++;
                            segments.splice(j, 0, new TextSegment({
                                content: bounds.end,
                                localGui: args.dir,
                                isSeparator: true
                            }));
                        }
                        if (end + bounds.end.length < content.length) {
                            segments.splice(j + 1, 0, new TextSegment({
                                content: content.substring(end + bounds.end.length, content.length),
                                localGui: args.dir,
                                keep: true
                            }));
                        }
                        if (!bounds.loops) {
                            break;
                        }
                    }
                }
                for (i = 0; i < segments.length; i++) {
                    segments[i].inBounds = false;
                }
                return segments;
            },

            handleCases: function (segments, args, cases, origContent, locale) {
                if (cases.length === 0) {
                    return segments;
                }
                var hArgs = {};
                for (var prop in args) {
                    if (args.hasOwnProperty(prop)) {
                        hArgs[prop] = args[prop];
                    }
                }
                for (var i =  0; i < cases.length; i++) {
                    if (!cases[i].handler || typeof(cases[i].handler.handle) !== "function") {
                        cases[i].handler = args.commonHandler;
                    }
                    if (cases[i].args) {
                        hArgs.cases = cases[i].args.cases;
                        hArgs.points = cases[i].args.points;
                        hArgs.bounds = cases[i].args.bounds;
                        hArgs.subs = cases[i].args.subs;
                    } else {
                        hArgs.cases = [];
                        hArgs.points = [];
                        hArgs.bounds = [];
                        hArgs.subs = {};
                    }
                    cases[i].handler.handle(origContent, segments, hArgs, locale);
                }
                return segments;
            },

            handlePoints: function (segments, args, points, origContent, locale) { //jshint unused: false
                for (var i = 0; i < points.length; i++) {
                    for (var j = 0; true; j++) {
                        if (j >= segments.length) {
                            break;
                        }
                        if (segments[j].isParsed || segments[j].keep || segments[j].isSeparator) {
                            continue;
                        }
                        var content = segments[j].content;
                        var pos = content.indexOf(points[i]);
                        if (pos >= 0) {
                            segments.splice(j, 1);
                            if (pos > 0) {
                                segments.splice(j, 0, new TextSegment({
                                    content: content.substring(0, pos),
                                    textDirection: args.subDir,
                                    localGui: args.dir,
                                    inPoints: true
                                }));
                                j++;
                            }
                            segments.splice(j, 0, new TextSegment({
                                content: points[i],
                                localGui: args.dir,
                                isSeparator: true
                            }));
                            if (pos + points[i].length + 1 <= content.length) {
                                segments.splice(j + 1, 0, new TextSegment({
                                    content: content.substring(pos + points[i].length),
                                    textDirection: args.subDir,
                                    localGui: args.dir,
                                    inPoints: true
                                }));
                            }
                        }
                    }
                }
                for (i = 0; i < segments.length; i++) {
                    if (segments[i].keep) {
                        segments[i].keep = false;
                    } else if(segments[i].inPoints){
                        segments[i].isParsed = true;
                        segments[i].inPoints = false;
                    }
                }
                return segments;
            }
        };
    })();

    var common = (function() {
        return {
            handle: function (content, segments, args, locale) {
                var cases = [];
                if (Array.isArray(args.cases)) {
                    cases = args.cases;
                }
                var points = [];
                if (typeof(args.points) !== "undefined") {
                    if (Array.isArray(args.points)) {
                        points = args.points;
                    } else if (typeof(args.points) === "string") {
                        points = args.points.split("");
                    }
                }
                var subs = {};
                if (typeof(args.subs) === "object") {
                    subs = args.subs;
                }
                var aBounds = [];
                if (Array.isArray(args.bounds)) {
                    aBounds = args.bounds;
                }

                tools.handleBounds(segments, args, aBounds, content, locale);
                tools.handleSubcontents(segments, args, subs, content, locale);
                tools.handleCases(segments, args, cases, content, locale);
                tools.handlePoints(segments, args, points, content, locale);
                return segments;
            }
        };
    })();

    var misc = (function() {
        var isBidiLocale = function (locale) {
            var lang = !locale ? "" : locale.split("-")[0];
            if (!lang || lang.length < 2) {
                return false;
            }
            return ["iw", "he", "ar", "fa", "ur"].some(function (bidiLang) {
                return bidiLang === lang;
            });
        };
        var LRE = "\u202A";
        var RLE = "\u202B";
        var PDF = "\u202C";
        var LRM = "\u200E";
        var RLM = "\u200F";
        var LRO = "\u202D";
        var RLO = "\u202E";

        return {
            LRE: LRE,
            RLE: RLE,
            PDF: PDF,
            LRM: LRM,
            RLM: RLM,
            LRO: LRO,
            RLO: RLO,

            getLocaleDetails: function (locale) {
                if (!locale) {
                    locale = typeof navigator === "undefined" ? "" :
                        (navigator.language ||
                        navigator.userLanguage ||
                        "");
                }
                locale = locale.toLowerCase();
                if (isBidiLocale(locale)) {
                    var full = locale.split("-");
                    return {lang: full[0], country: full[1] ? full[1] : ""};
                }
                return {lang: "not-bidi"};
            },

            removeUcc: function (text) {
                if (text) {
                    return text.replace(/[\u200E\u200F\u202A-\u202E]/g, "");
                }
                return text;
            },

            removeTags: function (text) {
                if (text) {
                    return text.replace(/<[^<]*>/g, "");
                }
                return text;
            },

            getDirection: function (text, dir, guiDir, checkEnd) {
                if (dir !== "auto" && (/^(rtl|ltr)$/i).test(dir)) {
                    return dir;
                }
                guiDir = (/^(rtl|ltr)$/i).test(guiDir) ? guiDir : "ltr";
                var txt = !checkEnd ? text : text.split("").reverse().join("");
                var fdc = /[A-Za-z\u05d0-\u065f\u066a-\u06ef\u06fa-\u07ff\ufb1d-\ufdff\ufe70-\ufefc]/.exec(txt);
                return fdc ? (fdc[0] <= "z" ? "ltr" : "rtl") : guiDir;
            },

            hasArabicChar: function (text) {
                var fdc = /[\u0600-\u065f\u066a-\u06ef\u06fa-\u07ff\ufb1d-\ufdff\ufe70-\ufefc]/.exec(text);
                return !!fdc;
            },

            showMarks: function (text, guiDir) {
                var result = "";
                for (var i = 0; i < text.length; i++) {
                    var c = "" + text.charAt(i);
                    switch (c) {
                    case LRM:
                        result += "<LRM>";
                        break;
                    case RLM:
                        result += "<RLM>";
                        break;
                    case LRE:
                        result += "<LRE>";
                        break;
                    case RLE:
                        result += "<RLE>";
                        break;
                    case LRO:
                        result += "<LRO>";
                        break;
                    case RLO:
                        result += "<RLO>";
                        break;
                    case PDF:
                        result += "<PDF>";
                        break;
                    default:
                        result += c;
                    }
                }
                var mark = typeof(guiDir) === "undefined" || !((/^(rtl|ltr)$/i).test(guiDir)) ? "" :
                    guiDir === "rtl" ? RLO : LRO;
                return mark + result + (mark === "" ? "" : PDF);
            },

            hideMarks: function (text) {
                var txt = text.replace(/<LRM>/g, this.LRM).replace(/<RLM>/g, this.RLM).replace(/<LRE>/g, this.LRE);
                return txt.replace(/<RLE>/g, this.RLE).replace(/<LRO>/g, this.LRO).replace(/<RLO>/g, this.RLO).replace(/<PDF>/g, this.PDF);
            },

            showTags: function (text) {
                return "<xmp>" + text + "</xmp>";
            },

            hideTags: function (text) {
                return text.replace(/<xmp>/g,"").replace(/<\/xmp>/g,"");
            }
        };
    })();

    var stext = (function() {
        var stt = {};

        // args
        //   handler: main handler (default - dbidi/stt/handlers/common)
        //   guiDir: GUI direction (default - "ltr")
        //   dir: main stt direction (default - guiDir)
        //   subDir: direction of subsegments
        //   points: array of delimiters (default - [])
        //   bounds: array of definitions of bounds in which handler works
        //   subs: object defines special handling for some substring if found
        //   cases: array of additional modules with their args for handling special cases (default - [])
        function parseAndDisplayStructure(content, fArgs, isHtml, locale) {
            if (!content || !fArgs) {
                return content;
            }
            return displayStructure(parseStructure(content, fArgs, locale), fArgs, isHtml);
        }

        function checkArguments(fArgs, fullCheck) {
            var args = Array.isArray(fArgs)? fArgs[0] : fArgs;
            if (!args.guiDir) {
                args.guiDir = "ltr";
            }
            if (!args.dir) {
                args.dir = args.guiDir;
            }
            if (!fullCheck) {
                return args;
            }
            if (typeof(args.points) === "undefined") {
                args.points = [];
            }
            if (!args.cases) {
                args.cases = [];
            }
            if (!args.bounds) {
                args.bounds = [];
            }
            args.commonHandler = common;
            return args;
        }

        function parseStructure(content, fArgs, locale) {
            if (!content || !fArgs) {
                return new TextSegment({content: ""});
            }
            var args = checkArguments(fArgs, true);
            var segments = [new TextSegment(
                {
                    content: content,
                    actual: content,
                    localGui: args.dir
                })];
            var parse = common.handle;
            if (args.handler && typeof(args.handler) === "function") {
                parse = args.handler.handle;
            }
            parse(content, segments, args, locale);
            return segments;
        }

        function displayStructure(segments, fArgs, isHtml) {
            var args = checkArguments(fArgs, false);
            if (isHtml) {
                return getResultWithHtml(segments, args);
            }
            else {
                return getResultWithUcc(segments, args);
            }
        }

        function getResultWithUcc(segments, args, isHtml) {
            var result = "";
            var checkedDir = "";
            var prevDir = "";
            var stop = false;
            for (var i = 0; i < segments.length; i++) {
                if (segments[i].isVisible) {
                    var dir = segments[i].textDirection;
                    var lDir = segments[i].localGui;
                    if (lDir !== "" && prevDir === "") {
                        result += (lDir === "rtl" ? misc.RLE : misc.LRE);
                    }
                    else if(prevDir !== "" && (lDir === "" || lDir !== prevDir || stop)) {
                        result += misc.PDF + (i == segments.length - 1 && lDir !== ""? "" : args.dir === "rtl" ? misc.RLM : misc.LRM);
                        if (lDir !== "") {
                            result += (lDir === "rtl" ? misc.RLE : misc.LRE);
                        }
                    }
                    if (dir === "auto") {
                        dir = misc.getDirection(segments[i].content, dir, args.guiDir);
                    }
                    if ((/^(rtl|ltr)$/i).test(dir)) {
                        result += (dir === "rtl" ? misc.RLE : misc.LRE) + segments[i].content + misc.PDF;
                        checkedDir = dir;
                    }
                    else {
                        result += segments[i].content;
                        checkedDir = misc.getDirection(segments[i].content, dir, args.guiDir, true);
                    }
                    if (i < segments.length - 1) {
                        var locDir = lDir && segments[i+1].localGui? lDir : args.dir;
                        result += locDir === "rtl" ? misc.RLM : misc.LRM;
                    }
                    else if(prevDir !== "") {
                        result += misc.PDF;
                    }
                    prevDir = lDir;
                    stop = false;
                }
                else {
                    stop = true;
                }
            }
            var sttDir = args.dir === "auto" ? misc.getDirection(segments[0].actual, args.dir, args.guiDir) : args.dir;
            if (sttDir !== args.guiDir) {
                result = (sttDir === "rtl" ? misc.RLE : misc.LRE) + result + misc.PDF;
            }
            return result;
        }

        function getResultWithHtml(segments, args, isHtml) {
            var result = "";
            var checkedDir = "";
            var prevDir = "";
            for (var i = 0; i < segments.length; i++) {
                if (segments[i].isVisible) {
                    var dir = segments[i].textDirection;
                    var lDir = segments[i].localGui;
                    if (lDir !== "" && prevDir === "") {
                        result += "<bdi dir='" + (lDir === "rtl" ? "rtl" : "ltr") + "'>";
                    }
                    else if(prevDir !== "" && (lDir === "" || lDir !== prevDir || stop)) {
                        result += "</bdi>" + (i == segments.length - 1 && lDir !== ""? "" : "<span style='unicode-bidi: embed; direction: " + (args.dir === "rtl" ? "rtl" : "ltr") + ";'></span>");
                        if (lDir !== "") {
                            result += "<bdi dir='" + (lDir === "rtl" ? "rtl" : "ltr") + "'>";
                        }
                    }

                    if (dir === "auto") {
                        dir = misc.getDirection(segments[i].content, dir, args.guiDir);
                    }
                    if ((/^(rtl|ltr)$/i).test(dir)) {
                        //result += "<span style='unicode-bidi: embed; direction: " + (dir === "rtl" ? "rtl" : "ltr") + ";'>" + segments[i].content + "</span>";
                        result += "<bdi dir='" + (dir === "rtl" ? "rtl" : "ltr") + "'>" + segments[i].content + "</bdi>";
                        checkedDir = dir;
                    }
                    else {
                        result += segments[i].content;
                        checkedDir = misc.getDirection(segments[i].content, dir, args.guiDir, true);
                    }
                    if (i < segments.length - 1) {
                        var locDir = lDir && segments[i+1].localGui? lDir : args.dir;
                        result += "<span style='unicode-bidi: embed; direction: " + (locDir === "rtl" ? "rtl" : "ltr") + ";'></span>";
                    }
                    else if(prevDir !== "") {
                        result += "</bdi>";
                    }
                    prevDir = lDir;
                    stop = false;
                }
                else {
                    stop = true;
                }
            }
            var sttDir = args.dir === "auto" ? misc.getDirection(segments[0].actual, args.dir, args.guiDir) : args.dir;
            if (sttDir !== args.guiDir) {
                result = "<bdi dir='" + (sttDir === "rtl" ? "rtl" : "ltr") + "'>" + result + "</bdi>";
            }
            return result;
        }

        //TBD ?
        function restore(text, isHtml) {
            return text;
        }

        stt.parseAndDisplayStructure = parseAndDisplayStructure;
        stt.parseStructure = parseStructure;
        stt.displayStructure = displayStructure;
        stt.restore = restore;

        return stt;
    })();

    var breadcrumb = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: args.dir ? args.dir : isRtl ? "rtl" : "ltr",
                        subs: {
                            content: ">",
                            continued: true,
                            subDir: isRtl ? "rtl" : "ltr"
                        },
                        cases: [{
                            args: {
                                subs: {
                                    content: "<",
                                    continued: true,
                                    subDir: isRtl ? "ltr" : "rtl"
                                }
                            }
                        }]
                };

                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var comma = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: ","
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var email = (function() {
        function getDir(text, locale) {
            if (misc.getLocaleDetails(locale).lang !== "ar") {
                return "ltr";
            }
            var ind = text.indexOf("@");
            if (ind > 0 && ind < text.length - 1) {
                return misc.hasArabicChar(text.substring(ind + 1)) ? "rtl" : "ltr";
            }
            return "ltr";
        }

        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: getDir(text, locale),
                        points: "<>.:,;@",
                        cases: [{
                            handler: common,
                            args: {
                                bounds: [{
                                    startAfter: "\"",
                                    endBefore: "\""
                                },
                                {
                                    startAfter: "(",
                                    endBefore: ")"
                                }
                                ],
                                points: ""
                            }
                        }]
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var filepath = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: "/\\:."
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var formula = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: " /%^&[]<>=!?~:.,|()+-*{}",
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();


    var sql = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: "\t!#%&()*+,-./:;<=>?|[]{}",
                        cases: [{
                            handler: common,
                            args: {
                                bounds: [{
                                    startAfter: "/*",
                                    endBefore: "*/"
                                },
                                {
                                    startAfter: "--",
                                    end: "\n"
                                },
                                {
                                    startAfter: "--"
                                }
                                ]
                            }
                        },
                        {
                            handler: common,
                            args: {
                                subs: {
                                    content: " ",
                                    continued: true
                                }
                            }
                        },
                        {
                            handler: common,
                            args: {
                                bounds: [{
                                    startAfter: "'",
                                    endBefore: "'"
                                },
                                {
                                    startAfter: "\"",
                                    endBefore: "\""
                                }
                                ]
                            }
                        }
                        ]
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var underscore = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: "_"
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var url = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: ":?#/@.[]="
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var word = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: args.dir ? args.dir : isRtl ? "rtl" : "ltr",
                        points: " ,.!?;:",
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var xpath = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var fArgs =
                {
                        guiDir: isRtl ? "rtl" : "ltr",
                        dir: "ltr",
                        points: " /[]<>=!:@.|()+-*",
                        cases: [{
                            handler: common,
                            args: {
                                bounds: [{
                                    startAfter: "\"",
                                    endBefore: "\""
                                },
                                {
                                    startAfter: "'",
                                    endBefore: "'"
                                }
                                ],
                                points: ""
                            }
                        }
                        ]
                };
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, fArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, fArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var custom = (function() {
        return {
            format: function (text, args, isRtl, isHtml, locale, parseOnly) {
                var hArgs = {};
                var prop = "";
                var sArgs = Array.isArray(args)? args[0] : args;
                for (prop in sArgs) {
                    if (sArgs.hasOwnProperty(prop)) {
                        hArgs[prop] = sArgs[prop];
                    }
                }
                hArgs.guiDir = isRtl ? "rtl" : "ltr";
                hArgs.dir = hArgs.dir ? hArgs.dir : hArgs.guiDir;
                if (!parseOnly) {
                    return stext.parseAndDisplayStructure(text, hArgs, !!isHtml, locale);
                }
                else {
                    return stext.parseStructure(text, hArgs, !!isHtml, locale);
                }
            }
        };
    })();

    var message = (function() {
        var params = {msgLang: "en", msgDir: "", phLang: "", phDir: "", phPacking: ["{","}"], phStt: {type: "none", args: {}}, guiDir: ""};
        var parametersChecked = false;

        function getDirectionOfLanguage(lang) {
            if (lang === "he" || lang === "iw" || lang === "ar") {
                return "rtl";
            }
            return "ltr";
        }

        function checkParameters(obj) {
            if (obj.msgDir.length === 0) {
                obj.msgDir = getDirectionOfLanguage(obj.msgLang);
            }
            obj.msgDir = obj.msgDir !== "ltr" && obj.msgDir !== "rtl" && obj.msgDir != "auto"? "ltr" : obj.msgDir;
            if (obj.guiDir.length === 0) {
                obj.guiDir = obj.msgDir;
            }
            obj.guiDir = obj.guiDir !== "rtl"? "ltr" : "rtl";
            if (obj.phDir.length === 0) {
                obj.phDir = obj.phLang.length === 0? obj.msgDir : getDirectionOfLanguage(obj.phLang);
            }
            obj.phDir = obj.phDir !== "ltr" && obj.phDir !== "rtl" && obj.phDir != "auto"? "ltr" : obj.phDir;
            if (typeof (obj.phPacking) === "string") {
                obj.phPacking = obj.phPacking.split("");
            }
            if (obj.phPacking.length < 2) {
                obj.phPacking = ["{","}"];
            }
        }

        return {
            setDefaults: function (args) {
                for (var prop in args) {
                    if (params.hasOwnProperty(prop)) {
                        params[prop] = args[prop];
                    }
                }
                checkParameters(params);
                parametersChecked = true;
            },

            format: function (text) {
                if (!parametersChecked) {
                    checkParameters(params);
                    parametersChecked = true;
                }
                var isHtml = false;
                var hasHtmlArg = false;
                var spLength = params.phPacking[0].length;
                var epLength = params.phPacking[1].length;
                if (arguments.length > 0) {
                    var last = arguments[arguments.length-1];
                    if (typeof (last) === "boolean") {
                        isHtml = last;
                        hasHtmlArg = true;
                    }
                }
                //Message
                var re = new RegExp(params.phPacking[0] + "\\d+" + params.phPacking[1]);
                var m;
                var tSegments = [];
                var offset = 0;
                var txt = text;
                while ((m = re.exec(txt)) != null) {
                    var lastIndex = txt.indexOf(m[0]) + m[0].length;
                    if (lastIndex > m[0].length) {
                        tSegments.push({text: txt.substring(0, lastIndex - m[0].length), ph: false});
                    }
                    tSegments.push({text: m[0], ph: true});
                    offset += lastIndex;
                    txt = txt.substring(lastIndex, txt.length);
                }
                if (offset < text.length) {
                    tSegments.push({text: text.substring(offset, text.length), ph: false});
                }
                //Parameters
                var tArgs = [];
                for (var i = 1; i < arguments.length - (hasHtmlArg? 1 : 0); i++) {
                    var arg = arguments[i];
                    var checkArr = arg;
                    var inLoop = false;
                    var indArr = 0;
                    if (Array.isArray(checkArr)) {
                        arg = checkArr[0];
                        if (typeof(arg) === "undefined") {
                            continue;
                        }
                        inLoop = true;
                    }
                    do {
                        if (typeof (arg) === "string") {
                            tArgs.push({text: arg, dir: params.phDir, stt: params.stt});
                        }
                        else if(typeof (arg) === "boolean") {
                            isHtml = arg;
                        }
                        else if(typeof (arg) === "object") {
                            tArgs.push(arg);
                            if (!arg.hasOwnProperty("text")) {
                                tArgs[tArgs.length-1].text = "{???}";
                            }
                            if (!arg.hasOwnProperty("dir") || arg.dir.length === 0) {
                                tArgs[tArgs.length-1].dir = params.phDir;
                            }
                            if (!arg.hasOwnProperty("stt") || (typeof (arg.stt) === "string" && arg.stt.length === 0) ||
                                (typeof (arg.stt) === "object" && Object.keys(arg.stt).length === 0)) {
                                tArgs[tArgs.length-1].stt = params.phStt;
                            }
                        }
                        else {
                            tArgs.push({text: "" + arg, dir: params.phDir, stt: params.phStt});
                        }
                        if (inLoop) {
                            indArr++;
                            if (indArr == checkArr.length) {
                                inLoop = false;
                            }
                            else {
                                arg = checkArr[indArr];
                            }
                        }
                    } while(inLoop);
                }
                //Indexing
                var segments = [];
                for (i = 0; i < tSegments.length; i++) {
                    var t = tSegments[i];
                    if (!t.ph) {
                        segments.push(new TextSegment({content: t.text, textDirection: params.msgDir}));
                    }
                    else {
                        var ind = parseInt(t.text.substring(spLength, t.text.length - epLength));
                        if (isNaN(ind) || ind >= tArgs.length) {
                            segments.push(new TextSegment({content: t.text, textDirection: params.msgDir}));
                            continue;
                        }
                        var sttType = "none";
                        if (!tArgs[ind].stt) {
                            tArgs[ind].stt = params.phStt;
                        }
                        if (tArgs[ind].stt) {
                            if (typeof (tArgs[ind].stt) === "string") {
                                sttType = tArgs[ind].stt;
                            }
                            else if(tArgs[ind].stt.hasOwnProperty("type")) {
                                sttType = tArgs[ind].stt.type;
                            }
                        }
                        if (sttType.toLowerCase() !== "none") {
                            var sttSegs =  getHandler(sttType).format(tArgs[ind].text, tArgs[ind].stt.args || {},
                                    params.msgDir === "rtl", false, params.msgLang, true);
                            for (var j = 0; j < sttSegs.length; j++) {
                                segments.push(sttSegs[j]);
                            }
                            segments.push(new TextSegment({isVisible: false}));
                        }
                        else {
                            segments.push(new TextSegment({content: tArgs[ind].text, textDirection: (tArgs[ind].dir? tArgs[ind].dir : params.phDir)}));
                        }
                    }
                }
                var result =  stext.displayStructure(segments, {guiDir: params.guiDir, dir: params.msgDir}, isHtml);
                return result;
            }
        };
    })();

    var event = null;

    function getHandler(type) {
        switch (type) {
        case "breadcrumb" :
            return breadcrumb;
        case "comma" :
            return comma;
        case "email" :
            return email;
        case "filepath" :
            return filepath;
        case "formula" :
            return formula;
        case "sql" :
            return sql;
        case "underscore" :
            return underscore;
        case "url" :
            return url;
        case "word" :
            return word;
        case "xpath" :
            return xpath;
        default:
            return custom;
        }
    }

    function isInputEventSupported(element) {
        var agent = window.navigator.userAgent;
        if (agent.indexOf("MSIE") >=0 || agent.indexOf("Trident") >=0 || agent.indexOf("Edge") >=0) {
            return false;
        }
        var checked = document.createElement(element.tagName);
        checked.contentEditable = true;
        var isSupported = ("oninput" in checked);
        if (!isSupported) {
          checked.setAttribute('oninput', 'return;');
          isSupported = typeof checked['oninput'] == 'function';
        }
        checked = null;
        return isSupported;
    }

    function attachElement(element, type, args, isRtl, locale) {
        //if (!element || element.nodeType != 1 || !element.isContentEditable)
        if (!element || element.nodeType != 1) {
            return false;
        }
        if (!event) {
            event = document.createEvent('Event');
            event.initEvent('TF', true, true);
        }
        element.setAttribute("data-tf-type", type);
        var sArgs = args === "undefined"? "{}" : JSON.stringify(Array.isArray(args)? args[0] : args);
        element.setAttribute("data-tf-args", sArgs);
        var dir = "ltr";
        if (isRtl === "undefined") {
            if (element.dir) {
                dir = element.dir;
            }
            else if(element.style && element.style.direction) {
                dir = element.style.direction;
            }
            isRtl = dir.toLowerCase() === "rtl";
        }
        element.setAttribute("data-tf-dir", isRtl);
        element.setAttribute("data-tf-locale", misc.getLocaleDetails(locale).lang);
        if (isInputEventSupported(element)) {
            var ehandler = element.oninput;
            element.oninput = function(event) {
                displayWithStructure(event.target);
            };
        }
        else {
            element.onkeyup = function(e) {
                displayWithStructure(e.target);
                element.dispatchEvent(event);
            };
            element.onmouseup = function(e) {
                displayWithStructure(e.target);
                element.dispatchEvent(event);
            };
        }
        displayWithStructure(element);

        return true;
    }

    function detachElement(element) {
        if (!element || element.nodeType != 1) {
            return;
        }
        element.removeAttribute("data-tf-type");
        element.removeAttribute("data-tf-args");
        element.removeAttribute("data-tf-dir");
        element.removeAttribute("data-tf-locale");
        element.innerHTML = element.textContent || "";
    }

    function displayWithStructure(element) {
        var txt = element.textContent || "";
        var selection = document.getSelection();
        if (txt.length === 0 || !selection || selection.rangeCount <= 0) {
            element.dispatchEvent(event);
            return;
        }

        var range = selection.getRangeAt(0);
        var tempRange = range.cloneRange(), startNode, startOffset;
        startNode = range.startContainer;
        startOffset = range.startOffset;
        var textOffset = 0;
        if (startNode.nodeType === 3) {
            textOffset += startOffset;
        }
        tempRange.setStart(element,0);
        tempRange.setEndBefore(startNode);
        var div = document.createElement('div');
        div.appendChild(tempRange.cloneContents());
        textOffset += div.textContent.length;

        element.innerHTML = getHandler(element.getAttribute("data-tf-type")).
            format(txt, JSON.parse(element.getAttribute("data-tf-args")), (element.getAttribute("data-tf-dir") === "true"? true : false),
            true, element.getAttribute("data-tf-locale"));
        var parent = element;
        var node = element;
        var newOffset = 0;
        var inEnd = false;
        selection.removeAllRanges();
        range.setStart(element,0);
        range.setEnd(element,0);
        while (node) {
            if (node.nodeType === 3) {
                if (newOffset + node.nodeValue.length >= textOffset) {
                    range.setStart(node, textOffset - newOffset);
                    break;
                }
                else {
                    newOffset += node.nodeValue.length;
                    node = node.nextSibling;
                }
            }
            else if(node.hasChildNodes()) {
                parent = node;
                node = parent.firstChild;
                continue;
            }
            else {
                node = node.nextSibling;
            }
            while (!node) {
                if (parent === element) {
                    inEnd = true;
                    break;
                }
                node = parent.nextSibling;
                parent = parent.parentNode;
            }
            if (inEnd) {
                break;
            }
        }

        selection.addRange(range);
        element.dispatchEvent(event);
    }

    return {
        /*!
        * Returns the HTML representation of a given structured text
        * @param text - the structured text
        * @param type - could be one of filepath, url, email
        * @param args - pass additional arguments to the handler. generally null.
        * @param isRtl - indicates if the GUI is mirrored
        * @param locale - the browser locale
        */
        getHtml: function (text, type, args, isRtl, locale) {
            return getHandler(type).format(text, args, isRtl, true, locale);
        },
        /*!
        * Handle Structured text correct display for a given HTML element.
        * @param element - the element  : should be of type div contenteditable=true
        * @param type - could be one of filepath, url, email
        * @param args - pass additional arguments to the handler. generally null.
        * @param isRtl - indicates if the GUI is mirrored
        * @param locale - the browser locale
        */
        attach: function (element, type, args, isRtl, locale) {
            return attachElement(element, type, args, isRtl, locale);
        }
    };
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
RED.state = {
    DEFAULT: 0,
    MOVING: 1,
    JOINING: 2,
    MOVING_ACTIVE: 3,
    ADDING: 4,
    EDITING: 5,
    EXPORT: 6,
    IMPORT: 7,
    IMPORT_DRAGGING: 8,
    QUICK_JOINING: 9,
    PANNING: 10,
    SELECTING_NODE: 11,
    GROUP_DRAGGING: 12,
    GROUP_RESIZE: 13,
    DETACHED_DRAGGING: 14,
    SLICING: 15,
    SLICING_JUNCTION: 16
}
;RED.plugins = (function() {
    var plugins = {};
    var pluginsByType = {};
    var moduleList = {};

    function registerPlugin(id,definition) {
        plugins[id] = definition;
        if (definition.type) {
            pluginsByType[definition.type] = pluginsByType[definition.type] || [];
            pluginsByType[definition.type].push(definition);
        }
        if (RED._loadingModule) {
            definition.module = RED._loadingModule;
            definition["_"] = function() {
                var args = Array.prototype.slice.call(arguments);
                var originalKey = args[0];
                if (!/:/.test(args[0])) {
                    args[0] = definition.module+":"+args[0];
                }
                var result = RED._.apply(null,args);
                if (result === args[0]) {
                    return originalKey;
                }
                return result;
            }
        } else {
            definition["_"] = RED["_"]
        }
        if (definition.onadd && typeof definition.onadd === 'function') {
            definition.onadd();
        }
        RED.events.emit("registry:plugin-added",id);
    }

    function getPlugin(id) {
        return plugins[id]
    }

    function getPluginsByType(type) {
        return pluginsByType[type] || [];
    }

    function setPluginList(list) {
        for(let i=0;i<list.length;i++) {
            let p = list[i];
            addPlugin(p);
        }
    }

    function addPlugin(p) {

        moduleList[p.module] = moduleList[p.module] || {
            name:p.module,
            version:p.version,
            local:p.local,
            sets:{},
            plugin: true,
            id: p.id
        };
        if (p.pending_version) {
            moduleList[p.module].pending_version = p.pending_version;
        }
        moduleList[p.module].sets[p.name] = p;

        RED.events.emit("registry:plugin-module-added",p.module);
    }

    function getModule(module) {
        return moduleList[module];
    }

    return {
        registerPlugin: registerPlugin,
        getPlugin: getPlugin,
        getPluginsByType: getPluginsByType,

        setPluginList: setPluginList,
        addPlugin: addPlugin,
        getModule: getModule
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

/**
 * An Interface to nodes and utility functions for creating/adding/deleting nodes and links
 * @namespace RED.nodes
*/
RED.nodes = (function() {
    var PORT_TYPE_INPUT = 1;
    var PORT_TYPE_OUTPUT = 0;

    var node_defs = {};
    var linkTabMap = {};

    var configNodes = {};
    var links = [];
    var nodeLinks = {};
    var defaultWorkspace;
    var workspaces = {};
    var workspacesOrder =[];
    var subflows = {};
    var loadedFlowVersion = null;

    var groups = {};
    var groupsByZ = {};

    var junctions = {};
    var junctionsByZ = {};

    var initialLoad;

    var dirty = false;

    const internalProperties = [
        "changed",
        "dirty",
        "id",
        "inputLabels",
        "moved",
        "outputLabels",
        "selected",
        "type",
        "users",
        "valid",
        "validationErrors",
        "wires",
        "a",
        "b",
        "c",
        "d",
        "e",
        "f",
        "g",
        "h",
        "i",
        "j",
        "k",
        "l",
        "m",
        "n",
        "o",
        "p",
        "q",
        "r",
        "s",
        "t",
        "u",
        "v",
        "w",
        "x",
        "y",
        "z",
        "_",
        "_config",
        "_def",
        "_orig"
    ];

    function setDirty(d) {
        dirty = d;
        if (!d) {
            allNodes.clearState()
        }
        RED.events.emit("workspace:dirty",{dirty:dirty});
    }

    // The registry holds information about all node types.
    var registry = (function() {
        var moduleList = {};
        var nodeList = [];
        var nodeSets = {};
        var typeToId = {};
        var nodeDefinitions = {};
        var iconSets = {};

        nodeDefinitions['tab'] = {
            defaults: {
                label: {value:""},
                disabled: {value: false},
                locked: {value: false},
                info: {value: ""},
                env: {value: []}
            }
        };

        var exports = {
            setModulePendingUpdated: function(module,version) {
                if (!!RED.plugins.getModule(module)) {
                    // The module updated is a plugin
                    RED.plugins.getModule(module).pending_version = version;
                } else {
                    moduleList[module].pending_version = version;
                }

                RED.events.emit("registry:module-updated",{module:module,version:version});
            },
            getModule: function(module) {
                return moduleList[module];
            },
            getNodeSetForType: function(nodeType) {
                return exports.getNodeSet(typeToId[nodeType]);
            },
            getModuleList: function() {
                return moduleList;
            },
            getNodeList: function() {
                return nodeList;
            },
            getNodeTypes: function() {
                return Object.keys(nodeDefinitions);
            },
            /**
             * Get an array of node definitions
             * @param {Object} options - options object
             * @param {boolean} [options.configOnly] - if true, only return config nodes
             * @param {function} [options.filter] - a filter function to apply to the list of nodes
             * @returns array of node definitions
             */
            getNodeDefinitions: function(options) {
                const result = []
                const configOnly = (options && options.configOnly)
                const filter = (options && options.filter)
                const keys = Object.keys(nodeDefinitions)
                for (const key of keys) {
                    const def = nodeDefinitions[key]
                    if(!def) { continue }
                    if (configOnly && def.category !== "config") {
                            continue
                    }
                    if (filter && !filter(nodeDefinitions[key])) {
                        continue
                    }
                    result.push(nodeDefinitions[key])
                }
                return result
            },
            setNodeList: function(list) {
                nodeList = [];
                for(var i=0;i<list.length;i++) {
                    var ns = list[i];
                    exports.addNodeSet(ns);
                }
            },
            addNodeSet: function(ns) {
                if (!ns.types) {
                    // A node has been loaded without any types. Ignore it.
                    return;
                }
                ns.added = false;
                nodeSets[ns.id] = ns;
                for (var j=0;j<ns.types.length;j++) {
                    typeToId[ns.types[j]] = ns.id;
                }
                nodeList.push(ns);

                moduleList[ns.module] = moduleList[ns.module] || {
                    name:ns.module,
                    version:ns.version,
                    local:ns.local,
                    sets:{}
                };
                if (ns.pending_version) {
                    moduleList[ns.module].pending_version = ns.pending_version;
                }
                moduleList[ns.module].sets[ns.name] = ns;
                RED.events.emit("registry:node-set-added",ns);
            },
            removeNodeSet: function(id) {
                var ns = nodeSets[id];
                if (!ns) { return {} }

                for (var j=0;j<ns.types.length;j++) {
                    delete typeToId[ns.types[j]];
                }
                delete nodeSets[id];
                for (var i=0;i<nodeList.length;i++) {
                    if (nodeList[i].id === id) {
                        nodeList.splice(i,1);
                        break;
                    }
                }
                delete moduleList[ns.module].sets[ns.name];
                if (Object.keys(moduleList[ns.module].sets).length === 0) {
                    delete moduleList[ns.module];
                }
                RED.events.emit("registry:node-set-removed",ns);
                return ns;
            },
            getNodeSet: function(id) {
                return nodeSets[id];
            },
            enableNodeSet: function(id) {
                var ns = nodeSets[id];
                ns.enabled = true;
                RED.events.emit("registry:node-set-enabled",ns);
            },
            disableNodeSet: function(id) {
                var ns = nodeSets[id];
                ns.enabled = false;
                RED.events.emit("registry:node-set-disabled",ns);
            },
            registerNodeType: function(nt,def) {
                if (nt.substring(0,8) != "subflow:") {
                    if (!nodeSets[typeToId[nt]]) {
                        var error = "";
                        var fullType = nt;
                        if (RED._loadingModule) {
                            fullType = "["+RED._loadingModule+"] "+nt;
                            if (nodeSets[RED._loadingModule]) {
                                error = nodeSets[RED._loadingModule].err || "";
                            } else {
                                error = "Unknown error";
                            }
                        }
                        RED.notify(RED._("palette.event.unknownNodeRegistered",{type:fullType, error:error}), "error");
                        return;
                    }
                    def.set = nodeSets[typeToId[nt]];
                    nodeSets[typeToId[nt]].added = true;
                    nodeSets[typeToId[nt]].enabled = true;

                    var ns;
                    if (def.set.module === "node-red") {
                        ns = "node-red";
                    } else {
                        ns = def.set.id;
                    }
                    def["_"] = function() {
                        var args = Array.prototype.slice.call(arguments, 0);
                        var original = args[0];
                        if (args[0].indexOf(":") === -1) {
                            args[0] = ns+":"+args[0];
                        }
                        var result = RED._.apply(null,args);
                        if (result === args[0]) {
                            result = original;
                        }
                        return result;
                    }
                    // TODO: too tightly coupled into palette UI
                }

                def.type = nt;
                nodeDefinitions[nt] = def;

                if (def.defaults) {
                    for (var d in def.defaults) {
                        if (def.defaults.hasOwnProperty(d)) {
                            if (def.defaults[d].type) {
                                try {
                                    def.defaults[d]._type = parseNodePropertyTypeString(def.defaults[d].type)
                                } catch(err) {
                                    console.warn(err);
                                }
                            }

                            if (internalProperties.includes(d)) {
                                console.warn(`registerType: ${nt}: the property "${d}" is internal and cannot be used.`);
                                delete def.defaults[d];
                            }
                        }
                    }
                }


                RED.events.emit("registry:node-type-added",nt);
            },
            removeNodeType: function(nt) {
                if (nt.substring(0,8) != "subflow:") {
                    // NON-NLS - internal debug message
                    throw new Error("this api is subflow only. called with:",nt);
                }
                delete nodeDefinitions[nt];
                RED.events.emit("registry:node-type-removed",nt);
            },
            getNodeType: function(nt) {
                return nodeDefinitions[nt];
            },
            setIconSets: function(sets) {
                iconSets = sets;
                iconSets["font-awesome"] = RED.nodes.fontAwesome.getIconList();
            },
            getIconSets: function() {
                return iconSets;
            }
        };
        return exports;
    })();

    // allNodes holds information about the Flow nodes.
    var allNodes = (function() {
        // Map node.id -> node
        var nodes = {};
        // Map tab.id -> Array of nodes on that tab
        var tabMap = {};
        // Map tab.id -> Set of dirty object ids on that tab
        var tabDirtyMap = {};
        // Map tab.id -> Set of object ids of things deleted from the tab that weren't otherwise dirty
        var tabDeletedNodesMap = {};
        // Set of object ids of things added to a tab after initial import
        var addedDirtyObjects = new Set()

        function changeCollectionDepth(tabNodes, toMove, direction, singleStep) {
            const result = []
            const moved = new Set();
            const startIndex = direction ? tabNodes.length - 1 : 0
            const endIndex = direction ? -1 : tabNodes.length
            const step = direction ? -1 : 1
            let target = startIndex // Only used for all-the-way moves
            for (let i = startIndex; i != endIndex; i += step) {
                if (toMove.size === 0) {
                    break;
                }
                const n = tabNodes[i]
                if (toMove.has(n)) {
                    if (singleStep) {
                        if (i !== startIndex && !moved.has(tabNodes[i - step])) {
                            tabNodes.splice(i, 1)
                            tabNodes.splice(i - step, 0, n)
                            n._reordered = true
                            result.push(n)
                        }
                    } else {
                        if (i !== target) {
                            tabNodes.splice(i, 1)
                            tabNodes.splice(target, 0, n)
                            n._reordered = true
                            result.push(n)
                        }
                        target += step
                    }
                    toMove.delete(n);
                    moved.add(n);
                }
            }
            return result
        }

        var api = {
            addTab: function(id) {
                tabMap[id] = [];
                tabDirtyMap[id] = new Set();
                tabDeletedNodesMap[id] = new Set();
            },
            hasTab: function(z) {
                return tabMap.hasOwnProperty(z)
            },
            removeTab: function(id) {
                delete tabMap[id];
                delete tabDirtyMap[id];
                delete tabDeletedNodesMap[id];
            },
            addNode: function(n) {
                nodes[n.id] = n;
                if (tabMap.hasOwnProperty(n.z)) {
                    tabMap[n.z].push(n);
                    api.addObjectToWorkspace(n.z, n.id, n.changed || n.moved)
                } else {
                    console.warn("Node added to unknown tab/subflow:",n);
                    tabMap["_"] = tabMap["_"] || [];
                    tabMap["_"].push(n);
                }
            },
            removeNode: function(n) {
                delete nodes[n.id]
                if (tabMap.hasOwnProperty(n.z)) {
                    var i = tabMap[n.z].indexOf(n);
                    if (i > -1) {
                        tabMap[n.z].splice(i,1);
                    }
                    api.removeObjectFromWorkspace(n.z, n.id)
                }
            },
            /**
             * Add an object to our dirty/clean tracking state
             * @param {String} z 
             * @param {String} id 
             * @param {Boolean} isDirty 
             */
            addObjectToWorkspace: function (z, id, isDirty) {
                if (isDirty) {
                    addedDirtyObjects.add(id)
                }
                if (tabDeletedNodesMap[z].has(id)) {
                    tabDeletedNodesMap[z].delete(id)
                }
                api.markNodeDirty(z, id, isDirty)
            },
            /**
             * Remove an object from our dirty/clean tracking state
             * @param {String} z 
             * @param {String} id 
             */
            removeObjectFromWorkspace: function (z, id) {
                if (!addedDirtyObjects.has(id)) {
                    tabDeletedNodesMap[z].add(id)
                } else {
                    addedDirtyObjects.delete(id)
                }
                api.markNodeDirty(z, id, false)
            },
            hasNode: function(id) {
                return nodes.hasOwnProperty(id);
            },
            getNode: function(id) {
                return nodes[id]
            },
            moveNode: function(n, newZ) {
                api.removeNode(n);
                n.z = newZ;
                api.addNode(n)
            },
            /**
             * @param {array} nodes 
             * @param {boolean} direction true:forwards false:back
             * @param {boolean} singleStep true:single-step false:all-the-way
             */
            changeDepth: function(nodes, direction, singleStep) {
                if (!Array.isArray(nodes)) {
                    nodes = [nodes]
                }
                let result = []
                const tabNodes = tabMap[nodes[0].z];
                const toMove = new Set(nodes.filter(function(n) { return n.type !== "group" && n.type !== "subflow" }));
                if (toMove.size > 0) {
                    result = result.concat(changeCollectionDepth(tabNodes, toMove, direction, singleStep))
                    if (result.length > 0) {
                        RED.events.emit('nodes:reorder',{
                            z: nodes[0].z,
                            nodes: result
                        });
                    }
                }

                const groupNodes = groupsByZ[nodes[0].z] || []
                const groupsToMove = new Set(nodes.filter(function(n) { return n.type === 'group'}))
                if (groupsToMove.size > 0) {
                    const groupResult = changeCollectionDepth(groupNodes, groupsToMove, direction, singleStep)
                    if (groupResult.length > 0) {
                        result = result.concat(groupResult)
                        RED.events.emit('groups:reorder',{
                            z: nodes[0].z,
                            nodes: groupResult
                        });
                    }
                }
                RED.view.redraw(true)
                return result
            },
            moveNodesForwards: function(nodes) {
                return api.changeDepth(nodes, true, true)
            },
            moveNodesBackwards: function(nodes) {
                return api.changeDepth(nodes, false, true)
            },
            moveNodesToFront: function(nodes) {
                return api.changeDepth(nodes, true, false)
            },
            moveNodesToBack: function(nodes) {
                return api.changeDepth(nodes, false, false)
            },
            getNodes: function(z) {
                return tabMap[z];
            },
            clear: function() {
                nodes = {};
                tabMap = {};
                tabDirtyMap = {};
                tabDeletedNodesMap = {};
                addedDirtyObjects = new Set();
            },
            /**
             * Clear all internal state on what is dirty.
             */
            clearState: function () {
                // Called when a deploy happens, we can forget about added/remove
                // items as they have now been deployed.
                addedDirtyObjects = new Set()
                const flowsToCheck = new Set()
                for (const [z, set] of Object.entries(tabDeletedNodesMap)) {
                    if (set.size > 0) {
                        set.clear()
                        flowsToCheck.add(z)
                    }
                }
                for (const [z, set] of Object.entries(tabDirtyMap)) {
                    if (set.size > 0) {
                        set.clear()
                        flowsToCheck.add(z)
                    }
                }
                for (const z of flowsToCheck) {
                    api.checkTabState(z)
                }
            },
            eachNode: function(cb) {
                var nodeList,i,j;
                for (i in subflows) {
                    if (subflows.hasOwnProperty(i)) {
                        nodeList = tabMap[i];
                        for (j = 0; j < nodeList.length; j++) {
                            if (cb(nodeList[j]) === false) {
                                return;
                            }
                        }
                    }
                }
                for (i = 0; i < workspacesOrder.length; i++) {
                    nodeList = tabMap[workspacesOrder[i]];
                    for (j = 0; j < nodeList.length; j++) {
                        if (cb(nodeList[j]) === false) {
                            return;
                        }
                    }
                }
                // Flow nodes that do not have a valid tab/subflow
                if (tabMap["_"]) {
                    nodeList = tabMap["_"];
                    for (j = 0; j < nodeList.length; j++) {
                        if (cb(nodeList[j]) === false) {
                            return;
                        }
                    }
                }
            },
            filterNodes: function(filter) {
                var result = [];
                var searchSet = null;
                var doZFilter = false;
                if (filter.hasOwnProperty("z")) {
                    if (tabMap.hasOwnProperty(filter.z)) {
                        searchSet = tabMap[filter.z];
                    } else {
                        doZFilter = true;
                    }
                }
                var objectLookup = false;
                if (searchSet === null) {
                    searchSet = Object.keys(nodes);
                    objectLookup = true;
                }


                for (var n=0;n<searchSet.length;n++) {
                    var node = searchSet[n];
                    if (objectLookup) {
                        node = nodes[node];
                    }
                    if (filter.hasOwnProperty("type") && node.type !== filter.type) {
                        continue;
                    }
                    if (doZFilter && node.z !== filter.z) {
                        continue;
                    }
                    result.push(node);
                }
                return result;
            },
            getNodeOrder: function(z) {
                return (groupsByZ[z] || []).concat(tabMap[z]).map(n => n.id)
            },
            setNodeOrder: function(z, order) {
                var orderMap = {};
                order.forEach(function(id,i) {
                    orderMap[id] = i;
                })
                tabMap[z].sort(function(A,B) {
                    A._reordered = true;
                    B._reordered = true;
                    return orderMap[A.id] - orderMap[B.id];
                })
                if (groupsByZ[z]) {
                    groupsByZ[z].sort(function(A,B) {
                        return orderMap[A.id] - orderMap[B.id];
                    })
                }
            },
            /**
             * Update our records if an object is dirty or not
             * @param {String} z tab id
             * @param {String} id object id
             * @param {Boolean} dirty whether the object is dirty or not
             */
            markNodeDirty: function(z, id, dirty) {
                if (tabDirtyMap[z]) {
                    if (dirty) {
                        tabDirtyMap[z].add(id)
                    } else {
                        tabDirtyMap[z].delete(id)
                    }
                    api.checkTabState(z)
                }
            },
            /**
             * Check if a tab should update its contentsChange flag
             * @param {String} z tab id
             */
            checkTabState: function (z) {
                const ws = workspaces[z] || subflows[z]
                if (ws) {
                    const contentsChanged = tabDirtyMap[z].size > 0 || tabDeletedNodesMap[z].size > 0
                    if (Boolean(ws.contentsChanged) !== contentsChanged) {
                        ws.contentsChanged = contentsChanged
                        if (ws.type === 'tab') {
                            RED.events.emit("flows:change", ws);
                        } else {
                            RED.events.emit("subflows:change", ws);
                        }
                    }
                }
            }
        }
        return api;
    })()

    function getID() {
        var bytes = [];
        for (var i=0;i<8;i++) {
            bytes.push(Math.round(0xff*Math.random()).toString(16).padStart(2,'0'));
        }
        return bytes.join("");
    }

    function parseNodePropertyTypeString(typeString) {
        typeString = typeString.trim();
        var c;
        var pos = 0;
        var isArray = /\[\]$/.test(typeString);
        if (isArray) {
            typeString = typeString.substring(0,typeString.length-2);
        }

        var l = typeString.length;
        var inBrackets = false;
        var inToken = false;
        var currentToken = "";
        var types = [];
        while (pos < l) {
            c = typeString[pos];
            if (inToken) {
                if (c === "|") {
                    types.push(currentToken.trim())
                    currentToken = "";
                    inToken = false;
                } else if (c === ")") {
                    types.push(currentToken.trim())
                    currentToken = "";
                    inBrackets = false;
                    inToken = false;
                } else {
                    currentToken += c;
                }
            } else {
                if (c === "(") {
                    if (inBrackets) {
                        throw new Error("Invalid character '"+c+"' at position "+pos)
                    }
                    inBrackets = true;
                } else if (c !== " ") {
                    inToken = true;
                    currentToken = c;
                }
            }
            pos++;
        }
        currentToken = currentToken.trim();
        if (currentToken.length > 0) {
            types.push(currentToken)
        }
        return {
            types: types,
            array: isArray
        }
    }

    const nodeProxyHandler = {
        get(node, prop) {
            if (prop === '__isProxy__') {
                return true
            } else if (prop == '__node__') {
                return node
            }
            return node[prop]
        },
        set(node, prop, value) {
            if (node.z && (RED.nodes.workspace(node.z)?.locked || RED.nodes.subflow(node.z)?.locked)) {
                if (
                    node._def.defaults[prop] ||
                    prop === 'z' ||
                    prop === 'l' ||
                    prop === 'd' ||
                    (prop === 'changed' && (!!node.changed) !== (!!value)) || // jshint ignore:line
                    ((prop === 'x' || prop === 'y') && !node.resize && node.type !== 'group')
                ) {
                    throw new Error(`Cannot modified property '${prop}' of locked object '${node.type}:${node.id}'`)
                }
            }
            if (node.z && (prop === 'changed' || prop === 'moved')) {
                setTimeout(() => {
                    allNodes.markNodeDirty(node.z, node.id, node.changed || node.moved)
                }, 0)
            }
            node[prop] = value;
            return true
        }
    }

    function addNode(n, opt) {
        let newNode
        if (!n.__isProxy__) {
            newNode = new Proxy(n, nodeProxyHandler)
        } else {
            newNode = n
        }

        if (n.type.indexOf("subflow") !== 0) {
            n["_"] = n._def._;
        } else {
            var subflowId = n.type.substring(8);
            var sf = RED.nodes.subflow(subflowId);
            if (sf) {
                sf.instances.push(newNode);
            }
            n["_"] = RED._;
        }

        // Both node and config node can use a config node
        updateConfigNodeUsers(newNode, { action: "add" });

        if (n._def.category == "config") {
            configNodes[n.id] = newNode;
        } else {
            if (n.wires && (n.wires.length > n.outputs)) { n.outputs = n.wires.length; }
            n.dirty = true;
            if (n._def.category == "subflows" && typeof n.i === "undefined") {
                var nextId = 0;
                RED.nodes.eachNode(function(node) {
                    nextId = Math.max(nextId,node.i||0);
                });
                n.i = nextId+1;
            }
            allNodes.addNode(newNode);
            if (!nodeLinks[n.id]) {
                nodeLinks[n.id] = {in:[],out:[]};
            }
        }
        RED.events.emit('nodes:add',newNode, opt);
        return newNode
    }
    function addLink(l) {
        if (nodeLinks[l.source.id]) {
            const isUnique = nodeLinks[l.source.id].out.every(function(link) {
                return link.sourcePort !== l.sourcePort || link.target.id !== l.target.id
            })
            if (!isUnique) {
                return
            }
        }
        links.push(l);
        if (l.source) {
            // Possible the node hasn't been added yet
            if (!nodeLinks[l.source.id]) {
                nodeLinks[l.source.id] = {in:[],out:[]};
            }
            nodeLinks[l.source.id].out.push(l);
        }
        if (l.target) {
            if (!nodeLinks[l.target.id]) {
                nodeLinks[l.target.id] = {in:[],out:[]};
            }
            nodeLinks[l.target.id].in.push(l);
        }
        if (l.source.z === l.target.z && linkTabMap[l.source.z]) {
            linkTabMap[l.source.z].push(l);
            allNodes.addObjectToWorkspace(l.source.z, getLinkId(l), true)
        }
        RED.events.emit("links:add",l);
    }

    function getLinkId(link) {
        return link.source.id + ':' + link.sourcePort + ':' + link.target.id
    }


    function getNode(id) {
        if (id in configNodes) {
            return configNodes[id];
        }
        return allNodes.getNode(id);
    }

    function removeNode(id) {
        var removedLinks = [];
        var removedNodes = [];
        var node;

        if (id in configNodes) {
            node = configNodes[id];
            delete configNodes[id];
            updateConfigNodeUsers(node, { action: "remove" });
            RED.events.emit('nodes:remove',node);
            RED.workspaces.refresh();
        } else if (allNodes.hasNode(id)) {
            node = allNodes.getNode(id);
            allNodes.removeNode(node);
            delete nodeLinks[id];
            removedLinks = links.filter(function(l) { return (l.source === node) || (l.target === node); });
            removedLinks.forEach(removeLink);
            updateConfigNodeUsers(node, { action: "remove" });

            // TODO: Legacy code for exclusive config node
            var updatedConfigNode = false;
            for (var d in node._def.defaults) {
                if (node._def.defaults.hasOwnProperty(d)) {
                    var property = node._def.defaults[d];
                    if (property.type) {
                        var type = registry.getNodeType(property.type);
                        if (type && type.category == "config") {
                            var configNode = configNodes[node[d]];
                            if (configNode) {
                                updatedConfigNode = true;
                                if (configNode._def.exclusive) {
                                    removeNode(node[d]);
                                    removedNodes.push(configNode);
                                }
                            }
                        }
                    }
                }
            }

            if (node.type.indexOf("subflow:") === 0) {
                var subflowId = node.type.substring(8);
                var sf = RED.nodes.subflow(subflowId);
                if (sf) {
                    sf.instances.splice(sf.instances.indexOf(node),1);
                }
            }

            if (updatedConfigNode) {
                RED.workspaces.refresh();
            }
            try {
                if (node._def.oneditdelete) {
                    node._def.oneditdelete.call(node);
                }
            } catch(err) {
                console.log("oneditdelete",node.id,node.type,err.toString());
            }
            RED.events.emit('nodes:remove',node);
        }



        if (node && node._def.onremove) {
            // Deprecated: never documented but used by some early nodes
            console.log("Deprecated API warning: node type ",node.type," has an onremove function - should be oneditdelete - please report");
            node._def.onremove.call(node);
        }
        return {links:removedLinks,nodes:removedNodes};
    }

    function moveNodesForwards(nodes) {
        return allNodes.moveNodesForwards(nodes);
    }
    function moveNodesBackwards(nodes) {
        return allNodes.moveNodesBackwards(nodes);
    }
    function moveNodesToFront(nodes) {
        return allNodes.moveNodesToFront(nodes);
    }
    function moveNodesToBack(nodes) {
        return allNodes.moveNodesToBack(nodes);
    }

    function getNodeOrder(z) {
        return allNodes.getNodeOrder(z);
    }
    function setNodeOrder(z, order) {
        allNodes.setNodeOrder(z,order);
    }

    function moveNodeToTab(node, z) {
        if (node.type === "group") {
            moveGroupToTab(node,z);
            return;
        }
        if (node.type === "junction") {
            moveJunctionToTab(node,z);
            return;
        }
        var oldZ = node.z;
        allNodes.moveNode(node,z);
        var nl = nodeLinks[node.id];
        if (nl) {
            nl.in.forEach(function(l) {
                var idx = linkTabMap[oldZ].indexOf(l);
                if (idx != -1) {
                    linkTabMap[oldZ].splice(idx, 1);
                }
                if ((l.source.z === z) && linkTabMap[z]) {
                    linkTabMap[z].push(l);
                }
            });
            nl.out.forEach(function(l) {
                var idx = linkTabMap[oldZ].indexOf(l);
                if (idx != -1) {
                    linkTabMap[oldZ].splice(idx, 1);
                }
                if ((l.target.z === z) && linkTabMap[z]) {
                    linkTabMap[z].push(l);
                }
            });
        }
        RED.events.emit("nodes:change",node);
    }
    function moveGroupToTab(group, z) {
        var index = groupsByZ[group.z].indexOf(group);
        groupsByZ[group.z].splice(index,1);
        groupsByZ[z] = groupsByZ[z] || [];
        groupsByZ[z].push(group);
        group.z = z;
        RED.events.emit("groups:change",group);
    }

    function moveJunctionToTab(junction, z) {
        var index = junctionsByZ[junction.z].indexOf(junction);
        junctionsByZ[junction.z].splice(index,1);
        junctionsByZ[z] = junctionsByZ[z] || [];
        junctionsByZ[z].push(junction);

        var oldZ = junction.z;
        junction.z = z;

        var nl = nodeLinks[junction.id];
        if (nl) {
            nl.in.forEach(function(l) {
                var idx = linkTabMap[oldZ].indexOf(l);
                if (idx != -1) {
                    linkTabMap[oldZ].splice(idx, 1);
                }
                if ((l.source.z === z) && linkTabMap[z]) {
                    linkTabMap[z].push(l);
                }
            });
            nl.out.forEach(function(l) {
                var idx = linkTabMap[oldZ].indexOf(l);
                if (idx != -1) {
                    linkTabMap[oldZ].splice(idx, 1);
                }
                if ((l.target.z === z) && linkTabMap[z]) {
                    linkTabMap[z].push(l);
                }
            });
        }
        RED.events.emit("junctions:change",junction);
    }

    function removeLink(l) {
        var index = links.indexOf(l);
        if (index != -1) {
            links.splice(index,1);
            if (l.source && nodeLinks[l.source.id]) {
                var sIndex = nodeLinks[l.source.id].out.indexOf(l)
                if (sIndex !== -1) {
                    nodeLinks[l.source.id].out.splice(sIndex,1)
                }
            }
            if (l.target && nodeLinks[l.target.id]) {
                var tIndex = nodeLinks[l.target.id].in.indexOf(l)
                if (tIndex !== -1) {
                    nodeLinks[l.target.id].in.splice(tIndex,1)
                }
            }
            if (l.source.z === l.target.z && linkTabMap[l.source.z]) {
                index = linkTabMap[l.source.z].indexOf(l);
                if (index !== -1) {
                    linkTabMap[l.source.z].splice(index,1)
                }
                allNodes.removeObjectFromWorkspace(l.source.z, getLinkId(l))
            }
        }
        RED.events.emit("links:remove",l);
    }

    function addWorkspace(ws,targetIndex) {
        workspaces[ws.id] = ws;
        allNodes.addTab(ws.id);
        linkTabMap[ws.id] = [];

        ws._def = RED.nodes.getType('tab');
        if (targetIndex === undefined) {
            workspacesOrder.push(ws.id);
        } else {
            workspacesOrder.splice(targetIndex,0,ws.id);
        }
        RED.events.emit('flows:add',ws);
        if (targetIndex !== undefined) {
            RED.events.emit('flows:reorder',workspacesOrder)
        }
    }
    function getWorkspace(id) {
        return workspaces[id];
    }
    function removeWorkspace(id) {
        var ws = workspaces[id];
        var removedNodes = [];
        var removedLinks = [];
        var removedGroups = [];
        var removedJunctions = [];
        if (ws) {
            delete workspaces[id];
            delete linkTabMap[id];
            workspacesOrder.splice(workspacesOrder.indexOf(id),1);
            var i;
            var node;

            if (allNodes.hasTab(id)) {
                removedNodes = allNodes.getNodes(id).slice()
            }
            for (i in configNodes) {
                if (configNodes.hasOwnProperty(i)) {
                    node = configNodes[i];
                    if (node.z == id) {
                        removedNodes.push(node);
                    }
                }
            }
            removedJunctions = RED.nodes.junctions(id)

            for (i=0;i<removedNodes.length;i++) {
                var result = removeNode(removedNodes[i].id);
                removedLinks = removedLinks.concat(result.links);
            }
            for (i=0;i<removedJunctions.length;i++) {
                var result = removeJunction(removedJunctions[i])
                removedLinks = removedLinks.concat(result.links)
            }

            // Must get 'removedGroups' in the right order.
            //  - start with the top-most groups
            //  - then recurse into them
            removedGroups = (groupsByZ[id] || []).filter(function(g) { return !g.g; });
            for (i=0;i<removedGroups.length;i++) {
                removedGroups[i].nodes.forEach(function(n) {
                    if (n.type === "group") {
                        removedGroups.push(n);
                    }
                });
            }
            // Now remove them in the reverse order
            for (i=removedGroups.length-1; i>=0; i--) {
                removeGroup(removedGroups[i]);
            }
            allNodes.removeTab(id);
            RED.events.emit('flows:remove',ws);
        }
        return {nodes:removedNodes,links:removedLinks, groups: removedGroups, junctions: removedJunctions};
    }

    /**
      * Add a Subflow to the Workspace
      *
      * @param {object} sf The Subflow to add.
      * @param {boolean|undefined} createNewIds Whether to update the name.
      */
    function addSubflow(sf, createNewIds) {
        if (createNewIds) {
            // Update the Subflow name to highlight that this is a copy
            const subflowNames = Object.keys(subflows).map(function (sfid) {
                return subflows[sfid].name || "";
            })
            subflowNames.sort()

            let copyNumber = 1;
            let subflowName = sf.name;
            subflowNames.forEach(function(name) {
                if (subflowName == name) {
                    subflowName = sf.name + " (" + copyNumber + ")";
                    copyNumber++;
                }
            });

            sf.name = subflowName;
        }

        sf.instances = [];

        subflows[sf.id] = sf;
        allNodes.addTab(sf.id);
        linkTabMap[sf.id] = [];

        RED.nodes.registerType("subflow:"+sf.id, {
            defaults:{
                name:{value:""},
                env:{value:[], validate: function(value) {
                    const errors = []
                    if (value) {
                        value.forEach(env => {
                            const r = RED.utils.validateTypedProperty(env.value, env.type)
                            if (r !== true) {
                                errors.push(env.name+': '+r)
                            }
                        })
                    }
                    if (errors.length === 0) {
                        return true
                    } else {
                        return errors
                    }
                }}
            },
            icon: function() { return sf.icon||"subflow.svg" },
            category: sf.category || "subflows",
            inputs: sf.in.length,
            outputs: sf.out.length,
            color: sf.color || "#DDAA99",
            label: function() { return this.name||RED.nodes.subflow(sf.id).name },
            labelStyle: function() { return this.name?"red-ui-flow-node-label-italic":""; },
            paletteLabel: function() { return RED.nodes.subflow(sf.id).name },
            inputLabels: function(i) { return sf.inputLabels?sf.inputLabels[i]:null },
            outputLabels: function(i) { return sf.outputLabels?sf.outputLabels[i]:null },
            oneditprepare: function() {
                if (this.type !== 'subflow') {
                    // A subflow instance node
                    RED.subflow.buildEditForm("subflow",this);
                } else {
                    // A subflow template node
                    RED.subflow.buildEditForm("subflow-template", this);
                }
            },
            oneditresize: function(size) {
                if (this.type === 'subflow') {
                    $("#node-input-env-container").editableList('height',size.height - 80);
                }
            },
            set:{
                module: "node-red"
            }
        });

        sf._def = RED.nodes.getType("subflow:"+sf.id);
        RED.events.emit("subflows:add",sf);
    }
    function getSubflow(id) {
        return subflows[id];
    }
    function removeSubflow(sf) {
        if (subflows[sf.id]) {
            delete subflows[sf.id];
            allNodes.removeTab(sf.id);
            registry.removeNodeType("subflow:"+sf.id);
            RED.events.emit("subflows:remove",sf);
        }
    }

    function subflowContains(sfid,nodeid) {
        var sfNodes = allNodes.getNodes(sfid);
        for (var i = 0; i<sfNodes.length; i++) {
            var node = sfNodes[i];
            var m = /^subflow:(.+)$/.exec(node.type);
            if (m) {
                if (m[1] === nodeid) {
                    return true;
                } else {
                    var result = subflowContains(m[1],nodeid);
                    if (result) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    function getDownstreamNodes(node) {
        const downstreamLinks = nodeLinks[node.id].out
        const downstreamNodes = new Set(downstreamLinks.map(l => l.target))
        return Array.from(downstreamNodes)
    }
    function getAllDownstreamNodes(node) {
        return getAllFlowNodes(node,'down').filter(function(n) { return n !== node });
    }
    function getAllUpstreamNodes(node) {
        return getAllFlowNodes(node,'up').filter(function(n) { return n !== node });
    }
    function getAllFlowNodes(node, direction) {
        var selection = RED.view.selection();
        var visited = new Set();
        var nodes = [node];
        var initialNode = true;
        while(nodes.length > 0) {
            var n = nodes.shift();
            visited.add(n);
            var links = [];
            if (!initialNode || !direction || (initialNode && direction === 'up')) {
                links = links.concat(nodeLinks[n.id].in);
            }
            if (!initialNode || !direction || (initialNode && direction === 'down')) {
                links = links.concat(nodeLinks[n.id].out);
            }
            initialNode = false;
            links.forEach(function(l) {
                if (!visited.has(l.source)) {
                    nodes.push(l.source);
                }
                if (!visited.has(l.target)) {
                    nodes.push(l.target);
                }
            })
        }
        return Array.from(visited);
    }


    function convertWorkspace(n,opts) {
        var exportCreds = true;
        if (opts) {
            if (opts.hasOwnProperty("credentials")) {
                exportCreds = opts.credentials;
            }
        }
        var node = {};
        node.id = n.id;
        node.type = n.type;
        for (var d in n._def.defaults) {
            if (n._def.defaults.hasOwnProperty(d)) {
                if (d === 'locked' && !n.locked) {
                    continue
                }
                node[d] = n[d];
            }
        }
        if (exportCreds) {
            var credentialSet = {};
            if (n.credentials) {
                for (var tabCred in n.credentials) {
                    if (n.credentials.hasOwnProperty(tabCred)) {
                        if (!n.credentials._ ||
                            n.credentials["has_"+tabCred] != n.credentials._["has_"+tabCred] ||
                            (n.credentials["has_"+tabCred] && n.credentials[tabCred])) {
                            credentialSet[tabCred] = n.credentials[tabCred];
                        }
                    }
                }
                if (Object.keys(credentialSet).length > 0) {
                    node.credentials = credentialSet;
                }
            }
        }
        return node;
    }
    /**
     * Converts a node to an exportable JSON Object
     **/
    function convertNode(n, opts) {
        var exportCreds = true;
        var exportDimensions = false;
        if (opts === false) {
            exportCreds = false;
        } else if (typeof opts === "object") {
            if (opts.hasOwnProperty("credentials")) {
                exportCreds = opts.credentials;
            }
            if (opts.hasOwnProperty("dimensions")) {
                exportDimensions = opts.dimensions;
            }
        }

        if (n.type === 'tab') {
            return convertWorkspace(n, { credentials: exportCreds });
        }
        var node = {};
        node.id = n.id;
        node.type = n.type;
        node.z = n.z;
        if (node.z === 0 || node.z === "") {
            delete node.z;
        }
        if (n.d === true) {
            node.d = true;
        }
        if (n.g) {
            node.g = n.g;
        }
        if (node.type == "unknown") {
            for (var p in n._orig) {
                if (n._orig.hasOwnProperty(p)) {
                    node[p] = n._orig[p];
                }
            }
        } else {
            for (var d in n._def.defaults) {
                if (n._def.defaults.hasOwnProperty(d)) {
                    node[d] = n[d];
                }
            }
            if (exportCreds) {
                var credentialSet = {};
                if ((/^subflow:/.test(node.type) ||
                     (node.type === "group")) &&
                    n.credentials) {
                    // A subflow instance/group node can have arbitrary creds
                    for (var sfCred in n.credentials) {
                        if (n.credentials.hasOwnProperty(sfCred)) {
                            if (!n.credentials._ ||
                                n.credentials["has_"+sfCred] != n.credentials._["has_"+sfCred] ||
                                (n.credentials["has_"+sfCred] && n.credentials[sfCred])) {
                                credentialSet[sfCred] = n.credentials[sfCred];
                            }
                        }
                    }
                } else if (n.credentials) {
                    // All other nodes have a well-defined list of possible credentials
                    for (var cred in n._def.credentials) {
                        if (n._def.credentials.hasOwnProperty(cred)) {
                            if (n._def.credentials[cred].type == 'password') {
                                if (!n.credentials._ ||
                                    n.credentials["has_"+cred] != n.credentials._["has_"+cred] ||
                                    (n.credentials["has_"+cred] && n.credentials[cred])) {
                                    credentialSet[cred] = n.credentials[cred];
                                }
                            } else if (n.credentials[cred] != null && (!n.credentials._ || n.credentials[cred] != n.credentials._[cred])) {
                                credentialSet[cred] = n.credentials[cred];
                            }
                        }
                    }
                }
                if (Object.keys(credentialSet).length > 0) {
                    node.credentials = credentialSet;
                }
            }
        }
        if (n.type === "group") {
            node.x = n.x;
            node.y = n.y;
            node.w = n.w;
            node.h = n.h;
            // In 1.1.0, we have seen an instance of this array containing `undefined`
            // Until we know how that can happen, add a filter here to remove them
            node.nodes = node.nodes.filter(function(n) { return !!n }).map(function(n) { return n.id });
        }
        if (n.type === "tab" || n.type === "group") {
            if (node.env && node.env.length === 0) {
                delete node.env;
            }
        }
        if (n._def.category != "config" || n.type === 'junction') {
            node.x = n.x;
            node.y = n.y;
            if (exportDimensions) {
                if (!n.hasOwnProperty('w')) {
                    // This node has not yet been drawn in the view. So we need
                    // to explicitly calculate its dimensions. Store the result
                    // on the node as if it had been drawn will save us doing
                    // it again
                    var dimensions = RED.view.calculateNodeDimensions(n);
                    n.w = dimensions[0];
                    n.h = dimensions[1];
                }
                node.w = n.w;
                node.h = n.h;
            }
            node.wires = [];
            for(var i=0;i<n.outputs;i++) {
                node.wires.push([]);
            }
            var wires = links.filter(function(d){return d.source === n;});
            for (var j=0;j<wires.length;j++) {
                var w = wires[j];
                if (w.target.type != "subflow") {
                    if (w.sourcePort < node.wires.length) {
                        node.wires[w.sourcePort].push(w.target.id);
                    }
                }
            }

            if (n.inputs > 0 && n.inputLabels && !/^\s*$/.test(n.inputLabels.join("")))  {
                node.inputLabels = n.inputLabels.slice();
            }
            if (n.outputs > 0 && n.outputLabels && !/^\s*$/.test(n.outputLabels.join(""))) {
                node.outputLabels = n.outputLabels.slice();
            }
            if ((!n._def.defaults || !n._def.defaults.hasOwnProperty("icon")) && n.icon) {
                var defIcon = RED.utils.getDefaultNodeIcon(n._def, n);
                if (n.icon !== defIcon.module+"/"+defIcon.file) {
                    node.icon = n.icon;
                }
            }
            if ((!n._def.defaults || !n._def.defaults.hasOwnProperty("l")) && n.hasOwnProperty('l')) {
                var showLabel = n._def.hasOwnProperty("showLabel")?n._def.showLabel:true;
                if (showLabel != n.l) {
                    node.l = n.l;
                }
            }
        }
        if (n.info) {
            node.info = n.info;
        }
        return node;
    }

    function convertSubflow(n, opts) {
        var exportCreds = true;
        var exportDimensions = false;
        if (opts === false) {
            exportCreds = false;
        } else if (typeof opts === "object") {
            if (opts.hasOwnProperty("credentials")) {
                exportCreds = opts.credentials;
            }
            if (opts.hasOwnProperty("dimensions")) {
                exportDimensions = opts.dimensions;
            }
        }


        var node = {};
        node.id = n.id;
        node.type = n.type;
        node.name = n.name;
        node.info = n.info;
        node.category = n.category;
        node.in = [];
        node.out = [];
        node.env = n.env;
        node.meta = n.meta;

        if (exportCreds) {
            var credentialSet = {};
            // A subflow node can have arbitrary creds
            for (var sfCred in n.credentials) {
                if (n.credentials.hasOwnProperty(sfCred)) {
                    if (!n.credentials._ ||
                        n.credentials["has_"+sfCred] != n.credentials._["has_"+sfCred] ||
                        (n.credentials["has_"+sfCred] && n.credentials[sfCred])) {
                        credentialSet[sfCred] = n.credentials[sfCred];
                    }
                }
            }
            if (Object.keys(credentialSet).length > 0) {
                node.credentials = credentialSet;
            }
        }

        node.color = n.color;

        n.in.forEach(function(p) {
            var nIn = {x:p.x,y:p.y,wires:[]};
            var wires = links.filter(function(d) { return d.source === p });
            for (var i=0;i<wires.length;i++) {
                var w = wires[i];
                if (w.target.type != "subflow") {
                    nIn.wires.push({id:w.target.id})
                }
            }
            node.in.push(nIn);
        });
        n.out.forEach(function(p,c) {
            var nOut = {x:p.x,y:p.y,wires:[]};
            var wires = links.filter(function(d) { return d.target === p });
            for (i=0;i<wires.length;i++) {
                if (wires[i].source.type != "subflow") {
                    nOut.wires.push({id:wires[i].source.id,port:wires[i].sourcePort})
                } else {
                    nOut.wires.push({id:n.id,port:0})
                }
            }
            node.out.push(nOut);
        });

        if (node.in.length > 0 && n.inputLabels && !/^\s*$/.test(n.inputLabels.join("")))  {
            node.inputLabels = n.inputLabels.slice();
        }
        if (node.out.length > 0 && n.outputLabels && !/^\s*$/.test(n.outputLabels.join(""))) {
            node.outputLabels = n.outputLabels.slice();
        }
        if (n.icon) {
            if (n.icon !== "node-red/subflow.svg") {
                node.icon = n.icon;
            }
        }
        if (n.status) {
            node.status = {x: n.status.x, y: n.status.y, wires:[]};
            links.forEach(function(d) {
                if (d.target === n.status) {
                    if (d.source.type != "subflow") {
                        node.status.wires.push({id:d.source.id, port:d.sourcePort})
                    } else {
                        node.status.wires.push({id:n.id, port:0})
                    }
                }
            });
        }

        return node;
    }

    function createExportableSubflow(id) {
        var sf = getSubflow(id);
        var nodeSet;
        var sfNodes = allNodes.getNodes(sf.id);
        if (sfNodes) {
            nodeSet = sfNodes.slice();
            nodeSet.unshift(sf);
        } else {
            nodeSet = [sf];
        }
        return createExportableNodeSet(nodeSet);
    }
    /**
     * Converts the current node selection to an exportable JSON Object
     * @param {Array<Node>} set the node selection to export
     * @param {Object} options
     * @param {Record<string, boolean>} [options.exportedIds]
     * @param {Record<string, boolean>} [options.exportedSubflows]
     * @param {Record<string, boolean>} [options.exportedConfigNodes]
     * @param {boolean} [options.includeModuleConfig]
     * @returns {Array<Node>}
     */
    function createExportableNodeSet(set, {
        exportedIds,
        exportedSubflows,
        exportedConfigNodes,
        includeModuleConfig = false
    } = {}) {
        var nns = [];

        exportedIds = exportedIds || {};
        set = set.filter(function(n) {
            if (exportedIds[n.id]) {
                return false;
            }
            exportedIds[n.id] = true;
            return true;
        })

        exportedConfigNodes = exportedConfigNodes || {};
        exportedSubflows = exportedSubflows || {};
        for (var n=0;n<set.length;n++) {
            var node = set[n];
            if (node.type.substring(0,8) == "subflow:") {
                var subflowId = node.type.substring(8);
                if (!exportedSubflows[subflowId]) {
                    exportedSubflows[subflowId] = true;
                    var subflow = getSubflow(subflowId);
                    var subflowSet = allNodes.getNodes(subflowId).slice();
                    subflowSet.unshift(subflow);

                    RED.nodes.eachConfig(function(n) {
                        if (n.z == subflowId) {
                            subflowSet.push(n);
                            exportedConfigNodes[n.id] = true;
                        }
                    });

                    subflowSet = subflowSet.concat(RED.nodes.junctions(subflowId))
                    subflowSet = subflowSet.concat(RED.nodes.groups(subflowId))

                    var exportableSubflow = createExportableNodeSet(subflowSet, { exportedIds, exportedSubflows, exportedConfigNodes });
                    nns = exportableSubflow.concat(nns);
                }
            }
            if (node.type !== "subflow") {
                var convertedNode = RED.nodes.convertNode(node, { credentials: false });
                for (var d in node._def.defaults) {
                    if (node._def.defaults[d].type) {
                        var nodeList = node[d];
                        if (!Array.isArray(nodeList)) {
                            nodeList = [nodeList];
                        }
                        nodeList = nodeList.filter(function(id) {
                            if (id in configNodes) {
                                var confNode = configNodes[id];
                                if (confNode._def.exportable !== false) {
                                    if (!(id in exportedConfigNodes)) {
                                        exportedConfigNodes[id] = true;
                                        set.push(confNode);
                                    }
                                    return true;
                                }
                                return false;
                            }
                            return true;
                        })
                        if (nodeList.length === 0) {
                            convertedNode[d] = Array.isArray(node[d])?[]:""
                        } else {
                            convertedNode[d] = Array.isArray(node[d])?nodeList:nodeList[0]
                        }
                    }
                }
                nns.push(convertedNode);
                if (node.type === "group") {
                    nns = nns.concat(createExportableNodeSet(node.nodes, { exportedIds, exportedSubflows, exportedConfigNodes }));
                }
            } else if (!node.direction) {
                // node.direction indicates its a virtual subflow node (input/output/status) that cannot be
                // exported so should be skipped over.
                var convertedSubflow = convertSubflow(node, { credentials: false });
                nns.push(convertedSubflow);
            }
        }
        if (includeModuleConfig) {
            updateGlobalConfigModuleList(nns)
        }
        return nns;
    }

    /**
     * Converts the current configuration to an exportable JSON Object
     * @param {object} opts
     * @param {boolean} [opts.credentials] whether to include (known) credentials. Default `true`.
     * @param {boolean} [opts.dimensions] whether to include node dimensions. Default `false`.
     * @param {boolean} [opts.includeModuleConfig] whether to include modules. Default `false`.
     * @returns {Array<object>}
     */
    function createCompleteNodeSet(opts) {
        var nns = [];
        var i;
        for (i=0;i<workspacesOrder.length;i++) {
            if (workspaces[workspacesOrder[i]].type == "tab") {
                nns.push(convertWorkspace(workspaces[workspacesOrder[i]], opts));
            }
        }
        for (i in subflows) {
            if (subflows.hasOwnProperty(i)) {
                nns.push(convertSubflow(subflows[i], opts));
            }
        }
        for (i in groups) {
            if (groups.hasOwnProperty(i)) {
                nns.push(convertNode(groups[i], opts));
            }
        }
        for (i in junctions) {
            if (junctions.hasOwnProperty(i)) {
                nns.push(convertNode(junctions[i], opts));
            }
        }
        for (i in configNodes) {
            if (configNodes.hasOwnProperty(i)) {
                nns.push(convertNode(configNodes[i], opts));
            }
        }
        RED.nodes.eachNode(function(n) {
            nns.push(convertNode(n, opts));
        })
        if (opts?.includeModuleConfig) {
            updateGlobalConfigModuleList(nns);
        }
        return nns;
    }

    function checkForMatchingSubflow(subflow,subflowNodes) {
        subflowNodes = subflowNodes || [];
        var i;
        var match = null;
        RED.nodes.eachSubflow(function(sf) {
            if (sf.name != subflow.name ||
                sf.info != subflow.info ||
                sf.in.length != subflow.in.length ||
                sf.out.length != subflow.out.length) {
                    return;
            }
            var sfNodes = RED.nodes.filterNodes({z:sf.id});
            if (sfNodes.length != subflowNodes.length) {
                return;
            }

            var subflowNodeSet = [subflow].concat(subflowNodes);
            var sfNodeSet = [sf].concat(sfNodes);

            var exportableSubflowNodes = JSON.stringify(subflowNodeSet);
            var exportableSFNodes = JSON.stringify(createExportableNodeSet(sfNodeSet));
            var nodeMap = {};
            for (i=0;i<sfNodes.length;i++) {
                exportableSubflowNodes = exportableSubflowNodes.replace(new RegExp("\""+subflowNodes[i].id+"\"","g"),'"'+sfNodes[i].id+'"');
            }
            exportableSubflowNodes = exportableSubflowNodes.replace(new RegExp("\""+subflow.id+"\"","g"),'"'+sf.id+'"');

            if (exportableSubflowNodes !== exportableSFNodes) {
                return;
            }

            match = sf;
            return false;
        });
        return match;
    }
    function compareNodes(nodeA,nodeB,idMustMatch) {
        if (idMustMatch && nodeA.id != nodeB.id) {
            return false;
        }
        if (nodeA.type != nodeB.type) {
            return false;
        }
        var def = nodeA._def;
        for (var d in def.defaults) {
            if (def.defaults.hasOwnProperty(d)) {
                var vA = nodeA[d];
                var vB = nodeB[d];
                if (typeof vA !== typeof vB) {
                    return false;
                }
                if (vA === null || typeof vA === "string" || typeof vA === "number") {
                    if (vA !== vB) {
                        return false;
                    }
                } else {
                    if (JSON.stringify(vA) !== JSON.stringify(vB)) {
                        return false;
                    }
                }
            }
        }
        return true;
    }

    function identifyImportConflicts(importedNodes) {
        var imported = {
            tabs: {},
            subflows: {},
            groups: {},
            junctions: {},
            configs: {},
            nodes: {},
            all: [],
            conflicted: {},
            zMap: {},
        }

        importedNodes.forEach(function(n) {
            imported.all.push(n);
            if (n.type === "tab") {
                imported.tabs[n.id] = n;
            } else if (n.type === "subflow") {
                imported.subflows[n.id] = n;
            } else if (n.type === "group") {
                imported.groups[n.id] = n;
            } else if (n.type === "junction") {
                imported.junctions[n.id] = n;
            } else if (n.hasOwnProperty("x") && n.hasOwnProperty("y")) {
                imported.nodes[n.id] = n;
            } else {
                imported.configs[n.id] = n;
            }
            var nodeZ = n.z || "__global__";
            imported.zMap[nodeZ] = imported.zMap[nodeZ] || [];
            imported.zMap[nodeZ].push(n)
            if (allNodes.hasNode(n.id) || configNodes[n.id] || workspaces[n.id] || subflows[n.id] || groups[n.id] || junctions[n.id]) {
                imported.conflicted[n.id] = n;
            }
        })
        return imported;

    }

    /**
     * Replace the provided nodes.
     * This must contain complete Subflow defs or complete Flow Tabs.
     * It does not replace an individual node in the middle of a flow.
     */
    function replaceNodes(newNodes) {
        var zMap = {};
        var newSubflows = {};
        var newConfigNodes = {};
        var removedNodes = [];
        // Figure out what we're being asked to replace - subflows/configNodes
        // TODO: config nodes
        newNodes.forEach(function(n) {
            if (n.type === "subflow") {
                newSubflows[n.id] = n;
            } else if (!n.hasOwnProperty('x') && !n.hasOwnProperty('y')) {
                newConfigNodes[n.id] = n;
            }
            if (n.z) {
                zMap[n.z] = zMap[n.z] || [];
                zMap[n.z].push(n);
            }
        })

        // Filter out config nodes inside a subflow def that is being replaced
        var configNodeIds = Object.keys(newConfigNodes);
        configNodeIds.forEach(function(id) {
            var n = newConfigNodes[id];
            if (newSubflows[n.z]) {
                // This config node is in a subflow to be replaced.
                //  - remove from the list as it'll get handled with the subflow
                delete newConfigNodes[id];
            }
        });
        // Rebuild the list of ids
        configNodeIds = Object.keys(newConfigNodes);

        // ------------------------------
        // Replace subflow definitions
        //
        // For each of the subflows to be replaced:
        var newSubflowIds = Object.keys(newSubflows);
        newSubflowIds.forEach(function(id) {
            var n = newSubflows[id];
            // Get a snapshot of the existing subflow definition
            removedNodes = removedNodes.concat(createExportableSubflow(id));
            // Remove the old subflow definition - but leave the instances in place
            var removalResult = RED.subflow.removeSubflow(n.id, true);
            // Create the list of nodes for the new subflow def
            // Need to sort the list in order to remove missing nodes
            var subflowNodes = [n].concat(zMap[n.id]).filter((s) => !!s);
            // Import the new subflow - no clashes should occur as we've removed
            // the old version
            var result = importNodes(subflowNodes);
            newSubflows[id] = getSubflow(id);
        })

        // Having replaced the subflow definitions, now need to update the
        // instance nodes.
        RED.nodes.eachNode(function(n) {
            if (/^subflow:/.test(n.type)) {
                var sfId = n.type.substring(8);
                if (newSubflows[sfId]) {
                    // This is an instance of one of the replaced subflows
                    //  - update the new def's instances array to include this one
                    newSubflows[sfId].instances.push(n);
                    //  - update the instance's _def to point to the new def
                    n._def = RED.nodes.getType(n.type);
                    //  - set all the flags so the view refreshes properly
                    n.dirty = true;
                    n.changed = true;
                    n._colorChanged = true;
                }
            }
        })

        newSubflowIds.forEach(function(id) {
            var n = newSubflows[id];
            RED.events.emit("subflows:change",n);
        })
        // Just in case the imported subflow changed color.
        RED.utils.clearNodeColorCache();

        // ------------------------------
        // Replace config nodes
        //
        configNodeIds.forEach(function(id) {
            const configNode = getNode(id);
            const currentUserCount = configNode.users;

            // Add a snapshot of the Config Node
            removedNodes = removedNodes.concat(convertNode(configNode));

            // Remove the Config Node instance
            removeNode(id);

            // Import the new one
            importNodes([newConfigNodes[id]]);

            // Re-attributes the user count
            getNode(id).users = currentUserCount;
        });

        return {
            removedNodes: removedNodes
        }

    }

    /**
     * Options:
     *  - generateIds - whether to replace all node ids
     *  - addFlow - whether to import nodes to a new tab
     *  - markChanged - whether to set changed=true on all newly imported objects
     *  - reimport - if node has a .z property, dont overwrite it
     *               Only applicible when `generateIds` is false
     *  - importMap - how to resolve any conflicts.
     *       - id:import - import as-is
     *       - id:copy - import with new id
     *       - id:replace - import over the top of existing
     *  - modules: map of module:version - hints for unknown nodes
     *  - applyNodeDefaults - whether to apply default values to the imported nodes (default: false)
     *  - eventContext - context to include in the `nodes:add` event
     */
    function importNodes(newNodesObj,options) { // createNewIds,createMissingWorkspace) {
        const defOpts = {
            generateIds: false,
            addFlow: false,
            markChanged: false,
            reimport: false,
            importMap: {},
            applyNodeDefaults: false,
            eventContext: null
        }
        options = Object.assign({}, defOpts, options)
        options.importMap = options.importMap || {}
        const createNewIds = options.generateIds;
        const reimport = (!createNewIds && !!options.reimport)
        const createMissingWorkspace = options.addFlow;
        const applyNodeDefaults = options.applyNodeDefaults;
        var i;
        var n;
        var newNodes;
        var nodeZmap = {};
        var recoveryWorkspace;
        if (typeof newNodesObj === "string") {
            if (newNodesObj === "") {
                return;
            }
            try {
                newNodes = JSON.parse(newNodesObj);
            } catch(err) {
                var e = new Error(RED._("clipboard.invalidFlow",{message:err.message}));
                e.code = "NODE_RED";
                throw e;
            }
        } else {
            newNodes = newNodesObj;
        }

        if (!Array.isArray(newNodes)) {
            newNodes = [newNodes];
        }

        // Scan for any duplicate nodes and remove them. This is a temporary
        // fix to help resolve corrupted flows caused by 0.20.0 where multiple
        // copies of the flow would get loaded at the same time.
        // If the user hit deploy they would have saved those duplicates.
        var seenIds = {};
        var existingNodes = [];
        var nodesToReplace = [];

        newNodes = newNodes.filter(function(n) {
            var id = n.id;
            if (seenIds[n.id]) {
                return false;
            }
            seenIds[n.id] = true;

            if (!options.generateIds) {
                if (!options.importMap[id]) {
                    // No conflict resolution for this node
                    var existing = allNodes.getNode(id) || configNodes[id] || workspaces[id] || subflows[id] || groups[id] || junctions[id];
                    if (existing) {
                        existingNodes.push({existing:existing, imported:n});
                    }
                } else if (options.importMap[id] === "replace") {
                    nodesToReplace.push(n);
                    return false;
                }
            }

            return true;
        })

        if (existingNodes.length > 0) {
            var errorMessage = RED._("clipboard.importDuplicate",{count:existingNodes.length});
            var nodeList = $("<ul>");
            var existingNodesCount = Math.min(5,existingNodes.length);
            for (var i=0;i<existingNodesCount;i++) {
                var conflict = existingNodes[i];
                $("<li>").text(
                    conflict.existing.id+
                    " [ "+conflict.existing.type+ ((conflict.imported.type !== conflict.existing.type)?" | "+conflict.imported.type:"")+" ]").appendTo(nodeList)
            }
            if (existingNodesCount !== existingNodes.length) {
                $("<li>").text(RED._("deploy.confirm.plusNMore",{count:existingNodes.length-existingNodesCount})).appendTo(nodeList)
            }
            var wrapper = $("<p>").append(nodeList);

            var existingNodesError = new Error(errorMessage+wrapper.html());
            existingNodesError.code = "import_conflict";
            existingNodesError.importConfig = identifyImportConflicts(newNodes);
            throw existingNodesError;
        }
        var removedNodes;
        if (nodesToReplace.length > 0) {
            var replaceResult = replaceNodes(nodesToReplace);
            removedNodes = replaceResult.removedNodes;
        }


        var isInitialLoad = false;
        if (!initialLoad) {
            isInitialLoad = true;
            initialLoad = JSON.parse(JSON.stringify(newNodes));
        }
        var unknownTypes = [];
        for (i=0;i<newNodes.length;i++) {
            n = newNodes[i];
            var id = n.id;
            // TODO: remove workspace in next release+1
            if (n.type != "workspace" &&
                n.type != "tab" &&
                n.type != "subflow" &&
                n.type != "group" &&
                n.type != 'junction' &&
                !registry.getNodeType(n.type) &&
                n.type.substring(0,8) != "subflow:" &&
                unknownTypes.indexOf(n.type)==-1) {
                    unknownTypes.push(n.type);
            }
            if (n.z) {
                nodeZmap[n.z] = nodeZmap[n.z] || [];
                nodeZmap[n.z].push(n);
            } else if (isInitialLoad && n.hasOwnProperty('x') && n.hasOwnProperty('y') && !n.z) {
                // Hit the rare issue where node z values get set to 0.
                // Repair the flow - but we really need to track that down.
                if (!recoveryWorkspace) {
                    recoveryWorkspace = {
                        id: RED.nodes.id(),
                        type: "tab",
                        disabled: false,
                        label: RED._("clipboard.recoveredNodes"),
                        info: RED._("clipboard.recoveredNodesInfo"),
                        env: []
                    }
                    addWorkspace(recoveryWorkspace);
                    RED.workspaces.add(recoveryWorkspace);
                    nodeZmap[recoveryWorkspace.id] = [];
                }
                n.z = recoveryWorkspace.id;
                nodeZmap[recoveryWorkspace.id].push(n);
            }

        }
        if (!isInitialLoad && unknownTypes.length > 0) {
            const notificationOptions = {
                type: "error",
                fixed: false,
                timeout: 10000,
            }
            let unknownNotification
            let missingModules = []
            if (options.modules) {
                missingModules = Object.keys(options.modules).filter(module => !RED.nodes.registry.getModule(module))
            }
            if (missingModules.length > 0) {
                notificationOptions.fixed = true
                delete notificationOptions.timeout
                // We have module hint list from imported global-config
                // Provide option to install missing modules
                notificationOptions.buttons = [
                    {
                        text: RED._("palette.editor.installAll"),
                        class: "primary",
                        click: function(e) {
                            unknownNotification.close();

                            RED.actions.invoke('core:manage-palette', {
                                autoInstall: true,
                                modules: missingModules.reduce((modules, moduleName) => {
                                    modules[moduleName] = options.modules[moduleName];
                                    return modules;
                                }, {}),
                            });
                        }
                    },
                    {
                        text: RED._("palette.editor.manageModules"),
                        class: "pull-left",
                        click: function(e) {
                            unknownNotification.close();

                            RED.actions.invoke('core:manage-palette', {
                                view: 'install',
                                filter: '"' + missingModules.join('", "') + '"'
                            });
                        }
                    }
                ]
                let moduleList = $("<ul>");
                missingModules.forEach(function(t) {
                    $("<li>").text(t).appendTo(moduleList);
                })
                moduleList = moduleList[0].outerHTML;
                unknownNotification = RED.notify(
                    "<p>"+RED._("clipboard.importWithModuleInfo")+"</p>"+
                    "<p>"+RED._("clipboard.importWithModuleInfoDesc")+"</p>"+
                    moduleList,
                    notificationOptions
                );
            } else {
                var typeList = $("<ul>");
                unknownTypes.forEach(function(t) {
                    $("<li>").text(t).appendTo(typeList);
                })
                typeList = typeList[0].outerHTML;
                
                unknownNotification = RED.notify(
                    "<p>"+RED._("clipboard.importUnrecognised",{count:unknownTypes.length})+"</p>"+typeList,
                    notificationOptions
                );
            }
        }

        var activeWorkspace = RED.workspaces.active();
        //TODO: check the z of the subflow instance and check _that_ if it exists
        var activeSubflow = getSubflow(activeWorkspace);
        for (i=0;i<newNodes.length;i++) {
            var m = /^subflow:(.+)$/.exec(newNodes[i].type);
            if (m) {
                var subflowId = m[1];
                var parent = getSubflow(activeWorkspace);
                if (parent) {
                    var err;
                    if (subflowId === parent.id) {
                        err = new Error(RED._("notification.errors.cannotAddSubflowToItself"));
                    }
                    if (subflowContains(subflowId,parent.id)) {
                        err = new Error(RED._("notification.errors.cannotAddCircularReference"));
                    }
                    if (err) {
                        // TODO: standardise error codes
                        err.code = "NODE_RED";
                        throw err;
                    }
                }
            }
        }

        var new_workspaces = [];
        var workspace_map = {};
        var new_subflows = [];
        var subflow_map = {};
        var subflow_denylist = {};
        var node_map = {};
        var new_nodes = [];
        var new_links = [];
        var new_groups = [];
        var new_junctions = [];
        var new_group_set = new Set();
        var nid;
        var def;
        var configNode;
        var missingWorkspace = null;
        var d;

        if (recoveryWorkspace) {
            new_workspaces.push(recoveryWorkspace);
        }

        // Find all tabs and subflow templates
        for (i=0;i<newNodes.length;i++) {
            n = newNodes[i];
            // TODO: remove workspace in next release+1
            if (n.type === "workspace" || n.type === "tab") {
                if (n.type === "workspace") {
                    n.type = "tab";
                }
                if (defaultWorkspace == null) {
                    defaultWorkspace = n;
                }
                if (activeWorkspace === 0) {
                    activeWorkspace = n.id;
                }
                if (createNewIds || options.importMap[n.id] === "copy") {
                    nid = getID();
                    workspace_map[n.id] = nid;
                    n.id = nid;
                } else {
                    workspace_map[n.id] = n.id;
                }
                addWorkspace(n);
                RED.workspaces.add(n);
                new_workspaces.push(n);
            } else if (n.type === "subflow") {
                var matchingSubflow;
                if (!options.importMap[n.id]) {
                    matchingSubflow = checkForMatchingSubflow(n,nodeZmap[n.id]);
                }
                if (matchingSubflow) {
                    subflow_denylist[n.id] = matchingSubflow;
                } else {
                    const oldId = n.id;

                    subflow_map[n.id] = n;
                    if (createNewIds || options.importMap[n.id] === "copy") {
                        nid = getID();
                        n.id = nid;
                    }
                    // TODO: handle createNewIds - map old to new subflow ids
                    n.in.forEach(function(input,i) {
                        input.type = "subflow";
                        input.direction = "in";
                        input.z = n.id;
                        input.i = i;
                        input.id = getID();
                    });
                    n.out.forEach(function(output,i) {
                        output.type = "subflow";
                        output.direction = "out";
                        output.z = n.id;
                        output.i = i;
                        output.id = getID();
                    });
                    if (n.status) {
                        n.status.type = "subflow";
                        n.status.direction = "status";
                        n.status.z = n.id;
                        n.status.id = getID();
                    }
                    new_subflows.push(n);
                    addSubflow(n,createNewIds || options.importMap[oldId] === "copy");
                }
            }
        }

        // Add a tab if there isn't one there already
        if (defaultWorkspace == null) {
            defaultWorkspace = { type:"tab", id:getID(), disabled: false, info:"",  label:RED._('workspace.defaultName',{number:1}), env:[]};
            addWorkspace(defaultWorkspace);
            RED.workspaces.add(defaultWorkspace);
            new_workspaces.push(defaultWorkspace);
            activeWorkspace = RED.workspaces.active();
        }

        const pendingConfigNodes = []
        const pendingConfigNodeIds = new Set()
        // Find all config nodes and add them
        for (i=0;i<newNodes.length;i++) {
            n = newNodes[i];
            def = registry.getNodeType(n.type);
            if (def && def.category == "config") {
                var existingConfigNode = null;
                if (createNewIds || options.importMap[n.id] === "copy") {
                    if (n.z) {
                        if (subflow_denylist[n.z]) {
                            continue;
                        } else if (subflow_map[n.z]) {
                            n.z = subflow_map[n.z].id;
                        } else {
                            n.z = workspace_map[n.z];
                            if (!workspaces[n.z]) {
                                if (createMissingWorkspace) {
                                    if (missingWorkspace === null) {
                                        missingWorkspace = RED.workspaces.add(null,true);
                                        new_workspaces.push(missingWorkspace);
                                    }
                                    n.z = missingWorkspace.id;
                                } else {
                                    n.z = activeWorkspace;
                                }
                            }
                        }
                    }
                    if (options.importMap[n.id] !== "copy") {
                        existingConfigNode = RED.nodes.node(n.id);
                        if (existingConfigNode) {
                            if (n.z && existingConfigNode.z !== n.z) {
                                existingConfigNode = null;
                                // Check the config nodes on n.z
                                for (var cn in configNodes) {
                                    if (configNodes.hasOwnProperty(cn)) {
                                        if (configNodes[cn].z === n.z && compareNodes(configNodes[cn],n,false)) {
                                            existingConfigNode = configNodes[cn];
                                            node_map[n.id] = configNodes[cn];
                                            break;
                                        }
                                    }
                                }
                            }
                        }
                    }
                } else {
                    const keepNodesCurrentZ = reimport && n.z && (RED.workspaces.contains(n.z) || RED.nodes.subflow(n.z))
                    if (!keepNodesCurrentZ && n.z && !workspace_map[n.z] && !subflow_map[n.z]) {
                        n.z = activeWorkspace;
                    }
                }

                if (!existingConfigNode || existingConfigNode._def.exclusive) { //} || !compareNodes(existingConfigNode,n,true) || existingConfigNode.z !== n.z) {
                    configNode = {
                        id:n.id,
                        z:n.z,
                        type:n.type,
                        info: n.info,
                        users:[],
                        _config:{},
                        _configNodeReferences: new Set()
                    };
                    if (!n.z) {
                        delete configNode.z;
                    }
                    if (options.markChanged) {
                        configNode.changed = true
                    }
                    if (n.hasOwnProperty('d')) {
                        configNode.d = n.d;
                    }
                    for (d in def.defaults) {
                        if (def.defaults.hasOwnProperty(d)) {
                            configNode[d] = n[d];
                            if (applyNodeDefaults && n[d] === undefined) {
                                // If the node has a default value, but the imported node does not
                                // set it, then set it to the default value
                                if (def.defaults[d].value !== undefined) {
                                    configNode[d] = JSON.parse(JSON.stringify(def.defaults[d].value))
                                }
                            }
                            configNode._config[d] = JSON.stringify(n[d]);
                            if (def.defaults[d].type) {
                                configNode._configNodeReferences.add(n[d])
                            }
                        }
                    }
                    if (def.hasOwnProperty('credentials') && n.hasOwnProperty('credentials')) {
                        configNode.credentials = {};
                        for (d in def.credentials) {
                            if (def.credentials.hasOwnProperty(d) && n.credentials.hasOwnProperty(d)) {
                                configNode.credentials[d] = n.credentials[d];
                            }
                        }
                    }
                    configNode.label = def.label;
                    configNode._def = def;
                    if (createNewIds || options.importMap[n.id] === "copy") {
                        configNode.id = getID();
                    }
                    node_map[n.id] = configNode;
                    pendingConfigNodes.push(configNode);
                    pendingConfigNodeIds.add(configNode.id)
                }
            }
        }

        // We need to sort new_nodes (which only contains config nodes at this point)
        // to ensure they get added in the right order. If NodeA depends on NodeB, then
        // NodeB must be added first.
        
        // Limit us to 5 full iterations of the list - this should be more than
        // enough to process the list as config->config node relationships are
        // not very common
        let iterationLimit = pendingConfigNodes.length * 5
        const handledConfigNodes = new Set()
        while (pendingConfigNodes.length > 0 && iterationLimit > 0) {
            const node = pendingConfigNodes.shift()
            let hasPending = false
            // Loop through the nodes referenced by this node to see if anything
            // is pending
            node._configNodeReferences.forEach(id => {
                if (pendingConfigNodeIds.has(id) && !handledConfigNodes.has(id)) {
                    // This reference is for a node we know is in this import, but
                    // it isn't added yet - flag as pending
                    hasPending = true
                }
            })
            if (!hasPending) {
                // This node has no pending config node references - safe to add
                delete node._configNodeReferences
                new_nodes.push(node)
                handledConfigNodes.add(node.id)
            } else {
                // This node has pending config node references
                // Put to the back of the queue
                pendingConfigNodes.push(node)
            }
            iterationLimit--
        }
        if (pendingConfigNodes.length > 0) {
            // We exceeded the iteration count. Could be due to reference loops
            // between the config nodes. At this point, just add the remaining
            // nodes as-is
            pendingConfigNodes.forEach(node => {
                delete node._configNodeReferences
                new_nodes.push(node)
            })
        }

        // Find regular flow nodes and subflow instances
        for (i=0;i<newNodes.length;i++) {
            n = newNodes[i];
            // TODO: remove workspace in next release+1
            if (n.type !== "workspace" && n.type !== "tab" && n.type !== "subflow") {
                def = registry.getNodeType(n.type);
                if (!def || def.category != "config") {
                    var node = {
                        x:parseFloat(n.x || 0),
                        y:parseFloat(n.y || 0),
                        z:n.z,
                        type: n.type,
                        info: n.info,
                        changed:false,
                        _config:{}
                    }
                    if (n.type !== "group" && n.type !== 'junction') {
                        node.wires = n.wires||[];
                        node.inputLabels = n.inputLabels;
                        node.outputLabels = n.outputLabels;
                        node.icon = n.icon;
                    }
                    if (n.type === 'junction') {
                        node.wires = n.wires||[];
                    }
                    if (n.hasOwnProperty('l')) {
                        node.l = n.l;
                    }
                    if (n.hasOwnProperty('d')) {
                        node.d = n.d;
                    }
                    if (n.hasOwnProperty('g')) {
                        node.g = n.g;
                    }
                    if (options.markChanged) {
                        node.changed = true
                    }
                    if (createNewIds || options.importMap[n.id] === "copy") {
                        if (subflow_denylist[n.z]) {
                            continue;
                        } else if (subflow_map[node.z]) {
                            node.z = subflow_map[node.z].id;
                        } else {
                            node.z = workspace_map[node.z];
                            if (!workspaces[node.z]) {
                                if (createMissingWorkspace) {
                                    if (missingWorkspace === null) {
                                        missingWorkspace = RED.workspaces.add(null,true);
                                        new_workspaces.push(missingWorkspace);
                                    }
                                    node.z = missingWorkspace.id;
                                } else {
                                    node.z = activeWorkspace;
                                }
                            }
                        }
                        node.id = getID();
                    } else {
                        node.id = n.id;
                        const keepNodesCurrentZ = reimport && node.z && (RED.workspaces.contains(node.z) || RED.nodes.subflow(node.z))
                        if (!keepNodesCurrentZ && (node.z == null || (!workspace_map[node.z] && !subflow_map[node.z]))) {
                            if (createMissingWorkspace) {
                                if (missingWorkspace === null) {
                                    missingWorkspace = RED.workspaces.add(null,true);
                                    new_workspaces.push(missingWorkspace);
                                }
                                node.z = missingWorkspace.id;
                            } else {
                                node.z = activeWorkspace;
                            }
                        }
                    }
                    node._def = def;
                    if (node.type === "group") {
                        node._def = RED.group.def;
                        for (d in node._def.defaults) {
                            if (node._def.defaults.hasOwnProperty(d) && d !== 'inputs' && d !== 'outputs') {
                                node[d] = n[d];
                                node._config[d] = JSON.stringify(n[d]);
                            }
                        }
                        node._config.x = node.x;
                        node._config.y = node.y;
                        if (n.hasOwnProperty('w')) {
                            node.w = n.w
                        }
                        if (n.hasOwnProperty('h')) {
                            node.h = n.h
                        }
                    } else if (n.type.substring(0,7) === "subflow") {
                        var parentId = n.type.split(":")[1];
                        var subflow = subflow_denylist[parentId]||subflow_map[parentId]||getSubflow(parentId);
                        if (!subflow){
                            node._def = {
                                color:"#fee",
                                defaults: {},
                                label: "unknown: "+n.type,
                                labelStyle: "red-ui-flow-node-label-italic",
                                outputs: n.outputs|| (n.wires && n.wires.length) || 0,
                                set: registry.getNodeSet("node-red/unknown")
                            }
                            var orig = {};
                            for (var p in n) {
                                if (n.hasOwnProperty(p) && p!="x" && p!="y" && p!="z" && p!="id" && p!="wires") {
                                    orig[p] = n[p];
                                }
                            }
                            node._orig = orig;
                            node.name = n.type;
                            node.type = "unknown";
                        } else {
                            if (subflow_denylist[parentId] || createNewIds || options.importMap[n.id] === "copy") {
                                parentId = subflow.id;
                                node.type = "subflow:"+parentId;
                                node._def = registry.getNodeType(node.type);
                                delete node.i;
                            }
                            node.name = n.name;
                            node.outputs = subflow.out.length;
                            node.inputs = subflow.in.length;
                            node.env = n.env;
                        }
                    } else if (n.type === 'junction') {
                         node._def = {defaults:{}}
                         node._config.x = node.x
                         node._config.y = node.y
                         node.inputs = 1
                         node.outputs = 1
                         node.w = 0;
                         node.h = 0;

                    } else {
                        if (!node._def) {
                            if (node.x && node.y) {
                                node._def = {
                                    color:"#fee",
                                    defaults: {},
                                    label: "unknown: "+n.type,
                                    labelStyle: "red-ui-flow-node-label-italic",
                                    outputs: n.outputs|| (n.wires && n.wires.length) || 0,
                                    set: registry.getNodeSet("node-red/unknown")
                                }
                            } else {
                                node._def = {
                                    category:"config",
                                    set: registry.getNodeSet("node-red/unknown")
                                };
                                node.users = [];
                                // This is a config node, so delete the default
                                // non-config node properties
                                delete node.x;
                                delete node.y;
                                delete node.wires;
                                delete node.inputLabels;
                                delete node.outputLabels;
                                if (!n.z) {
                                    delete node.z;
                                }
                            }
                            const unknownTypeDef = RED.nodes.getType('unknown')
                            node._def.oneditprepare = unknownTypeDef.oneditprepare

                            var orig = {};
                            for (var p in n) {
                                if (n.hasOwnProperty(p) && p!="x" && p!="y" && p!="z" && p!="id" && p!="wires") {
                                    orig[p] = n[p];
                                }
                            }
                            node._orig = orig;
                            node.name = n.type;
                            node.type = "unknown";
                            if (options.modules) {
                                // We have a module hint list. Attach to the unknown node so we can reference it later
                                node.modules = Object.keys(options.modules)
                            }
                        }
                        if (node._def.category != "config") {
                            if (n.hasOwnProperty('inputs') && node._def.defaults.hasOwnProperty("inputs")) {
                                node.inputs = parseInt(n.inputs, 10);
                                node._config.inputs = JSON.stringify(n.inputs);
                            } else {
                                node.inputs = node._def.inputs;
                            }
                            if (n.hasOwnProperty('outputs') && node._def.defaults.hasOwnProperty("outputs")) {
                                node.outputs = parseInt(n.outputs, 10);
                                node._config.outputs = JSON.stringify(n.outputs);
                            } else {
                                node.outputs = node._def.outputs;
                            }

                            // The node declares outputs in its defaults, but has not got a valid value
                            // Defer to the length of the wires array
                            if (node.hasOwnProperty('wires')) {
                                if (isNaN(node.outputs)) {
                                    node.outputs = node.wires.length;
                                } else if (node.wires.length > node.outputs) {
                                    // If 'wires' is longer than outputs, clip wires
                                    console.log("Warning: node.wires longer than node.outputs - trimming wires:", node.id, " wires:", node.wires.length, " outputs:", node.outputs);
                                    node.wires = node.wires.slice(0, node.outputs);
                                }
                            }

                            for (d in node._def.defaults) {
                                if (node._def.defaults.hasOwnProperty(d) && d !== 'inputs' && d !== 'outputs') {
                                    node[d] = n[d];
                                    if (applyNodeDefaults && n[d] === undefined) {
                                        // If the node has a default value, but the imported node does not
                                        // set it, then set it to the default value
                                        if (node._def.defaults[d].value !== undefined) {
                                            node[d] = JSON.parse(JSON.stringify(node._def.defaults[d].value))
                                        }
                                    }
                                    node._config[d] = JSON.stringify(n[d]);
                                }
                            }
                            node._config.x = node.x;
                            node._config.y = node.y;
                            if (node._def.hasOwnProperty('credentials') && n.hasOwnProperty('credentials')) {
                                node.credentials = {};
                                for (d in node._def.credentials) {
                                    if (node._def.credentials.hasOwnProperty(d) && n.credentials.hasOwnProperty(d)) {
                                        node.credentials[d] = n.credentials[d];
                                    }
                                }
                            }
                        }
                    }
                    node_map[n.id] = node;
                    // If an 'unknown' config node, it will not have been caught by the
                    // proper config node handling, so needs adding to new_nodes here
                    if (node.type === 'junction') {
                        new_junctions.push(node)
                    } else if (node.type === "unknown" || node._def.category !== "config") {
                        new_nodes.push(node);
                    } else if (node.type === "group") {
                        new_groups.push(node);
                        new_group_set.add(node.id);
                    }
                }
            }
        }

        // Remap all wires and config node references
        for (i=0;i<new_nodes.length+new_junctions.length;i++) {
            if (i<new_nodes.length) {
                n = new_nodes[i];
            } else {
                n = new_junctions[i - new_nodes.length]
            }
            if (n.wires) {
                for (var w1=0;w1<n.wires.length;w1++) {
                    var wires = (Array.isArray(n.wires[w1]))?n.wires[w1]:[n.wires[w1]];
                    for (var w2=0;w2<wires.length;w2++) {
                        if (node_map.hasOwnProperty(wires[w2])) {
                            if (n.z === node_map[wires[w2]].z) {
                                var link = {source:n,sourcePort:w1,target:node_map[wires[w2]]};
                                addLink(link);
                                new_links.push(link);
                            } else {
                                console.log("Warning: dropping link that crosses tabs:",n.id,"->",node_map[wires[w2]].id);
                            }
                        }
                    }
                }
                delete n.wires;
            }
            if (n.g && node_map[n.g]) {
                n.g = node_map[n.g].id;
            } else {
                delete n.g
            }
            // If importing a link node, ensure both ends of each link are either:
            // - not in a subflow
            // - both in the same subflow (not for link call node)
            if (/^link /.test(n.type) && n.links) {
                n.links = n.links.filter(function(id) {
                    const otherNode = node_map[id] || RED.nodes.node(id);
                    if (!otherNode) {
                        // Cannot find other end - remove the link
                        return false
                    }
                    if (otherNode.z === n.z) {
                        // Both ends in the same flow/subflow
                        return true
                    } else if (n.type === "link call" && !getSubflow(otherNode.z)) {
                        // Link call node can call out of a subflow as long as otherNode is
                        // not in a subflow
                        return true
                    } else if (!!getSubflow(n.z) || !!getSubflow(otherNode.z)) {
                        // One end is in a subflow - remove the link
                        return false
                    }
                    return true
                });
            }
            for (var d3 in n._def.defaults) {
                if (n._def.defaults.hasOwnProperty(d3)) {
                    if (n._def.defaults[d3].type) {
                        var nodeList = n[d3];
                        if (!Array.isArray(nodeList)) {
                            nodeList = [nodeList];
                        }
                        nodeList = nodeList.map(function(id) {
                            var node = node_map[id];
                            if (node) {
                                return node.id;
                            }
                            return id;
                        })
                        n[d3] = Array.isArray(n[d3])?nodeList:nodeList[0];
                    }
                }
            }
        }
        for (i=0;i<new_subflows.length;i++) {
            n = new_subflows[i];
            n.in.forEach(function(input) {
                input.wires.forEach(function(wire) {
                    if (node_map.hasOwnProperty(wire.id)) {
                        var link = {source:input, sourcePort:0, target:node_map[wire.id]};
                        addLink(link);
                        new_links.push(link);
                    }
                });
                delete input.wires;
            });
            n.out.forEach(function(output) {
                output.wires.forEach(function(wire) {
                    var link;
                    if (subflow_map[wire.id] && subflow_map[wire.id].id == n.id) {
                        link = {source:n.in[wire.port], sourcePort:wire.port,target:output};
                    } else if (node_map.hasOwnProperty(wire.id) || subflow_map.hasOwnProperty(wire.id)) {
                        link = {source:node_map[wire.id]||subflow_map[wire.id], sourcePort:wire.port,target:output};
                    }
                    if (link) {
                        addLink(link);
                        new_links.push(link);
                    }
                });
                delete output.wires;
            });
            if (n.status) {
                n.status.wires.forEach(function(wire) {
                    var link;
                    if (subflow_map[wire.id] && subflow_map[wire.id].id == n.id) {
                        link = {source:n.in[wire.port], sourcePort:wire.port,target:n.status};
                    } else if (node_map.hasOwnProperty(wire.id) || subflow_map.hasOwnProperty(wire.id)) {
                        link = {source:node_map[wire.id]||subflow_map[wire.id], sourcePort:wire.port,target:n.status};
                    }
                    if (link) {
                        addLink(link);
                        new_links.push(link);
                    }
                });
                delete n.status.wires;
            }
        }
        // Order the groups to ensure they are outer-most to inner-most
        var groupDepthMap = {};
        for (i=0;i<new_groups.length;i++) {
            n = new_groups[i];

            if (n.g && !new_group_set.has(n.g)) {
                delete n.g;
            }
            if (!n.g) {
                groupDepthMap[n.id] = 0;
            }
        }
        var changedDepth;
        do {
            changedDepth = false;
            for (i=0;i<new_groups.length;i++) {
                n = new_groups[i];
                if (n.g) {
                    if (groupDepthMap[n.id] !== groupDepthMap[n.g] + 1) {
                        groupDepthMap[n.id] = groupDepthMap[n.g] + 1;
                        changedDepth = true;
                    }
                }
            }
        } while(changedDepth);

        new_groups.sort(function(A,B) {
            return groupDepthMap[A.id] - groupDepthMap[B.id];
        });
        for (i=0;i<new_groups.length;i++) {
            new_groups[i] = addGroup(new_groups[i]);
            node_map[new_groups[i].id] = new_groups[i]
        }

        for (i=0;i<new_junctions.length;i++) {
            new_junctions[i] = addJunction(new_junctions[i]);
            node_map[new_junctions[i].id] = new_junctions[i]
        }


        // Now the nodes have been fully updated, add them.
        for (i=0;i<new_nodes.length;i++) {
            new_nodes[i] = addNode(new_nodes[i], options.eventContext)
            node_map[new_nodes[i].id] = new_nodes[i]
        }

        // Finally validate them all.
        // This has to be done after everything is added so that any checks for
        // dependent config nodes will pass
        for (i=0;i<new_nodes.length;i++) {
            var node = new_nodes[i];
            RED.editor.validateNode(node);
        }
        const lookupNode = (id) => {
            const mappedNode = node_map[id]
            if (!mappedNode) {
                return null
            }
            if (mappedNode.__isProxy__) {
                return mappedNode
            } else {
                return node_map[mappedNode.id]
            }
        }
        // Update groups to reference proxy node objects
        for (i=0;i<new_groups.length;i++) {
            n = new_groups[i];
            // bypass the proxy in case the flow is locked
            n.__node__.nodes = n.nodes.map(lookupNode)
            // Just in case the group references a node that doesn't exist for some reason
            n.__node__.nodes = n.nodes.filter(function(v) {
                if (v) {
                    // Repair any nodes that have forgotten they are in this group
                    if (v.g !== n.id) {
                        v.g = n.id;
                    }
                }
                return !!v
            });
        }

        // Update links to use proxy node objects
        for (i=0;i<new_links.length;i++) {
            new_links[i].source = lookupNode(new_links[i].source.id) || new_links[i].source
            new_links[i].target = lookupNode(new_links[i].target.id) || new_links[i].target
        }

        RED.workspaces.refresh();


        if (recoveryWorkspace) {
            var notification = RED.notify(RED._("clipboard.recoveredNodesNotification",{flowName:RED._("clipboard.recoveredNodes")}),{
                type:"warning",
                fixed:true,
                buttons: [
                    {text: RED._("common.label.close"), click: function() { notification.close() }}
                ]
            });
        }

        return {
            nodes:new_nodes,
            links:new_links,
            groups:new_groups,
            junctions: new_junctions,
            workspaces:new_workspaces,
            subflows:new_subflows,
            missingWorkspace: missingWorkspace,
            removedNodes: removedNodes,
            nodeMap: node_map
        }
    }

    // TODO: supports filter.z|type
    function filterNodes(filter) {
        return allNodes.filterNodes(filter);
    }

    function filterLinks(filter) {
        var result = [];
        var candidateLinks = [];
        var hasCandidates = false;
        var filterSZ = filter.source && filter.source.z;
        var filterTZ = filter.target && filter.target.z;
        var filterZ;
        if (filterSZ || filterTZ) {
            if (filterSZ === filterTZ) {
                filterZ = filterSZ;
            } else {
                filterZ = (filterSZ === undefined)?filterTZ:filterSZ
            }
        }
        if (filterZ) {
            candidateLinks = linkTabMap[filterZ] || [];
            hasCandidates = true;
        } else if (filter.source && filter.source.hasOwnProperty("id")) {
            if (nodeLinks[filter.source.id]) {
                hasCandidates = true;
                candidateLinks = candidateLinks.concat(nodeLinks[filter.source.id].out)
            }
        } else if (filter.target && filter.target.hasOwnProperty("id")) {
            if (nodeLinks[filter.target.id]) {
                hasCandidates = true;
                candidateLinks = candidateLinks.concat(nodeLinks[filter.target.id].in)
            }
        }
        if (!hasCandidates) {
            candidateLinks = links;
        }
        for (var n=0;n<candidateLinks.length;n++) {
            var link = candidateLinks[n];
            if (filter.source) {
                if (filter.source.hasOwnProperty("id") && link.source.id !== filter.source.id) {
                    continue;
                }
                if (filter.source.hasOwnProperty("z") && link.source.z !== filter.source.z) {
                    continue;
                }
            }
            if (filter.target) {
                if (filter.target.hasOwnProperty("id") && link.target.id !== filter.target.id) {
                    continue;
                }
                if (filter.target.hasOwnProperty("z") && link.target.z !== filter.target.z) {
                    continue;
                }
            }
            if (filter.hasOwnProperty("sourcePort") && link.sourcePort !== filter.sourcePort) {
                continue;
            }
            result.push(link);
        }
        return result;
    }

    /**
     * Update any config nodes referenced by the provided node to ensure
     * their 'users' list is correct.
     *
     * @param {object} node The node in which to check if it contains references
     * @param {object} options Options to apply.
     * @param {"add" | "remove"} [options.action] Add or remove the node from
     * the Config Node users list. Default `add`.
     * @param {boolean} [options.emitEvent] Emit the `nodes:changes` event.
     * Default true.
     */
    function updateConfigNodeUsers(node, options) {
        const defaultOptions = { action: "add", emitEvent: true };
        options = Object.assign({}, defaultOptions, options);

        for (var d in node._def.defaults) {
            if (node._def.defaults.hasOwnProperty(d)) {
                var property = node._def.defaults[d];
                if (property.type) {
                    var type = registry.getNodeType(property.type);
                    // Need to ensure the type is a config node to not treat links nodes
                    if (type && type.category == "config") {
                        var configNode = configNodes[node[d]];
                        if (configNode) {
                            if (options.action === "add") {
                                if (configNode.users.indexOf(node) === -1) {
                                    configNode.users.push(node);
                                    if (options.emitEvent) {
                                        RED.events.emit('nodes:change', configNode);
                                    }
                                }
                            } else if (options.action === "remove") {
                                if (configNode.users.indexOf(node) !== -1) {
                                    const users = configNode.users;
                                    users.splice(users.indexOf(node), 1);
                                    if (options.emitEvent) {
                                        RED.events.emit('nodes:change', configNode);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        // Subflows can have config node env
        if (node.type.indexOf("subflow:") === 0) {
            node.env?.forEach((prop) => {
                if (prop.type === "conf-type" && prop.value) {
                    // Add the node to the config node users
                    const configNode = getNode(prop.value);
                    if (configNode) {
                        if (options.action === "add") {
                            if (configNode.users.indexOf(node) === -1) {
                                configNode.users.push(node);
                                if (options.emitEvent) {
                                    RED.events.emit('nodes:change', configNode);
                                }
                            }
                        } else if (options.action === "remove") {
                            if (configNode.users.indexOf(node) !== -1) {
                                const users = configNode.users;
                                users.splice(users.indexOf(node), 1);
                                if (options.emitEvent) {
                                    RED.events.emit('nodes:change', configNode);
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    function flowVersion(version) {
        if (version !== undefined) {
            loadedFlowVersion = version;
        } else {
            return loadedFlowVersion;
        }
    }

    function clear() {
        links = [];
        linkTabMap = {};
        nodeLinks = {};
        configNodes = {};
        workspacesOrder = [];
        groups = {};
        groupsByZ = {};
        junctions = {};
        junctionsByZ = {};

        var workspaceIds = Object.keys(workspaces);
        // Ensure all workspaces are unlocked so we don't get any edit-protection
        // preventing removal
        workspaceIds.forEach(function(id) {
            workspaces[id].locked = false
        });

        var subflowIds = Object.keys(subflows);
        subflowIds.forEach(function(id) {
            RED.subflow.removeSubflow(id)
        });
        workspaceIds.forEach(function(id) {
            RED.workspaces.remove(workspaces[id]);
        });
        defaultWorkspace = null;
        initialLoad = null;
        workspaces = {};

        allNodes.clear();

        RED.nodes.dirty(false);
        RED.view.redraw(true, true);
        RED.palette.refresh();
        RED.workspaces.refresh();
        RED.sidebar.config.refresh();
        RED.sidebar.info.refresh();

        RED.events.emit("workspace:clear");
    }

    function addGroup(group) {
        if (!group.__isProxy__) {
            group = new Proxy(group, nodeProxyHandler)
        }
        groupsByZ[group.z] = groupsByZ[group.z] || [];
        groupsByZ[group.z].push(group);
        groups[group.id] = group;
        allNodes.addObjectToWorkspace(group.z, group.id, group.changed || group.moved)
        RED.events.emit("groups:add",group);
        return group
    }
    function removeGroup(group) {
        var i = groupsByZ[group.z].indexOf(group);
        groupsByZ[group.z].splice(i,1);
        if (groupsByZ[group.z].length === 0) {
            delete groupsByZ[group.z];
        }
        if (group.g) {
            if (groups[group.g]) {
                var index = groups[group.g].nodes.indexOf(group);
                groups[group.g].nodes.splice(index,1);
            }
        }
        RED.group.markDirty(group);
        allNodes.removeObjectFromWorkspace(group.z, group.id)
        delete groups[group.id];
        RED.events.emit("groups:remove",group);
    }
    function getGroupOrder(z) {
        const groups = groupsByZ[z]
        return groups.map(g => g.id)
    }

    function addJunction(junction) {
        if (!junction.__isProxy__) {
            junction = new Proxy(junction, nodeProxyHandler)
        }
        junctionsByZ[junction.z] = junctionsByZ[junction.z] || []
        junctionsByZ[junction.z].push(junction)
        junctions[junction.id] = junction;
        if (!nodeLinks[junction.id]) {
            nodeLinks[junction.id] = {in:[],out:[]};
        }
        allNodes.addObjectToWorkspace(junction.z, junction.id, junction.changed || junction.moved)
        RED.events.emit("junctions:add", junction)
        return junction
    }
    function removeJunction(junction) {
        var i = junctionsByZ[junction.z].indexOf(junction)
        junctionsByZ[junction.z].splice(i, 1)
        if (junctionsByZ[junction.z].length === 0) {
            delete junctionsByZ[junction.z]
        }
        delete junctions[junction.id]
        delete nodeLinks[junction.id];
        allNodes.removeObjectFromWorkspace(junction.z, junction.id)
        RED.events.emit("junctions:remove", junction)

        var removedLinks = links.filter(function(l) { return (l.source === junction) || (l.target === junction); });
        removedLinks.forEach(removeLink);
        return { links: removedLinks }
    }

    function getNodeHelp(type) {
        var helpContent = "";
        var helpElement = $("script[data-help-name='"+type+"']");
        if (helpElement) {
            helpContent = helpElement.html();
            var helpType = helpElement.attr("type");
            if (helpType === "text/markdown") {
                helpContent = RED.utils.renderMarkdown(helpContent);
            }
        }
        return helpContent;
    }

    function getNodeIslands(nodes) {
        var selectedNodes = new Set(nodes);
        // Maps node => island index
        var nodeToIslandIndex = new Map();
        // Maps island index => [nodes in island]
        var islandIndexToNodes = new Map();
        var internalLinks = new Set();
        nodes.forEach((node, index) => {
            nodeToIslandIndex.set(node,index);
            islandIndexToNodes.set(index, [node]);
            var inboundLinks = RED.nodes.getNodeLinks(node, PORT_TYPE_INPUT);
            var outboundLinks = RED.nodes.getNodeLinks(node, PORT_TYPE_OUTPUT);
            inboundLinks.forEach(l => {
                if (selectedNodes.has(l.source)) {
                    internalLinks.add(l)
                }
            })
            outboundLinks.forEach(l => {
                if (selectedNodes.has(l.target)) {
                    internalLinks.add(l)
                }
            })
        })

        internalLinks.forEach(l => {
            let source = l.source;
            let target = l.target;
            if (nodeToIslandIndex.get(source) !== nodeToIslandIndex.get(target)) {
                let sourceIsland = nodeToIslandIndex.get(source);
                let islandToMove = nodeToIslandIndex.get(target);
                let nodesToMove = islandIndexToNodes.get(islandToMove);
                nodesToMove.forEach(n => {
                    nodeToIslandIndex.set(n,sourceIsland);
                    islandIndexToNodes.get(sourceIsland).push(n);
                })
                islandIndexToNodes.delete(islandToMove);
            }
        })
        const result = [];
        islandIndexToNodes.forEach((nodes,index) => {
            result.push(nodes);
        })
        return result;
    }

    function detachNodes(nodes) {
        let allSelectedNodes = [];
        nodes.forEach(node => {
            if (node.type === 'group') {
                let groupNodes = RED.group.getNodes(node,true,true);
                allSelectedNodes = allSelectedNodes.concat(groupNodes);
            } else {
                allSelectedNodes.push(node);
            }
        })
        if (allSelectedNodes.length > 0 ) {
            const nodeIslands = RED.nodes.getNodeIslands(allSelectedNodes);
            let removedLinks = [];
            let newLinks = [];
            let createdLinkIds = new Set();

            nodeIslands.forEach(nodes => {
                let selectedNodes = new Set(nodes);
                let allInboundLinks = [];
                let allOutboundLinks = [];
                // Identify links that enter or exit this island of nodes
                nodes.forEach(node => {
                    var inboundLinks = RED.nodes.getNodeLinks(node, PORT_TYPE_INPUT);
                    var outboundLinks = RED.nodes.getNodeLinks(node, PORT_TYPE_OUTPUT);
                    inboundLinks.forEach(l => {
                        if (!selectedNodes.has(l.source)) {
                            allInboundLinks.push(l)
                        }
                    })
                    outboundLinks.forEach(l => {
                        if (!selectedNodes.has(l.target)) {
                            allOutboundLinks.push(l)
                        }
                    })
                });


                // Identify the links to restore
                allInboundLinks.forEach(inLink => {
                    // For Each inbound link,
                    //  - get source node.
                    //  - trace through to all outbound links
                    let sourceNode = inLink.source;
                    let targetNodes = new Set();
                    let visited = new Set();
                    let stack = [inLink.target];
                    while (stack.length > 0) {
                        let node = stack.pop(stack);
                        visited.add(node)
                        let links = RED.nodes.getNodeLinks(node, PORT_TYPE_OUTPUT);
                        links.forEach(l => {
                            if (visited.has(l.target)) {
                                return
                            }
                            visited.add(l.target);
                            if (selectedNodes.has(l.target)) {
                                // internal link
                                stack.push(l.target)
                            } else {
                                targetNodes.add(l.target)
                            }
                        })
                    }
                    targetNodes.forEach(target => {
                        let linkId = `${sourceNode.id}[${inLink.sourcePort}] -> ${target.id}`
                        if (!createdLinkIds.has(linkId)) {
                            createdLinkIds.add(linkId);
                            let link = {
                                source: sourceNode,
                                sourcePort: inLink.sourcePort,
                                target: target
                            }
                            let existingLinks = RED.nodes.filterLinks(link)
                            if (existingLinks.length === 0) {
                                newLinks.push(link);
                            }
                        }
                    })
                })

                // 2. delete all those links
                allInboundLinks.forEach(l => { RED.nodes.removeLink(l); removedLinks.push(l)})
                allOutboundLinks.forEach(l => { RED.nodes.removeLink(l); removedLinks.push(l)})
            })

            newLinks.forEach(l => RED.nodes.addLink(l));
            return {
                newLinks,
                removedLinks
            }
        }
    }

    /**
     * Gets the module list for the given nodes
     * @param {Array<Node>} nodes the nodes to search in
     * @returns {Record<string, string>} an object with {[moduleName]: moduleVersion}
     */
    function getModuleListForNodes(nodes) {
        const modules = {}
        const typeSet = new Set()
        nodes.forEach((n) => {
            if (!typeSet.has(n.type)) {
                typeSet.add(n.type)
                const nodeSet = RED.nodes.registry.getNodeSetForType(n.type)
                if (nodeSet) {
                    modules[nodeSet.module] = nodeSet.version
                    nodeSet.types.forEach((t) => typeSet.add(t))
                }
            }
        })
        return modules
    }

    function updateGlobalConfigModuleList(nodes) {
        const modules = getModuleListForNodes(nodes)
        delete modules['node-red']
        const hasModules = (Object.keys(modules).length > 0)
        let globalConfigNode = nodes.find((n) => n.type === 'global-config')
        if (!globalConfigNode && hasModules) {
            globalConfigNode = {
                id: RED.nodes.id(),
                type: 'global-config',
                env: [],
                modules
            }
            nodes.push(globalConfigNode)
        } else if (hasModules) {
            globalConfigNode.modules = modules
        }
    }
    return {
        init: function() {
            RED.events.on("registry:node-type-added",function(type) {
                var def = registry.getNodeType(type);
                var replaced = false;
                var replaceNodes = {};
                RED.nodes.eachNode(function(n) {
                    if (n.type === "unknown" && n.name === type) {
                        replaceNodes[n.id] = n;
                    }
                });
                RED.nodes.eachConfig(function(n) {
                    if (n.type === "unknown" && n.name === type) {
                        replaceNodes[n.id] = n;
                    }
                });

                const nodeGroupMap = {}
                var replaceNodeIds = Object.keys(replaceNodes);
                if (replaceNodeIds.length > 0) {
                    var reimportList = [];
                    replaceNodeIds.forEach(function(id) {
                        var n = replaceNodes[id];
                        if (configNodes.hasOwnProperty(n.id)) {
                            delete configNodes[n.id];
                        } else {
                            allNodes.removeNode(n);
                        }
                        if (n.g) {
                            // reimporting a node *without* including its group object
                            // will cause the g property to be cleared. Cache it
                            // here so we can restore it
                            nodeGroupMap[n.id] = n.g
                        }
                        reimportList.push(convertNode(n));
                        RED.events.emit('nodes:remove',n);
                    });

                    // Remove any links between nodes that are going to be reimported.
                    // This prevents a duplicate link from being added.
                    var removeLinks = [];
                    RED.nodes.eachLink(function(l) {
                        if (replaceNodes.hasOwnProperty(l.source.id) && replaceNodes.hasOwnProperty(l.target.id)) {
                            removeLinks.push(l);
                        }
                    });
                    removeLinks.forEach(removeLink);

                    // Force the redraw to be synchronous so the view updates
                    // *now* and removes the unknown node
                    RED.view.redraw(true, true);
                    var result = importNodes(reimportList,{generateIds:false, reimport: true});
                    var newNodeMap = {};
                    result.nodes.forEach(function(n) {
                        newNodeMap[n.id] = n;
                        if (nodeGroupMap[n.id]) {
                            // This node is in a group - need to substitute the
                            // node reference inside the group
                            n.g = nodeGroupMap[n.id]
                            const group = RED.nodes.group(n.g)
                            if (group) {
                                var index = group.nodes.findIndex(gn => gn.id === n.id)
                                if (index > -1) {
                                    group.nodes[index] = n
                                }
                            }
                        }
                    });
                    RED.nodes.eachLink(function(l) {
                        if (newNodeMap.hasOwnProperty(l.source.id)) {
                            l.source = newNodeMap[l.source.id];
                        }
                        if (newNodeMap.hasOwnProperty(l.target.id)) {
                            l.target = newNodeMap[l.target.id];
                        }
                    });
                    RED.view.redraw(true);
                }
            });
            RED.events.on('deploy', function () {
                allNodes.clearState()
            });
            RED.actions.add("core:trigger-selected-nodes-action", function () {
                const selectedNodes = RED.view.selection().nodes || [];
                // Triggers the button action of the selected nodes
                selectedNodes.forEach((node) => RED.view.clickNodeButton(node));
            });
        },
        registry:registry,
        setNodeList: registry.setNodeList,

        getNodeSet: registry.getNodeSet,
        addNodeSet: registry.addNodeSet,
        removeNodeSet: registry.removeNodeSet,
        enableNodeSet: registry.enableNodeSet,
        disableNodeSet: registry.disableNodeSet,

        setIconSets: registry.setIconSets,
        getIconSets: registry.getIconSets,

        registerType: registry.registerNodeType,
        getType: registry.getNodeType,
        getNodeHelp: getNodeHelp,
        convertNode: convertNode,
        add: addNode,
        remove: removeNode,
        clear: clear,
        detachNodes: detachNodes,
        moveNodesForwards: moveNodesForwards,
        moveNodesBackwards: moveNodesBackwards,
        moveNodesToFront: moveNodesToFront,
        moveNodesToBack: moveNodesToBack,
        getNodeOrder: getNodeOrder,
        setNodeOrder: setNodeOrder,

        moveNodeToTab: moveNodeToTab,

        addLink: addLink,
        removeLink: removeLink,
        getNodeLinks: function(id, portType) {
            if (typeof id !== 'string') {
                id = id.id;
            }
            if (nodeLinks[id]) {
                if (portType === 1) {
                    // Return cloned arrays so they can be safely modified by caller
                    return [].concat(nodeLinks[id].in)
                } else {
                    return [].concat(nodeLinks[id].out)
                }
            }
            return [];
        },
        addWorkspace: addWorkspace,
        removeWorkspace: removeWorkspace,
        getWorkspaceOrder: function() { return [...workspacesOrder] },
        setWorkspaceOrder: function(order) { workspacesOrder = order; },
        workspace: getWorkspace,

        addSubflow: addSubflow,
        removeSubflow: removeSubflow,
        subflow: getSubflow,
        subflowContains: subflowContains,

        addGroup: addGroup,
        removeGroup: removeGroup,
        group: function(id) { return groups[id] },
        groups: function(z) { return groupsByZ[z]?groupsByZ[z].slice():[] },

        addJunction: addJunction,
        removeJunction: removeJunction,
        junction: function(id) { return junctions[id] },
        junctions: function(z) { return junctionsByZ[z]?junctionsByZ[z].slice():[] },

        eachNode: function(cb) {
            allNodes.eachNode(cb);
        },
        eachLink: function(cb) {
            for (var l=0;l<links.length;l++) {
                if (cb(links[l]) === false) {
                    break;
                }
            }
        },
        eachConfig: function(cb) {
            for (var id in configNodes) {
                if (configNodes.hasOwnProperty(id)) {
                    if (cb(configNodes[id]) === false) {
                        break;
                    }
                }
            }
        },
        eachSubflow: function(cb) {
            for (var id in subflows) {
                if (subflows.hasOwnProperty(id)) {
                    if (cb(subflows[id]) === false) {
                        break;
                    }
                }
            }
        },
        eachWorkspace: function(cb) {
            for (var i=0;i<workspacesOrder.length;i++) {
                if (cb(workspaces[workspacesOrder[i]]) === false) {
                    break;
                }
            }
        },
        eachGroup: function(cb) {
            for (var group of Object.values(groups)) {
                if (cb(group) === false) {
                    break
                }
            }
        },
        eachJunction: function(cb) {
            for (var junction of Object.values(junctions)) {
                if (cb(junction) === false) {
                    break
                }
            }
        },

        node: getNode,

        version: flowVersion,
        originalFlow: function(flow) {
            if (flow === undefined) {
                return initialLoad;
            } else {
                initialLoad = flow;
            }
        },

        filterNodes: filterNodes,
        filterLinks: filterLinks,

        import: importNodes,

        identifyImportConflicts: identifyImportConflicts,

        getAllFlowNodes: getAllFlowNodes,
        getAllUpstreamNodes: getAllUpstreamNodes,
        getAllDownstreamNodes: getAllDownstreamNodes,
        getDownstreamNodes: getDownstreamNodes,
        getNodeIslands: getNodeIslands,
        createExportableNodeSet: createExportableNodeSet,
        createCompleteNodeSet: createCompleteNodeSet,
        updateConfigNodeUsers: updateConfigNodeUsers,
        id: getID,
        dirty: function(d) {
            if (d == null) {
                return dirty;
            } else {
                setDirty(d);
            }
        }
    };
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

RED.nodes.fontAwesome = (function() {

    var iconMap = {
        "fa-address-book-o": "\uf2ba",
        "fa-address-book": "\uf2b9",
        "fa-address-card-o": "\uf2bc",
        "fa-address-card": "\uf2bb",
        "fa-adjust": "\uf042",
        "fa-align-center": "\uf037",
        "fa-align-justify": "\uf039",
        "fa-align-left": "\uf036",
        "fa-align-right": "\uf038",
        "fa-ambulance": "\uf0f9",
        "fa-american-sign-language-interpreting": "\uf2a3",
        "fa-anchor": "\uf13d",
        "fa-angle-double-down": "\uf103",
        "fa-angle-double-left": "\uf100",
        "fa-angle-double-right": "\uf101",
        "fa-angle-double-up": "\uf102",
        "fa-angle-down": "\uf107",
        "fa-angle-left": "\uf104",
        "fa-angle-right": "\uf105",
        "fa-angle-up": "\uf106",
        "fa-archive": "\uf187",
        "fa-area-chart": "\uf1fe",
        "fa-arrow-circle-down": "\uf0ab",
        "fa-arrow-circle-left": "\uf0a8",
        "fa-arrow-circle-o-down": "\uf01a",
        "fa-arrow-circle-o-left": "\uf190",
        "fa-arrow-circle-o-right": "\uf18e",
        "fa-arrow-circle-o-up": "\uf01b",
        "fa-arrow-circle-right": "\uf0a9",
        "fa-arrow-circle-up": "\uf0aa",
        "fa-arrow-down": "\uf063",
        "fa-arrow-left": "\uf060",
        "fa-arrow-right": "\uf061",
        "fa-arrow-up": "\uf062",
        "fa-arrows-alt": "\uf0b2",
        "fa-arrows-h": "\uf07e",
        "fa-arrows-v": "\uf07d",
        "fa-arrows": "\uf047",
        "fa-asl-interpreting": "\uf2a3",
        "fa-assistive-listening-systems": "\uf2a2",
        "fa-asterisk": "\uf069",
        "fa-at": "\uf1fa",
        "fa-audio-description": "\uf29e",
        "fa-automobile": "\uf1b9",
        "fa-backward": "\uf04a",
        "fa-balance-scale": "\uf24e",
        "fa-ban": "\uf05e",
        "fa-bank": "\uf19c",
        "fa-bar-chart-o": "\uf080",
        "fa-bar-chart": "\uf080",
        "fa-barcode": "\uf02a",
        "fa-bars": "\uf0c9",
        "fa-bath": "\uf2cd",
        "fa-bathtub": "\uf2cd",
        "fa-battery-0": "\uf244",
        "fa-battery-1": "\uf243",
        "fa-battery-2": "\uf242",
        "fa-battery-3": "\uf241",
        "fa-battery-4": "\uf240",
        "fa-battery-empty": "\uf244",
        "fa-battery-full": "\uf240",
        "fa-battery-half": "\uf242",
        "fa-battery-quarter": "\uf243",
        "fa-battery-three-quarters": "\uf241",
        "fa-battery": "\uf240",
        "fa-bed": "\uf236",
        "fa-beer": "\uf0fc",
        "fa-bell-o": "\uf0a2",
        "fa-bell-slash-o": "\uf1f7",
        "fa-bell-slash": "\uf1f6",
        "fa-bell": "\uf0f3",
        "fa-bicycle": "\uf206",
        "fa-binoculars": "\uf1e5",
        "fa-birthday-cake": "\uf1fd",
        "fa-blind": "\uf29d",
        "fa-bold": "\uf032",
        "fa-bolt": "\uf0e7",
        "fa-bomb": "\uf1e2",
        "fa-book": "\uf02d",
        "fa-bookmark-o": "\uf097",
        "fa-bookmark": "\uf02e",
        "fa-braille": "\uf2a1",
        "fa-briefcase": "\uf0b1",
        "fa-bug": "\uf188",
        "fa-building-o": "\uf0f7",
        "fa-building": "\uf1ad",
        "fa-bullhorn": "\uf0a1",
        "fa-bullseye": "\uf140",
        "fa-bus": "\uf207",
        "fa-cab": "\uf1ba",
        "fa-calculator": "\uf1ec",
        "fa-calendar-check-o": "\uf274",
        "fa-calendar-minus-o": "\uf272",
        "fa-calendar-o": "\uf133",
        "fa-calendar-plus-o": "\uf271",
        "fa-calendar-times-o": "\uf273",
        "fa-calendar": "\uf073",
        "fa-camera-retro": "\uf083",
        "fa-camera": "\uf030",
        "fa-car": "\uf1b9",
        "fa-caret-down": "\uf0d7",
        "fa-caret-left": "\uf0d9",
        "fa-caret-right": "\uf0da",
        "fa-caret-square-o-down": "\uf150",
        "fa-caret-square-o-left": "\uf191",
        "fa-caret-square-o-right": "\uf152",
        "fa-caret-square-o-up": "\uf151",
        "fa-caret-up": "\uf0d8",
        "fa-cart-arrow-down": "\uf218",
        "fa-cart-plus": "\uf217",
        "fa-cc": "\uf20a",
        "fa-certificate": "\uf0a3",
        "fa-chain-broken": "\uf127",
        "fa-chain": "\uf0c1",
        "fa-check-circle-o": "\uf05d",
        "fa-check-circle": "\uf058",
        "fa-check-square-o": "\uf046",
        "fa-check-square": "\uf14a",
        "fa-check": "\uf00c",
        "fa-chevron-circle-down": "\uf13a",
        "fa-chevron-circle-left": "\uf137",
        "fa-chevron-circle-right": "\uf138",
        "fa-chevron-circle-up": "\uf139",
        "fa-chevron-down": "\uf078",
        "fa-chevron-left": "\uf053",
        "fa-chevron-right": "\uf054",
        "fa-chevron-up": "\uf077",
        "fa-child": "\uf1ae",
        "fa-circle-o-notch": "\uf1ce",
        "fa-circle-o": "\uf10c",
        "fa-circle-thin": "\uf1db",
        "fa-circle": "\uf111",
        "fa-clipboard": "\uf0ea",
        "fa-clock-o": "\uf017",
        "fa-clone": "\uf24d",
        "fa-close": "\uf00d",
        "fa-cloud-download": "\uf0ed",
        "fa-cloud-upload": "\uf0ee",
        "fa-cloud": "\uf0c2",
        "fa-cny": "\uf157",
        "fa-code-fork": "\uf126",
        "fa-code": "\uf121",
        "fa-coffee": "\uf0f4",
        "fa-cog": "\uf013",
        "fa-cogs": "\uf085",
        "fa-columns": "\uf0db",
        "fa-comment-o": "\uf0e5",
        "fa-comment": "\uf075",
        "fa-commenting-o": "\uf27b",
        "fa-commenting": "\uf27a",
        "fa-comments-o": "\uf0e6",
        "fa-comments": "\uf086",
        "fa-compass": "\uf14e",
        "fa-compress": "\uf066",
        "fa-copy": "\uf0c5",
        "fa-copyright": "\uf1f9",
        "fa-creative-commons": "\uf25e",
        "fa-credit-card-alt": "\uf283",
        "fa-credit-card": "\uf09d",
        "fa-crop": "\uf125",
        "fa-crosshairs": "\uf05b",
        "fa-cube": "\uf1b2",
        "fa-cubes": "\uf1b3",
        "fa-cut": "\uf0c4",
        "fa-cutlery": "\uf0f5",
        "fa-dashboard": "\uf0e4",
        "fa-database": "\uf1c0",
        "fa-deaf": "\uf2a4",
        "fa-deafness": "\uf2a4",
        "fa-dedent": "\uf03b",
        "fa-desktop": "\uf108",
        "fa-diamond": "\uf219",
        "fa-dollar": "\uf155",
        "fa-dot-circle-o": "\uf192",
        "fa-download": "\uf019",
        "fa-drivers-license-o": "\uf2c3",
        "fa-drivers-license": "\uf2c2",
        "fa-edit": "\uf044",
        "fa-eject": "\uf052",
        "fa-ellipsis-h": "\uf141",
        "fa-ellipsis-v": "\uf142",
        "fa-envelope-o": "\uf003",
        "fa-envelope-open-o": "\uf2b7",
        "fa-envelope-open": "\uf2b6",
        "fa-envelope-square": "\uf199",
        "fa-envelope": "\uf0e0",
        "fa-eraser": "\uf12d",
        "fa-eur": "\uf153",
        "fa-euro": "\uf153",
        "fa-exchange": "\uf0ec",
        "fa-exclamation-circle": "\uf06a",
        "fa-exclamation-triangle": "\uf071",
        "fa-exclamation": "\uf12a",
        "fa-expand": "\uf065",
        "fa-external-link-square": "\uf14c",
        "fa-external-link": "\uf08e",
        "fa-eye-slash": "\uf070",
        "fa-eye": "\uf06e",
        "fa-eyedropper": "\uf1fb",
        "fa-fast-backward": "\uf049",
        "fa-fast-forward": "\uf050",
        "fa-fax": "\uf1ac",
        "fa-feed": "\uf09e",
        "fa-female": "\uf182",
        "fa-fighter-jet": "\uf0fb",
        "fa-file-archive-o": "\uf1c6",
        "fa-file-audio-o": "\uf1c7",
        "fa-file-code-o": "\uf1c9",
        "fa-file-excel-o": "\uf1c3",
        "fa-file-image-o": "\uf1c5",
        "fa-file-movie-o": "\uf1c8",
        "fa-file-o": "\uf016",
        "fa-file-pdf-o": "\uf1c1",
        "fa-file-photo-o": "\uf1c5",
        "fa-file-picture-o": "\uf1c5",
        "fa-file-powerpoint-o": "\uf1c4",
        "fa-file-sound-o": "\uf1c7",
        "fa-file-text-o": "\uf0f6",
        "fa-file-text": "\uf15c",
        "fa-file-video-o": "\uf1c8",
        "fa-file-word-o": "\uf1c2",
        "fa-file-zip-o": "\uf1c6",
        "fa-file": "\uf15b",
        "fa-files-o": "\uf0c5",
        "fa-film": "\uf008",
        "fa-filter": "\uf0b0",
        "fa-fire-extinguisher": "\uf134",
        "fa-fire": "\uf06d",
        "fa-flag-checkered": "\uf11e",
        "fa-flag-o": "\uf11d",
        "fa-flag": "\uf024",
        "fa-flash": "\uf0e7",
        "fa-flask": "\uf0c3",
        "fa-floppy-o": "\uf0c7",
        "fa-folder-o": "\uf114",
        "fa-folder-open-o": "\uf115",
        "fa-folder-open": "\uf07c",
        "fa-folder": "\uf07b",
        "fa-font": "\uf031",
        "fa-forward": "\uf04e",
        "fa-frown-o": "\uf119",
        "fa-futbol-o": "\uf1e3",
        "fa-gamepad": "\uf11b",
        "fa-gavel": "\uf0e3",
        "fa-gbp": "\uf154",
        "fa-gear": "\uf013",
        "fa-gears": "\uf085",
        "fa-genderless": "\uf22d",
        "fa-gift": "\uf06b",
        "fa-glass": "\uf000",
        "fa-globe": "\uf0ac",
        "fa-graduation-cap": "\uf19d",
        "fa-group": "\uf0c0",
        "fa-h-square": "\uf0fd",
        "fa-hand-grab-o": "\uf255",
        "fa-hand-lizard-o": "\uf258",
        "fa-hand-o-down": "\uf0a7",
        "fa-hand-o-left": "\uf0a5",
        "fa-hand-o-right": "\uf0a4",
        "fa-hand-o-up": "\uf0a6",
        "fa-hand-paper-o": "\uf256",
        "fa-hand-peace-o": "\uf25b",
        "fa-hand-pointer-o": "\uf25a",
        "fa-hand-rock-o": "\uf255",
        "fa-hand-scissors-o": "\uf257",
        "fa-hand-spock-o": "\uf259",
        "fa-hand-stop-o": "\uf256",
        "fa-handshake-o": "\uf2b5",
        "fa-hard-of-hearing": "\uf2a4",
        "fa-hashtag": "\uf292",
        "fa-hdd-o": "\uf0a0",
        "fa-header": "\uf1dc",
        "fa-headphones": "\uf025",
        "fa-heart-o": "\uf08a",
        "fa-heart": "\uf004",
        "fa-heartbeat": "\uf21e",
        "fa-history": "\uf1da",
        "fa-home": "\uf015",
        "fa-hospital-o": "\uf0f8",
        "fa-hotel": "\uf236",
        "fa-hourglass-1": "\uf251",
        "fa-hourglass-2": "\uf252",
        "fa-hourglass-3": "\uf253",
        "fa-hourglass-end": "\uf253",
        "fa-hourglass-half": "\uf252",
        "fa-hourglass-o": "\uf250",
        "fa-hourglass-start": "\uf251",
        "fa-hourglass": "\uf254",
        "fa-i-cursor": "\uf246",
        "fa-id-badge": "\uf2c1",
        "fa-id-card-o": "\uf2c3",
        "fa-id-card": "\uf2c2",
        "fa-ils": "\uf20b",
        "fa-image": "\uf03e",
        "fa-inbox": "\uf01c",
        "fa-indent": "\uf03c",
        "fa-industry": "\uf275",
        "fa-info-circle": "\uf05a",
        "fa-info": "\uf129",
        "fa-inr": "\uf156",
        "fa-institution": "\uf19c",
        "fa-intersex": "\uf224",
        "fa-italic": "\uf033",
        "fa-jpy": "\uf157",
        "fa-key": "\uf084",
        "fa-keyboard-o": "\uf11c",
        "fa-krw": "\uf159",
        "fa-language": "\uf1ab",
        "fa-laptop": "\uf109",
        "fa-leaf": "\uf06c",
        "fa-legal": "\uf0e3",
        "fa-lemon-o": "\uf094",
        "fa-level-down": "\uf149",
        "fa-level-up": "\uf148",
        "fa-life-bouy": "\uf1cd",
        "fa-life-buoy": "\uf1cd",
        "fa-life-ring": "\uf1cd",
        "fa-life-saver": "\uf1cd",
        "fa-lightbulb-o": "\uf0eb",
        "fa-line-chart": "\uf201",
        "fa-link": "\uf0c1",
        "fa-list-alt": "\uf022",
        "fa-list-ol": "\uf0cb",
        "fa-list-ul": "\uf0ca",
        "fa-list": "\uf03a",
        "fa-location-arrow": "\uf124",
        "fa-lock": "\uf023",
        "fa-long-arrow-down": "\uf175",
        "fa-long-arrow-left": "\uf177",
        "fa-long-arrow-right": "\uf178",
        "fa-long-arrow-up": "\uf176",
        "fa-low-vision": "\uf2a8",
        "fa-magic": "\uf0d0",
        "fa-magnet": "\uf076",
        "fa-mail-forward": "\uf064",
        "fa-mail-reply-all": "\uf122",
        "fa-mail-reply": "\uf112",
        "fa-male": "\uf183",
        "fa-map-marker": "\uf041",
        "fa-map-o": "\uf278",
        "fa-map-pin": "\uf276",
        "fa-map-signs": "\uf277",
        "fa-map": "\uf279",
        "fa-mars-double": "\uf227",
        "fa-mars-stroke-h": "\uf22b",
        "fa-mars-stroke-v": "\uf22a",
        "fa-mars-stroke": "\uf229",
        "fa-mars": "\uf222",
        "fa-medkit": "\uf0fa",
        "fa-meh-o": "\uf11a",
        "fa-mercury": "\uf223",
        "fa-microchip": "\uf2db",
        "fa-microphone-slash": "\uf131",
        "fa-microphone": "\uf130",
        "fa-minus-circle": "\uf056",
        "fa-minus-square-o": "\uf147",
        "fa-minus-square": "\uf146",
        "fa-minus": "\uf068",
        "fa-mobile-phone": "\uf10b",
        "fa-mobile": "\uf10b",
        "fa-money": "\uf0d6",
        "fa-moon-o": "\uf186",
        "fa-mortar-board": "\uf19d",
        "fa-motorcycle": "\uf21c",
        "fa-mouse-pointer": "\uf245",
        "fa-music": "\uf001",
        "fa-navicon": "\uf0c9",
        "fa-neuter": "\uf22c",
        "fa-newspaper-o": "\uf1ea",
        "fa-object-group": "\uf247",
        "fa-object-ungroup": "\uf248",
        "fa-outdent": "\uf03b",
        "fa-paint-brush": "\uf1fc",
        "fa-paper-plane-o": "\uf1d9",
        "fa-paper-plane": "\uf1d8",
        "fa-paperclip": "\uf0c6",
        "fa-paragraph": "\uf1dd",
        "fa-paste": "\uf0ea",
        "fa-pause-circle-o": "\uf28c",
        "fa-pause-circle": "\uf28b",
        "fa-pause": "\uf04c",
        "fa-paw": "\uf1b0",
        "fa-pencil-square-o": "\uf044",
        "fa-pencil-square": "\uf14b",
        "fa-pencil": "\uf040",
        "fa-percent": "\uf295",
        "fa-phone-square": "\uf098",
        "fa-phone": "\uf095",
        "fa-photo": "\uf03e",
        "fa-picture-o": "\uf03e",
        "fa-pie-chart": "\uf200",
        "fa-plane": "\uf072",
        "fa-play-circle-o": "\uf01d",
        "fa-play-circle": "\uf144",
        "fa-play": "\uf04b",
        "fa-plug": "\uf1e6",
        "fa-plus-circle": "\uf055",
        "fa-plus-square-o": "\uf196",
        "fa-plus-square": "\uf0fe",
        "fa-plus": "\uf067",
        "fa-podcast": "\uf2ce",
        "fa-power-off": "\uf011",
        "fa-print": "\uf02f",
        "fa-puzzle-piece": "\uf12e",
        "fa-qrcode": "\uf029",
        "fa-question-circle-o": "\uf29c",
        "fa-question-circle": "\uf059",
        "fa-question": "\uf128",
        "fa-quote-left": "\uf10d",
        "fa-quote-right": "\uf10e",
        "fa-random": "\uf074",
        "fa-recycle": "\uf1b8",
        "fa-refresh": "\uf021",
        "fa-registered": "\uf25d",
        "fa-remove": "\uf00d",
        "fa-reorder": "\uf0c9",
        "fa-repeat": "\uf01e",
        "fa-reply-all": "\uf122",
        "fa-reply": "\uf112",
        "fa-retweet": "\uf079",
        "fa-rmb": "\uf157",
        "fa-road": "\uf018",
        "fa-rocket": "\uf135",
        "fa-rotate-left": "\uf0e2",
        "fa-rotate-right": "\uf01e",
        "fa-rouble": "\uf158",
        "fa-rss-square": "\uf143",
        "fa-rss": "\uf09e",
        "fa-rub": "\uf158",
        "fa-ruble": "\uf158",
        "fa-rupee": "\uf156",
        "fa-s15": "\uf2cd",
        "fa-save": "\uf0c7",
        "fa-scissors": "\uf0c4",
        "fa-search-minus": "\uf010",
        "fa-search-plus": "\uf00e",
        "fa-search": "\uf002",
        "fa-send-o": "\uf1d9",
        "fa-send": "\uf1d8",
        "fa-server": "\uf233",
        "fa-share-square-o": "\uf045",
        "fa-share-square": "\uf14d",
        "fa-share": "\uf064",
        "fa-shekel": "\uf20b",
        "fa-sheqel": "\uf20b",
        "fa-shield": "\uf132",
        "fa-ship": "\uf21a",
        "fa-shopping-bag": "\uf290",
        "fa-shopping-basket": "\uf291",
        "fa-shopping-cart": "\uf07a",
        "fa-shower": "\uf2cc",
        "fa-sign-in": "\uf090",
        "fa-sign-language": "\uf2a7",
        "fa-sign-out": "\uf08b",
        "fa-signal": "\uf012",
        "fa-signing": "\uf2a7",
        "fa-sitemap": "\uf0e8",
        "fa-sliders": "\uf1de",
        "fa-smile-o": "\uf118",
        "fa-snowflake-o": "\uf2dc",
        "fa-soccer-ball-o": "\uf1e3",
        "fa-sort-alpha-asc": "\uf15d",
        "fa-sort-alpha-desc": "\uf15e",
        "fa-sort-amount-asc": "\uf160",
        "fa-sort-amount-desc": "\uf161",
        "fa-sort-asc": "\uf0de",
        "fa-sort-desc": "\uf0dd",
        "fa-sort-down": "\uf0dd",
        "fa-sort-numeric-asc": "\uf162",
        "fa-sort-numeric-desc": "\uf163",
        "fa-sort-up": "\uf0de",
        "fa-sort": "\uf0dc",
        "fa-space-shuttle": "\uf197",
        "fa-spinner": "\uf110",
        "fa-spoon": "\uf1b1",
        "fa-square-o": "\uf096",
        "fa-square": "\uf0c8",
        "fa-star-half-empty": "\uf123",
        "fa-star-half-full": "\uf123",
        "fa-star-half-o": "\uf123",
        "fa-star-half": "\uf089",
        "fa-star-o": "\uf006",
        "fa-star": "\uf005",
        "fa-step-backward": "\uf048",
        "fa-step-forward": "\uf051",
        "fa-stethoscope": "\uf0f1",
        "fa-sticky-note-o": "\uf24a",
        "fa-sticky-note": "\uf249",
        "fa-stop-circle-o": "\uf28e",
        "fa-stop-circle": "\uf28d",
        "fa-stop": "\uf04d",
        "fa-street-view": "\uf21d",
        "fa-strikethrough": "\uf0cc",
        "fa-subscript": "\uf12c",
        "fa-subway": "\uf239",
        "fa-suitcase": "\uf0f2",
        "fa-sun-o": "\uf185",
        "fa-superscript": "\uf12b",
        "fa-support": "\uf1cd",
        "fa-table": "\uf0ce",
        "fa-tablet": "\uf10a",
        "fa-tachometer": "\uf0e4",
        "fa-tag": "\uf02b",
        "fa-tags": "\uf02c",
        "fa-tasks": "\uf0ae",
        "fa-taxi": "\uf1ba",
        "fa-television": "\uf26c",
        "fa-terminal": "\uf120",
        "fa-text-height": "\uf034",
        "fa-text-width": "\uf035",
        "fa-th-large": "\uf009",
        "fa-th-list": "\uf00b",
        "fa-th": "\uf00a",
        "fa-thermometer-0": "\uf2cb",
        "fa-thermometer-1": "\uf2ca",
        "fa-thermometer-2": "\uf2c9",
        "fa-thermometer-3": "\uf2c8",
        "fa-thermometer-4": "\uf2c7",
        "fa-thermometer-empty": "\uf2cb",
        "fa-thermometer-full": "\uf2c7",
        "fa-thermometer-half": "\uf2c9",
        "fa-thermometer-quarter": "\uf2ca",
        "fa-thermometer-three-quarters": "\uf2c8",
        "fa-thermometer": "\uf2c7",
        "fa-thumb-tack": "\uf08d",
        "fa-thumbs-down": "\uf165",
        "fa-thumbs-o-down": "\uf088",
        "fa-thumbs-o-up": "\uf087",
        "fa-thumbs-up": "\uf164",
        "fa-ticket": "\uf145",
        "fa-times-circle-o": "\uf05c",
        "fa-times-circle": "\uf057",
        "fa-times-rectangle-o": "\uf2d4",
        "fa-times-rectangle": "\uf2d3",
        "fa-times": "\uf00d",
        "fa-tint": "\uf043",
        "fa-toggle-down": "\uf150",
        "fa-toggle-left": "\uf191",
        "fa-toggle-off": "\uf204",
        "fa-toggle-on": "\uf205",
        "fa-toggle-right": "\uf152",
        "fa-toggle-up": "\uf151",
        "fa-trademark": "\uf25c",
        "fa-train": "\uf238",
        "fa-transgender-alt": "\uf225",
        "fa-transgender": "\uf224",
        "fa-trash-o": "\uf014",
        "fa-trash": "\uf1f8",
        "fa-tree": "\uf1bb",
        "fa-trophy": "\uf091",
        "fa-truck": "\uf0d1",
        "fa-try": "\uf195",
        "fa-tty": "\uf1e4",
        "fa-turkish-lira": "\uf195",
        "fa-tv": "\uf26c",
        "fa-umbrella": "\uf0e9",
        "fa-underline": "\uf0cd",
        "fa-undo": "\uf0e2",
        "fa-universal-access": "\uf29a",
        "fa-university": "\uf19c",
        "fa-unlink": "\uf127",
        "fa-unlock-alt": "\uf13e",
        "fa-unlock": "\uf09c",
        "fa-unsorted": "\uf0dc",
        "fa-upload": "\uf093",
        "fa-usd": "\uf155",
        "fa-user-circle-o": "\uf2be",
        "fa-user-circle": "\uf2bd",
        "fa-user-md": "\uf0f0",
        "fa-user-o": "\uf2c0",
        "fa-user-plus": "\uf234",
        "fa-user-secret": "\uf21b",
        "fa-user-times": "\uf235",
        "fa-user": "\uf007",
        "fa-users": "\uf0c0",
        "fa-vcard-o": "\uf2bc",
        "fa-vcard": "\uf2bb",
        "fa-venus-double": "\uf226",
        "fa-venus-mars": "\uf228",
        "fa-venus": "\uf221",
        "fa-video-camera": "\uf03d",
        "fa-volume-control-phone": "\uf2a0",
        "fa-volume-down": "\uf027",
        "fa-volume-off": "\uf026",
        "fa-volume-up": "\uf028",
        "fa-warning": "\uf071",
        "fa-wheelchair-alt": "\uf29b",
        "fa-wheelchair": "\uf193",
        "fa-wifi": "\uf1eb",
        "fa-window-close-o": "\uf2d4",
        "fa-window-close": "\uf2d3",
        "fa-window-maximize": "\uf2d0",
        "fa-window-minimize": "\uf2d1",
        "fa-window-restore": "\uf2d2",
        "fa-won": "\uf159",
        "fa-wrench": "\uf0ad",
        "fa-yen": "\uf157"
    };

    var brandIconMap = {
        "fa-500px": "\uf26e",
        "fa-adn": "\uf170",
        "fa-amazon": "\uf270",
        "fa-android": "\uf17b",
        "fa-angellist": "\uf209",
        "fa-apple": "\uf179",
        "fa-bandcamp": "\uf2d5",
        "fa-behance-square": "\uf1b5",
        "fa-behance": "\uf1b4",
        "fa-bitbucket-square": "\uf172",
        "fa-bitbucket": "\uf171",
        "fa-bitcoin": "\uf15a",
        "fa-black-tie": "\uf27e",
        "fa-bluetooth-b": "\uf294",
        "fa-bluetooth": "\uf293",
        "fa-btc": "\uf15a",
        "fa-buysellads": "\uf20d",
        "fa-cc-amex": "\uf1f3",
        "fa-cc-diners-club": "\uf24c",
        "fa-cc-discover": "\uf1f2",
        "fa-cc-jcb": "\uf24b",
        "fa-cc-mastercard": "\uf1f1",
        "fa-cc-paypal": "\uf1f4",
        "fa-cc-stripe": "\uf1f5",
        "fa-cc-visa": "\uf1f0",
        "fa-chrome": "\uf268",
        "fa-codepen": "\uf1cb",
        "fa-codiepie": "\uf284",
        "fa-connectdevelop": "\uf20e",
        "fa-contao": "\uf26d",
        "fa-css3": "\uf13c",
        "fa-dashcube": "\uf210",
        "fa-delicious": "\uf1a5",
        "fa-deviantart": "\uf1bd",
        "fa-digg": "\uf1a6",
        "fa-dribbble": "\uf17d",
        "fa-dropbox": "\uf16b",
        "fa-drupal": "\uf1a9",
        "fa-edge": "\uf282",
        "fa-eercast": "\uf2da",
        "fa-empire": "\uf1d1",
        "fa-envira": "\uf299",
        "fa-etsy": "\uf2d7",
        "fa-expeditedssl": "\uf23e",
        "fa-fa": "\uf2b4",
        "fa-facebook-f": "\uf09a",
        "fa-facebook-official": "\uf230",
        "fa-facebook-square": "\uf082",
        "fa-facebook": "\uf09a",
        "fa-firefox": "\uf269",
        "fa-first-order": "\uf2b0",
        "fa-flickr": "\uf16e",
        "fa-font-awesome": "\uf2b4",
        "fa-fonticons": "\uf280",
        "fa-fort-awesome": "\uf286",
        "fa-forumbee": "\uf211",
        "fa-foursquare": "\uf180",
        "fa-free-code-camp": "\uf2c5",
        "fa-ge": "\uf1d1",
        "fa-get-pocket": "\uf265",
        "fa-gg-circle": "\uf261",
        "fa-gg": "\uf260",
        "fa-git-square": "\uf1d2",
        "fa-git": "\uf1d3",
        "fa-github-alt": "\uf113",
        "fa-github-square": "\uf092",
        "fa-github": "\uf09b",
        "fa-gitlab": "\uf296",
        "fa-gittip": "\uf184",
        "fa-glide-g": "\uf2a6",
        "fa-glide": "\uf2a5",
        "fa-google-plus-circle": "\uf2b3",
        "fa-google-plus-official": "\uf2b3",
        "fa-google-plus-square": "\uf0d4",
        "fa-google-plus": "\uf0d5",
        "fa-google-wallet": "\uf1ee",
        "fa-google": "\uf1a0",
        "fa-gratipay": "\uf184",
        "fa-grav": "\uf2d6",
        "fa-hacker-news": "\uf1d4",
        "fa-houzz": "\uf27c",
        "fa-html5": "\uf13b",
        "fa-imdb": "\uf2d8",
        "fa-instagram": "\uf16d",
        "fa-internet-explorer": "\uf26b",
        "fa-ioxhost": "\uf208",
        "fa-joomla": "\uf1aa",
        "fa-jsfiddle": "\uf1cc",
        "fa-lastfm-square": "\uf203",
        "fa-lastfm": "\uf202",
        "fa-leanpub": "\uf212",
        "fa-linkedin-square": "\uf08c",
        "fa-linkedin": "\uf0e1",
        "fa-linode": "\uf2b8",
        "fa-linux": "\uf17c",
        "fa-maxcdn": "\uf136",
        "fa-meanpath": "\uf20c",
        "fa-medium": "\uf23a",
        "fa-meetup": "\uf2e0",
        "fa-mixcloud": "\uf289",
        "fa-modx": "\uf285",
        "fa-odnoklassniki-square": "\uf264",
        "fa-odnoklassniki": "\uf263",
        "fa-opencart": "\uf23d",
        "fa-openid": "\uf19b",
        "fa-opera": "\uf26a",
        "fa-optin-monster": "\uf23c",
        "fa-pagelines": "\uf18c",
        "fa-paypal": "\uf1ed",
        "fa-pied-piper-alt": "\uf1a8",
        "fa-pied-piper-pp": "\uf1a7",
        "fa-pied-piper": "\uf2ae",
        "fa-pinterest-p": "\uf231",
        "fa-pinterest-square": "\uf0d3",
        "fa-pinterest": "\uf0d2",
        "fa-product-hunt": "\uf288",
        "fa-qq": "\uf1d6",
        "fa-quora": "\uf2c4",
        "fa-ra": "\uf1d0",
        "fa-ravelry": "\uf2d9",
        "fa-rebel": "\uf1d0",
        "fa-reddit-alien": "\uf281",
        "fa-reddit-square": "\uf1a2",
        "fa-reddit": "\uf1a1",
        "fa-renren": "\uf18b",
        "fa-resistance": "\uf1d0",
        "fa-safari": "\uf267",
        "fa-scribd": "\uf28a",
        "fa-sellsy": "\uf213",
        "fa-share-alt-square": "\uf1e1",
        "fa-share-alt": "\uf1e0",
        "fa-shirtsinbulk": "\uf214",
        "fa-simplybuilt": "\uf215",
        "fa-skyatlas": "\uf216",
        "fa-skype": "\uf17e",
        "fa-slack": "\uf198",
        "fa-slideshare": "\uf1e7",
        "fa-snapchat-ghost": "\uf2ac",
        "fa-snapchat-square": "\uf2ad",
        "fa-snapchat": "\uf2ab",
        "fa-soundcloud": "\uf1be",
        "fa-spotify": "\uf1bc",
        "fa-stack-exchange": "\uf18d",
        "fa-stack-overflow": "\uf16c",
        "fa-steam-square": "\uf1b7",
        "fa-steam": "\uf1b6",
        "fa-stumbleupon-circle": "\uf1a3",
        "fa-stumbleupon": "\uf1a4",
        "fa-superpowers": "\uf2dd",
        "fa-telegram": "\uf2c6",
        "fa-tencent-weibo": "\uf1d5",
        "fa-themeisle": "\uf2b2",
        "fa-trello": "\uf181",
        "fa-tripadvisor": "\uf262",
        "fa-tumblr-square": "\uf174",
        "fa-tumblr": "\uf173",
        "fa-twitch": "\uf1e8",
        "fa-twitter-square": "\uf081",
        "fa-twitter": "\uf099",
        "fa-usb": "\uf287",
        "fa-viacoin": "\uf237",
        "fa-viadeo-square": "\uf2aa",
        "fa-viadeo": "\uf2a9",
        "fa-vimeo-square": "\uf194",
        "fa-vimeo": "\uf27d",
        "fa-vine": "\uf1ca",
        "fa-vk": "\uf189",
        "fa-wechat": "\uf1d7",
        "fa-weibo": "\uf18a",
        "fa-weixin": "\uf1d7",
        "fa-whatsapp": "\uf232",
        "fa-wikipedia-w": "\uf266",
        "fa-windows": "\uf17a",
        "fa-wordpress": "\uf19a",
        "fa-wpbeginner": "\uf297",
        "fa-wpexplorer": "\uf2de",
        "fa-wpforms": "\uf298",
        "fa-xing-square": "\uf169",
        "fa-xing": "\uf168",
        "fa-y-combinator-square": "\uf1d4",
        "fa-y-combinator": "\uf23b",
        "fa-yahoo": "\uf19e",
        "fa-yc-square": "\uf1d4",
        "fa-yc": "\uf23b",
        "fa-yelp": "\uf1e9",
        "fa-yoast": "\uf2b1",
        "fa-youtube-play": "\uf16a",
        "fa-youtube-square": "\uf166",
        "fa-youtube": "\uf167",
    };

    var iconList = Object.keys(iconMap);

    return {
        getIconUnicode: function(name) {
            return iconMap[name] || brandIconMap[name];
        },
        getIconList: function() {
            return iconList;
        },
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

/**
 * An API for undo / redo history buffer
 * @namespace RED.history
*/
RED.history = (function() {
    var undoHistory = [];
    var redoHistory = [];

    function nodeOrJunction(id) {
        var node = RED.nodes.node(id);
        if (node) {
            return node;
        }
        return RED.nodes.junction(id);
    }
    function ensureUnlocked(id, flowsToLock) {
        const flow = id && (RED.nodes.workspace(id) || RED.nodes.subflow(id) || null);
        const isLocked = flow ? flow.locked : false;
        if (flow && isLocked) {
            flow.locked = false;
            flowsToLock.add(flow)
        }
    }
    function undoEvent(ev) {
        var i;
        var len;
        var node;
        var group;
        var subflow;
        var modifiedTabs = {};
        var inverseEv;
        if (ev) {
            if (ev.t == 'multi') {
                inverseEv = {
                    t: 'multi',
                    events: []
                };
                len = ev.events.length;
                for (i=len-1;i>=0;i--) {
                    var r = undoEvent(ev.events[i]);
                    inverseEv.events.push(r);
                }
            } else if (ev.t == 'replace') {
                if (ev.complete) {
                    // This is a replace of everything. We can short-cut
                    // the logic by clearing everyting first, then importing
                    // the ev.config.
                    // Used by RED.diff.mergeDiff
                    inverseEv = {
                        t: 'replace',
                        config: RED.nodes.createCompleteNodeSet(),
                        changed: {},
                        moved: {},
                        complete: true,
                        rev: RED.nodes.version(),
                        dirty: RED.nodes.dirty()
                    };
                    var selectedTab = RED.workspaces.active();
                    inverseEv.config.forEach(n => {
                        const node = RED.nodes.node(n.id)
                        if (node) {
                            inverseEv.changed[n.id] = node.changed
                            inverseEv.moved[n.id] = node.moved
                        }
                    })
                    RED.nodes.clear();
                    var imported = RED.nodes.import(ev.config);
                    // Clear all change flags from the import
                    RED.nodes.dirty(false);

                    const flowsToLock = new Set()
                    
                    imported.nodes.forEach(function(n) {
                        if (ev.changed[n.id]) {
                            ensureUnlocked(n.z, flowsToLock)
                            n.changed = true;
                        }
                        if (ev.moved[n.id]) {
                            ensureUnlocked(n.z, flowsToLock)
                            n.moved = true;
                        }
                    })
                    flowsToLock.forEach(flow => {
                        flow.locked = true
                    })

                    RED.nodes.version(ev.rev);
                    RED.view.redraw(true);
                    RED.palette.refresh();
                    RED.workspaces.refresh();
                    RED.workspaces.show(selectedTab, true);
                    RED.sidebar.config.refresh();
                } else {
                    var importMap = {};
                    ev.config.forEach(function(n) {
                        importMap[n.id] = "replace";
                    })
                    var importedResult = RED.nodes.import(ev.config,{importMap: importMap})
                    inverseEv = {
                        t: 'replace',
                        config: importedResult.removedNodes,
                        dirty: RED.nodes.dirty()
                    }
                }
            } else if (ev.t == 'add') {
                inverseEv = {
                    t: "delete",
                    dirty: RED.nodes.dirty()
                };
                if (ev.nodes) {
                    inverseEv.nodes = [];
                    for (i=0;i<ev.nodes.length;i++) {
                        node = RED.nodes.node(ev.nodes[i]);
                        if (node.z) {
                            modifiedTabs[node.z] = true;
                        }
                        inverseEv.nodes.push(node);
                        RED.nodes.remove(ev.nodes[i]);
                        if (node.g) {
                            var group = RED.nodes.group(node.g);
                            var index = group.nodes.indexOf(node);
                            if (index !== -1) {
                                group.nodes.splice(index,1);
                                RED.group.markDirty(group);
                            }
                        }
                    }
                }
                if (ev.links) {
                    inverseEv.links = [];
                    for (i=0;i<ev.links.length;i++) {
                        inverseEv.links.push(ev.links[i]);
                        RED.nodes.removeLink(ev.links[i]);
                    }
                }
                if (ev.junctions) {
                    inverseEv.junctions = [];
                    for (i=0;i<ev.junctions.length;i++) {
                        inverseEv.junctions.push(ev.junctions[i]);
                        RED.nodes.removeJunction(ev.junctions[i]);
                        if (ev.junctions[i].g) {
                            var group = RED.nodes.group(ev.junctions[i].g);
                            var index = group.nodes.indexOf(ev.junctions[i]);
                            if (index !== -1) {
                                group.nodes.splice(index,1);
                                RED.group.markDirty(group);
                            }
                        }


                    }
                }
                if (ev.groups) {
                    inverseEv.groups = [];
                    for (i = ev.groups.length - 1;i>=0;i--) {
                        group = ev.groups[i];
                        modifiedTabs[group.z] = true;
                        // The order of groups is important
                        //  - to invert the action, the order is reversed
                        inverseEv.groups.unshift(group);
                        RED.nodes.removeGroup(group);
                    }
                }
                if (ev.workspaces) {
                    inverseEv.workspaces = [];
                    for (i=0;i<ev.workspaces.length;i++) {
                        var workspaceOrder = RED.nodes.getWorkspaceOrder();
                        ev.workspaces[i]._index = workspaceOrder.indexOf(ev.workspaces[i].id);
                        inverseEv.workspaces.push(ev.workspaces[i]);
                        RED.nodes.removeWorkspace(ev.workspaces[i].id);
                        RED.workspaces.remove(ev.workspaces[i]);
                    }
                }
                if (ev.subflows) {
                    inverseEv.subflows = [];
                    for (i=0;i<ev.subflows.length;i++) {
                        inverseEv.subflows.push(ev.subflows[i]);
                        RED.nodes.removeSubflow(ev.subflows[i]);
                        RED.workspaces.remove(ev.subflows[i]);
                    }
                }
                if (ev.subflow) {
                    inverseEv.subflow = {};
                    if (ev.subflow.instances) {
                        inverseEv.subflow.instances = [];
                        ev.subflow.instances.forEach(function(n) {
                            inverseEv.subflow.instances.push(n);
                            var node = RED.nodes.node(n.id);
                            if (node) {
                                node.changed = n.changed;
                                node.dirty = true;
                            }
                        });
                    }
                    if (ev.subflow.hasOwnProperty('changed')) {
                        subflow = RED.nodes.subflow(ev.subflow.id);
                        if (subflow) {
                            subflow.changed = ev.subflow.changed;
                        }
                    }
                }
                if (ev.removedLinks) {
                    inverseEv.createdLinks = [];
                    for (i=0;i<ev.removedLinks.length;i++) {
                        inverseEv.createdLinks.push(ev.removedLinks[i]);
                        RED.nodes.addLink(ev.removedLinks[i]);
                    }
                }

            } else if (ev.t == "delete") {
                inverseEv = {
                    t: "add",
                    dirty: RED.nodes.dirty()
                };
                if (ev.workspaces) {
                    inverseEv.workspaces = [];
                    for (i=0;i<ev.workspaces.length;i++) {
                        inverseEv.workspaces.push(ev.workspaces[i]);
                        RED.nodes.addWorkspace(ev.workspaces[i],ev.workspaces[i]._index);
                        RED.workspaces.add(ev.workspaces[i],undefined,ev.workspaces[i]._index);
                        delete ev.workspaces[i]._index;
                    }
                }
                if (ev.subflows) {
                    inverseEv.subflows = [];
                    for (i=0;i<ev.subflows.length;i++) {
                        inverseEv.subflows.push(ev.subflows[i]);
                        RED.nodes.addSubflow(ev.subflows[i]);
                    }
                }
                if (ev.subflowInputs && ev.subflowInputs.length > 0) {
                    subflow = RED.nodes.subflow(ev.subflowInputs[0].z);
                    subflow.in.push(ev.subflowInputs[0]);
                    subflow.in[0].dirty = true;
                }
                if (ev.subflowOutputs && ev.subflowOutputs.length > 0) {
                    subflow = RED.nodes.subflow(ev.subflowOutputs[0].z);
                    ev.subflowOutputs.sort(function(a,b) { return a.i-b.i});
                    for (i=0;i<ev.subflowOutputs.length;i++) {
                        var output = ev.subflowOutputs[i];
                        subflow.out.splice(output.i,0,output);
                        for (var j=output.i+1;j<subflow.out.length;j++) {
                            subflow.out[j].i++;
                            subflow.out[j].dirty = true;
                        }
                        RED.nodes.eachLink(function(l) {
                            if (l.source.type == "subflow:"+subflow.id) {
                                if (l.sourcePort >= output.i) {
                                    l.sourcePort++;
                                }
                            }
                        });
                    }
                }
                if (ev.subflow) {
                    inverseEv.subflow = {};
                    if (ev.subflow.hasOwnProperty('instances')) {
                        inverseEv.subflow.instances = [];
                        ev.subflow.instances.forEach(function(n) {
                            inverseEv.subflow.instances.push(n);
                            var node = RED.nodes.node(n.id);
                            if (node) {
                                node.changed = n.changed;
                                node.dirty = true;
                            }
                        });
                    }
                    if (ev.subflow.hasOwnProperty('status')) {
                        subflow = RED.nodes.subflow(ev.subflow.id);
                        subflow.status = ev.subflow.status;
                    }
                }
                if (subflow) {
                    RED.nodes.filterNodes({type:"subflow:"+subflow.id}).forEach(function(n) {
                        n.inputs = subflow.in.length;
                        n.outputs = subflow.out.length;
                        n.resize = true;
                        n.dirty = true;
                    });
                }
                if (ev.groups) {
                    inverseEv.groups = [];
                    var groupsToAdd = {};
                    ev.groups.forEach(function(g) { groupsToAdd[g.id] = g; });
                    for (i = ev.groups.length - 1;i>=0;i--) {
                        RED.nodes.addGroup(ev.groups[i])
                        modifiedTabs[ev.groups[i].z] = true;
                        // The order of groups is important
                        //  - to invert the action, the order is reversed
                        inverseEv.groups.unshift(ev.groups[i]);
                        if (ev.groups[i].g) {
                            if (!groupsToAdd[ev.groups[i].g]) {
                                group = RED.nodes.group(ev.groups[i].g);
                            } else {
                                group = groupsToAdd[ev.groups[i].g];
                            }
                            if (group.nodes.indexOf(ev.groups[i]) === -1) {
                                group.nodes.push(ev.groups[i]);
                            }
                            RED.group.markDirty(ev.groups[i])
                        }
                    }
                }
                if (ev.nodes) {
                    inverseEv.nodes = [];
                    for (i=0;i<ev.nodes.length;i++) {
                        RED.nodes.add(ev.nodes[i]);
                        modifiedTabs[ev.nodes[i].z] = true;
                        inverseEv.nodes.push(ev.nodes[i].id);
                        if (ev.nodes[i].g) {
                            group = RED.nodes.group(ev.nodes[i].g);
                            if (group.nodes.indexOf(ev.nodes[i]) === -1) {
                                group.nodes.push(ev.nodes[i]);
                            }
                            RED.group.markDirty(group)
                        }
                    }
                }
                if (ev.junctions) {
                    inverseEv.junctions = [];
                    for (i=0;i<ev.junctions.length;i++) {
                        inverseEv.junctions.push(ev.junctions[i]);
                        RED.nodes.addJunction(ev.junctions[i]);
                        if (ev.junctions[i].g) {
                            group = RED.nodes.group(ev.junctions[i].g);
                            if (group.nodes.indexOf(ev.junctions[i]) === -1) {
                                group.nodes.push(ev.junctions[i]);
                            }
                            RED.group.markDirty(group)
                        }

                    }
                }
                if (ev.links) {
                    inverseEv.links = [];
                    for (i=0;i<ev.links.length;i++) {
                        RED.nodes.addLink(ev.links[i]);
                        inverseEv.links.push(ev.links[i]);
                    }
                }
                if (ev.createdLinks) {
                    inverseEv.removedLinks = [];
                    for (i=0;i<ev.createdLinks.length;i++) {
                        inverseEv.removedLinks.push(ev.createdLinks[i]);
                        RED.nodes.removeLink(ev.createdLinks[i]);
                    }
                }
                if (ev.changes) {
                    for (i in ev.changes) {
                        if (ev.changes.hasOwnProperty(i)) {
                            node = RED.nodes.node(i);
                            if (node) {
                                for (var d in ev.changes[i]) {
                                    if (ev.changes[i].hasOwnProperty(d)) {
                                        node[d] = ev.changes[i][d];
                                    }
                                }
                                node.dirty = true;
                            }
                            RED.events.emit("nodes:change",node);
                        }
                    }
                }
                if (subflow) {
                    RED.events.emit("subflows:change", subflow);
                }
            } else if (ev.t == "move") {
                inverseEv = {
                    t: 'move',
                    nodes: [],
                    dirty: RED.nodes.dirty()
                };
                for (i=0;i<ev.nodes.length;i++) {
                    var n = ev.nodes[i];
                    var rn = {n: n.n, ox: n.n.x, oy: n.n.y, dirty: true, moved: n.n.moved};
                    inverseEv.nodes.push(rn);
                    n.n.x = n.ox;
                    n.n.y = n.oy;
                    n.n.dirty = true;
                    n.n.moved = n.moved;
                }
                // A move could have caused a link splice
                if (ev.links) {
                    inverseEv.removedLinks = [];
                    for (i=0;i<ev.links.length;i++) {
                        inverseEv.removedLinks.push(ev.links[i]);
                        RED.nodes.removeLink(ev.links[i]);
                    }
                }
                if (ev.removedLinks) {
                    inverseEv.links = [];
                    for (i=0;i<ev.removedLinks.length;i++) {
                        inverseEv.links.push(ev.removedLinks[i]);
                        RED.nodes.addLink(ev.removedLinks[i]);
                    }
                }
                if (ev.addToGroup) {
                    RED.group.removeFromGroup(ev.addToGroup,ev.nodes.map(function(n) { return n.n }),false);
                    inverseEv.removeFromGroup = ev.addToGroup;
                }
                if (ev.removeFromGroup) {
                    RED.group.addToGroup(ev.removeFromGroup,ev.nodes.map(function(n) { return n.n }));
                    inverseEv.addToGroup = ev.removeFromGroup;
                }
            } else if (ev.t == "edit") {
                inverseEv = {
                    t: "edit",
                    changes: {},
                    changed: ev.node.changed,
                    dirty: RED.nodes.dirty()
                };
                inverseEv.node = ev.node;
                for (i in ev.changes) {
                    if (ev.changes.hasOwnProperty(i)) {
                        inverseEv.changes[i] = ev.node[i];
                        if (ev.node._def.defaults && ev.node._def.defaults[i] && ev.node._def.defaults[i].type) {
                            // This property is a reference to another node or nodes.
                            var nodeList = ev.node[i];
                            if (!Array.isArray(nodeList)) {
                                nodeList = [nodeList];
                            }
                            nodeList.forEach(function(id) {
                                var currentConfigNode = RED.nodes.node(id);
                                if (currentConfigNode && currentConfigNode._def.category === "config") {
                                    currentConfigNode.users.splice(currentConfigNode.users.indexOf(ev.node),1);
                                    RED.events.emit("nodes:change",currentConfigNode);
                                }
                            });
                            nodeList = ev.changes[i];
                            if (!Array.isArray(nodeList)) {
                                nodeList = [nodeList];
                            }
                            nodeList.forEach(function(id) {
                                var newConfigNode = RED.nodes.node(id);
                                if (newConfigNode && newConfigNode._def.category === "config") {
                                    newConfigNode.users.push(ev.node);
                                    RED.events.emit("nodes:change",newConfigNode);
                                }
                            });
                        } else if (i === "env" && ev.node.type.indexOf("subflow:") === 0) {
                            // Subflow can have config node in node.env
                            let nodeList = ev.node.env || [];
                            nodeList = nodeList.reduce((list, prop) => {
                                if (prop.type === "conf-type" && prop.value) {
                                    list.push(prop.value);
                                }
                                return list;
                            }, []);

                            nodeList.forEach(function(id) {
                                const configNode = RED.nodes.node(id);
                                if (configNode) {
                                    if (configNode.users.indexOf(ev.node) !== -1) {
                                        configNode.users.splice(configNode.users.indexOf(ev.node), 1);
                                        RED.events.emit("nodes:change", configNode);
                                    }
                                }
                            });

                            nodeList = ev.changes.env || [];
                            nodeList = nodeList.reduce((list, prop) => {
                                if (prop.type === "conf-type" && prop.value) {
                                    list.push(prop.value);
                                }
                                return list;
                            }, []);

                            nodeList.forEach(function(id) {
                                const configNode = RED.nodes.node(id);
                                if (configNode) {
                                    if (configNode.users.indexOf(ev.node) === -1) {
                                        configNode.users.push(ev.node);
                                        RED.events.emit("nodes:change", configNode);
                                    }
                                }
                            });
                        } else if (i === "color" && ev.node.type === "subflow") {
                            // Handle the Subflow definition color change
                            RED.utils.clearNodeColorCache();
                            const subflowDef = RED.nodes.getType("subflow:" + ev.node.id);
                            if (subflowDef) {
                                subflowDef.color = ev.changes[i] || "#DDAA99";
                            }
                        }
                        if (i === "credentials" && ev.changes[i]) {
                            // Reset - Only want to keep the changes
                            inverseEv.changes[i] = {};
                            for (const [key, value] of Object.entries(ev.changes[i])) {
                                // Edge case: node.credentials is cleared after a deploy, so we can't
                                // capture values for the inverse event when undoing past a deploy
                                if (ev.node.credentials) {
                                    inverseEv.changes[i][key] = ev.node.credentials[key];
                                }
                                ev.node.credentials[key] = value;
                            }
                        } else {
                            ev.node[i] = ev.changes[i];
                        }
                    }
                }
                if (ev.node.type === 'subflow') {
                    // Ensure ports get a refresh in case of a label change
                    if (ev.changes.inputLabels) {
                        ev.node.in.forEach(function(input) { input.dirty = true; });
                    }
                    if (ev.changes.outputLabels) {
                        ev.node.out.forEach(function(output) { output.dirty = true; });
                    }
                }
                ev.node.dirty = true;
                ev.node.changed = ev.changed;

                var eventType;
                switch(ev.node.type) {
                    case 'tab': eventType = "flows"; break;
                    case 'group': eventType = "groups"; break;
                    case 'subflow': eventType = "subflows"; break;
                    default: eventType = "nodes"; break;
                }
                eventType += ":change";
                RED.events.emit(eventType,ev.node);


                if (ev.node.type === 'tab' && ev.changes.hasOwnProperty('disabled')) {
                    $("#red-ui-tab-"+(ev.node.id.replace(".","-"))).toggleClass('red-ui-workspace-disabled',!!ev.node.disabled);
                }
                if (ev.node.type === 'tab' && ev.changes.hasOwnProperty('locked')) {
                    $("#red-ui-tab-"+(ev.node.id.replace(".","-"))).toggleClass('red-ui-workspace-locked',!!ev.node.locked);
                }
                if (ev.subflow) {
                    inverseEv.subflow = {};
                    if (ev.subflow.hasOwnProperty('inputCount')) {
                        inverseEv.subflow.inputCount = ev.node.in.length;
                        if (ev.node.in.length > ev.subflow.inputCount) {
                            inverseEv.subflow.inputs = ev.node.in.slice(ev.subflow.inputCount);
                            ev.node.in.splice(ev.subflow.inputCount);
                        } else if (ev.subflow.inputs.length > 0) {
                            ev.node.in = ev.node.in.concat(ev.subflow.inputs);
                        }
                    }
                    if (ev.subflow.hasOwnProperty('outputCount')) {
                        inverseEv.subflow.outputCount = ev.node.out.length;
                        if (ev.node.out.length > ev.subflow.outputCount) {
                            inverseEv.subflow.outputs = ev.node.out.slice(ev.subflow.outputCount);
                            ev.node.out.splice(ev.subflow.outputCount);
                        } else if (ev.subflow.outputs.length > 0) {
                            ev.node.out = ev.node.out.concat(ev.subflow.outputs);
                        }
                    }
                    if (ev.subflow.hasOwnProperty('instances')) {
                        inverseEv.subflow.instances = [];
                        ev.subflow.instances.forEach(function(n) {
                            inverseEv.subflow.instances.push(n);
                            var node = RED.nodes.node(n.id);
                            if (node) {
                                node.changed = n.changed;
                                node.dirty = true;

                                if (ev.changes && ev.changes.hasOwnProperty('color')) {
                                    node._colorChanged = true;
                                }
                            }
                        });
                    }
                    if (ev.subflow.hasOwnProperty('status')) {
                        if (ev.subflow.status) {
                            delete ev.node.status;
                        }
                    }
                    RED.editor.validateNode(ev.node);
                    RED.nodes.filterNodes({type:"subflow:"+ev.node.id}).forEach(function(n) {
                        n.inputs = ev.node.in.length;
                        n.outputs = ev.node.out.length;
                        RED.editor.updateNodeProperties(n);
                        RED.editor.validateNode(n);
                    });
                }

                let outputMap;
                if (ev.outputMap) {
                    outputMap = {};
                    inverseEv.outputMap = {};
                    for (var port in ev.outputMap) {
                        if (ev.outputMap.hasOwnProperty(port) && ev.outputMap[port] !== "-1") {
                            outputMap[ev.outputMap[port]] = port;
                            inverseEv.outputMap[ev.outputMap[port]] = port;
                        }
                    }
                }
                ev.node.__outputs = inverseEv.changes.outputs;
                RED.editor.updateNodeProperties(ev.node,outputMap);
                RED.editor.validateNode(ev.node);

                // If it's a Config Node, validate user nodes too.
                // NOTE: The Config Node must be validated before validating users.
                if (ev.node.users) {
                    const validatedNodes = new Set();
                    const userStack = ev.node.users.slice();

                    validatedNodes.add(ev.node.id);
                    while (userStack.length) {
                        const node = userStack.pop();
                        if (!validatedNodes.has(node.id)) {
                            validatedNodes.add(node.id);
                            if (node.users) {
                                userStack.push(...node.users);
                            }
                            RED.editor.validateNode(node);
                        }
                    }
                }
                if (ev.links) {
                    inverseEv.createdLinks = [];
                    for (i=0;i<ev.links.length;i++) {
                        RED.nodes.addLink(ev.links[i]);
                        inverseEv.createdLinks.push(ev.links[i]);
                    }
                }
                if (ev.createdLinks) {
                    inverseEv.links = [];
                    for (i=0;i<ev.createdLinks.length;i++) {
                        RED.nodes.removeLink(ev.createdLinks[i]);
                        inverseEv.links.push(ev.createdLinks[i]);
                    }
                }
            } else if (ev.t == "createSubflow") {
                inverseEv = {
                    t: "deleteSubflow",
                    activeWorkspace: ev.activeWorkspace,
                    dirty: RED.nodes.dirty()
                };
                if (ev.nodes) {
                    inverseEv.movedNodes = [];
                    var z = ev.activeWorkspace;
                    var fullNodeList = RED.nodes.filterNodes({z:ev.subflow.subflow.id});
                    fullNodeList = fullNodeList.concat(RED.nodes.groups(ev.subflow.subflow.id))
                    fullNodeList = fullNodeList.concat(RED.nodes.junctions(ev.subflow.subflow.id))
                    fullNodeList.forEach(function(n) {
                        n.x += ev.subflow.offsetX;
                        n.y += ev.subflow.offsetY;
                        n.dirty = true;
                        inverseEv.movedNodes.push(n.id);
                        RED.nodes.moveNodeToTab(n, z);
                    });
                    inverseEv.subflows = [];
                    for (i=0;i<ev.nodes.length;i++) {
                        inverseEv.subflows.push(nodeOrJunction(ev.nodes[i]));
                        RED.nodes.remove(ev.nodes[i]);
                    }
                }
                if (ev.links) {
                    inverseEv.links = [];
                    for (i=0;i<ev.links.length;i++) {
                        inverseEv.links.push(ev.links[i]);
                        RED.nodes.removeLink(ev.links[i]);
                    }
                }

                inverseEv.subflow = ev.subflow;
                RED.nodes.removeSubflow(ev.subflow.subflow);
                RED.workspaces.remove(ev.subflow.subflow);

                if (ev.removedLinks) {
                    inverseEv.createdLinks = [];
                    for (i=0;i<ev.removedLinks.length;i++) {
                        inverseEv.createdLinks.push(ev.removedLinks[i]);
                        RED.nodes.addLink(ev.removedLinks[i]);
                    }
                }
            } else if (ev.t == "deleteSubflow") {
                inverseEv = {
                    t: "createSubflow",
                    activeWorkspace: ev.activeWorkspace,
                    dirty: RED.nodes.dirty(),
                };
                if (ev.subflow) {
                    RED.nodes.addSubflow(ev.subflow.subflow);
                    inverseEv.subflow = ev.subflow;
                    if (ev.subflow.subflow.g) {
                        RED.group.addToGroup(RED.nodes.group(ev.subflow.subflow.g),ev.subflow.subflow);
                    }
                }
                if (ev.subflows) {
                    inverseEv.nodes = [];
                    for (i=0;i<ev.subflows.length;i++) {
                        RED.nodes.add(ev.subflows[i]);
                        inverseEv.nodes.push(ev.subflows[i].id);
                    }
                }
                if (ev.movedNodes) {
                    ev.movedNodes.forEach(function(nid) {
                        nn = RED.nodes.node(nid);
                        if (!nn) {
                            nn = RED.nodes.group(nid);
                        }
                        nn.x -= ev.subflow.offsetX;
                        nn.y -= ev.subflow.offsetY;
                        nn.dirty = true;
                        RED.nodes.moveNodeToTab(nn, ev.subflow.subflow.id);
                    });
                }
                if (ev.links) {
                    inverseEv.links = [];
                    for (i=0;i<ev.links.length;i++) {
                        inverseEv.links.push(ev.links[i]);
                        RED.nodes.addLink(ev.links[i]);
                    }
                }
                if (ev.createdLinks) {
                    inverseEv.removedLinks = [];
                    for (i=0;i<ev.createdLinks.length;i++) {
                        inverseEv.removedLinks.push(ev.createdLinks[i]);
                        RED.nodes.removeLink(ev.createdLinks[i]);
                    }
                }
            } else if (ev.t == "reorder") {
                inverseEv = {
                    t: 'reorder',
                    dirty: RED.nodes.dirty()
                };
                if (ev.workspaces) {
                    inverseEv.workspaces = {
                        from: ev.workspaces.to,
                        to: ev.workspaces.from
                    }
                    RED.workspaces.order(ev.workspaces.from);
                }
                if (ev.nodes) {
                    inverseEv.nodes = {
                        z: ev.nodes.z,
                        from: ev.nodes.to,
                        to: ev.nodes.from
                    }
                    RED.nodes.setNodeOrder(ev.nodes.z,ev.nodes.from);
                }
            } else if (ev.t == "createGroup") {
                inverseEv = {
                    t: "ungroup",
                    dirty: RED.nodes.dirty(),
                    groups: []
                }
                if (ev.groups) {
                    for (i=0;i<ev.groups.length;i++) {
                        inverseEv.groups.push(ev.groups[i]);
                        RED.group.ungroup(ev.groups[i]);
                    }
                }
            } else if (ev.t == "ungroup") {
                inverseEv = {
                    t: "createGroup",
                    dirty: RED.nodes.dirty(),
                    groups: []
                }
                if (ev.groups) {
                    for (i=0;i<ev.groups.length;i++) {
                        inverseEv.groups.push(ev.groups[i]);
                        var nodes = ev.groups[i].nodes.slice();
                        ev.groups[i].nodes = [];
                        RED.nodes.addGroup(ev.groups[i]);
                        RED.group.addToGroup(ev.groups[i],nodes);
                        if (ev.groups[i].g) {
                            const parentGroup = RED.nodes.group(ev.groups[i].g)
                            if (parentGroup) {
                                RED.group.addToGroup(parentGroup, ev.groups[i])
                            }
                        }
                    }
                }
            } else if (ev.t == "addToGroup") {
                inverseEv = {
                    t: "removeFromGroup",
                    dirty: RED.nodes.dirty(),
                    group: ev.group,
                    nodes: ev.nodes,
                    reparent: ev.reparent
                }
                if (ev.nodes) {
                    RED.group.removeFromGroup(ev.group,ev.nodes,(ev.hasOwnProperty('reparent')&&ev.hasOwnProperty('reparent')!==undefined)?ev.reparent:true);
                }
            } else if (ev.t == "removeFromGroup") {
                inverseEv = {
                    t: "addToGroup",
                    dirty: RED.nodes.dirty(),
                    group: ev.group,
                    nodes: ev.nodes,
                    reparent: ev.reparent
                }
                if (ev.nodes) {
                    RED.group.addToGroup(ev.group,ev.nodes);
                }
            }

            if(ev.callback && typeof ev.callback === 'function') {
                inverseEv.callback = ev.callback;
                ev.callback(ev);
            }

            Object.keys(modifiedTabs).forEach(function(id) {
                var subflow = RED.nodes.subflow(id);
                if (subflow) {
                    RED.editor.validateNode(subflow);
                }
            });

            RED.nodes.dirty(ev.dirty);
            RED.view.updateActive();
            RED.view.select(null);
            RED.workspaces.refresh();
            RED.sidebar.config.refresh();
            RED.subflow.refresh();

            return inverseEv;
        }

    }

    return {
        //TODO: this function is a placeholder until there is a 'save' event that can be listened to
        markAllDirty: function() {
            for (const event of [...undoHistory, ...redoHistory]) {
                event.dirty = true;
            }
        },
        list: function() {
            return undoHistory;
        },
        listRedo: function() {
            return redoHistory;
        },
        depth: function() {
            return undoHistory.length;
        },
        push: function(ev) {
            undoHistory.push(ev);
            redoHistory = [];
            RED.menu.setDisabled("menu-item-edit-undo", false);
            RED.menu.setDisabled("menu-item-edit-redo", true);
        },
        pop: function() {
            var ev = undoHistory.pop();
            var rev = undoEvent(ev);
            if (rev) {
                redoHistory.push(rev);
            }
            RED.menu.setDisabled("menu-item-edit-undo", undoHistory.length === 0);
            RED.menu.setDisabled("menu-item-edit-redo", redoHistory.length === 0);
        },
        peek: function() {
            return undoHistory[undoHistory.length-1];
        },
        replace: function(ev) {
            if (undoHistory.length === 0) {
                RED.history.push(ev);
            } else {
                undoHistory[undoHistory.length-1] = ev;
            }
        },
        clear: function() {
            undoHistory = [];
            redoHistory = [];
            RED.menu.setDisabled("menu-item-edit-undo", true);
            RED.menu.setDisabled("menu-item-edit-redo", true);
        },
        redo: function() {
            var ev = redoHistory.pop();
            if (ev) {
                var uev = undoEvent(ev);
                if (uev) {
                    undoHistory.push(uev);
                }
            }
            RED.menu.setDisabled("menu-item-edit-undo", undoHistory.length === 0);
            RED.menu.setDisabled("menu-item-edit-redo", redoHistory.length === 0);
        }
    }

})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
RED.validators = {
    number: function(blankAllowed,mopt){
        return function(v, opt) {
            if (blankAllowed && (v === '' || v === undefined)) {
                return true
            }
            if (v !== '') {
                if (/^NaN$|^[+-]?[0-9]*\.?[0-9]*([eE][-+]?[0-9]+)?$|^[+-]?(0b|0B)[01]+$|^[+-]?(0o|0O)[0-7]+$|^[+-]?(0x|0X)[0-9a-fA-F]+$/.test(v)) {
                    return true
                }
                if (/^\${[^}]+}$/.test(v)) {
                    // Allow ${ENV_VAR} value
                    return true
                }
            }
            if (!isNaN(v)) {
                return true
            }
            if (opt && opt.label) {
                return RED._("validator.errors.invalid-num-prop", {
                    prop: opt.label
                });
            }
            return opt ? RED._("validator.errors.invalid-num") : false;
        };
    },
    regex: function(re, mopt) {
        return function(v, opt) {
            if (re.test(v)) {
                return true;
            }
            if (opt && opt.label) {
                return RED._("validator.errors.invalid-regex-prop", {
                    prop: opt.label
                });
            }
            return opt ? RED._("validator.errors.invalid-regexp") : false;
        };
    },
    typedInput: function(ptypeName, isConfig, mopt) {
        let options = ptypeName
        if (typeof ptypeName === 'string' ) {
            options = {}
            options.typeField = ptypeName
            options.isConfig = isConfig
            options.allowBlank = false
        }
        return function(v, opt) {
            let ptype = options.type
            if (!ptype && options.typeField) {
                ptype = $("#node-"+(options.isConfig?"config-":"")+"input-"+options.typeField).val() || this[options.typeField];
            }
            if (options.allowBlank && v === '') {
                return true
            }
            if (options.allowUndefined && v === undefined) {
                return true
            }
            const result = RED.utils.validateTypedProperty(v, ptype, opt)
            if (result === true || opt) {
                // Valid, or opt provided - return result as-is
                return result
            }
            // No opt - need to return false for backwards compatibilty
            return false
        }
    }
};;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

RED.utils = (function() {
    const listOfTypedArrays = ['Int8Array','Uint8Array','Uint8ClampedArray','Int16Array','Uint16Array','Int32Array','Uint32Array','Float32Array','Float64Array','BigInt64Array','BigUint64Array'];

    window._marked = window.marked;
    window.marked = function(txt) {
        console.warn("Use of 'marked()' is deprecated. Use RED.utils.renderMarkdown() instead");
        return renderMarkdown(txt);
    }

    const descriptionList = {
        name: 'descriptionList',
        level: 'block',                                     // Is this a block-level or inline-level tokenizer?
        start(src) {
            if (!src) { return null; }
            let m = src.match(/:[^:\n]/g);
            return m && m.index; // Hint to Marked.js to stop and check for a match
        },
        tokenizer(src, tokens) {
            if (!src) { return null; }
            const rule = /^(?::[^:\n]+:[^:\n]*(?:\n|$))+/;    // Regex for the complete token
            const match = rule.exec(src);
            if (match) {
                return {                                        // Token to generate
                    type: 'descriptionList',                      // Should match "name" above
                    raw: match[0],                                // Text to consume from the source
                    text: match[0].trim(),                        // Additional custom properties
                    tokens: this.lexer.inlineTokens(match[0].trim())    // inlineTokens to process **bold**, *italics*, etc.
                };
            }
        },
        renderer(token) {
            return `<dl class="message-properties">${this.parser.parseInline(token.tokens)}\n</dl>`; // parseInline to turn child tokens into HTML
        }
    };

    const description = {
        name: 'description',
        level: 'inline',           // Is this a block-level or inline-level tokenizer?
        start(src) {
            if (!src) { return null; }
            let m = src.match(/:/g);
            return m && m.index;   // Hint to Marked.js to stop and check for a match
        },
        tokenizer(src, tokens) {
            if (!src) { return null; }
            const rule = /^:([^:\n]+)\(([^:\n]+)\).*?:([^:\n]*)(?:\n|$)/;  // Regex for the complete token
            const match = rule.exec(src);
            if (match) {
                return {                                       // Token to generate
                    type: 'description',                       // Should match "name" above
                    raw: match[0],                             // Text to consume from the source
                    dt: this.lexer.inlineTokens(match[1].trim()),    // Additional custom properties
                    types: this.lexer.inlineTokens(match[2].trim()),
                    dd: this.lexer.inlineTokens(match[3].trim()),
                };
            }
        },
        renderer(token) {
            return `\n<dt>${this.parser.parseInline(token.dt)}<span class="property-type">${this.parser.parseInline(token.types)}</span></dt><dd>${this.parser.parseInline(token.dd)}</dd>`;
        },
        childTokens: ['dt', 'dd'],                 // Any child tokens to be visited by walkTokens
        walkTokens(token) {                        // Post-processing on the completed token tree
            if (token.type === 'strong') {
                token.text += ' walked';
            }
        }
    };


    const renderer = new window._marked.Renderer();

    //override list creation - add node-ports to order lists
    renderer.list = function (body, ordered, start) {
        let addClass = /dl.*?class.*?message-properties.*/.test(body);
        if (addClass && ordered) {
            return '<ol class="node-ports">' + body + '</ol>';
        } else if (ordered) {
            return '<ol>' + body + '</ol>';
        } else {
            return '<ul>' + body + '</ul>'
        }
    }

    var mermaidIsInitialized = false;
    var mermaidIsEnabled /* = undefined */;

    renderer.code = function (code, lang) {
        if(lang === "mermaid") {
            return `<pre style='word-break: unset;' data-c64='${btoa(code)}' class='mermaid'>${code}</pre>`;
        } else {
            return "<pre><code>" +code +"</code></pre>";
        }
    };

    window._marked.setOptions({
        renderer: renderer,
        gfm: true,
        tables: true,
        breaks: false,
        pedantic: false,
        smartLists: true,
        smartypants: false
    });

    window._marked.use({extensions: [descriptionList, description] } );

    function renderMarkdown(txt) {
        var rendered = _marked.parse(txt);
        const cleaned = DOMPurify.sanitize(rendered);
        return cleaned;
    }

    function formatString(str) {
        return str.replace(/\r?\n/g,"&crarr;").replace(/\t/g,"&rarr;");
    }

    function sanitize(m) {
        return m.replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;");
    }

    /**
     * Truncates a string to a specified maximum length, adding ellipsis
     * if truncated.
     *
     * @param {string} str The string to be truncated.
     * @param {number} [maxLength = 120] The maximum length of the truncated
     * string. Default `120`.
     * @returns {string} The truncated string with ellipsis if it exceeds the
     * maximum length.
     */
    function truncateString(str, maxLength = 120) {
        return str.length > maxLength ? str.slice(0, maxLength) + "..." : str;
    }

    function buildMessageSummaryValue(value) {
        var result;
        if (Array.isArray(value)) {
            result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-meta"></span>').text('array['+value.length+']');
        } else if (value === null) {
            result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-null">null</span>');
        } else if (typeof value === 'object') {
            if (value.hasOwnProperty('type') && value.type === 'undefined') {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-null">undefined</span>');
            } else if (value.hasOwnProperty('type') && value.type === 'Buffer' && value.hasOwnProperty('data')) {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-meta"></span>').text('buffer['+value.length+']');
            } else if (value.hasOwnProperty('type') && value.type === 'array' && value.hasOwnProperty('data')) {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-meta"></span>').text('array['+value.length+']');
            } else if (value.hasOwnProperty('type') && value.type === 'set' && value.hasOwnProperty('data')) {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-meta"></span>').text('set['+value.length+']');
            } else if (value.hasOwnProperty('type') && value.type === 'map' && value.hasOwnProperty('data')) {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-meta"></span>').text('map');
            } else if (value.hasOwnProperty('type') && value.type === 'function') {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-meta"></span>').text('function');
            } else if (value.hasOwnProperty('type') && (value.type === 'number' || value.type === 'bigint')) {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-number"></span>').text(value.data);
            } else if (value.hasOwnProperty('type') && value.type === 'regexp') {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-string"></span>').text(value.data);
            } else if (value.hasOwnProperty('type') && value.hasOwnProperty('data') && listOfTypedArrays.includes(value.type)) {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-meta"></span>').text(value.type + '['+value.length+']');
            } else {
                result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-meta">object</span>');
            }
        } else if (typeof value === 'string') {
            var subvalue;
            if (value.length > 30) {
                subvalue = sanitize(value.substring(0,30))+"&hellip;";
            } else {
                subvalue = sanitize(value);
            }
            result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-string"></span>').html('"'+formatString(subvalue)+'"');
        } else if (typeof value === 'number') {
            result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-number"></span>').text(""+value);
        } else {
            result = $('<span class="red-ui-debug-msg-object-value red-ui-debug-msg-type-other"></span>').text(""+value);
        }
        return result;
    }
    function makeExpandable(el,onbuild,ontoggle,expand) {
        el.addClass("red-ui-debug-msg-expandable");
        el.prop('toggle',function() {
            return function(state) {
                var parent = el.parent();
                if (parent.hasClass('collapsed')) {
                    if (state) {
                        if (onbuild && !parent.hasClass('built')) {
                            onbuild();
                            parent.addClass('built');
                        }
                        parent.removeClass('collapsed');
                        return true;
                    }
                } else {
                    if (!state) {
                        parent.addClass('collapsed');
                        return true;
                    }
                }
                return false;
            }
        });
        el.on("click", function(e) {
            var parent = $(this).parent();
            var currentState = !parent.hasClass('collapsed');
            if ($(this).prop('toggle')(!currentState)) {
                if (ontoggle) {
                    ontoggle(!currentState);
                }
            }
            // if (parent.hasClass('collapsed')) {
            //     if (onbuild && !parent.hasClass('built')) {
            //         onbuild();
            //         parent.addClass('built');
            //     }
            //     if (ontoggle) {
            //         ontoggle(true);
            //     }
            //     parent.removeClass('collapsed');
            // } else {
            //     parent.addClass('collapsed');
            //     if (ontoggle) {
            //         ontoggle(false);
            //     }
            // }
            e.preventDefault();
        });
        if (expand) {
            el.trigger("click");
        }

    }

    var pinnedPaths = {};
    var formattedPaths = {};

    function addMessageControls(obj,sourceId,key,msg,rootPath,strippedKey,extraTools,enablePinning) {
        if (!pinnedPaths.hasOwnProperty(sourceId)) {
            pinnedPaths[sourceId] = {}
        }
        var tools = $('<span class="red-ui-debug-msg-tools"></span>').appendTo(obj);
        var copyTools = $('<span class="red-ui-debug-msg-tools-copy button-group"></span>').appendTo(tools);
        if (!!key) {
            var copyPath = $('<button class="red-ui-button red-ui-button-small"><i class="fa fa-terminal"></i></button>').appendTo(copyTools).on("click", function(e) {
                e.preventDefault();
                e.stopPropagation();
                RED.clipboard.copyText(key,copyPath,"clipboard.copyMessagePath");
            })
            RED.popover.tooltip(copyPath,RED._("node-red:debug.sidebar.copyPath"));
        }
        var copyPayload = $('<button class="red-ui-button red-ui-button-small"><i class="fa fa-clipboard"></i></button>').appendTo(copyTools).on("click", function(e) {
            e.preventDefault();
            e.stopPropagation();
            RED.clipboard.copyText(msg,copyPayload,"clipboard.copyMessageValue");
        })
        RED.popover.tooltip(copyPayload,RED._("node-red:debug.sidebar.copyPayload"));
        if (enablePinning && strippedKey !== undefined && strippedKey !== '') {
            var isPinned = pinnedPaths[sourceId].hasOwnProperty(strippedKey);

            var pinPath = $('<button class="red-ui-button red-ui-button-small red-ui-debug-msg-tools-pin"><i class="fa fa-map-pin"></i></button>').appendTo(tools).on("click", function(e) {
                e.preventDefault();
                e.stopPropagation();
                if (pinnedPaths[sourceId].hasOwnProperty(strippedKey)) {
                    delete pinnedPaths[sourceId][strippedKey];
                    $(this).removeClass("selected");
                    obj.removeClass("red-ui-debug-msg-row-pinned");
                } else {
                    var rootedPath = "$"+(strippedKey[0] === '['?"":".")+strippedKey;
                    pinnedPaths[sourceId][strippedKey] = normalisePropertyExpression(rootedPath);
                    $(this).addClass("selected");
                    obj.addClass("red-ui-debug-msg-row-pinned");
                }
            }).toggleClass("selected",isPinned);
            obj.toggleClass("red-ui-debug-msg-row-pinned",isPinned);
            RED.popover.tooltip(pinPath,RED._("node-red:debug.sidebar.pinPath"));
        }
        if (extraTools) {
            var t = extraTools;
            if (typeof t === 'function') {
                t = t(key,msg);
            }
            if (t) {
                t.addClass("red-ui-debug-msg-tools-other");
                t.appendTo(tools);
            }
        }
    }
    function checkExpanded(strippedKey, expandPaths, { minRange, maxRange, expandLeafNodes }) {
        if (expandPaths && expandPaths.length > 0) {
            if (strippedKey === '' && minRange === undefined) {
                return true;
            }
            for (var i=0;i<expandPaths.length;i++) {
                var p = expandPaths[i];
                if (expandLeafNodes && p === strippedKey) {
                    return true
                }
                if (p.indexOf(strippedKey) === 0 && (p[strippedKey.length] === "." ||  p[strippedKey.length] === "[") ) {

                    if (minRange !== undefined && p[strippedKey.length] === "[") {
                        var subkey = p.substring(strippedKey.length);
                        var m = (/\[(\d+)\]/.exec(subkey));
                        if (m) {
                            var index = parseInt(m[1]);
                            return minRange<=index && index<=maxRange;
                        }
                    } else {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    function formatNumber(element,obj,sourceId,path,cycle,initialFormat) {
        var format = (formattedPaths[sourceId] && formattedPaths[sourceId][path] && formattedPaths[sourceId][path]['number']) || initialFormat || "dec";
        if (cycle) {
            if (format === 'dec') {
                if ((obj.toString().length===13) && (obj<=2147483647000)) {
                    format = 'dateMS';
                } else if ((obj.toString().length===10) && (obj<=2147483647)) {
                    format = 'dateS';
                } else {
                    format = 'hex'
                }
            } else if (format === 'dateMS' || format == 'dateS') {
                if ((obj.toString().length===13) && (obj<=2147483647000)) {
                    format = 'dateML';
                } else if ((obj.toString().length===10) && (obj<=2147483647)) {
                    format = 'dateL';
                } else {
                    format = 'hex'
                }
            } else if (format === 'dateML' || format == 'dateL') {
                format = 'hex';
            } else {
                format = 'dec';
            }
            formattedPaths[sourceId] = formattedPaths[sourceId]||{};
            formattedPaths[sourceId][path] = formattedPaths[sourceId][path]||{};
            formattedPaths[sourceId][path]['number'] = format;
        } else if (initialFormat !== undefined){
            formattedPaths[sourceId] = formattedPaths[sourceId]||{};
            formattedPaths[sourceId][path] = formattedPaths[sourceId][path]||{};
            formattedPaths[sourceId][path]['number'] = format;
        }
        if (format === 'dec') {
            element.text(""+obj);
        } else if (format === 'dateMS') {
            element.text((new Date(obj)).toISOString());
        } else if (format === 'dateS') {
            element.text((new Date(obj*1000)).toISOString());
        } else if (format === 'dateML') {
            var dd = new Date(obj);
            element.text(dd.toLocaleString() + "  [UTC" + ( dd.getTimezoneOffset()/-60 <=0?"":"+" ) + dd.getTimezoneOffset()/-60 +"]");
        } else if (format === 'dateL') {
            var ddl = new Date(obj*1000);
            element.text(ddl.toLocaleString() + "  [UTC" + ( ddl.getTimezoneOffset()/-60 <=0?"":"+" ) + ddl.getTimezoneOffset()/-60 +"]");
        } else if (format === 'hex') {
            element.text("0x"+(obj).toString(16));
        }
    }

    function formatBuffer(element,button,sourceId,path,cycle) {
        var format = (formattedPaths[sourceId] && formattedPaths[sourceId][path] && formattedPaths[sourceId][path]['buffer']) || "raw";
        if (cycle) {
            if (format === 'raw') {
                format = 'string';
            } else {
                format = 'raw';
            }
            formattedPaths[sourceId] = formattedPaths[sourceId]||{};
            formattedPaths[sourceId][path] = formattedPaths[sourceId][path]||{};
            formattedPaths[sourceId][path]['buffer'] = format;
        }
        if (format === 'raw') {
            button.text('raw');
            element.removeClass('red-ui-debug-msg-buffer-string').addClass('red-ui-debug-msg-buffer-raw');
        } else if (format === 'string') {
            button.text('string');
            element.addClass('red-ui-debug-msg-buffer-string').removeClass('red-ui-debug-msg-buffer-raw');
        }
    }

    // Max string length before truncating
    const MAX_STRING_LENGTH = 80;

    /**
     * Create a DOM element representation of obj - as used by Debug sidebar etc
     *
     * @params obj - the data to display
     * @params options - a bag of options
     *
     * - If you want the Copy Value button, then set `sourceId`
     * - If you want the Copy Path button, also set `path` to the value to be copied
     */
    function createObjectElement(obj,options) {
        options = options || {};
        var key = options.key;
        var typeHint = options.typeHint;
        var hideKey = options.hideKey;
        var path = options.path;
        var sourceId = options.sourceId;
        var rootPath = options.rootPath;
        var expandPaths = options.expandPaths;
        const enablePinning = options.enablePinning
        const expandLeafNodes = options.expandLeafNodes;
        var ontoggle = options.ontoggle;
        var exposeApi = options.exposeApi;
        var tools = options.tools;

        var subElements = {};
        var i;
        var e;
        var entryObj;
        var expandableHeader;
        var header;
        var headerHead;
        var value;
        var strippedKey;
        if (path !== undefined && rootPath !== undefined) {
             strippedKey = path.substring(rootPath.length+(path[rootPath.length]==="."?1:0));
        }
        var element = $('<span class="red-ui-debug-msg-element"></span>');
        element.collapse = function() {
            element.find(".red-ui-debug-msg-expandable").parent().addClass("collapsed");
        }
        header = $('<span class="red-ui-debug-msg-row"></span>').appendTo(element);
        if (sourceId) {
            addMessageControls(header,sourceId,path,obj,rootPath,strippedKey,tools, enablePinning);
        }
        if (!key) {
            element.addClass("red-ui-debug-msg-top-level");
            if (sourceId && !expandPaths) {
                var pinned = pinnedPaths[sourceId];
                expandPaths = [];
                if (pinned) {
                    for (var pinnedPath in pinned) {
                        if (pinned.hasOwnProperty(pinnedPath)) {
                            try {
                                var res = getMessageProperty({$:obj},pinned[pinnedPath]);
                                if (res !== undefined) {
                                    expandPaths.push(pinnedPath);
                                }
                            } catch(err) {
                            }
                        }
                    }
                    expandPaths.sort();
                }
                element.clearPinned = function() {
                    element.find(".red-ui-debug-msg-row-pinned").removeClass("red-ui-debug-msg-row-pinned");
                    pinnedPaths[sourceId] = {};
                }
            }
        } else {
            if (!hideKey) {
                $('<span class="red-ui-debug-msg-object-key"></span>').text(key).appendTo(header);
                $('<span>: </span>').appendTo(header);
            }
        }
        entryObj = $('<span class="red-ui-debug-msg-object-value"></span>').appendTo(header);

        var isArray = Array.isArray(obj);
        var isArrayObject = false;
        if (obj && typeof obj === 'object' && obj.hasOwnProperty('type') && obj.hasOwnProperty('data') && ((obj.__enc__ && obj.type === 'set') || (obj.__enc__ && obj.type === 'array') || (obj.__enc__ && listOfTypedArrays.includes(obj.type)) || obj.type === 'Buffer')) {
            isArray = true;
            isArrayObject = true;
        }
        if (obj === null || obj === undefined) {
            $('<span class="red-ui-debug-msg-type-null">'+obj+'</span>').appendTo(entryObj);
        } else if (obj.__enc__ && obj.type === 'undefined') {
            $('<span class="red-ui-debug-msg-type-null">undefined</span>').appendTo(entryObj);
        } else if (obj.__enc__ && (obj.type === 'number' || obj.type === 'bigint')) {
            e = $('<span class="red-ui-debug-msg-type-number red-ui-debug-msg-object-header"></span>').text(obj.data).appendTo(entryObj);
        } else if (typeHint === "regexp" || (obj.__enc__ && obj.type === 'regexp')) {
            e = $('<span class="red-ui-debug-msg-type-string red-ui-debug-msg-object-header"></span>').text((typeof obj === "string")?obj:obj.data).appendTo(entryObj);
        } else if (typeHint === "function" || (obj.__enc__ && obj.type === 'function')) {
            e = $('<span class="red-ui-debug-msg-type-meta red-ui-debug-msg-object-header"></span>').text("function").appendTo(entryObj);
        } else if (typeHint === "internal" || (obj.__enc__ && obj.type === 'internal')) {
            e = $('<span class="red-ui-debug-msg-type-meta red-ui-debug-msg-object-header"></span>').text("[internal]").appendTo(entryObj);
        } else if (typeof obj === 'string') {
            if (/[\t\n\r]/.test(obj) || obj.length > MAX_STRING_LENGTH) {
                element.addClass('collapsed');
                $('<i class="fa fa-caret-right red-ui-debug-msg-object-handle"></i> ').prependTo(header);
                makeExpandable(header, function() {
                    $('<span class="red-ui-debug-msg-type-meta red-ui-debug-msg-object-type-header"></span>').text(typeHint||'string').appendTo(header);
                    var row = $('<div class="red-ui-debug-msg-object-entry collapsed"></div>').appendTo(element);
                    $('<pre class="red-ui-debug-msg-type-string"></pre>').text(obj).appendTo(row);
                },function(state) {if (ontoggle) { ontoggle(path,state);}}, checkExpanded(strippedKey, expandPaths, { expandLeafNodes }));
            }
            e = $('<span class="red-ui-debug-msg-type-string red-ui-debug-msg-object-header"></span>').html('"'+formatString(sanitize(truncateString(obj, MAX_STRING_LENGTH)))+'"').appendTo(entryObj);
            if (/^#[0-9a-f]{6}$/i.test(obj)) {
                $('<span class="red-ui-debug-msg-type-string-swatch"></span>').css('backgroundColor',obj).appendTo(e);
            }

            let n = RED.nodes.node(obj) ?? RED.nodes.workspace(obj);
            if (n) {
                if (options.nodeSelector && "function" == typeof options.nodeSelector) {
                    e.css('cursor', 'pointer').on("click", function(evt) {
                        evt.preventDefault();
                        options.nodeSelector(n.id);
                    })
                }
            }

        } else if (typeof obj === 'number') {
            e = $('<span class="red-ui-debug-msg-type-number"></span>').appendTo(entryObj);

            if (Number.isInteger(obj) && (obj >= 0)) { // if it's a +ve integer
                e.addClass("red-ui-debug-msg-type-number-toggle");
                e.on("click", function(evt) {
                    evt.preventDefault();
                    formatNumber($(this), obj, sourceId, path, true);
                });
            }
            formatNumber(e,obj,sourceId,path,false,typeHint==='hex'?'hex':undefined);

        } else if (isArray) {
            element.addClass('collapsed');

            var originalLength = obj.length;
            if (typeHint) {
                var m = /\[(\d+)\]/.exec(typeHint);
                if (m) {
                    originalLength = parseInt(m[1]);
                }
            }
            var data = obj;
            var type = 'array';
            if (isArrayObject) {
                data = obj.data;
                if (originalLength === undefined) {
                    originalLength = data.length;
                }
                if (data.__enc__) {
                    data = data.data;
                }
                type = obj.type.toLowerCase();
            } else if (/buffer/.test(typeHint)) {
                type = 'buffer';
            }
            var fullLength = data.length;

            if (originalLength > 0) {
                $('<i class="fa fa-caret-right red-ui-debug-msg-object-handle"></i> ').prependTo(header);
                var arrayRows = $('<div class="red-ui-debug-msg-array-rows"></div>').appendTo(element);
                element.addClass('red-ui-debug-msg-buffer-raw');
            }
            if (key) {
                headerHead = $('<span class="red-ui-debug-msg-type-meta"></span>').text(typeHint||(type+'['+originalLength+']')).appendTo(entryObj);
            } else {
                headerHead = $('<span class="red-ui-debug-msg-object-header"></span>').appendTo(entryObj);
                $('<span>[ </span>').appendTo(headerHead);
                var arrayLength = Math.min(originalLength,10);
                for (i=0;i<arrayLength;i++) {
                    buildMessageSummaryValue(data[i]).appendTo(headerHead);
                    if (i < arrayLength-1) {
                        $('<span>, </span>').appendTo(headerHead);
                    }
                }
                if (originalLength > arrayLength) {
                    $('<span> &hellip;</span>').appendTo(headerHead);
                }
                if (arrayLength === 0) {
                    $('<span class="red-ui-debug-msg-type-meta">empty</span>').appendTo(headerHead);
                }
                $('<span> ]</span>').appendTo(headerHead);
            }
            if (originalLength > 0) {

                makeExpandable(header,function() {
                    if (!key) {
                        headerHead = $('<span class="red-ui-debug-msg-type-meta red-ui-debug-msg-object-type-header"></span>').text(typeHint||(type+'['+originalLength+']')).appendTo(header);
                    }
                    if (type === 'buffer') {
                        var stringRow = $('<div class="red-ui-debug-msg-string-rows"></div>').appendTo(element);
                        var sr = $('<div class="red-ui-debug-msg-object-entry collapsed"></div>').appendTo(stringRow);
                        var stringEncoding = "";
                        try {
                            stringEncoding = String.fromCharCode.apply(null, new Uint16Array(data))
                        } catch(err) {
                            console.log(err);
                        }
                        $('<pre class="red-ui-debug-msg-type-string"></pre>').text(stringEncoding).appendTo(sr);
                        var bufferOpts = $('<span class="red-ui-debug-msg-buffer-opts"></span>').appendTo(headerHead);
                        var switchFormat = $('<a class="red-ui-button red-ui-button-small" href="#"></a>').text('raw').appendTo(bufferOpts).on("click", function(e) {
                            e.preventDefault();
                            e.stopPropagation();
                            formatBuffer(element,$(this),sourceId,path,true);
                        });
                        formatBuffer(element,switchFormat,sourceId,path,false);

                    }
                    var row;
                    if (fullLength <= 10) {
                        for (i=0;i<fullLength;i++) {
                            row = $('<div class="red-ui-debug-msg-object-entry collapsed"></div>').appendTo(arrayRows);
                            subElements[path+"["+i+"]"] = createObjectElement(
                                data[i],
                                {
                                    key: ""+i,
                                    typeHint: type==='buffer'?'hex':false,
                                    hideKey: false,
                                    path: path+"["+i+"]",
                                    sourceId,
                                    rootPath,
                                    expandPaths,
                                    expandLeafNodes,
                                    ontoggle,
                                    exposeApi,
                                    // tools: tools // Do not pass tools down as we
                                                    // keep them attached to the top-level header
                                    nodeSelector: options.nodeSelector,
                                    enablePinning
                                }
                            ).appendTo(row);
                        }
                    } else {
                        for (i=0;i<fullLength;i+=10) {
                            var minRange = i;
                            row = $('<div class="red-ui-debug-msg-object-entry collapsed"></div>').appendTo(arrayRows);
                            header = $('<span></span>').appendTo(row);
                            $('<i class="fa fa-caret-right red-ui-debug-msg-object-handle"></i> ').appendTo(header);
                            makeExpandable(header, (function() {
                                var min = minRange;
                                var max = Math.min(fullLength-1,(minRange+9));
                                var parent = row;
                                return function() {
                                    for (var i=min;i<=max;i++) {
                                        var row = $('<div class="red-ui-debug-msg-object-entry collapsed"></div>').appendTo(parent);
                                        subElements[path+"["+i+"]"] = createObjectElement(
                                            data[i],
                                            {
                                                key: ""+i,
                                                typeHint: type==='buffer'?'hex':false,
                                                hideKey: false,
                                                path: path+"["+i+"]",
                                                sourceId,
                                                rootPath,
                                                expandPaths,
                                                expandLeafNodes,
                                                ontoggle,
                                                exposeApi,
                                                // tools: tools // Do not pass tools down as we
                                                                // keep them attached to the top-level header
                                                nodeSelector: options.nodeSelector,
                                                enablePinning
                                            }
                                        ).appendTo(row);
                                    }
                                }
                            })(),
                            (function() { var path = path+"["+i+"]"; return function(state) {if (ontoggle) { ontoggle(path,state);}}})(),
                            checkExpanded(strippedKey,expandPaths,{ minRange, maxRange: Math.min(fullLength-1,(minRange+9)), expandLeafNodes}));
                            $('<span class="red-ui-debug-msg-object-key"></span>').html("["+minRange+" &hellip; "+Math.min(fullLength-1,(minRange+9))+"]").appendTo(header);
                        }
                        if (fullLength < originalLength) {
                             $('<div class="red-ui-debug-msg-object-entry collapsed"><span class="red-ui-debug-msg-object-key">['+fullLength+' &hellip; '+originalLength+']</span></div>').appendTo(arrayRows);
                        }
                    }
                },
                function(state) {if (ontoggle) { ontoggle(path,state);}},
                checkExpanded(strippedKey, expandPaths, { expandLeafNodes }));
            }
        } else if (typeof obj === 'object') {
            element.addClass('collapsed');
            var data = obj;
            var type = "object";
            if (data.__enc__) {
                data = data.data;
                type = obj.type.toLowerCase();
            }
            var keys = Object.keys(data);
            if (key || keys.length > 0) {
                $('<i class="fa fa-caret-right red-ui-debug-msg-object-handle"></i> ').prependTo(header);
                makeExpandable(header, function() {
                    if (!key) {
                        $('<span class="red-ui-debug-msg-type-meta red-ui-debug-msg-object-type-header"></span>').text(type).appendTo(header);
                    }
                    for (i=0;i<keys.length;i++) {
                        var row = $('<div class="red-ui-debug-msg-object-entry collapsed"></div>').appendTo(element);
                        var newPath = path;
                        if (newPath !== undefined) {
                            if (/^[a-zA-Z_$][0-9a-zA-Z_$]*$/.test(keys[i])) {
                                newPath += (newPath.length > 0?".":"")+keys[i];
                            } else {
                                newPath += "[\""+keys[i].replace(/"/,"\\\"")+"\"]"
                            }
                        }
                        subElements[newPath] = createObjectElement(
                            data[keys[i]],
                            {
                                key: keys[i],
                                typeHint: false,
                                hideKey: false,
                                path: newPath,
                                sourceId,
                                rootPath,
                                expandPaths,
                                expandLeafNodes,
                                ontoggle,
                                exposeApi,
                                // tools: tools // Do not pass tools down as we
                                                // keep them attached to the top-level header
                                nodeSelector: options.nodeSelector,
                                enablePinning
                            }
                        ).appendTo(row);
                    }
                    if (keys.length === 0) {
                        $('<div class="red-ui-debug-msg-object-entry red-ui-debug-msg-type-meta collapsed"></div>').text("empty").appendTo(element);
                    }
                },
                function(state) {if (ontoggle) { ontoggle(path,state);}},
                checkExpanded(strippedKey, expandPaths, { expandLeafNodes }));
            }
            if (key) {
                $('<span class="red-ui-debug-msg-type-meta"></span>').text(type).appendTo(entryObj);
            } else {
                headerHead = $('<span class="red-ui-debug-msg-object-header"></span>').appendTo(entryObj);
                $('<span>{ </span>').appendTo(headerHead);
                var keysLength = Math.min(keys.length,5);
                for (i=0;i<keysLength;i++) {
                    $('<span class="red-ui-debug-msg-object-key"></span>').text(keys[i]).appendTo(headerHead);
                    $('<span>: </span>').appendTo(headerHead);
                    buildMessageSummaryValue(data[keys[i]]).appendTo(headerHead);
                    if (i < keysLength-1) {
                        $('<span>, </span>').appendTo(headerHead);
                    }
                }
                if (keys.length > keysLength) {
                    $('<span> &hellip;</span>').appendTo(headerHead);
                }
                if (keysLength === 0) {
                    $('<span class="red-ui-debug-msg-type-meta">empty</span>').appendTo(headerHead);
                }
                $('<span> }</span>').appendTo(headerHead);
            }
        } else {
            $('<span class="red-ui-debug-msg-type-other"></span>').text(""+obj).appendTo(entryObj);
        }
        if (exposeApi) {
            element.prop('expand', function() { return function(targetPath, state) {
                if (path === targetPath) {
                    if (header.prop('toggle')) {
                        header.prop('toggle')(state);
                    }
                } else if (subElements[targetPath] && subElements[targetPath].prop('expand') ) {
                    subElements[targetPath].prop('expand')(targetPath,state);
                } else {
                    for (var p in subElements) {
                        if (subElements.hasOwnProperty(p)) {
                            if (targetPath.indexOf(p) === 0) {
                                if (subElements[p].prop('expand') ) {
                                    subElements[p].prop('expand')(targetPath,state);
                                }
                                break;
                            }
                        }
                    }
                }
            }});
        }
        return element;
    }

    function createError(code, message) {
        var e = new Error(message);
        e.code = code;
        return e;
    }

    function normalisePropertyExpression(str,msg) {
        // This must be kept in sync with validatePropertyExpression
        // in editor/js/ui/utils.js

        var length = str.length;
        if (length === 0) {
            throw createError("INVALID_EXPR","Invalid property expression: zero-length");
        }
        var parts = [];
        var start = 0;
        var inString = false;
        var inBox = false;
        var boxExpression = false;
        var quoteChar;
        var v;
        for (var i=0;i<length;i++) {
            var c = str[i];
            if (!inString) {
                if (c === "'" || c === '"') {
                    if (i != start) {
                        throw createError("INVALID_EXPR","Invalid property expression: unexpected "+c+" at position "+i);
                    }
                    inString = true;
                    quoteChar = c;
                    start = i+1;
                } else if (c === '.') {
                    if (i===0) {
                        throw createError("INVALID_EXPR","Invalid property expression: unexpected . at position 0");
                    }
                    if (start != i) {
                        v = str.substring(start,i);
                        if (/^\d+$/.test(v)) {
                            parts.push(parseInt(v));
                        } else {
                            parts.push(v);
                        }
                    }
                    if (i===length-1) {
                        throw createError("INVALID_EXPR","Invalid property expression: unterminated expression");
                    }
                    // Next char is first char of an identifier: a-z 0-9 $ _
                    if (!/[a-z0-9\$\_]/i.test(str[i+1])) {
                        throw createError("INVALID_EXPR","Invalid property expression: unexpected "+str[i+1]+" at position "+(i+1));
                    }
                    start = i+1;
                } else if (c === '[') {
                    if (i === 0) {
                        throw createError("INVALID_EXPR","Invalid property expression: unexpected "+c+" at position "+i);
                    }
                    if (start != i) {
                        parts.push(str.substring(start,i));
                    }
                    if (i===length-1) {
                        throw createError("INVALID_EXPR","Invalid property expression: unterminated expression");
                    }
                    // Start of a new expression. If it starts with msg it is a nested expression
                    // Need to scan ahead to find the closing bracket
                    if (/^msg[.\[]/.test(str.substring(i+1))) {
                        var depth = 1;
                        var inLocalString = false;
                        var localStringQuote;
                        for (var j=i+1;j<length;j++) {
                            if (/["']/.test(str[j])) {
                                if (inLocalString) {
                                    if (str[j] === localStringQuote) {
                                        inLocalString = false
                                    }
                                } else {
                                    inLocalString = true;
                                    localStringQuote = str[j]
                                }
                            }
                            if (str[j] === '[') {
                                depth++;
                            } else if (str[j] === ']') {
                                depth--;
                            }
                            if (depth === 0) {
                                try {
                                    if (msg) {
                                        parts.push(getMessageProperty(msg, str.substring(i+1,j)))
                                    } else {
                                        parts.push(normalisePropertyExpression(str.substring(i+1,j), msg));
                                    }
                                    inBox = false;
                                    i = j;
                                    start = j+1;
                                    break;
                                } catch(err) {
                                    throw createError("INVALID_EXPR","Invalid expression started at position "+(i+1))
                                }
                            }
                        }
                        if (depth > 0) {
                            throw createError("INVALID_EXPR","Invalid property expression: unmatched '[' at position "+i);
                        }
                        continue;
                    } else if (!/["'\d]/.test(str[i+1])) {
                        // Next char is either a quote or a number
                        throw createError("INVALID_EXPR","Invalid property expression: unexpected "+str[i+1]+" at position "+(i+1));
                    }
                    start = i+1;
                    inBox = true;
                } else if (c === ']') {
                    if (!inBox) {
                        throw createError("INVALID_EXPR","Invalid property expression: unexpected "+c+" at position "+i);
                    }
                    if (start != i) {
                        v = str.substring(start,i);
                        if (/^\d+$/.test(v)) {
                            parts.push(parseInt(v));
                        } else {
                            throw createError("INVALID_EXPR","Invalid property expression: unexpected array expression at position "+start);
                        }
                    }
                    start = i+1;
                    inBox = false;
                } else if (c === ' ') {
                    throw createError("INVALID_EXPR","Invalid property expression: unexpected ' ' at position "+i);
                }
            } else {
                if (c === quoteChar) {
                    if (i-start === 0) {
                        throw createError("INVALID_EXPR","Invalid property expression: zero-length string at position "+start);
                    }
                    parts.push(str.substring(start,i));
                    // If inBox, next char must be a ]. Otherwise it may be [ or .
                    if (inBox && !/\]/.test(str[i+1])) {
                        throw createError("INVALID_EXPR","Invalid property expression: unexpected array expression at position "+start);
                    } else if (!inBox && i+1!==length && !/[\[\.]/.test(str[i+1])) {
                        throw createError("INVALID_EXPR","Invalid property expression: unexpected "+str[i+1]+" expression at position "+(i+1));
                    }
                    start = i+1;
                    inString = false;
                }
            }

        }
        if (inBox || inString) {
            throw new createError("INVALID_EXPR","Invalid property expression: unterminated expression");
        }
        if (start < length) {
            parts.push(str.substring(start));
        }
        return parts;
    }

    /**
     * Validate a property expression
     * @param {*} str - the property value
     * @returns {boolean|string} whether the node proprty is valid. `true`: valid `false|String`: invalid
     */
    function validatePropertyExpression(str, opt) {
        try {
            const parts = normalisePropertyExpression(str);
            return true;
        } catch(err) {
            // If the validator has opt, it is a 3.x validator that
            // can return a String to mean 'invalid' and provide a reason
            if (opt) {
                if (opt.label) {
                    return opt.label + ': ' + err.message;
                }
                return err.message;
            }
            // Otherwise, a 2.x returns a false value
            return false;
        }
    }

    /**
     * Checks a typed property is valid according to the type.
     * Returns true if valid.
     * Return String error message if invalid
     * @param {*} propertyType 
     * @param {*} propertyValue 
     * @returns true if valid, String if invalid
     */
    function validateTypedProperty(propertyValue, propertyType, opt) {
        if (propertyValue && /^\${[^}]+}$/.test(propertyValue)) {
            // Allow ${ENV_VAR} value
            return true
        }
        let error;
        if (propertyType === 'json') {
            try {
                JSON.parse(propertyValue);
            } catch(err) {
                error = RED._("validator.errors.invalid-json", {
                    error: err.message
                });
            }
        } else if (propertyType === 'msg' || propertyType === 'flow' || propertyType === 'global' ) {
            // To avoid double label
            const valid = RED.utils.validatePropertyExpression(propertyValue, opt ? {} : null);
            if (valid !== true) {
                error = opt ? valid : RED._("validator.errors.invalid-prop");
            }
        } else if (propertyType === 'num') {
            if (!/^NaN$|^[+-]?[0-9]*\.?[0-9]*([eE][-+]?[0-9]+)?$|^[+-]?(0b|0B)[01]+$|^[+-]?(0o|0O)[0-7]+$|^[+-]?(0x|0X)[0-9a-fA-F]+$/.test(propertyValue)) {
                error = RED._("validator.errors.invalid-num");
            }
        } else if (propertyType === 'jsonata') {
            try { 
                jsonata(propertyValue)
            } catch(err) {
                error = RED._("validator.errors.invalid-expr", {
                    error: err.message
                });
            }
        }
        if (error) {
            if (opt && opt.label) {
                return opt.label + ': ' + error;
            }
            return error;
        }
        return true;
    }

    function getMessageProperty(msg,expr) {
        var result = null;
        var msgPropParts;

        if (typeof expr === 'string') {
            if (expr.indexOf('msg.')===0) {
                expr = expr.substring(4);
            }
            msgPropParts = normalisePropertyExpression(expr);
        } else {
            msgPropParts = expr;
        }
        var m;
        msgPropParts.reduce(function(obj, key) {
            result = (typeof obj[key] !== "undefined" ? obj[key] : undefined);
            if (result === undefined && obj.hasOwnProperty('type') && obj.hasOwnProperty('data')&& obj.hasOwnProperty('length')) {
                result = (typeof obj.data[key] !== "undefined" ? obj.data[key] : undefined);
            }
            return result;
        }, msg);
        return result;
    }

    function setMessageProperty(msg,prop,value,createMissing) {
        if (typeof createMissing === 'undefined') {
            createMissing = (typeof value !== 'undefined');
        }
        if (prop.indexOf('msg.')===0) {
            prop = prop.substring(4);
        }
        var msgPropParts = normalisePropertyExpression(prop);
        var depth = 0;
        var length = msgPropParts.length;
        var obj = msg;
        var key;
        for (var i=0;i<length-1;i++) {
            key = msgPropParts[i];
            if (typeof key === 'string' || (typeof key === 'number' && !Array.isArray(obj))) {
                if (obj.hasOwnProperty(key)) {
                    obj = obj[key];
                } else if (createMissing) {
                    if (typeof msgPropParts[i+1] === 'string') {
                        obj[key] = {};
                    } else {
                        obj[key] = [];
                    }
                    obj = obj[key];
                } else {
                    return null;
                }
            } else if (typeof key === 'number') {
                // obj is an array
                if (obj[key] === undefined) {
                    if (createMissing) {
                        if (typeof msgPropParts[i+1] === 'string') {
                            obj[key] = {};
                        } else {
                            obj[key] = [];
                        }
                        obj = obj[key];
                    } else {
                        return null;
                    }
                } else {
                    obj = obj[key];
                }
            }
        }
        key = msgPropParts[length-1];
        if (typeof value === "undefined") {
            if (typeof key === 'number' && Array.isArray(obj)) {
                obj.splice(key,1);
            } else {
                delete obj[key]
            }
        } else {
            obj[key] = value;
        }
    }

    function separateIconPath(icon) {
        var result = {module: "", file: ""};
        if (icon) {
            var index = icon.indexOf(RED.settings.apiRootUrl+'icons/');
            if (index === 0) {
                icon = icon.substring((RED.settings.apiRootUrl+'icons/').length);
            }
            var match = /^((?:@[^/]+\/)?[^/]+)\/(.*)$/.exec(icon);
            if (match) {
                result.module = match[1];
                result.file = match[2];
            } else {
                result.file = icon;
            }
        }
        return result;
    }

    function getDefaultNodeIcon(def,node) {
        def = def || {};
        var icon_url;
        if (node && node.type === "subflow") {
            icon_url = "node-red/subflow.svg";
        } else if (typeof def.icon === "function") {
            try {
                icon_url = def.icon.call(node);
            } catch(err) {
                console.log("Definition error: "+def.type+".icon",err);
                icon_url = "arrow-in.svg";
            }
        } else {
            icon_url = def.icon;
        }

        var iconPath = separateIconPath(icon_url);
        if (!iconPath.module) {
            if (def.set) {
                iconPath.module = def.set.module;
            } else {
                // Handle subflow instance nodes that don't have def.set
                iconPath.module = "node-red";
            }
        }
        return iconPath;
    }

    function isIconExists(iconPath) {
        var iconSets = RED.nodes.getIconSets();
        var iconFileList = iconSets[iconPath.module];
        if (iconFileList && iconFileList.indexOf(iconPath.file) !== -1) {
            return true;
        } else {
            return false;
        }
    }

    function getNodeIcon(def,node) {
        def = def || {};
        if (node && node.type === '_selection_') {
            return "font-awesome/fa-object-ungroup";
        } else if (node && node.type === 'group') {
            return "font-awesome/fa-object-group"
        } else if ((node && node.type === 'junction') || (def.type === "junction") ) {
            return "font-awesome/fa-circle-o"
        } else if (def.category === 'config') {
            return RED.settings.apiRootUrl+"icons/node-red/cog.svg"
        } else if ((node && /^_action_:/.test(node.type)) || /^_action_:/.test(def.type)) {
            return "font-awesome/fa-cogs"
        } else if (node && node.type === 'tab') {
            return "red-ui-icons/red-ui-icons-flow"
            // return RED.settings.apiRootUrl+"images/subflow_tab.svg"
        } else if (node && node.type === 'unknown') {
            return RED.settings.apiRootUrl+"icons/node-red/alert.svg"
        } else if (node && node.icon) {
            var iconPath = separateIconPath(node.icon);
            if (isIconExists(iconPath)) {
                if (iconPath.module === "font-awesome") {
                    return node.icon;
                } else {
                    return RED.settings.apiRootUrl+"icons/" + node.icon;
                }
            } else if (iconPath.module !== "font-awesome" && /.png$/i.test(iconPath.file)) {
                iconPath.file = iconPath.file.replace(/.png$/,".svg");
                if (isIconExists(iconPath)) {
                    return RED.settings.apiRootUrl+"icons/" + node.icon.replace(/.png$/,".svg");
                }
            }
        }

        var iconPath = getDefaultNodeIcon(def, node);
        if (isIconExists(iconPath)) {
            if (iconPath.module === "font-awesome") {
                return iconPath.module+"/"+iconPath.file;
            } else {
                return RED.settings.apiRootUrl+"icons/"+iconPath.module+"/"+iconPath.file;
            }
        }

        if (/.png$/i.test(iconPath.file)) {
            var originalFile = iconPath.file;
            iconPath.file = iconPath.file.replace(/.png$/,".svg");
            if (isIconExists(iconPath)) {
                return RED.settings.apiRootUrl+"icons/"+iconPath.module+"/"+iconPath.file;
            }
            iconPath.file = originalFile;
        }

        // This could be a non-core node trying to use a core icon.
        iconPath.module = 'node-red';
        if (isIconExists(iconPath)) {
            return RED.settings.apiRootUrl+"icons/"+iconPath.module+"/"+iconPath.file;
        }
        if (/.png$/i.test(iconPath.file)) {
            iconPath.file = iconPath.file.replace(/.png$/,".svg");
            if (isIconExists(iconPath)) {
                return RED.settings.apiRootUrl+"icons/"+iconPath.module+"/"+iconPath.file;
            }
        }
        if (def.category === 'subflows') {
            return RED.settings.apiRootUrl+"icons/node-red/subflow.svg";
        }
        return RED.settings.apiRootUrl+"icons/node-red/arrow-in.svg";
    }

    function getNodeLabel(node,defaultLabel) {
        defaultLabel = defaultLabel||"";
        var l;
        if (node.type === 'tab') {
            l = node.label || defaultLabel
        } else if (node.type === 'group') {
            l = node.name || defaultLabel
        } else if (node.type === 'junction') {
            l = 'junction'
        } else {
            l = node._def.label;
            try {
                l = (typeof l === "function" ? l.call(node) : l)||defaultLabel;
            } catch(err) {
                console.log("Definition error: "+node.type+".label",err);
                l = defaultLabel;
            }
        }
        return RED.text.bidi.enforceTextDirectionWithUCC(l);
    }

    function getPaletteLabel(nodeType, def) {
        var label = nodeType;
        if (typeof def.paletteLabel !== "undefined") {
            try {
                label = (typeof def.paletteLabel === "function" ? def.paletteLabel.call(def) : def.paletteLabel)||"";
            } catch(err) {
                console.log("Definition error: "+nodeType+".paletteLabel",err);
            }
        }
        return label
    }

    var nodeColorCache = {};
    function clearNodeColorCache() {
        nodeColorCache = {};
    }

    function getNodeColor(type, def) {
        def = def || {};
        var result = def.color;
        var paletteTheme = RED.settings.theme('palette.theme') || [];
        if (paletteTheme.length > 0) {
            if (!nodeColorCache.hasOwnProperty(type)) {
                nodeColorCache[type] = def.color;
                var l = paletteTheme.length;
                for (var i = 0; i < l; i++ ){
                    var themeRule = paletteTheme[i];
                    if (themeRule.hasOwnProperty('category')) {
                        if (!themeRule.hasOwnProperty('_category')) {
                            themeRule._category = new RegExp(themeRule.category);
                        }
                        if (!themeRule._category.test(def.category)) {
                            continue;
                        }
                    }
                    if (themeRule.hasOwnProperty('type')) {
                        if (!themeRule.hasOwnProperty('_type')) {
                            themeRule._type = new RegExp(themeRule.type);
                        }
                        if (!themeRule._type.test(type)) {
                            continue;
                        }
                    }
                    nodeColorCache[type] = themeRule.color || def.color;
                    break;
                }
            }
            result = nodeColorCache[type];
        }
        if (result) {
            return result;
        } else {
            return "#ddd";
        }
    }

    function addSpinnerOverlay(container,contain) {
        var spinner = $('<div class="red-ui-component-spinner "><img src="red/images/spin.svg"/></div>').appendTo(container);
        if (contain) {
            spinner.addClass('red-ui-component-spinner-contain');
        }
        return spinner;
    }

    function decodeObject(payload,format) {
        if ((format === 'number') && (payload === "NaN")) {
            payload = Number.NaN;
        } else if ((format === 'number') && (payload === "Infinity")) {
            payload = Infinity;
        } else if ((format === 'number') && (payload === "-Infinity")) {
            payload = -Infinity;
        } else if (format === 'Object' || /^(array|set|map)/.test(format) || format === 'boolean' || format === 'number' || listOfTypedArrays.includes(format)) {
            payload = JSON.parse(payload);
        } else if (/error/i.test(format)) {
            payload = JSON.parse(payload);
        } else if (format === 'null') {
            payload = null;
        } else if (format === 'undefined') {
            payload = undefined;
        } else if (/^buffer/.test(format)) {
            var buffer = payload;
            payload = [];
            for (var c = 0; c < buffer.length; c += 2) {
                payload.push(parseInt(buffer.substr(c, 2), 16));
            }
        }
        return payload;
    }

    function parseContextKey(key, defaultStore) {
        var parts = {};
        var m = /^#:\((\S+?)\)::(.*)$/.exec(key);
        if (m) {
            parts.store = m[1];
            parts.key = m[2];
        } else {
            parts.key = key;
            if (defaultStore) {
                parts.store = defaultStore;
            } else if (RED.settings.context) {
                parts.store = RED.settings.context.default;
            }
        }
        return parts;
    }

     /**
      * Create or update an icon element and append it to iconContainer.
      * @param iconUrl - Url of icon.
      * @param iconContainer - Icon container element with red-ui-palette-icon-container class.
      * @param isLarge - Whether the icon size is large.
      */
    function createIconElement(iconUrl, iconContainer, isLarge) {
        // Removes the previous icon when icon was changed.
        var iconElement = iconContainer.find(".red-ui-palette-icon");
        if (iconElement.length !== 0) {
            iconElement.remove();
        }
        var faIconElement = iconContainer.find("i");
        if (faIconElement.length !== 0) {
            faIconElement.remove();
        }

        // Show either icon image or font-awesome icon
        var iconPath = separateIconPath(iconUrl);
        if (iconPath.module === "font-awesome") {
            var fontAwesomeUnicode = RED.nodes.fontAwesome.getIconUnicode(iconPath.file);
            if (fontAwesomeUnicode) {
                var faIconElement = $('<i/>').appendTo(iconContainer);
                var faLarge = isLarge ? "fa-lg " : "";
                faIconElement.addClass("red-ui-palette-icon-fa fa fa-fw " + faLarge + iconPath.file);
                return;
            }
            // If the specified name is not defined in font-awesome, show arrow-in icon.
            iconUrl = RED.settings.apiRootUrl+"icons/node-red/arrow-in.svg"
        } else if (iconPath.module === "red-ui-icons") {
            var redIconElement = $('<i/>').appendTo(iconContainer);
            redIconElement.addClass("red-ui-palette-icon red-ui-icons " + iconPath.file);
            return;
        }
        var imageIconElement = $('<div/>',{class:"red-ui-palette-icon"}).appendTo(iconContainer);
        imageIconElement.css("backgroundImage", "url("+iconUrl+")");
    }

    function createNodeIcon(node, includeLabel) {
        var container = $('<span class="red-ui-node-icon-container">');

        var def = node._def;
        var nodeDiv = $('<div>',{class:"red-ui-node-icon"})
        if (node.type === "_selection_") {
            nodeDiv.addClass("red-ui-palette-icon-selection");
        } else if (node.type === "group") {
            nodeDiv.addClass("red-ui-palette-icon-group");
        } else if (node.type === "junction") {
            nodeDiv.addClass("red-ui-palette-icon-junction");
        } else if (node.type === 'tab') {
            nodeDiv.addClass("red-ui-palette-icon-flow");
        } else {
            var colour = RED.utils.getNodeColor(node.type,def);
            // if (node.type === 'tab') {
            //     colour = "#C0DEED";
            // }
            nodeDiv.css('backgroundColor',colour);
            var borderColor = getDarkerColor(colour);
            if (borderColor !== colour) {
                nodeDiv.css('border-color',borderColor)
            }
        }

        var icon_url = RED.utils.getNodeIcon(def,node);
        RED.utils.createIconElement(icon_url, nodeDiv, true);

        nodeDiv.appendTo(container);

        if (includeLabel) {
            var labelText = RED.utils.getNodeLabel(node,node.name || (node.type+": "+node.id));
            var label = $('<div>',{class:"red-ui-node-label"}).appendTo(container);
            if (labelText) {
                label.text(labelText)
            } else {
                label.html("&nbsp;")
            }
        }
        return container;
    }

    function getDarkerColor(c) {
        var r,g,b;
        if (/^#[a-f0-9]{6}$/i.test(c)) {
            r = parseInt(c.substring(1, 3), 16);
            g = parseInt(c.substring(3, 5), 16);
            b = parseInt(c.substring(5, 7), 16);
        } else if (/^#[a-f0-9]{3}$/i.test(c)) {
            r = parseInt(c.substring(1, 2)+c.substring(1, 2), 16);
            g = parseInt(c.substring(2, 3)+c.substring(2, 3), 16);
            b = parseInt(c.substring(3, 4)+c.substring(3, 4), 16);
        } else {
            return c;
        }
        var l = 0.3 * r/255 + 0.59 * g/255 + 0.11 * b/255 ;
        r = Math.max(0,r-50);
        g = Math.max(0,g-50);
        b = Math.max(0,b-50);
        var s = ((r<<16) + (g<<8) + b).toString(16);
        return '#'+'000000'.slice(0, 6-s.length)+s;
    }

    function parseModuleList(list) {
        list = list || ["*"];
        return list.map(function(rule) {
            var m = /^(.+?)(?:@(.*))?$/.exec(rule);
            var wildcardPos = m[1].indexOf("*");
            wildcardPos = wildcardPos===-1?Infinity:wildcardPos;

            return {
                module: new RegExp("^"+m[1].replace(/\*/g,".*")+"$"),
                version: m[2],
                wildcardPos: wildcardPos
            }
        })
    }

    function checkAgainstList(module,version,list) {
        for (var i=0;i<list.length;i++) {
            var rule = list[i];
            if (rule.module.test(module)) {
                // Without a full semver library in the editor,
                // we skip the version check.
                // Not ideal - but will get caught in the runtime
                // if the user tries to install.
                return rule;
            }
        }
    }

    function checkModuleAllowed(module,version,allowList,denyList) {
        if (!allowList && !denyList) {
            // Default to allow
            return true;
        }
        if (allowList.length === 0 && denyList.length === 0) {
            return true;
        }

        var allowedRule = checkAgainstList(module,version,allowList);
        var deniedRule = checkAgainstList(module,version,denyList);
        // console.log("A",allowedRule)
        // console.log("D",deniedRule)

        if (allowedRule && !deniedRule) {
            return true;
        }
        if (!allowedRule && deniedRule) {
            return false;
        }
        if (!allowedRule && !deniedRule) {
            return true;
        }
        if (allowedRule.wildcardPos !== deniedRule.wildcardPos) {
            return allowedRule.wildcardPos > deniedRule.wildcardPos
        } else {
            // First wildcard in same position.
            // Go with the longer matching rule. This isn't going to be 100%
            // right, but we are deep into edge cases at this point.
            return allowedRule.module.toString().length > deniedRule.module.toString().length
        }
        return false;
    }

    function getBrowserInfo() {
        var r = {}
        try {
            var ua = navigator.userAgent;
            r.ua = ua;
            r.browser = /Edge\/\d+/.test(ua) ? 'ed' : /MSIE 9/.test(ua) ? 'ie9' : /MSIE 10/.test(ua) ? 'ie10' : /MSIE 11/.test(ua) ? 'ie11' : /MSIE\s\d/.test(ua) ? 'ie?' : /rv\:11/.test(ua) ? 'ie11' : /Firefox\W\d/.test(ua) ? 'ff' : /Chrom(e|ium)\W\d|CriOS\W\d/.test(ua) ? 'gc' : /\bSafari\W\d/.test(ua) ? 'sa' : /\bOpera\W\d/.test(ua) ? 'op' : /\bOPR\W\d/i.test(ua) ? 'op' : typeof MSPointerEvent !== 'undefined' ? 'ie?' : '';
            r.os = /Windows NT 10/.test(ua) ? "win10" : /Windows NT 6\.0/.test(ua) ? "winvista" : /Windows NT 6\.1/.test(ua) ? "win7" : /Windows NT 6\.\d/.test(ua) ? "win8" : /Windows NT 5\.1/.test(ua) ? "winxp" : /Windows NT [1-5]\./.test(ua) ? "winnt" : /Mac/.test(ua) ? "mac" : /Linux/.test(ua) ? "linux" : /X11/.test(ua) ? "nix" : "";
            r.touch = 'ontouchstart' in document.documentElement;
            r.mobile = /IEMobile|Windows Phone|Lumia/i.test(ua) ? 'w' : /iPhone|iP[oa]d/.test(ua) ? 'i' : /Android/.test(ua) ? 'a' : /BlackBerry|PlayBook|BB10/.test(ua) ? 'b' : /Mobile Safari/.test(ua) ? 's' : /webOS|Mobile|Tablet|Opera Mini|\bCrMo\/|Opera Mobi/i.test(ua) ? 1 : 0;
            r.tablet = /Tablet|iPad/i.test(ua);
            r.ie = /MSIE \d|Trident.*rv:/.test(navigator.userAgent);
            r.android = /android/i.test(navigator.userAgent);
        } catch (error) { }
        return r;
    }

    return {
        createObjectElement: createObjectElement,
        getMessageProperty: getMessageProperty,
        setMessageProperty: setMessageProperty,
        normalisePropertyExpression: normalisePropertyExpression,
        validatePropertyExpression: validatePropertyExpression,
        separateIconPath: separateIconPath,
        getDefaultNodeIcon: getDefaultNodeIcon,
        getNodeIcon: getNodeIcon,
        getNodeLabel: getNodeLabel,
        getNodeColor: getNodeColor,
        getPaletteLabel: getPaletteLabel,
        clearNodeColorCache: clearNodeColorCache,
        addSpinnerOverlay: addSpinnerOverlay,
        decodeObject: decodeObject,
        parseContextKey: parseContextKey,
        createIconElement: createIconElement,
        sanitize: sanitize,
        truncateString: truncateString,
        renderMarkdown: renderMarkdown,
        createNodeIcon: createNodeIcon,
        getDarkerColor: getDarkerColor,
        parseModuleList: parseModuleList,
        checkModuleAllowed: checkModuleAllowed,
        getBrowserInfo: getBrowserInfo,
        validateTypedProperty: validateTypedProperty
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
(function($) {

/**
 * options:
 *   - addButton : boolean|string - text for add label, default 'add'
 *   - buttons : array - list of custom buttons (objects with fields 'id', 'label', 'icon', 'title', 'click')
 *   - height : number|'auto'
 *   - resize : function - called when list as a whole is resized
 *   - resizeItem : function(item) - called to resize individual item
 *   - sortable : boolean|string - string is the css selector for handle
 *   - sortItems : function(items) - when order of items changes
 *   - connectWith : css selector of other sortables
 *   - removable : boolean - whether to display delete button on items
 *   - addItem : function(row,index,itemData) - when an item is added
 *   - removeItem : function(itemData) - called when an item is removed
 *   - filter : function(itemData) - called for each item to determine if it should be shown
 *   - sort : function(itemDataA,itemDataB) - called to sort items
 *   - scrollOnAdd : boolean - whether to scroll to newly added items
 * methods:
 *   - addItem(itemData)
 *   - insertItemAt : function(data,index) - add an item at the specified index
 *   - removeItem(itemData, detach) - remove the item. Optionally detach to preserve any event handlers on the item's label
 *   - getItemAt(index)
 *   - indexOf(itemData)
 *   - width(width)
 *   - height(height)
 *   - items()
 *   - empty()
 *   - filter(filter)
 *   - sort(sort)
 *   - length()
 */
    $.widget( "nodered.editableList", {
        _create: function() {
            var that = this;

            this.element.addClass('red-ui-editableList-list');
            this.uiWidth = this.element.width();
            this.uiContainer = this.element
                .wrap( "<div>" )
                .parent();

            if (this.options.header) {
                this.options.header.addClass("red-ui-editableList-header");
                this.borderContainer = this.uiContainer.wrap("<div>").parent();
                this.borderContainer.prepend(this.options.header);
                this.topContainer = this.borderContainer.wrap("<div>").parent();
            } else {
                this.topContainer = this.uiContainer.wrap("<div>").parent();
            }
            this.topContainer.addClass('red-ui-editableList');
            if (this.options.class) {
                this.topContainer.addClass(this.options.class);
            }

            var buttons = this.options.buttons || [];

            if (this.options.addButton !== false) {
                var addLabel, addTitle;
                if (typeof this.options.addButton === 'string') {
                    addLabel = this.options.addButton
                } else {
                    if (RED && RED._) {
                        addLabel = RED._("editableList.add");
                        addTitle = RED._("editableList.addTitle");
                    } else {
                        addLabel = 'add';
                        addTitle = 'add new item';
                    }
                }
                buttons.unshift({
                    label: addLabel,
                    icon: "fa fa-plus",
                    click: function(evt) {
                        that.addItem({});
                    },
                    title: addTitle
                });
            }

            buttons.forEach(function(button) {
                var element = $('<button type="button" class="red-ui-button red-ui-button-small red-ui-editableList-addButton" style="margin-top: 4px; margin-right: 5px;"></button>')
                    .appendTo(that.topContainer)
                    .on("click", function(evt) {
                        evt.preventDefault();
                        if (button.click !== undefined) {
                            button.click(evt);
                        }
                    });

                if (button.id) {
                    element.attr("id", button.id);
                }
                if (button.title) {
                    element.attr("title", button.title);
                }
                if (button.icon) {
                    element.append($("<i></i>").attr("class", button.icon));
                }
                if (button.label) {
                    element.append($("<span></span>").text(" " + button.label));
                }
            });

            if (this.element.css("position") === "absolute") {
                ["top","left","bottom","right"].forEach(function(s) {
                    var v = that.element.css(s);
                    if (v!=="auto" && v!=="") {
                        that.topContainer.css(s,v);
                        that.uiContainer.css(s,"0");
                        if (s === "top" && that.options.header) {
                            that.uiContainer.css(s,"20px")
                        }
                        that.element.css(s,'auto');
                    }
                })
                this.element.css("position","static");
                this.topContainer.css("position","absolute");
                this.uiContainer.css("position","absolute");

            }
            if (this.options.header) {
                this.borderContainer.addClass("red-ui-editableList-border");
            } else {
                this.uiContainer.addClass("red-ui-editableList-border");
            }
            this.uiContainer.addClass("red-ui-editableList-container");

            this.uiHeight = this.element.height();

            this.activeFilter = this.options.filter||null;
            this.activeSort = this.options.sort||null;
            this.scrollOnAdd = this.options.scrollOnAdd;
            if (this.scrollOnAdd === undefined) {
                this.scrollOnAdd = true;
            }
            var minHeight = this.element.css("minHeight");
            if (minHeight !== '0px') {
                this.uiContainer.css("minHeight",minHeight);
                this.element.css("minHeight",0);
            }
            var maxHeight = this.element.css("maxHeight");
            if (maxHeight !== '0px') {
                this.uiContainer.css("maxHeight",maxHeight);
                this.element.css("maxHeight",null);
            }
            if (this.options.height !== 'auto') {
                this.uiContainer.css("overflow-y","auto");
                if (!isNaN(this.options.height)) {
                    this.uiHeight = this.options.height;
                }
            }
            this.element.height('auto');

            var attrStyle = this.element.attr('style');
            var m;
            if ((m = /width\s*:\s*(\d+%)/i.exec(attrStyle)) !== null) {
                this.element.width('100%');
                this.uiContainer.width(m[1]);
            }
            if (this.options.sortable) {
                var isCanceled = false; // Flag to track if an item has been canceled from being dropped into a different list
                var noDrop = false; // Flag to track if an item is being dragged into a different list
                var handle = (typeof this.options.sortable === 'string')?
                                this.options.sortable :
                                ".red-ui-editableList-item-handle";
                var sortOptions = {
                    axis: "y",
                    update: function( event, ui ) {
                        // dont trigger update if the item is being canceled
                        const targetList = $(event.target);
                        const draggedItem = ui.item;
                        const draggedItemParent = draggedItem.parent();
                        if (!targetList.is(draggedItemParent) && draggedItem.hasClass("red-ui-editableList-item-constrained")) {
                            noDrop = true;
                        }
                        if (isCanceled || noDrop) {
                            return;
                        }
                        if (that.options.sortItems) {
                            that.options.sortItems(that.items());
                        }
                    },
                    handle:handle,
                    cursor: "move",
                    tolerance: "pointer",
                    forcePlaceholderSize:true,
                    placeholder: "red-ui-editabelList-item-placeholder",
                    start: function (event, ui) {
                        isCanceled = false;
                        ui.placeholder.height(ui.item.height() - 4);
                        ui.item.css('cursor', 'grabbing'); // TODO: this doesn't seem to work, use a class instead?
                    },
                    stop: function (event, ui) {
                        ui.item.css('cursor', 'auto');
                    },
                    receive: function (event, ui) {
                        if (ui.item.hasClass("red-ui-editableList-item-constrained")) {
                            isCanceled = true;
                            $(ui.sender).sortable('cancel');
                        }
                    },
                    over: function (event, ui) {
                        // if the dragged item is constrained, prevent it from being dropped into a different list
                        const targetList = $(event.target);
                        const draggedItem = ui.item;
                        const draggedItemParent = draggedItem.parent();
                        if (!targetList.is(draggedItemParent) && draggedItem.hasClass("red-ui-editableList-item-constrained")) {
                            noDrop = true;
                            draggedItem.css('cursor', 'no-drop'); // TODO: this doesn't seem to work, use a class instead?
                        } else {
                            noDrop = false;
                            draggedItem.css('cursor', 'grabbing'); // TODO: this doesn't seem to work, use a class instead?
                        }
                    }
                };
                if (this.options.connectWith) {
                    sortOptions.connectWith = this.options.connectWith;
                }

                this.element.sortable(sortOptions);
            }

            this._resize();

            // this.menu = this._createMenu(this.types, function(v) { that.type(v) });
            // this.type(this.options.default||this.types[0].value);
        },
        _resize: function() {
            var currentFullHeight = this.topContainer.height();
            var innerHeight = this.uiContainer.height();
            var delta = currentFullHeight - innerHeight;
            if (this.uiHeight !== 0) {
                this.uiContainer.height(this.uiHeight-delta);
            }
            if (this.options.resize) {
                this.options.resize();
            }
            if (this.options.resizeItem) {
                var that = this;
                this.element.children().each(function(i) {
                    that.options.resizeItem($(this).children(".red-ui-editableList-item-content"),i);
                });
            }
        },
        _destroy: function() {
            if (this.topContainer) {
                var tc = this.topContainer;
                delete this.topContainer;
                tc.remove();
            }
        },
        _refreshFilter: function() {
            var that = this;
            var count = 0;
            if (!this.activeFilter) {
                return this.element.children().show();
            }
            var items = this.items();
            items.each(function (i,el) {
                var data = el.data('data');
                try {
                    if (that.activeFilter(data)) {
                        el.parent().show();
                        count++;
                    } else {
                        el.parent().hide();
                    }
                } catch(err) {
                    console.log(err);
                    el.parent().show();
                    count++;
                }
            });
            return count;
        },
        _refreshSort: function() {
            if (this.activeSort) {
                var items = this.element.children();
                var that = this;
                items.sort(function(A,B) {
                    return that.activeSort($(A).children(".red-ui-editableList-item-content").data('data'),$(B).children(".red-ui-editableList-item-content").data('data'));
                });
                $.each(items,function(idx,li) {
                    that.element.append(li);
                })
            }
        },
        width: function(desiredWidth) {
            this.uiWidth = desiredWidth;
            this._resize();
        },
        height: function(desiredHeight) {
            this.uiHeight = desiredHeight;
            this._resize();
        },
        getItemAt: function(index) {
            var items = this.items();
            if (index >= 0 && index < items.length) {
                return $(items[index]).data('data');
            } else {
                return;
            }
        },
        indexOf: function(data) {
            var items = this.items();
            for (var i=0;i<items.length;i++) {
                if ($(items[i]).data('data') === data) {
                    return i
                }
            }
            return -1
        },
        insertItemAt: function(data,index) {
            var that = this;
            data = data || {};
            var li = $('<li>');
            var row = $('<div/>').addClass("red-ui-editableList-item-content").appendTo(li);
            row.data('data',data);
            if (this.options.sortable === true) {
                $('<i class="red-ui-editableList-item-handle fa fa-bars"></i>').appendTo(li);
                li.addClass("red-ui-editableList-item-sortable");
            }
            if (this.options.removable) {
                var deleteButton = $('<a/>',{href:"#",class:"red-ui-editableList-item-remove red-ui-button red-ui-button-small"}).appendTo(li);
                $('<i/>',{class:"fa fa-remove"}).appendTo(deleteButton);
                li.addClass("red-ui-editableList-item-removable");
                deleteButton.on("click", function(evt) {
                    evt.preventDefault();
                    var data = row.data('data');
                    li.addClass("red-ui-editableList-item-deleting")
                    li.fadeOut(300, function() {
                        $(this).remove();
                        if (that.options.removeItem) {
                            that.options.removeItem(data);
                        }
                    });
                });
            }
            var added = false;
            if (this.activeSort) {
                var items = this.items();
                var skip = false;
                items.each(function(i,el) {
                    if (added) { return }
                    var itemData = el.data('data');
                    if (that.activeSort(data,itemData) < 0) {
                         li.insertBefore(el.closest("li"));
                         added = true;
                    }
                });
            }
            if (!added) {
                if (index <= 0) {
                    li.prependTo(this.element);
                } else if (index > that.element.children().length-1) {
                    li.appendTo(this.element);
                } else {
                    li.insertBefore(this.element.children().eq(index));
                }
            }
            if (this.options.addItem) {
                var index = that.element.children().length-1;
                // setTimeout(function() {
                    that.options.addItem(row,index,data);
                    if (that.activeFilter) {
                        try {
                            if (!that.activeFilter(data)) {
                                li.hide();
                            }
                        } catch(err) {
                        }
                    }

                    if (!that.activeSort && that.scrollOnAdd) {
                        setTimeout(function() {
                            that.uiContainer.scrollTop(that.element.height());
                        },0);
                    }
                // },0);
            }
        },
        addItem: function(data) {
            this.insertItemAt(data,this.element.children().length)
        },
        addItems: function(items) {
            for (var i=0; i<items.length;i++) {
                this.addItem(items[i]);
            }
        },
        removeItem: function(data,detach) {
            var items = this.element.children().filter(function(f) {
                return data === $(this).children(".red-ui-editableList-item-content").data('data');
            });
            if (detach) {
                items.detach();
            } else {
                items.remove();
            }
            if (this.options.removeItem) {
                this.options.removeItem(data);
            }
        },
        items: function() {
            return this.element.children().map(function(i) { return $(this).children(".red-ui-editableList-item-content"); });
        },
        empty: function() {
            this.element.empty();
            this.uiContainer.scrollTop(0);
        },
        filter: function(filter) {
            if (filter !== undefined) {
                this.activeFilter = filter;
            }
            return this._refreshFilter();
        },
        sort: function(sort) {
            if (sort !== undefined) {
                this.activeSort = sort;
            }
            return this._refreshSort();
        },
        length: function() {
            return this.element.children().length;
        },
        show: function(item) {
            var items = this.element.children().filter(function(f) {
                return item === $(this).children(".red-ui-editableList-item-content").data('data');
            });
            if (items.length > 0) {
                this.uiContainer.scrollTop(this.uiContainer.scrollTop()+items.position().top)
            }
        },
        getItem: function(li) {
            var el = li.children(".red-ui-editableList-item-content");
            if (el.length) {
                return el.data('data');
            } else {
                return null;
            }
        },
        cancel: function() {
            this.element.sortable("cancel");
        }
    });
})(jQuery);
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
(function($) {

/**
 * options:
 *   - data : array - initial items to display in tree
 *   - multi : boolean - if true, .selected will return an array of results
 *                       otherwise, returns the first selected item
 *   - sortable: boolean/string - TODO: see editableList
 *   - selectable: boolean - default true - whether individual items can be selected
 *   - rootSortable: boolean - if 'sortable' is set, then setting this to
 *                             false, prevents items being sorted to the
 *                             top level of the tree
 *   - autoSelect: boolean - default true - triggers item selection when navigating
 *                           list by keyboard. If the list has checkboxed items
 *                           you probably want to set this to false
 *
 * methods:
 *   - data(items) - clears existing items and replaces with new data
 *   - clearSelection - clears the selected items
 *   - filter(filterFunc) - filters the tree using the provided function
 * events:
 *   - treelistselect : function(event, item) {}
 *   - treelistconfirm : function(event,item) {}
 *   - treelistchangeparent: function(event,item, oldParent, newParent) {}
 *
 * data:
 * [
 *     {
 *         label: 'Local', // label for the item
 *         sublabel: 'Local', // a sub-label for the item
 *         icon: 'fa fa-rocket', // (optional) icon for the item
 *         checkbox: true/false, // (optional) if present, display checkbox accordingly
 *         radio: 'group-name',  // (optional) if present, display radio box - using group-name to set radio group
 *         selected: true/false, // (optional) whether the item is selected or not
 *         children: [] | function(done,item) // (optional) an array of child items, or a function
 *                                       // that will call the `done` callback with an array
 *                                       // of child items
 *         expanded: true/false, // show the child items by default
 *         deferBuild: true/false, // don't build any ui elements for the item's children
 *                                    until it is expanded by the user.
 *         element: // custom dom element to use for the item - ignored if `label` is set
 *         collapsible: true/false, // prevent a parent item from being collapsed. default true.
 *     }
 * ]
 *
 * var treeList = $("<div>").css({width: "100%", height: "100%"}).treeList({data:[...]})
 * treeList.on('treelistselect', function(e,item) { console.log(item)})
 * treeList.treeList('data',[ ... ] )
 *
 *
 * After `data` has been added to the tree, each item is augmented the following
 * properties and functions:
 *
 *   item.parent - set to the parent item
 *   item.depth - the depth in the tree (0 == root)
 *   item.treeList.container
 *   item.treeList.label - the label element for the item
 *   item.treeList.parentList  - the editableList instance this item is in
 *   item.treeList.remove(detach) - removes the item from the tree. Optionally detach to preserve any event handlers on the item's label
 *   item.treeList.makeLeaf(detachChildElements) - turns an element with children into a leaf node,
 *                                                 removing the UI decoration etc.
 *                                                 detachChildElements - any children with custom
 *                                                 elements will be detached rather than removed
 *                                                 so jQuery event handlers are preserved in case
 *                                                 the child elements need to be reattached later
 *   item.treeList.makeParent(children) - turns an element into a parent node, adding the necessary
 *                                        UI decoration.
 *   item.treeList.insertChildAt(newItem,position,select) - adds a child item an the specified position.
 *                                                          Optionally selects the item after adding.
 *   item.treeList.addChild(newItem,select) - appends a child item.
 *                                            Optionally selects the item after adding.
 *   item.treeList.expand(done) - expands the parent item to show children. Optional 'done' callback.
 *   item.treeList.collapse() - collapse the parent item to hide children.
 *   item.treeList.sortChildren(sortFunction) - does a one-time sort of the children using sortFunction
 *   item.treeList.replaceElement(element) - replace the custom element for the item
 *
 *
 */

    $.widget( "nodered.treeList", {
        _create: function() {
            var that = this;
            var autoSelect = true;
            if (that.options.autoSelect === false) {
                autoSelect = false;
            }
            this.element.addClass('red-ui-treeList');
            this.element.attr("tabIndex",0);
            var wrapper = $('<div>',{class:'red-ui-treeList-container'}).appendTo(this.element);
            this.element.on('keydown', function(evt) {
                var focussed = that._topList.find(".focus").parent().data('data');
                if (!focussed && (evt.keyCode === 40 || evt.keyCode === 38)) {
                    if (that._data[0]) {
                        if (autoSelect) {
                            that.select(that._data[0]);
                        } else {
                            that._topList.find(".focus").removeClass("focus")
                        }
                        that._data[0].treeList.label.addClass('focus')
                    }
                    return;
                }
                var target;
                switch(evt.keyCode) {
                    case 32: // SPACE
                    case 13: // ENTER
                        if (!that.options.selectable) { return }
                        if (evt.altKey || evt.ctrlKey || evt.metaKey || evt.shiftKey) {
                            return
                        }
                        evt.preventDefault();
                        evt.stopPropagation();
                        if (focussed.checkbox) {
                            focussed.treeList.checkbox.trigger("click");
                        } else if (focussed.radio) {
                            focussed.treeList.radio.trigger("click");
                        } else if (focussed.children) {
                            if (focussed.treeList.container.hasClass("expanded")) {
                                focussed.treeList.collapse()
                            } else {
                                focussed.treeList.expand()
                            }
                        } else {
                            that._trigger("confirm",null,focussed)
                        }
                    break;
                    case 37: // LEFT
                        evt.preventDefault();
                        evt.stopPropagation();
                        if (focussed.children&& focussed.treeList.container.hasClass("expanded")) {
                            focussed.treeList.collapse()
                        } else if (focussed.parent) {
                            target = focussed.parent;
                        }
                    break;
                    case 38: // UP
                        evt.preventDefault();
                        evt.stopPropagation();
                        target = that._getPreviousSibling(focussed);
                        if (target) {
                            target = that._getLastDescendant(target);
                        }
                        if (!target && focussed.parent) {
                            target = focussed.parent;
                        }
                    break;
                    case 39: // RIGHT
                        evt.preventDefault();
                        evt.stopPropagation();
                        if (focussed.children) {
                            if (!focussed.treeList.container.hasClass("expanded")) {
                                focussed.treeList.expand()
                            }
                        }
                    break
                    case 40: //DOWN
                        evt.preventDefault();
                        evt.stopPropagation();
                        if (focussed.children && Array.isArray(focussed.children) && focussed.children.length > 0 && focussed.treeList.container.hasClass("expanded")) {
                            target = focussed.children[0];
                        } else {
                            target = that._getNextSibling(focussed);
                            while (!target && focussed.parent) {
                                focussed = focussed.parent;
                                target = that._getNextSibling(focussed);
                            }
                        }
                    break
                }
                if (target) {
                    if (autoSelect) {
                        that.select(target);
                    } else {
                        that._topList.find(".focus").removeClass("focus")
                    }
                    target.treeList.label.addClass('focus')
                }
            });
            this._data = [];
            this._items = {};
            this._selected = new Set();
            this._topList = $('<ol class="red-ui-treeList-list">').css({
                position:'absolute',
                top:0,
                left:0,
                right:0,
                bottom:0
            }).appendTo(wrapper);

            var topListOptions = {
                addButton: false,
                scrollOnAdd: false,
                height: '100%',
                addItem: function(container,i,item) {
                    that._addSubtree(that._topList,container,item,0);
                }
            };
            if (this.options.header) {
                topListOptions.header = this.options.header;
            }
            if (this.options.rootSortable !== false && !!this.options.sortable) {
                topListOptions.sortable = this.options.sortable;
                topListOptions.connectWith = '.red-ui-treeList-sortable';
                this._topList.addClass('red-ui-treeList-sortable');
            }
            this._topList.editableList(topListOptions)


            if (this.options.data) {
                this.data(this.options.data);
            }
        },
        _getLastDescendant: function(item) {
            // Gets the last visible descendant of the item
            if (!item.children || !item.treeList.container.hasClass("expanded") || item.children.length === 0) {
                return item;
            }
            return this._getLastDescendant(item.children[item.children.length-1]);
        },
        _getPreviousSibling: function(item) {
            var candidates;
            if (!item.parent) {
                candidates = this._data;
            } else {
                candidates = item.parent.children;
            }
            var index = candidates.indexOf(item);
            if (index === 0) {
                return null;
            } else {
                return candidates[index-1];
            }
        },
        _getNextSibling: function(item) {
            var candidates;
            if (!item.parent) {
                candidates = this._data;
            } else {
                candidates = item.parent.children;
            }
            var index = candidates.indexOf(item);
            if (index === candidates.length - 1) {
                return null;
            } else {
                return candidates[index+1];
            }
        },
        _addChildren: function(container,parent,children,depth,onCompleteChildren) {
            var that = this;
            var subtree = $('<ol class="red-ui-treeList-list">').appendTo(container).editableList({
                connectWith: ".red-ui-treeList-sortable",
                sortable: that.options.sortable,
                addButton: false,
                scrollOnAdd: false,
                height: 'auto',
                addItem: function(container,i,item) {
                    that._addSubtree(subtree,container,item,depth+1);
                },
                sortItems: function(data) {
                    var children = [];
                    var reparented = [];
                    data.each(function() {
                        var child = $(this).data('data');
                        children.push(child);
                        var evt = that._fixDepths(parent,child);
                        if (evt) {
                            reparented.push(evt);
                        }
                    })
                    if (Array.isArray(parent.children)) {
                        parent.children = children;
                    }
                    reparented.forEach(function(evt) {
                        that._trigger("changeparent",null,evt);
                    });
                    that._trigger("sort",null,parent);
                },
                filter: parent.treeList.childFilter
            });
            if (!!that.options.sortable) {
                subtree.addClass('red-ui-treeList-sortable');
            }
            var sliceSize = 30;
            var index = 0;
            var addSlice = function() {
                var start = index;
                for (var i=0;i<sliceSize;i++) {
                    index = start+i;
                    if (index === children.length) {
                        setTimeout(function() {
                            if (onCompleteChildren) {
                                onCompleteChildren();
                            }
                        },10);
                        return;
                    }
                    children[index].parent = parent;
                    subtree.editableList('addItem',children[index])
                }
                index++;
                if (index < children.length) {
                    setTimeout(function() {
                        addSlice();
                    },10);
                }
            }
            addSlice();
            subtree.hide()
            return subtree;
        },
        _fixDepths: function(parent,child) {
            // If child has just been moved into parent in the UI
            // this will fix up the internal data structures to match.
            // The calling function must take care of getting child
            // into the parent.children array. The rest is up to us.
            var that = this;
            var reparentedEvent = null;
            if (child.parent !== parent) {
                reparented = true;
                var oldParent = child.parent;
                child.parent = parent;
                reparentedEvent = {
                    item: child,
                    old: oldParent,
                }
            }
            if (child.depth !== parent.depth+1) {
                child.depth = parent.depth+1;
                // var labelPaddingWidth = ((child.gutter ? child.gutter[0].offsetWidth + 2 : 0) + (child.depth * 20));
                var labelPaddingWidth = (((child.gutter&&!child.gutter.hasClass("red-ui-treeList-gutter-float"))?child.gutter.width()+2:0)+(child.depth*20));
                child.treeList.labelPadding.width(labelPaddingWidth+'px');
                if (child.element) {
                    $(child.element).css({
                        width: "calc(100% - "+(labelPaddingWidth+20+(child.icon?20:0))+"px)"
                    })
                }
                // This corrects all child item depths
                if (child.children && Array.isArray(child.children)) {
                    child.children.forEach(function(item) {
                        that._fixDepths(child,item);
                    })
                }
            }
            return reparentedEvent;
        },
        _initItem: function(item,depth) {
            if (item.treeList) {
                return;
            }
            var that = this;
            this._items[item.id] = item;
            item.treeList = {};
            item.depth = depth;
            item.treeList.remove = function(detach) {
                if (item.treeList.parentList) {
                    item.treeList.parentList.editableList('removeItem',item,detach);
                }
                if (item.parent) {
                    var index = item.parent.children.indexOf(item);
                    item.parent.children.splice(index,1)
                    that._trigger("sort",null,item.parent);
                }
                that._selected.delete(item);
                delete item.treeList;
                delete that._items[item.id];
                if(item.depth === 0) {
                    for(var key in that._items) {
                        if (that._items.hasOwnProperty(key)) {
                            var child = that._items[key];
                            if(child.parent && child.parent.id === item.id) {
                                delete that._items[key].treeList;
                                delete that._items[key];
                            }
                        }
                    }
                    that._data = that._data.filter(function(data) { return data.id !== item.id})
                }
            }
            item.treeList.insertChildAt = function(newItem,position,select) {
                newItem.parent = item;
                item.children.splice(position,0,newItem);
                var processChildren = function(parent,i) {
                    that._initItem(i,parent.depth+1)
                    i.parent = parent;
                    if (i.children && typeof i.children !== 'function') {
                        i.children.forEach(function(item) {
                            processChildren(i, item, parent.depth+2)
                        });
                    }
                }
                processChildren(item,newItem);

                if (!item.deferBuild && item.treeList.childList) {
                    item.treeList.childList.editableList('insertItemAt',newItem,position)
                    if (select) {
                        setTimeout(function() {
                            that.select(newItem)
                        },100);
                    }
                    that._trigger("sort",null,item);

                    if (that.activeFilter) {
                        that.filter(that.activeFilter);
                    }
                }
            }
            item.treeList.addChild = function(newItem,select) {
                item.treeList.insertChildAt(newItem,item.children.length,select);
            }
            item.treeList.expand = function(done) {
                if (!item.children) {
                    if (done) { done(false) }
                    return;
                }
                if (!item.treeList.container) {
                    item.expanded = true;
                    if (done) { done(false) }
                    return;
                }
                var container = item.treeList.container;
                if (container.hasClass("expanded")) {
                    if (done) { done(false) }
                    return;
                }

                if (!container.hasClass("built") && (item.deferBuild || typeof item.children === 'function')) {
                    container.addClass('built');
                    var childrenAdded = false;
                    var spinner;
                    var startTime = 0;
                    var started = Date.now();
                    var completeBuild = function(children) {
                        childrenAdded = true;
                        item.treeList.childList = that._addChildren(container,item,children,depth, function() {
                            if (done) { done(true) }
                            that._trigger("childrenloaded",null,item)
                        });
                        var delta = Date.now() - startTime;
                        if (delta < 400) {
                            setTimeout(function() {
                                item.treeList.childList.slideDown('fast');
                                if (spinner) {
                                    spinner.remove();
                                }
                            },400-delta);
                        } else {
                            item.treeList.childList.slideDown('fast');
                            if (spinner) {
                                spinner.remove();
                            }
                        }
                        item.expanded = true;
                    }
                    if (typeof item.children === 'function') {
                        item.children(completeBuild,item);
                    } else {
                        delete item.deferBuild;
                        completeBuild(item.children);
                    }
                    if (!childrenAdded) {
                        startTime = Date.now();
                        spinner = $('<div class="red-ui-treeList-spinner">').css({
                            "background-position": (35+depth*20)+'px 50%'
                        }).appendTo(container);
                    }

                } else {
                    if (that._loadingData || item.children.length > 20) {
                        item.treeList.childList.show();
                    } else {
                        item.treeList.childList.slideDown('fast');
                    }
                    item.expanded = true;
                    if (done) { done(!that._loadingData) }
                }
                container.addClass("expanded");
            }
            item.treeList.collapse = function() {
                if (item.collapsible === false) {
                    return
                }
                if (!item.children) {
                    return;
                }
                item.expanded = false;
                if (item.treeList.container) {
                    if (item.children.length < 20) {
                        item.treeList.childList.slideUp('fast');
                    } else {
                        item.treeList.childList.hide();
                    }
                    item.treeList.container.removeClass("expanded");
                }
            }
            item.treeList.sortChildren = function(sortFunc) {
                if (!item.children) {
                    return;
                }
                item.children.sort(sortFunc);
                if (item.treeList.childList) {
                    // Do a one-off sort of the list, which means calling sort twice:
                    // 1. first with the desired sort function
                    item.treeList.childList.editableList('sort',sortFunc);
                    // 2. and then with null to remove it
                    item.treeList.childList.editableList('sort',null);
                }
            }
            item.treeList.replaceElement = function (element) {
                if (item.element) {
                    if (item.treeList.container) {
                        $(item.element).remove();
                        $(element).appendTo(item.treeList.label);
                        // using the JQuery Object, the gutter width will
                        // be wrong when the element is reattached the second time
                        var labelPaddingWidth = (item.gutter ? item.gutter[0].offsetWidth + 2 : 0) + (item.depth * 20);

                        $(element).css({
                            width: "calc(100% - "+(labelPaddingWidth+20+(item.icon?20:0))+"px)"
                        })
                    }
                    item.element = element;
                }
            }

            if (item.children && typeof item.children !== "function") {
                item.children.forEach(function(i) {
                    that._initItem(i,depth+1);
                })
            }
        },
        _addSubtree: function(parentList, container, item, depth) {
            var that = this;
            this._initItem(item,depth);
            // item.treeList = {};
            // item.treeList.depth = depth;
            item.treeList.container = container;

            item.treeList.parentList = parentList;

            var label = $("<div>",{class:"red-ui-treeList-label"});
            label.appendTo(container);
            item.treeList.label = label;
            if (item.class) {
                label.addClass(item.class);
            }
            if (item.gutter) {
                item.gutter.css({
                    position: 'absolute'
                }).appendTo(label)

            }

            var labelPaddingWidth = ((item.gutter&&!item.gutter.hasClass("red-ui-treeList-gutter-float"))?item.gutter.width()+2:0)+(depth*20);

            item.treeList.labelPadding = $('<span>').css({
                display: "inline-block",
                "flex-shrink": 0,
                width:  labelPaddingWidth+'px'
            }).appendTo(label);

            label.on('mouseover',function(e) { that._trigger('itemmouseover',e,item); })
            label.on('mouseout',function(e) { that._trigger('itemmouseout',e,item); })
            label.on('mouseenter',function(e) { that._trigger('itemmouseenter',e,item); })
            label.on('mouseleave',function(e) { that._trigger('itemmouseleave',e,item); })

            item.treeList.makeLeaf = function(detachChildElements) {
                if (!treeListIcon.children().length) {
                    // Already a leaf
                    return
                }
                if (detachChildElements && item.children) {
                    var detachChildren = function(item) {
                        if (item.children) {
                            item.children.forEach(function(child) {
                                if (child.element) {
                                    child.element.detach();
                                }
                                if (child.gutter) {
                                    child.gutter.detach();
                                }
                                detachChildren(child);
                            });
                        }
                    }
                    detachChildren(item);
                }
                treeListIcon.empty();
                if (!item.deferBuild) {
                    item.treeList.childList.remove();
                    delete item.treeList.childList;
                }
                label.off("click.red-ui-treeList-expand");
                treeListIcon.off("click.red-ui-treeList-expand");
                delete item.children;
                container.removeClass("expanded");
                delete item.expanded;
            }
            item.treeList.makeParent = function(children) {
                if (treeListIcon.children().length) {
                    // Already a parent because we've got the angle-right icon
                    return;
                }
                $('<i class="fa fa-angle-right" />').toggleClass("hide",item.collapsible === false).appendTo(treeListIcon);
                treeListIcon.on("click.red-ui-treeList-expand", function(e) {
                        e.stopPropagation();
                        e.preventDefault();
                        if (container.hasClass("expanded")) {
                            item.treeList.collapse();
                        } else {
                            item.treeList.expand();
                        }
                    });
                // $('<span class="red-ui-treeList-icon"><i class="fa fa-folder-o" /></span>').appendTo(label);
                label.on("click.red-ui-treeList-expand", function(e) {
                    if (container.hasClass("expanded")) {
                        if (item.hasOwnProperty('selected') || label.hasClass("selected")) {
                            item.treeList.collapse();
                        }
                    } else {
                        item.treeList.expand();
                    }
                })
                if (!item.children) {
                    item.children = children||[];
                    item.treeList.childList = that._addChildren(container,item,item.children,depth);
                }
            }

            var treeListIcon = $('<span class="red-ui-treeList-icon"></span>').appendTo(label);
            if (item.children) {
                item.treeList.makeParent();
            }

            if (item.checkbox) {
                var selectWrapper = $('<span class="red-ui-treeList-icon"></span>');
                var cb = $('<input class="red-ui-treeList-checkbox" type="checkbox">').prop('checked',item.selected).appendTo(selectWrapper);
                cb.on('click', function(e) {
                    e.stopPropagation();
                });
                cb.on('change', function(e) {
                    item.selected = this.checked;
                    if (item.selected) {
                        that._selected.add(item);
                    } else {
                        that._selected.delete(item);
                    }
                    label.toggleClass("selected",this.checked);
                    that._trigger("select",e,item);
                })
                if (!item.children) {
                    label.on("click", function(e) {
                        e.stopPropagation();
                        cb.trigger("click");
                        that._topList.find(".focus").removeClass("focus")
                        label.addClass('focus')
                    })
                }
                item.treeList.select = function(v) {
                    if (v !== item.selected) {
                        cb.trigger("click");
                    }
                }
                item.treeList.checkbox = cb;
                selectWrapper.appendTo(label)
            } else if (item.radio) {
                var selectWrapper = $('<span class="red-ui-treeList-icon"></span>');
                var cb = $('<input class="red-ui-treeList-radio" type="radio">').prop('name', item.radio).prop('checked',item.selected).appendTo(selectWrapper);
                cb.on('click', function(e) {
                    e.stopPropagation();
                });
                cb.on('change', function(e) {
                    item.selected = this.checked;
                    that._selected.forEach(function(selectedItem) {
                        if (selectedItem.radio === item.radio) {
                            selectedItem.treeList.label.removeClass("selected");
                            selectedItem.selected = false;
                            that._selected.delete(selectedItem);
                        }
                    })
                    if (item.selected) {
                        that._selected.add(item);
                    } else {
                        that._selected.delete(item);
                    }
                    label.toggleClass("selected",this.checked);
                    that._trigger("select",e,item);
                })
                if (!item.children) {
                    label.on("click", function(e) {
                        e.stopPropagation();
                        cb.trigger("click");
                        that._topList.find(".focus").removeClass("focus")
                        label.addClass('focus')
                    })
                }
                item.treeList.select = function(v) {
                    if (v !== item.selected) {
                        cb.trigger("click");
                    }
                }
                selectWrapper.appendTo(label)
                item.treeList.radio = cb;
            } else {
                label.on("click", function(e) {
                    if (!that.options.multi) {
                        that.clearSelection();
                    }
                    label.addClass("selected");
                    that._selected.add(item);
                    that._topList.find(".focus").removeClass("focus")
                    label.addClass('focus')

                    that._trigger("select",e,item)
                })
                label.on("dblclick", function(e) {
                    that._topList.find(".focus").removeClass("focus")
                    label.addClass('focus')
                    if (!item.children) {
                        that._trigger("confirm",e,item);
                    }
                })
                item.treeList.select = function(v) {
                    if (!that.options.multi) {
                        that.clearSelection();
                    }
                    label.toggleClass("selected",v);
                    if (v) {
                        that._selected.add(item);
                        that._trigger("select",null,item)
                    } else {
                        that._selected.delete(item);
                    }
                    that.reveal(item);
                }
            }
            label.toggleClass("selected",!!item.selected);
            if (item.selected) {
                that._selected.add(item);
            }
            if (item.icon) {
                if (typeof item.icon === "string") {
                    $('<span class="red-ui-treeList-icon"><i class="'+item.icon+'" /></span>').appendTo(label);
                } else {
                    $('<span class="red-ui-treeList-icon">').appendTo(label).append(item.icon);
                }
            }
            if (item.hasOwnProperty('label') || item.hasOwnProperty('sublabel')) {
                if (item.hasOwnProperty('label')) {
                    $('<span class="red-ui-treeList-label-text"></span>').text(item.label).appendTo(label);
                }
                if (item.hasOwnProperty('sublabel')) {
                    $('<span class="red-ui-treeList-sublabel-text"></span>').text(item.sublabel).appendTo(label);
                }

            } else if (item.element) {
                $(item.element).appendTo(label);
                $(item.element).css({
                    width: "calc(100% - "+(labelPaddingWidth+20+(item.icon?20:0))+"px)"
                })
            }
            if (item.children) {
                if (Array.isArray(item.children) && !item.deferBuild) {
                    item.treeList.childList = that._addChildren(container,item,item.children,depth);
                }
                if (item.expanded) {
                    item.treeList.expand();
                }
            }
            // label.appendTo(container);
        },
        empty: function() {
            this._topList.editableList('empty');
        },
        data: function(items) {
            var that = this;
            if (items !== undefined) {
                this._data = items;
                this._items = {};
                this._topList.editableList('empty');
                this._loadingData = true;
                for (var i=0; i<items.length;i++) {
                    this._topList.editableList('addItem',items[i]);
                }
                setTimeout(function() {
                    delete that._loadingData;
                },200);
                this._trigger("select")

            } else {
                return this._data;
            }
        },
        show: function(item, done) {
            if (typeof item === "string") {
                item = this._items[item]
            }
            if (!item) {
                return;
            }
            var that = this;
            var stack = [];
            var i = item;
            while(i) {
                stack.unshift(i);
                i = i.parent;
            }
            var isOpening = false;
            var handleStack = function(opening) {
                isOpening = isOpening ||opening
                var item = stack.shift();
                if (stack.length === 0) {
                    setTimeout(function() {
                        that.reveal(item);
                        if (done) { done(); }
                    },isOpening?200:0);
                } else {
                    item.treeList.expand(handleStack)
                }
            }
            handleStack();
        },
        reveal: function(item) {
            if (typeof item === "string") {
                item = this._items[item]
            }
            if (!item) {
                return;
            }
            var listOffset = this._topList.offset().top;
            var itemOffset = item.treeList.label.offset().top;
            var scrollTop = this._topList.parent().scrollTop();
            itemOffset -= listOffset+scrollTop;
            var treeHeight = this._topList.parent().height();
            var itemHeight = item.treeList.label.outerHeight();
            if (itemOffset < itemHeight/2) {
                this._topList.parent().scrollTop(scrollTop+itemOffset-itemHeight/2-itemHeight)
            } else if (itemOffset+itemHeight > treeHeight) {
                this._topList.parent().scrollTop(scrollTop+((itemOffset+2.5*itemHeight)-treeHeight));
            }
        },
        select: function(item, triggerEvent, deselectExisting) {
            var that = this;
            if (!this.options.multi && deselectExisting !== false) {
                this.clearSelection();
            }
            if (Array.isArray(item)) {
                item.forEach(function(i) {
                    that.select(i,triggerEvent,false);
                })
                return;
            }
            if (typeof item === "string") {
                item = this._items[item]
            }
            if (!item) {
                return;
            }
            // this.show(item.id);
            item.selected = true;
            this._selected.add(item);

            if (item.treeList.label) {
                item.treeList.label.addClass("selected");
            }

            that._topList.find(".focus").removeClass("focus");

            if (triggerEvent !== false) {
                this._trigger("select",null,item)
            }
        },
        clearSelection: function() {
            this._selected.forEach(function(item) {
                item.selected = false;
                if (item.treeList.checkbox) {
                    item.treeList.checkbox.prop('checked',false)
                }
                if (item.treeList.label) {
                    item.treeList.label.removeClass("selected")
                }
            });
            this._selected.clear();
        },
        selected: function() {
            var selected = [];
            this._selected.forEach(function(item) {
                selected.push(item);
            })
            if (this.options.multi) {
                return selected;
            }
            if (selected.length) {
                return selected[0]
            } else {
                // TODO: This may be a bug.. it causes the call to return itself
                // not undefined.
                return undefined;
            }
        },
        filter: function(filterFunc) {
            this.activeFilter = filterFunc;
            var totalCount = 0;
            var filter = function(item) {
                var matchCount = 0;
                if (filterFunc && filterFunc(item)) {
                    matchCount++;
                    totalCount++;
                }
                var childCount = 0;
                if (item.children && typeof item.children !== "function") {
                    if (item.treeList.childList) {
                        childCount = item.treeList.childList.editableList('filter', filter);
                    } else {
                        item.treeList.childFilter = filter;
                        if (filterFunc) {
                            item.children.forEach(function(i) {
                                if (filter(i)) {
                                    childCount++;
                                }
                            })

                        }
                    }
                    matchCount += childCount;
                    if (filterFunc && childCount > 0) {
                        setTimeout(function() {
                            item.treeList.expand();
                        },10);
                    }
                }
                if (!filterFunc) {
                    totalCount++;
                    return true
                }
                return matchCount > 0
            }
            this._topList.editableList('filter', filter);
            return totalCount;
        },
        get: function(id) {
            return this._items[id] || null;
        }
    });

})(jQuery);
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
(function($) {
    $.widget( "nodered.checkboxSet", {
        _create: function() {
            var that = this;
            this.uiElement = this.element.wrap( "<span>" ).parent();
            this.uiElement.addClass("red-ui-checkboxSet");
            if (this.options.parent) {
                this.parent = this.options.parent;
                this.parent.checkboxSet('addChild',this.element);
            }
            this.children = [];
            this.partialFlag = false;
            this.stateValue = 0;
            var initialState = this.element.prop('checked');
            this.states = [
                $('<span class="red-ui-checkboxSet-option hide"><i class="fa fa-square-o"></i></span>').appendTo(this.uiElement),
                $('<span class="red-ui-checkboxSet-option hide"><i class="fa fa-check-square-o"></i></span>').appendTo(this.uiElement),
                $('<span class="red-ui-checkboxSet-option hide"><i class="fa fa-minus-square-o"></i></span>').appendTo(this.uiElement)
            ];
            if (initialState) {
                this.states[1].show();
            } else {
                this.states[0].show();
            }

            this.element.on("change", function() {
                if (this.checked) {
                    that.states[0].hide();
                    that.states[1].show();
                    that.states[2].hide();
                } else {
                    that.states[1].hide();
                    that.states[0].show();
                    that.states[2].hide();
                }
                var isChecked = this.checked;
                that.children.forEach(function(child) {
                    child.checkboxSet('state',isChecked,false,true);
                })
            })
            this.uiElement.on("click", function(e) {
                e.stopPropagation();
                // state returns null for a partial state. Clicking on that should
                // result in false.
                that.state((that.state()===false)?true:false);
            })
            if (this.parent) {
                this.parent.checkboxSet('updateChild',this);
            }
        },
        _destroy: function() {
            if (this.parent) {
                this.parent.checkboxSet('removeChild',this.element);
            }
        },
        addChild: function(child) {
            var that = this;
            this.children.push(child);
        },
        removeChild: function(child) {
            var index = this.children.indexOf(child);
            if (index > -1) {
                this.children.splice(index,1);
            }
        },
        updateChild: function(child) {
            var checkedCount = 0;
            this.children.forEach(function(c,i) {
                if (c.checkboxSet('state') === true) {
                    checkedCount++;
                }
            });
            if (checkedCount === 0) {

                this.state(false,true);
            } else if (checkedCount === this.children.length) {
                this.state(true,true);
            } else {
                this.state(null,true);
            }
        },
        disable: function() {
            this.uiElement.addClass('disabled');
        },
        state: function(state,suppressEvent,suppressParentUpdate) {

            if (arguments.length === 0) {
                return this.partialFlag?null:this.element.is(":checked");
            } else {
                this.partialFlag = (state === null);
                var trueState = this.partialFlag||state;
                this.element.prop('checked',trueState);
                if (state === true) {
                    this.states[0].hide();
                    this.states[1].show();
                    this.states[2].hide();
                } else if (state === false) {
                    this.states[2].hide();
                    this.states[1].hide();
                    this.states[0].show();
                } else if (state === null) {
                    this.states[0].hide();
                    this.states[1].hide();
                    this.states[2].show();
                }
                if (!suppressEvent) {
                    this.element.trigger('change',null);
                }
                if (!suppressParentUpdate && this.parent) {
                    this.parent.checkboxSet('updateChild',this);
                }
            }
        }
    })

})(jQuery);
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
RED.menu = (function() {

    var menuItems = {};
    let menuItemCount = 0

    function createMenuItem(opt) {
        var item;

        if (opt !== null && opt.id) {
            var themeSetting = RED.settings.theme("menu."+opt.id);
            if (themeSetting === false) {
                return null;
            }
        }

        function setInitialState() {
            var savedStateActive = RED.settings.get("menu-" + opt.id);
            if (opt.setting) {
                // May need to migrate pre-0.17 setting

                if (savedStateActive !== null) {
                    RED.settings.set(opt.setting,savedStateActive);
                    RED.settings.remove("menu-" + opt.id);
                } else {
                    savedStateActive = RED.settings.get(opt.setting);
                }
            }
            if (savedStateActive) {
                link.addClass("active");
                triggerAction(opt.id,true);
            } else if (savedStateActive === false) {
                link.removeClass("active");
                triggerAction(opt.id,false);
            } else if (opt.hasOwnProperty("selected")) {
                if (opt.selected) {
                    link.addClass("active");
                } else {
                    link.removeClass("active");
                }
                triggerAction(opt.id,opt.selected);
            }
        }

        if (opt === null) {
            item = $('<li class="red-ui-menu-divider"></li>');
        } else {
            item = $('<li></li>');
            if (!opt.id) {
                opt.id = 'red-ui-menu-item-'+(menuItemCount++)
            }
            if (opt.group) {
                item.addClass("red-ui-menu-group-"+opt.group);
            }
            var linkContent = '<a '+(opt.id?'id="'+opt.id+'" ':'')+'tabindex="-1" href="#">';
            if (opt.toggle) {
                linkContent += '<i class="fa fa-square'+(opt.direction!=='right'?" pull-left":"")+'"></i>';
                linkContent += '<i class="fa fa-check-square'+(opt.direction!=='right'?" pull-left":"")+'"></i>';

            }
            if (opt.icon !== undefined) {
                if (/\.(png|svg)/.test(opt.icon)) {
                    linkContent += '<img src="'+opt.icon+'"/> ';
                } else {
                    linkContent += '<i class="'+(opt.icon?opt.icon:'" style="display: inline-block;"')+'"></i> ';
                }
            }
            let label = opt.label
            if (!opt.label && typeof opt.onselect === 'string') {
                label = RED.actions.getLabel(opt.onselect)
            }
            if (opt.sublabel) {
                linkContent += '<span class="red-ui-menu-label-container"><span class="red-ui-menu-label">'+label+'</span>'+
                               '<span class="red-ui-menu-sublabel">'+opt.sublabel+'</span></span>'
            } else {
                linkContent += '<span class="red-ui-menu-label"><span>'+label+'</span></span>'
            }

            linkContent += '</a>';

            var link = $(linkContent).appendTo(item);
            opt.link = link;
            if (typeof opt.onselect === 'string' || opt.shortcut) {
                var shortcut = opt.shortcut || RED.keyboard.getShortcut(opt.onselect);
                if (shortcut && shortcut.key) {
                    opt.shortcutSpan = $('<span class="red-ui-popover-key">'+RED.keyboard.formatKey(shortcut.key, true)+'</span>').appendTo(link.find(".red-ui-menu-label"));
                }
            }

            menuItems[opt.id] = opt;

            if (opt.onselect) {
                link.on("click", function(e) {
                    e.preventDefault();
                    if ($(this).parent().hasClass("disabled")) {
                        return;
                    }
                    if (opt.toggle) {
                        if (opt.toggle === true) {
                            setSelected(opt.id, !isSelected(opt.id));
                        } else {
                            setSelected(opt.id, true);
                        }
                    } else {
                        triggerAction(opt.id);
                    }
                });
                if (opt.toggle) {
                    setInitialState();
                }
            } else if (opt.href) {
                link.attr("target","_blank").attr("href",opt.href);
            } else if (!opt.options) {
                item.addClass("disabled");
                link.on("click", function(event) {
                    event.preventDefault();
                });
            }
            if (opt.options) {
                item.addClass("red-ui-menu-dropdown-submenu"+(opt.direction!=='right'?" pull-left":""));
                var submenu = $('<ul id="'+opt.id+'-submenu" class="red-ui-menu-dropdown"></ul>').appendTo(item);
                var hasIcons = false
                var hasSubmenus = false

                for (var i=0;i<opt.options.length;i++) {

                    if (opt.options[i]) {
                        if (opt.onpreselect && opt.options[i].onpreselect === undefined) {
                            opt.options[i].onpreselect = opt.onpreselect
                        }
                        if (opt.onpostselect && opt.options[i].onpostselect === undefined) {
                            opt.options[i].onpostselect = opt.onpostselect
                        }
                        opt.options[i].direction = opt.direction
                        hasIcons = hasIcons || (opt.options[i].icon);
                        hasSubmenus = hasSubmenus || (opt.options[i].options);
                    }

                    var li = createMenuItem(opt.options[i]);
                    if (li) {
                        li.appendTo(submenu);
                    }
                }
                if (!hasIcons) {
                    submenu.addClass("red-ui-menu-dropdown-noicons")
                }
                if (hasSubmenus) {
                    submenu.addClass("red-ui-menu-dropdown-submenus")
                }


            }
            if (opt.disabled) {
                item.addClass("disabled");
            }
            if (opt.visible === false) {
                item.addClass("hide");
            }
        }


        return item;

    }
    function createMenu(options) {
        var topMenu = $("<ul/>",{class:"red-ui-menu red-ui-menu-dropdown pull-right"});
        if (options.direction) {
            topMenu.addClass("red-ui-menu-dropdown-direction-"+options.direction)
        }
        if (options.id) {
            topMenu.attr({id:options.id+"-submenu"});
            var menuParent = $("#"+options.id);
            if (menuParent.length === 1) {
                topMenu.insertAfter(menuParent);
                menuParent.on("click", function(evt) {
                    evt.stopPropagation();
                    evt.preventDefault();
                    if (topMenu.is(":visible")) {
                        $(document).off("click.red-ui-menu");
                        topMenu.hide();
                    } else {
                        $(document).on("click.red-ui-menu", function(evt) {
                            $(document).off("click.red-ui-menu");
                            activeMenu = null;
                            topMenu.hide();
                        });
                        $(".red-ui-menu.red-ui-menu-dropdown").hide();
                        topMenu.show();
                    }
                })
            }
        }

        var lastAddedSeparator = false;
        var hasSubmenus = false;
        var hasIcons = false;
        for (var i=0;i<options.options.length;i++) {
            var opt = options.options[i];
            if (opt) {
                if (options.onpreselect && opt.onpreselect === undefined) {
                    opt.onpreselect = options.onpreselect
                }
                if (options.onpostselect && opt.onpostselect === undefined) {
                    opt.onpostselect = options.onpostselect
                }
                opt.direction = options.direction || 'left'
            }
            if (opt !== null || !lastAddedSeparator) {
                hasIcons = hasIcons || (opt && opt.icon);
                hasSubmenus = hasSubmenus || (opt && opt.options);
                var li = createMenuItem(opt);
                if (li) {
                    li.appendTo(topMenu);
                    lastAddedSeparator = (opt === null);
                }
            }
        }
        if (!hasIcons) {
            topMenu.addClass("red-ui-menu-dropdown-noicons")
        }
        if (hasSubmenus) {
            topMenu.addClass("red-ui-menu-dropdown-submenus")
        }
        return topMenu;
    }

    function triggerAction(id, args) {
        var opt = menuItems[id];
        var callback = opt.onselect;
        if (opt.onpreselect) {
            opt.onpreselect.call(opt,args)
        }
        if (typeof opt.onselect === 'string') {
            callback = RED.actions.get(opt.onselect);
        }
        if (callback) {
            callback.call(opt,args);
        } else {
            console.log("No callback for",id,opt.onselect);
        }
        if (opt.onpostselect) {
            opt.onpostselect.call(opt,args)
        }
    }

    function isSelected(id) {
        return $("#" + id).hasClass("active");
    }

    function setSelected(id,state) {
        var alreadySet = false;
        if (isSelected(id) == state) {
            alreadySet = true;
        }
        var opt = menuItems[id];
        if (state) {
            $("#"+id).addClass("active");
        } else {
            $("#"+id).removeClass("active");
        }
        if (opt) {
            if (opt.toggle && typeof opt.toggle === "string") {
                if (state) {
                    for (var m in menuItems) {
                        if (menuItems.hasOwnProperty(m)) {
                            var mi = menuItems[m];
                            if (mi.id != opt.id && opt.toggle == mi.toggle) {
                                setSelected(mi.id,false);
                            }
                        }
                    }
                }
            }
            if (!alreadySet && opt.onselect) {
                triggerAction(opt.id,state);
            }
            if (!opt.local && !alreadySet) {
                RED.settings.set(opt.setting||("menu-"+opt.id), state);
            }
        }
    }

    function toggleSelected(id) {
        setSelected(id,!isSelected(id));
    }

    function setDisabled(id,state) {
        if (state) {
            $("#"+id).parent().addClass("disabled");
        } else {
            $("#"+id).parent().removeClass("disabled");
        }
    }

    function setVisible(id,state) {
        if (!state) {
            $("#"+id).parent().addClass("hide");
        } else {
            $("#"+id).parent().removeClass("hide");
        }
    }

    function addItem(id,opt) {
        var item = createMenuItem(opt);
        if (opt !== null && opt.group) {
            var groupItems = $("#"+id+"-submenu").children(".red-ui-menu-group-"+opt.group);
            if (groupItems.length === 0) {
                item.appendTo("#"+id+"-submenu");
            } else {
                for (var i=0;i<groupItems.length;i++) {
                    var groupItem = groupItems[i];
                    var label = $(groupItem).find(".red-ui-menu-label").html();
                    if (opt.label < label) {
                        $(groupItem).before(item);
                        break;
                    }
                }
                if (i === groupItems.length) {
                    item.appendTo("#"+id+"-submenu");
                }
            }
        } else {
            item.appendTo("#"+id+"-submenu");
        }
    }
    function removeItem(id) {
        $("#"+id).parent().remove();
    }

    function setAction(id,action) {
        var opt = menuItems[id];
        if (opt) {
            opt.onselect = action;
        }
    }

    function refreshShortcuts() {
        for (var id in menuItems) {
            if (menuItems.hasOwnProperty(id)) {
                var opt = menuItems[id];
                if (typeof opt.onselect === "string" && opt.shortcutSpan) {
                    opt.shortcutSpan.remove();
                    delete opt.shortcutSpan;
                    var shortcut = RED.keyboard.getShortcut(opt.onselect);
                    if (shortcut && shortcut.key) {
                        opt.shortcutSpan = $('<span class="red-ui-popover-key">'+RED.keyboard.formatKey(shortcut.key, true)+'</span>').appendTo(opt.link.find(".red-ui-menu-label"));
                    }
                }
            }
        }
    }

    return {
        init: createMenu,
        setSelected: setSelected,
        isSelected: isSelected,
        toggleSelected: toggleSelected,
        setDisabled: setDisabled,
        setVisible: setVisible,
        addItem: addItem,
        removeItem: removeItem,
        setAction: setAction,
        refreshShortcuts: refreshShortcuts
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/


RED.panels = (function() {

    function createPanel(options) {
        var container = options.container || $("#"+options.id);
        var children = container.children();
        if (children.length !== 2) {
            console.log(options.id);
            throw new Error("Container must have exactly two children");
        }
        var vertical = (!options.dir || options.dir === "vertical");
        container.addClass("red-ui-panels");
        if (!vertical) {
            container.addClass("red-ui-panels-horizontal");
        }

        $(children[0]).addClass("red-ui-panel");
        $(children[1]).addClass("red-ui-panel");

        var separator = $('<div class="red-ui-panels-separator"></div>').insertAfter(children[0]);
        var startPosition;
        var panelSizes = [];
        var modifiedSizes = false;
        var panelRatio = 0.5;
        separator.draggable({
                axis: vertical?"y":"x",
                containment: container,
                scroll: false,
                start:function(event,ui) {
                    startPosition = vertical?ui.position.top:ui.position.left;

                    panelSizes = [
                        vertical?$(children[0]).height():$(children[0]).width(),
                        vertical?$(children[1]).height():$(children[1]).width()
                    ];
                },
                drag: function(event,ui) {
                    var size = vertical?container.height():container.width();
                    var delta = (vertical?ui.position.top:ui.position.left)-startPosition;
                    var newSizes = [panelSizes[0]+delta,panelSizes[1]-delta];
                    if (vertical) {
                        $(children[0]).height(newSizes[0]);
                        // $(children[1]).height(newSizes[1]);
                        ui.position.top -= delta;
                    } else {
                        $(children[0]).width(newSizes[0]);
                        // $(children[1]).width(newSizes[1]);
                        ui.position.left -= delta;
                    }
                    if (options.resize) {
                        options.resize(newSizes[0],newSizes[1]);
                    }
                    panelRatio = newSizes[0]/(size-8);
                },
                stop:function(event,ui) {
                    modifiedSizes = true;
                }
        });

        var panel = {
            ratio: function(ratio) {
                if (ratio === undefined) {
                    return panelRatio;
                }
                panelRatio = ratio;
                modifiedSizes = true;
                if (ratio === 0 || ratio === 1) {
                    separator.hide();
                } else {
                    separator.show();
                }
                if (vertical) {
                    panel.resize(container.height());
                } else {
                    panel.resize(container.width());
                }
            },
            resize: function(size) {
                var panelSizes;
                if (vertical) {
                    panelSizes = [$(children[0]).outerHeight(),$(children[1]).outerHeight()];
                    container.height(size);
                } else {
                    panelSizes = [$(children[0]).outerWidth(),$(children[1]).outerWidth()];
                    container.width(size);
                }
                if (modifiedSizes) {
                    var topPanelSize = panelRatio*(size-8);
                    var bottomPanelSize = size - topPanelSize - 8;
                    panelSizes = [topPanelSize,bottomPanelSize];
                    if (vertical) {
                        $(children[0]).outerHeight(panelSizes[0]);
                        // $(children[1]).outerHeight(panelSizes[1]);
                    } else {
                        $(children[0]).outerWidth(panelSizes[0]);
                        // $(children[1]).outerWidth(panelSizes[1]);
                    }
                }
                if (options.resize) {
                    if (vertical) {
                        panelSizes = [$(children[0]).height(),$(children[1]).height()];
                    } else {
                        panelSizes = [$(children[0]).width(),$(children[1]).width()];
                    }
                    options.resize(panelSizes[0],panelSizes[1]);
                }
            }
        }
        return panel;
    }

    return {
        create: createPanel
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
/*
 * RED.popover.create(options) - create a popover callout box
 * RED.popover.tooltip(target,content, action) - add a tooltip to an element
 * RED.popover.menu(options) - create a dropdown menu
 * RED.popover.panel(content) - create a dropdown container element
 */


/*
 * RED.popover.create(options)
 *
 *  options
 *    - target : DOM element - the element to target with the popover
 *    - direction : string - position of the popover relative to target
 *                  'top', 'right'(default), 'bottom', 'left', 'inset-[top,right,bottom,left]'
 *    - trigger : string - what triggers the popover to be displayed
 *                  'hover' - display when hovering the target
 *                  'click' - display when target is clicked
 *                  'modal' - hmm not sure, need to find where we use that mode
 *    - content : string|function - contents of the popover. If a string, handled
 *                                  as raw HTML, so take care.
 *                                  If a function, can return a String to be added
 *                                  as text (not HTML), or a DOM element to append
 *    - delay : object - sets show/hide delays after mouseover/out events
 *                  { show: 750, hide: 50 }
 *    - autoClose : number - delay before closing the popover in some cases
 *                     if trigger is click - delay after mouseout
 *                     else if trigger not hover/modal - delay after showing
 *    - width : number - width of popover, default 'auto'
 *    - maxWidth : number - max width of popover, default 'auto'
 *    - size : string - scale of popover. 'default', 'small'
 *    - offset : number - px offset from target
 *    - tooltip : boolean - if true, clicking on popover closes it
 *    - class : string - optional css class to apply to popover
 *    - interactive : if trigger is 'hover' and this is set to true, allow the mouse
 *                    to move over the popover without hiding it.
 *
 * Returns the popover object with the following properties/functions:
 *   properties:
 *    - element : DOM element - the popover dom element
 *   functions:
 *    - setContent(content) - change the popover content. This only works if the
 *                            popover is not currently displayed. It does not
 *                            change the content of a visible popover.
 *    - open(instant) - show the popover. If 'instant' is true, don't fade in
 *    - close(instant) - hide the popover. If 'instant' is true, don't fade out
 *    - move(options) - move the popover. The options parameter can take many
 *                      of the options detailed above including:
 *                       target,direction,content,width,offset
 *                      Other settings probably won't work because we haven't needed to change them
 */

/*
 * RED.popover.tooltip(target,content, action)
 *
 *  - target : DOM element - the element to apply the tooltip to
 *  - content : string - the text of the tooltip
 *  - action : string - *optional* the name of an Action this tooltip is tied to
 *                      For example, it 'target' is a button that triggers a particular action.
 *                      The tooltip will include the keyboard shortcut for the action
 *                      if one is defined
 *
 */

/*
 * RED.popover.menu(options)
 *
 *  options
 *    - options : array - list of menu options - see below for format
 *    - width : number - width of the menu. Default: 'auto'
 *    - class : string - class to apply to the menu container
 *    - maxHeight : number - maximum height of menu before scrolling items. Default: none
 *    - onselect : function(item) - called when a menu item is selected, if that item doesn't
 *                                  have its own onselect function
 *    - onclose : function(cancelled) - called when the menu is closed
 *    - disposeOnClose : boolean - by default, the menu is discarded when it closes
 *                                 and mustbe rebuilt to redisplay. Setting this to 'false'
 *                                 keeps the menu on the DOM so it can be shown again.
 *
 *  Menu Options array:
 *  [
 *      label : string|DOM element - the label of the item. Can be custom DOM element
 *      onselect : function - called when the item is selected
 *  ]
 *
 * Returns the menu object with the following functions:
 *
 *  - options([menuItems]) - if menuItems is undefined, returns the current items.
 *                           otherwise, sets the current menu items
 *  - show(opts) - shows the menu. `opts` is an object of options. See  RED.popover.panel.show(opts)
 *                 for the full list of options. In most scenarios, this just needs:
 *                  - target : DOM element - the element to display the menu below
 *  - hide(cancelled) - hide the menu
 */

/*
 * RED.popover.panel(content)
 *  Create a UI panel that can be displayed relative to any target element.
 *  Handles auto-closing when mouse clicks outside the panel
 *
 *  - 'content' - DOM element to display in the panel
 *
 * Returns the panel object with the following functions:
 *
 *  properties:
 *    - container : DOM element - the panel element
 *
 *  functions:
 *    - show(opts) - show the panel.
 *       opts:
 *          - onclose : function - called when the panel closes
 *          - closeButton : DOM element - if the panel is closeable by a click of a button,
 *                                        by providing a reference to it here, we can
 *                                        handle the events properly to hide the panel
 *          - target : DOM element - the element to display the panel relative to
 *          - align : string - should the panel align to the left or right edge of target
 *                             default: 'right'
 *          - offset : Array - px offset to apply from the target. [width, height]
 *          - dispose : boolean - whether the panel should be removed from DOM when hidden
 *                                default: true
 *    - hide(dispose) - hide the panel.
 */

RED.popover = (function() {
    var deltaSizes = {
        "default": {
            x: 12,
            y: 12
        },
        "small": {
            x:8,
            y:8
        }
    }
    function createPopover(options) {
        var target = options.target;
        var direction = options.direction || "right";
        var trigger = options.trigger;
        var content = options.content;
        var delay = options.delay ||  { show: 750, hide: 50 };
        var autoClose = options.autoClose;
        var width = options.width||"auto";
        var maxWidth = options.maxWidth;
        var size = options.size||"default";
        var popupOffset = options.offset || 0;
        if (!deltaSizes[size]) {
            throw new Error("Invalid RED.popover size value:",size);
        }

        var timer = null;
        let isOpen = false
        var active;
        var div;
        var contentDiv;
        var currentStyle;

        var openPopup = function(instant) {
            if (isOpen) {
                return
            }
            if (active) {
                isOpen = true
                var existingPopover = target.data("red-ui-popover");
                if (options.tooltip && existingPopover) {
                    active = false;
                    return;
                }
                div = $('<div class="red-ui-popover"></div>');
                if (options.class) {
                    div.addClass(options.class);
                }
                contentDiv = $('<div class="red-ui-popover-content">').appendTo(div);
                if (size !== "default") {
                    div.addClass("red-ui-popover-size-"+size);
                }
                if (typeof content === 'function') {
                    var result = content.call(res);
                    if (result === null) {
                        return;
                    }
                    if (typeof result === 'string') {
                        contentDiv.text(result);
                    } else {
                        contentDiv.append(result);
                    }
                } else {
                    contentDiv.html(content);
                }
                div.appendTo("body");

                movePopup({target,direction,width,maxWidth});

                if (existingPopover) {
                    existingPopover.close(true);
                }
                if (options.trigger !== 'manual') {
                    target.data("red-ui-popover",res)
                }
                if (options.tooltip) {
                    div.on("mousedown", function(evt) {
                        closePopup(true);
                    });
                }
                if (/*trigger === 'hover' && */options.interactive) {
                    div.on('mouseenter', function(e) {
                        clearTimeout(timer);
                        active = true;
                    })
                    div.on('mouseleave', function(e) {
                        if (timer) {
                            clearTimeout(timer);
                        }
                        if (active) {
                            timer = setTimeout(function() {
                                active = false;
                                closePopup();
                            },delay.hide);
                        }
                    })
                }
                if (instant) {
                    div.show();
                } else {
                    div.fadeIn("fast");
                }
            }
        }
        var movePopup = function(options) {
            target = options.target || target;
            direction = options.direction || direction || "right";
            popupOffset = options.offset || popupOffset;
            var transition = options.transition;

            var width = options.width||"auto";
            div.width(width);
            if (options.maxWidth) {
                div.css("max-width",options.maxWidth)
            } else {
                div.css("max-width", 'auto');
            }

            var targetPos = target[0].getBoundingClientRect();
            var targetHeight = targetPos.height;
            var targetWidth = targetPos.width;

            var divHeight = div.outerHeight();
            var divWidth = div.outerWidth();
            var paddingRight = 10;

            var viewportTop = $(window).scrollTop();
            var viewportLeft = $(window).scrollLeft();
            var viewportBottom = viewportTop + $(window).height();
            var viewportRight = viewportLeft + $(window).width();
            var top = 0;
            var left = 0;
            if (direction === 'right') {
                top = targetPos.top+targetHeight/2-divHeight/2;
                left = targetPos.left+targetWidth+deltaSizes[size].x+popupOffset;
            } else if (direction === 'left') {
                top = targetPos.top+targetHeight/2-divHeight/2;
                left = targetPos.left-deltaSizes[size].x-divWidth-popupOffset;
            } else if (direction === 'bottom') {
                top = targetPos.top+targetHeight+deltaSizes[size].y+popupOffset;
                left = targetPos.left+targetWidth/2-divWidth/2;
                if (left < 0) {
                    direction = "right";
                    top = targetPos.top+targetHeight/2-divHeight/2;
                    left = targetPos.left+targetWidth+deltaSizes[size].x+popupOffset;
                } else if (left+divWidth+paddingRight > viewportRight) {
                    direction = "left";
                    top = targetPos.top+targetHeight/2-divHeight/2;
                    left = targetPos.left-deltaSizes[size].x-divWidth-popupOffset;
                    if (top+divHeight+targetHeight/2 + 5 > viewportBottom) {
                        top -= (top+divHeight+targetHeight/2 - viewportBottom + 5)
                    }
                } else if (top+divHeight > viewportBottom) {
                    direction = 'top';
                    top = targetPos.top-deltaSizes[size].y-divHeight-popupOffset;
                    left = targetPos.left+targetWidth/2-divWidth/2;
                }
            } else if (direction === 'top') {
                top = targetPos.top-deltaSizes[size].y-divHeight-popupOffset;
                left = targetPos.left+targetWidth/2-divWidth/2;
                if (top < 0) {
                    direction = 'bottom';
                    top = targetPos.top+targetHeight+deltaSizes[size].y+popupOffset;
                    left = targetPos.left+targetWidth/2-divWidth/2;
                }
            } else if (/inset/.test(direction)) {
                top = targetPos.top + targetHeight/2 - divHeight/2;
                left = targetPos.left + targetWidth/2 - divWidth/2;

                if (/bottom/.test(direction)) {
                    top = targetPos.top + targetHeight - divHeight-popupOffset;
                }
                if (/top/.test(direction)) {
                    top = targetPos.top+popupOffset;
                }
                if (/left/.test(direction)) {
                    left = targetPos.left+popupOffset;
                }
                if (/right/.test(direction)) {
                    left = targetPos.left + targetWidth - divWidth-popupOffset;
                }
            }
            if (currentStyle) {
                div.removeClass(currentStyle);
            }
            if (transition) {
                div.css({
                    "transition": "0.6s ease",
                    "transition-property": "top,left,right,bottom"
                })
            }
            currentStyle = 'red-ui-popover-'+direction;
            div.addClass(currentStyle).css({top: top, left: left});
            if (transition) {
                setTimeout(function() {
                    div.css({
                        "transition": "none"
                    });
                },600);
            }

        }
        var closePopup = function(instant) {
            isOpen = false
            $(document).off('mousedown.red-ui-popover');
            if (!active) {
                if (div) {
                    if (instant) {
                        div.remove();
                    } else {
                        div.fadeOut("fast",function() {
                            $(this).remove();
                        });
                    }
                    div = null;
                    target.removeData("red-ui-popover",res)
                }
            }
        }

        target.on("remove", function (ev) {
            if (timer) {
                clearTimeout(timer);
            }
            if (active) {
                active = false;
                setTimeout(closePopup,delay.hide);
            }
        });

        if (trigger === 'hover') {
            target.on('mouseenter',function(e) {
                clearTimeout(timer);
                if (!active) {
                    active = true;
                    timer = setTimeout(openPopup,delay.show);
                }
            });
            target.on('mouseleave disabled', function(e) {
                if (timer) {
                    clearTimeout(timer);
                }
                if (active) {
                    active = false;
                    setTimeout(closePopup,delay.hide);
                }
            });
        } else if (trigger === 'click') {
            target.on("click", function(e) {
                e.preventDefault();
                e.stopPropagation();
                active = !active;
                if (!active) {
                    closePopup();
                } else {
                    openPopup();
                }
            });
            if (autoClose) {
                target.on('mouseleave disabled', function(e) {
                    if (timer) {
                        clearTimeout(timer);
                    }
                    if (active) {
                        active = false;
                        setTimeout(closePopup,autoClose);
                    }
                });
            }

        } else if (trigger === 'modal') {
            $(document).on('mousedown.red-ui-popover', function (event) {
                var target = event.target;
                while (target.nodeName !== 'BODY' && target !== div[0]) {
                    target = target.parentElement;
                }
                if (target.nodeName === 'BODY') {
                    active = false;
                    closePopup();
                }
            });
        } else if (autoClose) {
            setTimeout(function() {
                active = false;
                closePopup();
            },autoClose);
        }
        var res = {
            get element() { return div },
            setContent: function(_content) {
                content = _content;

                return res;
            },
            open: function (instant) {
                active = true;
                openPopup(instant);
                return res;
            },
            close: function (instant) {
                active = false;
                closePopup(instant);
                return res;
            },
            move: function(options) {
                movePopup(options);
                return
            }
        }
        return res;

    }

    return {
        create: createPopover,
        tooltip: function(target,content, action, interactive) {
            var label = function() {
                var label = content;
                if (typeof content === 'function') {
                    label = content()
                }
                if (action) {
                    var shortcut = RED.keyboard.getShortcut(action);
                    if (shortcut && shortcut.key) {
                        label = $('<span>'+content+' <span class="red-ui-popover-key">'+RED.keyboard.formatKey(shortcut.key, true)+'</span></span>');
                    }
                }
                return label;
            }
            var popover = RED.popover.create({
                tooltip: true,
                target:target,
                trigger: "hover",
                size: "small",
                direction: "bottom",
                content: label,
                interactive,
                delay: { show: 750, hide: 50 }
            });
            popover.setContent = function(newContent) {
                content = newContent;
            }
            popover.setAction = function(newAction) {
                action = newAction;
            }
            popover.delete = function() {
                popover.close(true)
                target.off("mouseenter");
                target.off("mouseleave");
            };
            return popover;

        },
        menu: function(options) {
            var list = $('<ul class="red-ui-menu"></ul>');
            if (options.style === 'compact') {
                list.addClass("red-ui-menu-compact");
            }
            var menuOptions = options.options || [];
            var first;

            var container = RED.popover.panel(list);
            if (options.width) {
                container.container.width(options.width);
            }
            if (options.class) {
                container.container.addClass(options.class);
            }
            if (options.maxHeight) {
                container.container.css({
                    "max-height": options.maxHeight,
                    "overflow-y": 'auto'
                })
            }
            var menu = {
                options: function(opts) {
                    if (opts === undefined) {
                        return menuOptions
                    }
                    menuOptions = opts || [];
                    list.empty();
                    menuOptions.forEach(function(opt) {
                        var item = $('<li>').appendTo(list);
                        var link = $('<a href="#"></a>').appendTo(item);
                        if (typeof opt.label == "string") {
                            link.text(opt.label)
                        } else if (opt.label){
                            opt.label.appendTo(link);
                        }
                        link.on("click", function(evt) {
                            evt.preventDefault();
                            if (opt.onselect) {
                                opt.onselect();
                            } else if (options.onselect) {
                                options.onselect(opt);
                            }
                            menu.hide();
                        })
                        if (!first) { first = link}
                    })
                },
                show: function(opts) {
                    $(document).on("keydown.red-ui-menu", function(evt) {
                        var currentItem = list.find(":focus").parent();
                        if (evt.keyCode === 40) {
                            evt.preventDefault();
                            // DOWN
                            if (currentItem.length > 0) {
                                if (currentItem.index() === menuOptions.length-1) {
                                    // Wrap to top of list
                                    list.children().first().children().first().focus();
                                } else {
                                    currentItem.next().children().first().focus();
                                }
                            } else {
                                list.children().first().children().first().focus();
                            }
                        } else if (evt.keyCode === 38) {
                            evt.preventDefault();
                            // UP
                            if (currentItem.length > 0) {
                                if (currentItem.index() === 0) {
                                    // Wrap to bottom of list
                                    list.children().last().children().first().focus();
                                } else {
                                    currentItem.prev().children().first().focus();
                                }
                            } else {
                                list.children().last().children().first().focus();
                            }
                        } else if (evt.keyCode === 27) {
                            // ESCAPE
                            evt.preventDefault();
                            menu.hide(true);
                        } else if (evt.keyCode === 9 && options.tabSelect) {
                            // TAB - with tabSelect enabled
                            evt.preventDefault();
                            currentItem.find("a").trigger("click");

                        }
                        evt.stopPropagation();
                    })
                    opts.onclose = function() {
                        $(document).off("keydown.red-ui-menu");
                        if (options.onclose) {
                            options.onclose(true);
                        }
                    }
                    container.show(opts);
                },
                hide: function(cancelled) {
                    $(document).off("keydown.red-ui-menu");
                    container.hide(options.disposeOnClose);
                    if (options.onclose) {
                        options.onclose(cancelled);
                    }
                }
            }
            menu.options(menuOptions);
            return menu;
        },
        panel: function(content) {
            var panel = $('<div class="red-ui-editor-dialog red-ui-popover-panel"></div>');
            panel.css({ display: "none" });
            panel.appendTo(document.body);
            content.appendTo(panel);

            function hide(dispose) {
                $(document).off("mousedown.red-ui-popover-panel-close");
                $(document).off("keydown.red-ui-popover-panel-close");
                panel.hide();
                panel.css({
                    height: "auto"
                });
                if (dispose !== false) {
                    panel.remove();
                }
            }
            function show(options) {
                var closeCallback = options.onclose;
                var closeButton = options.closeButton;
                var target = options.target;
                var align = options.align || "right";
                var offset = options.offset || [0,0];
                var xPos = options.x;
                var yPos = options.y;
                var isAbsolutePosition = (xPos !== undefined && yPos !== undefined)

                var pos = isAbsolutePosition?{left:xPos, top: yPos}:target.offset();
                var targetWidth = isAbsolutePosition?0:target.width();
                var targetHeight = isAbsolutePosition?0:target.outerHeight();
                var panelHeight = panel.height();
                var panelWidth = panel.width();

                var top = (targetHeight+pos.top) + offset[1];
                if (top+panelHeight-$(document).scrollTop() > $(window).height()) {
                    top -= (top+panelHeight)-$(window).height() + 5;
                }
                if (top < 0) {
                    panel.height(panelHeight+top)
                    top = 0;
                }
                if (align === "right") {
                    panel.css({
                        top: top+"px",
                        left: (pos.left+offset[0])+"px",
                    });
                } else if (align === "left") {
                    panel.css({
                        top: top+"px",
                        left: (pos.left-panelWidth+offset[0])+"px",
                    });
                }
                panel.slideDown(100);

                $(document).on("keydown.red-ui-popover-panel-close", function(event) {
                    if (event.keyCode === 27) {
                        // ESCAPE
                        if (closeCallback) {
                            closeCallback();
                        }
                        hide(options.dispose);
                    }
                });

                $(document).on("mousedown.red-ui-popover-panel-close", function(event) {
                    var hitCloseButton = closeButton && $(event.target).closest(closeButton).length;
                    if(!hitCloseButton && !$(event.target).closest(panel).length && !$(event.target).closest(".red-ui-editor-dialog").length) {
                        if (closeCallback) {
                            closeCallback();
                        }
                        hide(options.dispose);
                    }
                    // if ($(event.target).closest(target).length) {
                    //     event.preventDefault();
                    // }
                })
            }
            return  {
                container: panel,
                show:show,
                hide:hide
            }
        },
        dialog: function(options) {

            const dialogContent = $('<div style="position:relative"></div>');

            if (options.closeButton !== false) {
                $('<button type="button" class="red-ui-button red-ui-button-small" style="float: right; margin-top: -4px; margin-right: -4px;"><i class="fa fa-times"></i></button>').appendTo(dialogContent).click(function(evt) {
                    evt.preventDefault();
                    close();
                })
            }

            const dialogBody = $('<div class="red-ui-dialog-body"></div>').appendTo(dialogContent);
            if (options.title) {
                $('<h2>').text(options.title).appendTo(dialogBody);
            }
            $('<div>').css("text-align","left").html(options.content).appendTo(dialogBody);

            const stepToolbar = $('<div>',{class:"red-ui-dialog-toolbar"}).appendTo(dialogContent);

            if (options.buttons) {
                options.buttons.forEach(button => {
                    const btn = $('<button type="button" class="red-ui-button"></button>').text(button.text).appendTo(stepToolbar);
                    if (button.class) {
                        btn.addClass(button.class);
                    }
                    if (button.click) {
                        btn.on('click', function(evt) {
                            evt.preventDefault();
                            button.click();
                        })
                    }

                })
            }

            const width = 500;
            const maxWidth = Math.min($(window).width()-10,Math.max(width || 0, 300));

            let shade = $('<div class="red-ui-shade" style="z-index: 2000"></div>').appendTo(document.body);
            shade.fadeIn()

            let popover = RED.popover.create({
                target: $(".red-ui-editor"),
                width: width || "auto",
                maxWidth: maxWidth+"px",
                direction: "inset",
                class: "red-ui-dialog",
                trigger: "manual",
                content: dialogContent
            }).open()

            function close() {
                if (shade) {
                    shade.fadeOut(() => {
                        shade.remove()
                        shade = null
                    })
                }
                if (popover) {
                    popover.close()
                    popover = null
                }
            }

            return {
                close
            }
        }
    }

})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
(function($) {

/**
 * options:
 *   - minimumLength : the minimum length of text before firing a change event
 *   - delay : delay, in ms, after a keystroke before firing change event
 *
 * methods:
 *   - value([val]) - gets the current value, or, if `val` is provided, sets the value
 *   - count - sets or clears a sub-label on the input. This can be used to provide
 *             a feedback on the number of matches, or number of available entries to search
 *   - change - trigger a change event
 *
 */

    $.widget( "nodered.searchBox", {
        _create: function() {
            var that = this;

            this.currentTimeout = null;
            this.lastSent = "";
            this.element.val("");
            this.element.addClass("red-ui-searchBox-input");
            this.uiContainer = this.element.wrap("<div>").parent();
            this.uiContainer.addClass("red-ui-searchBox-container");

            if (this.options.style === "compact") {
                this.uiContainer.addClass("red-ui-searchBox-compact");
            }

            if (this.element.parents("form").length === 0) {
                var form = this.element.wrap("<form>").parent();
                form.addClass("red-ui-searchBox-form");
            }
            $('<i class="fa fa-search"></i>').prependTo(this.uiContainer);
            this.clearButton = $('<a class="red-ui-searchBox-clear" href="#"><i class="fa fa-times"></i></a>').appendTo(this.uiContainer);
            this.clearButton.on("click",function(e) {
                e.preventDefault();
                that.element.val("");
                that._change("",true);
                that.element.trigger("focus");
            });

            if (this.options.options) {
                this.uiContainer.addClass("red-ui-searchBox-has-options");
                this.optsButton = $('<a class="red-ui-searchBox-opts" href="#"><i class="fa fa-caret-down"></i></a>').appendTo(this.uiContainer);
                var menuShown = false;
                this.optsMenu = RED.popover.menu({
                    style: this.options.style,
                    options: this.options.options.map(function(opt) {
                        return {
                            label: opt.label,
                            onselect: function() {
                                that.element.val(opt.value+" ");
                                that._change(opt.value,true);
                            }
                        }
                    }),
                    onclose: function(cancelled) {
                        menuShown = false;
                        that.element.trigger("focus");
                    },
                    disposeOnClose: false
                });

                var showMenu = function() {
                    menuShown = true;
                    that.optsMenu.show({
                        target: that.optsButton,
                        align: "left",
                        offset: [that.optsButton.width()-2,-1],
                        dispose: false
                    })
                }
                this.optsButton.on("click",function(e) {
                    e.preventDefault();
                    if (!menuShown) {
                        showMenu();
                    } else {
                        // TODO: This doesn't quite work because the panel's own
                        // mousedown handler triggers a close before this click
                        // handler fires.
                        that.optsMenu.hide(true);
                    }
                });
                this.optsButton.on("keydown",function(e) {
                    if (!menuShown && e.keyCode === 40) {
                        //DOWN
                        showMenu();
                    }
                });
                this.element.on("keydown",function(e) {
                    if (!menuShown && e.keyCode === 40 && $(this).val() === '') {
                        //DOWN (only show menu if search field is emty)
                        showMenu();
                    }
                });
            }

            this.resultCount = $('<span>',{class:"red-ui-searchBox-resultCount hide"}).appendTo(this.uiContainer);

            this.element.val("");
            this.element.on("keydown",function(evt) {
                if (evt.keyCode === 27) {
                    that.element.val("");
                }
                if (evt.keyCode === 13) {
                    evt.preventDefault();
                }
            })
            this.element.on("keyup",function(evt) {
                that._change($(this).val());
            });

            this.element.on("focus",function() {
                $(document).one("mousedown",function() {
                    that.element.blur();
                });
            });

        },
        _change: function(val,instant) {
            var fireEvent = false;
            if (val === "") {
                this.clearButton.hide();
                fireEvent = true;
            } else {
                this.clearButton.show();
                fireEvent = (val.length >= (this.options.minimumLength||0));
            }
            var current = this.element.val();
            fireEvent = fireEvent && current !== this.lastSent;
            if (fireEvent) {
                if (!instant && this.options.delay > 0) {
                    clearTimeout(this.currentTimeout);
                    var that = this;
                    this.currentTimeout = setTimeout(function() {
                        that.lastSent = that.element.val();
                        that._trigger("change");
                    },this.options.delay);
                } else {
                    this.lastSent = this.element.val();
                    this._trigger("change");
                }
            }
        },
        value: function(val) {
            if (val === undefined) {
                return this.element.val();
            } else {
                this.element.val(val);
                this._change(val);
            }
        },
        count: function(val) {
            if (val === undefined || val === null || val === "") {
                this.resultCount.text("").hide();
            } else {
                this.resultCount.text(val).show();
            }
        },
        change: function() {
            this._trigger("change");
        }
    });
})(jQuery);
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/



RED.tabs = (function() {

    var defaultTabIcon = "fa fa-lemon-o";
    var dragActive = false;
    var dblClickTime;
    var dblClickArmed = false;

    function createTabs(options) {
        var tabs = {};
        var pinnedTabsCount = 0;
        var currentTabWidth;
        var currentActiveTabWidth = 0;
        var collapsibleMenu;
        var mousedownTab;
        var preferredOrder = options.order;
        var ul = options.element || $("#"+options.id);
        var wrapper = ul.wrap( "<div>" ).parent();
        var scrollContainer = ul.wrap( "<div>" ).parent();
        wrapper.addClass("red-ui-tabs");
        if (options.vertical) {
            wrapper.addClass("red-ui-tabs-vertical");
        }

        if (options.addButton) {
            wrapper.addClass("red-ui-tabs-add");
            var addButton = $('<div class="red-ui-tab-button red-ui-tabs-add"><a href="#"><i class="fa fa-plus"></i></a></div>').appendTo(wrapper);
            addButton.find('a').on("click", function(evt) {
                evt.preventDefault();
                if (typeof options.addButton === 'function') {
                    options.addButton();
                } else if (typeof options.addButton === 'string') {
                    RED.actions.invoke(options.addButton);
                }
            })
            if (typeof options.addButton === 'string') {
                var l = options.addButton;
                if (options.addButtonCaption) {
                    l = options.addButtonCaption
                }
                RED.popover.tooltip(addButton,l,options.addButton);
            }
            ul.on("dblclick", function(evt) {
                var existingTabs = ul.children();
                var clickX = evt.clientX;
                var targetIndex = 0;
                existingTabs.each(function(index) {
                    var pos = $(this).offset();
                    if (pos.left > clickX) {
                        return false;
                    }
                    targetIndex = index+1;
                })
                if (typeof options.addButton === 'function') {
                    options.addButton({index:targetIndex});
                } else if (typeof options.addButton === 'string') {
                    RED.actions.invoke(options.addButton,{index:targetIndex});
                }
            });
        }
        if (options.searchButton) {
            // This is soft-deprecated as we don't use this in the core anymore
            // We no use the `menu` option to provide a drop-down list of actions
            wrapper.addClass("red-ui-tabs-search");
            var searchButton = $('<div class="red-ui-tab-button red-ui-tabs-search"><a href="#"><i class="fa fa-list-ul"></i></a></div>').appendTo(wrapper);
            searchButton.find('a').on("click", function(evt) {
                evt.preventDefault();
                if (typeof options.searchButton === 'function') {
                    options.searchButton()
                } else if (typeof options.searchButton === 'string') {
                    RED.actions.invoke(options.searchButton);
                }
            })
            if (typeof options.searchButton === 'string') {
                var l = options.searchButton;
                if (options.searchButtonCaption) {
                    l = options.searchButtonCaption
                }
                RED.popover.tooltip(searchButton,l,options.searchButton);
            }

        }
        if (options.menu) {
            wrapper.addClass("red-ui-tabs-menu");
            var menuButton = $('<div class="red-ui-tab-button red-ui-tabs-menu"><a href="#"><i class="fa fa-caret-down"></i></a></div>').appendTo(wrapper);
            var menuButtonLink = menuButton.find('a')
            var menuOpen = false;
            var menu;
            menuButtonLink.on("click", function(evt) {
                evt.stopPropagation();
                evt.preventDefault();
                if (menuOpen) {
                    menu.remove();
                    menuOpen = false;
                    return;
                }
                menuOpen = true;
                var menuOptions = [];
                if (typeof options.searchButton === 'function') {
                    menuOptions = options.menu()
                } else if (Array.isArray(options.menu)) {
                    menuOptions = options.menu;
                } else if (typeof options.menu === 'function') {
                    menuOptions = options.menu();
                }
                menu = RED.menu.init({options: menuOptions});
                menu.attr("id",options.id+"-menu");
                menu.css({
                    position: "absolute"
                })
                menu.appendTo("body");
                var elementPos = menuButton.offset();
                menu.css({
                    top: (elementPos.top+menuButton.height()-2)+"px",
                    left: (elementPos.left - menu.width() + menuButton.width())+"px"
                })
                $(".red-ui-menu.red-ui-menu-dropdown").hide();
                $(document).on("click.red-ui-tabmenu", function(evt) {
                    $(document).off("click.red-ui-tabmenu");
                    menuOpen = false;
                    menu.remove();
                });
                menu.show();
            })
        }

        if (options.contextmenu) {
            wrapper.on('contextmenu', function(evt) {
                let clickedTab
                let target = evt.target
                while(target.nodeName !== 'A' && target.nodeName !== 'UL' && target.nodeName !== 'BODY') {
                    target = target.parentNode
                }
                if (target.nodeName === 'A') {
                    const href = target.getAttribute('href')
                    if (href) {
                        clickedTab = tabs[href.slice(1)]
                    }
                }
                evt.preventDefault()
                evt.stopPropagation()
                RED.contextMenu.show({
                    x:evt.clientX-5,
                    y:evt.clientY-5,
                    options: options.contextmenu(clickedTab)
                })
                return false
            })
        }

        var scrollLeft;
        var scrollRight;

        if (options.scrollable) {
            wrapper.addClass("red-ui-tabs-scrollable");
            scrollContainer.addClass("red-ui-tabs-scroll-container");
            scrollContainer.on("scroll",function(evt) {
                // Generated by trackpads - not mousewheel
                updateScroll(evt);
            });
            scrollContainer.on("wheel", function(evt) {
                if (evt.originalEvent.deltaX === 0) {
                    // Prevent the scroll event from firing
                    evt.preventDefault();

                    // Assume this is wheel event which might not trigger
                    // the scroll event, so do things manually
                    var sl = scrollContainer.scrollLeft();
                    sl += evt.originalEvent.deltaY;
                    scrollContainer.scrollLeft(sl);
                }
            })
            scrollLeft = $('<div class="red-ui-tab-button red-ui-tab-scroll red-ui-tab-scroll-left"><a href="#" style="display:none;"><i class="fa fa-caret-left"></i></a></div>').appendTo(wrapper).find("a");
            scrollLeft.on('mousedown',function(evt) {scrollEventHandler(evt, evt.shiftKey?('-='+scrollContainer.scrollLeft()):'-=150') }).on('click',function(evt){ evt.preventDefault();});
            scrollRight = $('<div class="red-ui-tab-button red-ui-tab-scroll red-ui-tab-scroll-right"><a href="#" style="display:none;"><i class="fa fa-caret-right"></i></a></div>').appendTo(wrapper).find("a");
            scrollRight.on('mousedown',function(evt) { scrollEventHandler(evt,evt.shiftKey?('+='+(scrollContainer[0].scrollWidth - scrollContainer.width()-scrollContainer.scrollLeft())):'+=150') }).on('click',function(evt){ evt.preventDefault();});
        }

        if (options.collapsible) {
            // var dropDown = $('<div>',{class:"red-ui-tabs-select"}).appendTo(wrapper);
            // ul.hide();
            wrapper.addClass("red-ui-tabs-collapsible");

            var collapsedButtonsRow = $('<div class="red-ui-tab-link-buttons"></div>').appendTo(wrapper);

            if (options.menu !== false) {
                var selectButton = $('<a href="#"><i class="fa fa-caret-down"></i></a>').appendTo(collapsedButtonsRow);
                selectButton.addClass("red-ui-tab-link-button-menu")
                selectButton.on("click", function(evt) {
                    evt.stopPropagation();
                    evt.preventDefault();
                    if (!collapsibleMenu) {
                        var pinnedOptions = [];
                        var options = [];
                        ul.children().each(function(i,el) {
                            var id = $(el).data('tabId');
                            var opt = {
                                id:"red-ui-tabs-menu-option-"+id,
                                icon: tabs[id].iconClass || defaultTabIcon,
                                label: tabs[id].name,
                                onselect: function() {
                                    activateTab(id);
                                }
                            };
                            // if (tabs[id].pinned) {
                            //     pinnedOptions.push(opt);
                            // } else {
                                options.push(opt);
                            // }
                        });
                        options = pinnedOptions.concat(options);
                        collapsibleMenu = RED.menu.init({options: options});
                        collapsibleMenu.css({
                            position: "absolute"
                        })
                        collapsibleMenu.appendTo("body");
                    }
                    var elementPos = selectButton.offset();
                    collapsibleMenu.css({
                        top: (elementPos.top+selectButton.height()-2)+"px",
                        left: (elementPos.left - collapsibleMenu.width() + selectButton.width())+"px"
                    })
                    if (collapsibleMenu.is(":visible")) {
                        $(document).off("click.red-ui-tabmenu");
                    } else {
                        $(".red-ui-menu.red-ui-menu-dropdown").hide();
                        $(document).on("click.red-ui-tabmenu", function(evt) {
                            $(document).off("click.red-ui-tabmenu");
                            collapsibleMenu.hide();
                        });
                    }
                    collapsibleMenu.toggle();
                })
            }

        }

        function scrollEventHandler(evt,dir) {
            evt.preventDefault();
            if ($(this).hasClass('disabled')) {
                return;
            }
            var currentScrollLeft = scrollContainer.scrollLeft();
            scrollContainer.animate( { scrollLeft: dir }, 100);
            var interval = setInterval(function() {
                var newScrollLeft = scrollContainer.scrollLeft()
                if (newScrollLeft === currentScrollLeft) {
                    clearInterval(interval);
                    return;
                }
                currentScrollLeft = newScrollLeft;
                scrollContainer.animate( { scrollLeft: dir }, 100);
            },100);
            $(this).one('mouseup',function() {
                clearInterval(interval);
            })
        }


        ul.children().first().addClass("active");
        ul.children().addClass("red-ui-tab");

        function getSelection() {
            var selection = ul.find("li.red-ui-tab.selected");
            var selectedTabs = [];
            selection.each(function() {
                selectedTabs.push(tabs[$(this).find('a').attr('href').slice(1)])
            })
            return selectedTabs;
        }

        function selectionChanged() {
            options.onselect(getSelection());
        }

        function onTabClick(evt) {
            if (dragActive) {
                return
            }
            if (evt.currentTarget !== mousedownTab) {
                mousedownTab = null;
                return;
            }
            mousedownTab = null;
            if (dblClickTime && Date.now()-dblClickTime < 400) {
                dblClickTime = 0;
                dblClickArmed = true;
                return onTabDblClick.call(this,evt);
            }
            dblClickTime = Date.now();

            var currentTab = ul.find("li.red-ui-tab.active");
            var thisTab = $(this).parent();
            var fireSelectionChanged = false;
            if (options.onselect) {
                if (evt.metaKey || evt.ctrlKey) {
                    if (thisTab.hasClass("selected")) {
                        thisTab.removeClass("selected");
                        if (thisTab[0] !== currentTab[0]) {
                            // Deselect background tab
                            // - don't switch to it
                            selectionChanged();
                            return;
                        } else {
                            // Deselect current tab
                            // - if nothing remains selected, do nothing
                            // - otherwise switch to first selected tab
                            var selection = ul.find("li.red-ui-tab.selected");
                            if (selection.length === 0) {
                                selectionChanged();
                                return;
                            }
                            thisTab = selection.first();
                        }
                    } else {
                        if (!currentTab.hasClass("selected")) {
                            var currentTabObj = tabs[currentTab.find('a').attr('href').slice(1)];
                            // Auto select current tab
                            currentTab.addClass("selected");
                        }
                        thisTab.addClass("selected");
                    }
                    fireSelectionChanged = true;
                } else if (evt.shiftKey) {
                    if (currentTab[0] !== thisTab[0]) {
                        var firstTab,lastTab;
                        if (currentTab.index() < thisTab.index()) {
                            firstTab = currentTab;
                            lastTab = thisTab;
                        } else {
                            firstTab = thisTab;
                            lastTab = currentTab;
                        }
                        ul.find("li.red-ui-tab").removeClass("selected");
                        firstTab.addClass("selected");
                        lastTab.addClass("selected");
                        firstTab.nextUntil(lastTab).addClass("selected");
                    }
                    fireSelectionChanged = true;
                } else {
                    var selection = ul.find("li.red-ui-tab.selected");
                    if (selection.length > 0) {
                        selection.removeClass("selected");
                        fireSelectionChanged = true;
                    }
                }
            }

            var thisTabA = thisTab.find("a");
            if (options.onclick) {
                options.onclick(tabs[thisTabA.attr('href').slice(1)], evt);
                if (evt.isDefaultPrevented() && evt.isPropagationStopped()) {
                    return false
                }
            }
            activateTab(thisTabA);
            if (fireSelectionChanged) {
                selectionChanged();
            }
        }

        function updateScroll() {
            if (ul.children().length !== 0) {
                var sl = scrollContainer.scrollLeft();
                var scWidth = scrollContainer.width();
                var ulWidth = ul.width();
                if (sl === 0) {
                    scrollLeft.hide();
                } else {
                    scrollLeft.show();
                }
                if (sl === ulWidth-scWidth) {
                    scrollRight.hide();
                } else {
                    scrollRight.show();
                }
            }
        }
        function onTabDblClick(evt) {
            evt.preventDefault();
            if (evt.metaKey || evt.shiftKey) {
                return;
            }
            if (options.ondblclick) {
                options.ondblclick(tabs[$(this).attr('href').slice(1)]);
            }
            return false;
        }

        function activateTab(link) {
            if (typeof link === "string") {
                link = ul.find("a[href='#"+link+"']");
            }
            if (link.length === 0) {
                return;
            }
            if (link.parent().hasClass("hide-tab")) {
                link.parent().removeClass("hide-tab").removeClass("hide");
                if (options.onshow) {
                    options.onshow(tabs[link.attr('href').slice(1)])
                }
            }
            if (!link.parent().hasClass("active")) {
                ul.children().removeClass("active");
                ul.children().css({"transition": "width 100ms"});
                link.parent().addClass("active");
                var parentId = link.parent().attr('id');
                wrapper.find(".red-ui-tab-link-button").removeClass("active selected");
                $("#"+parentId+"-link-button").addClass("active selected");
                if (options.scrollable) {
                    var pos = link.parent().position().left;
                    if (pos-21 < 0) {
                        scrollContainer.animate( { scrollLeft: '+='+(pos-50) }, 300);
                    } else if (pos + 120 > scrollContainer.width()) {
                        scrollContainer.animate( { scrollLeft: '+='+(pos + 140-scrollContainer.width()) }, 300);
                    }
                }
                if (options.onchange) {
                    options.onchange(tabs[link.attr('href').slice(1)]);
                }
                updateTabWidths();
                setTimeout(function() {
                    ul.children().css({"transition": ""});
                },100);
            }
        }
        function activatePreviousTab() {
            var previous = findPreviousVisibleTab();
            if (previous.length > 0) {
                activateTab(previous.find("a"));
            }
        }
        function activateNextTab() {
            var next = findNextVisibleTab();
            if (next.length > 0) {
                activateTab(next.find("a"));
            }
        }

        function updateTabWidths() {
            if (options.vertical) {
                return;
            }
            var allTabs = ul.find("li.red-ui-tab");
            var tabs = allTabs.filter(":not(.hide-tab)");
            var hiddenTabs = allTabs.filter(".hide-tab");
            var width = wrapper.width();
            var tabCount = tabs.length;
            var tabWidth;

            if (options.collapsible) {
                var availableCount = collapsedButtonsRow.children().length;
                var visibleCount = collapsedButtonsRow.children(":visible").length;
                tabWidth = width - collapsedButtonsRow.width()-10;
                var maxTabWidth = 198;
                var minTabWidth = 120;
                if (tabWidth <= minTabWidth || (tabWidth < maxTabWidth && visibleCount > 5)) {
                    // The tab is too small. Hide the next button to make room
                    // Start at the end of the button row, -1 for the menu button
                    var b = collapsedButtonsRow.find("a:last").prev();
                    var index = collapsedButtonsRow.children().length - 2;
                    // Work backwards to find the first visible button
                    while (b.is(":not(:visible)")) {
                        b = b.prev();
                        index--;
                    }
                    // If it isn't a pinned button, hide it to get the room
                    if (tabWidth <= minTabWidth || visibleCount>6) {//}!b.hasClass("red-ui-tab-link-button-pinned")) {
                        b.hide();
                    }
                    tabWidth = Math.max(minTabWidth,width - collapsedButtonsRow.width()-10);
                } else {
                    if (visibleCount !== availableCount) {
                        if (visibleCount < 6) {
                            tabWidth = minTabWidth;
                        } else {
                            tabWidth = maxTabWidth;
                        }
                    }
                    var space = width - tabWidth - collapsedButtonsRow.width();
                    if (space > 40) {
                        collapsedButtonsRow.find("a:not(:visible):first").show();
                    }
                    tabWidth = width - collapsedButtonsRow.width()-10;
                }
                tabs.css({width:tabWidth});

            } else {
                var tabWidth = (width-12-(tabCount*6))/tabCount;
                currentTabWidth = (100*tabWidth/width)+"%";
                currentActiveTabWidth = currentTabWidth+"%";
                if (options.scrollable) {
                    tabWidth = Math.max(tabWidth,140);
                    currentTabWidth = tabWidth+"px";
                    currentActiveTabWidth = 0;
                    var listWidth = Math.max(wrapper.width(),12+(tabWidth+6)*tabCount);
                    ul.width(listWidth);
                    updateScroll();
                } else if (options.hasOwnProperty("minimumActiveTabWidth")) {
                    if (tabWidth < options.minimumActiveTabWidth) {
                        tabCount -= 1;
                        tabWidth = (width-12-options.minimumActiveTabWidth-(tabCount*6))/tabCount;
                        currentTabWidth = (100*tabWidth/width)+"%";
                        currentActiveTabWidth = options.minimumActiveTabWidth+"px";
                    } else {
                        currentActiveTabWidth = 0;
                    }
                }
                // if (options.collapsible) {
                //     console.log(currentTabWidth);
                // }

                tabs.css({width:currentTabWidth});
                hiddenTabs.css({width:"0px"});
                if (tabWidth < 50) {
                    // ul.find(".red-ui-tab-close").hide();
                    ul.find(".red-ui-tab-icon").hide();
                    ul.find(".red-ui-tab-label").css({paddingLeft:Math.min(12,Math.max(0,tabWidth-38))+"px"})
                } else {
                    // ul.find(".red-ui-tab-close").show();
                    ul.find(".red-ui-tab-icon").show();
                    ul.find(".red-ui-tab-label").css({paddingLeft:""})
                }
                if (currentActiveTabWidth !== 0) {
                    ul.find("li.red-ui-tab.active").css({"width":options.minimumActiveTabWidth});
                    // ul.find("li.red-ui-tab.active .red-ui-tab-close").show();
                    ul.find("li.red-ui-tab.active .red-ui-tab-icon").show();
                    ul.find("li.red-ui-tab.active .red-ui-tab-label").css({paddingLeft:""})
                }
            }

        }

        ul.find("li.red-ui-tab a")
            .on("mousedown", function(evt) { mousedownTab = evt.currentTarget })
            .on("mouseup",onTabClick)
            // prevent browser-default middle-click behaviour
            .on("auxclick", function(evt) { evt.preventDefault() })
            .on("click", function(evt) {evt.preventDefault(); })
            .on("dblclick", function(evt) {evt.stopPropagation(); evt.preventDefault(); })

        setTimeout(function() {
            updateTabWidths();
        },0);


        function removeTab(id) {
            if (options.onselect) {
                var selection = ul.find("li.red-ui-tab.selected");
                if (selection.length > 0) {
                    selection.removeClass("selected");
                    selectionChanged();
                }
            }
            var li = ul.find("a[href='#"+id+"']").parent();
            if (li.hasClass("active")) {
                var tab = findPreviousVisibleTab(li);
                if (tab.length === 0) {
                    tab = findNextVisibleTab(li);
                }
                if (tab.length > 0) {
                    activateTab(tab.find("a"));
                } else {
                    if (options.onchange) {
                        options.onchange(null);
                    }
                }
            }

            li.remove();
            if (tabs[id].pinned) {
                pinnedTabsCount--;
            }
            if (options.onremove) {
                options.onremove(tabs[id]);
            }
            delete tabs[id];
            updateTabWidths();
            if (collapsibleMenu) {
                collapsibleMenu.remove();
                collapsibleMenu = null;
            }
        }

        function findPreviousVisibleTab(li) {
            if (!li) {
                li = ul.find("li.active");
            }
            var previous = li.prev();
            while(previous.length > 0 && previous.hasClass("hide-tab")) {
                previous = previous.prev();
            }
            return previous;
        }
        function findNextVisibleTab(li) {
            if (!li) {
                li = ul.find("li.active");
            }
            var next = li.next();
            while(next.length > 0 && next.hasClass("hide-tab")) {
                next = next.next();
            }
            return next;
        }
        function showTab(id) {
            if (tabs[id]) {
                var li = ul.find("a[href='#"+id+"']").parent();
                if (li.hasClass("hide-tab")) {
                    li.removeClass("hide-tab").removeClass("hide");
                    if (ul.find("li.red-ui-tab:not(.hide-tab)").length === 1) {
                        activateTab(li.find("a"))
                    }
                    updateTabWidths();
                    if (options.onshow) {
                        options.onshow(tabs[id])
                    }
                }
            }
        }
        function hideTab(id) {
            if (tabs[id]) {
                var li = ul.find("a[href='#"+id+"']").parent();
                if (!li.hasClass("hide-tab")) {
                    if (li.hasClass("active")) {
                        var tab = findPreviousVisibleTab(li);
                        if (tab.length === 0) {
                            tab = findNextVisibleTab(li);
                        }
                        if (tab.length > 0) {
                            activateTab(tab.find("a"));
                        } else {
                            if (options.onchange) {
                                options.onchange(null);
                            }
                        }
                    }
                    li.removeClass("active");
                    li.one("transitionend", function(evt) {
                        li.addClass("hide");
                        updateTabWidths();
                        if (options.onhide) {
                            options.onhide(tabs[id])
                        }
                        setTimeout(function() {
                            updateScroll()
                        },200)
                    })
                    li.addClass("hide-tab");
                    li.css({width:0})
                }
            }
        }

        var tabAPI =  {
            addTab: function(tab,targetIndex) {
                if (options.onselect) {
                    var selection = ul.find("li.red-ui-tab.selected");
                    if (selection.length > 0) {
                        selection.removeClass("selected");
                        selectionChanged();
                    }
                }
                tabs[tab.id] = tab;
                var li = $("<li/>",{class:"red-ui-tab"});
                if (ul.children().length === 0) {
                    targetIndex = undefined;
                }
                if (targetIndex === 0) {
                    li.prependTo(ul);
                } else if (targetIndex > 0) {
                    li.insertAfter(ul.find("li:nth-child("+(targetIndex)+")"));
                } else {
                    li.appendTo(ul);
                }
                li.attr('id',"red-ui-tab-"+(tab.id.replace(".","-")));
                li.data("tabId",tab.id);

                if (options.maximumTabWidth || tab.maximumTabWidth) {
                    li.css("maxWidth",(options.maximumTabWidth || tab.maximumTabWidth) +"px");
                }
                var link = $("<a/>",{href:"#"+tab.id, class:"red-ui-tab-label"}).appendTo(li);
                if (tab.icon) {
                    $('<i>',{class:"red-ui-tab-icon", style:"mask-image: url("+tab.icon+"); -webkit-mask-image: url("+tab.icon+");"}).appendTo(link);
                } else if (tab.iconClass) {
                    $('<i>',{class:"red-ui-tab-icon "+tab.iconClass}).appendTo(link);
                }
                var span = $('<span/>',{class:"red-ui-text-bidi-aware"}).text(tab.label).appendTo(link);
                span.attr('dir', RED.text.bidi.resolveBaseTextDir(tab.label));
                if (options.collapsible) {
                    li.addClass("red-ui-tab-pinned");
                    var pinnedLink = $('<a href="#'+tab.id+'" class="red-ui-tab-link-button"></a>');
                    if (tab.pinned) {
                        if (pinnedTabsCount === 0) {
                            pinnedLink.prependTo(collapsedButtonsRow)
                        } else {
                            pinnedLink.insertAfter(collapsedButtonsRow.find("a.red-ui-tab-link-button-pinned:last"));
                        }
                    } else {
                        if (options.menu !== false) {
                            pinnedLink.insertBefore(collapsedButtonsRow.find("a:last"));
                        } else {
                            pinnedLink.appendTo(collapsedButtonsRow);
                        }
                    }

                    pinnedLink.attr('id',li.attr('id')+"-link-button");
                    if (tab.iconClass) {
                        $('<i>',{class:tab.iconClass}).appendTo(pinnedLink);
                    } else {
                        $('<i>',{class:defaultTabIcon}).appendTo(pinnedLink);
                    }
                    pinnedLink.on("click", function(evt) {
                        evt.preventDefault();
                        activateTab(tab.id);
                    });
                    pinnedLink.data("tabId",tab.id)
                    if (tab.pinned) {
                        pinnedLink.addClass("red-ui-tab-link-button-pinned");
                        pinnedTabsCount++;
                    }
                    RED.popover.tooltip($(pinnedLink), tab.name, tab.action);
                    if (options.onreorder) {
                        var pinnedLinkIndex;
                        var pinnedLinks = [];
                        var startPinnedIndex;
                        pinnedLink.draggable({
                            distance: 10,
                            axis:"x",
                            containment: ".red-ui-tab-link-buttons",
                            start: function(event,ui) {
                                dragActive = true;
                                $(".red-ui-tab-link-buttons").width($(".red-ui-tab-link-buttons").width());
                                if (dblClickArmed) { dblClickArmed = false; return false }
                                collapsedButtonsRow.children().each(function(i) {
                                    pinnedLinks[i] = {
                                        el:$(this),
                                        text: $(this).text(),
                                        left: $(this).position().left,
                                        width: $(this).width(),
                                        menu: $(this).hasClass("red-ui-tab-link-button-menu")
                                    };
                                    if ($(this).is(pinnedLink)) {
                                        pinnedLinkIndex = i;
                                        startPinnedIndex = i;
                                    }
                                });
                                collapsedButtonsRow.children().each(function(i) {
                                    if (i!==pinnedLinkIndex) {
                                        $(this).css({
                                            position: 'absolute',
                                            left: pinnedLinks[i].left+"px",
                                            width: pinnedLinks[i].width+2,
                                            transition: "left 0.3s"
                                        });
                                    }
                                })
                                if (!pinnedLink.hasClass('active')) {
                                    pinnedLink.css({'zIndex':1});
                                }
                            },
                            drag: function(event,ui) {
                                ui.position.left += pinnedLinks[pinnedLinkIndex].left;
                                var tabCenter = ui.position.left + pinnedLinks[pinnedLinkIndex].width/2;
                                for (var i=0;i<pinnedLinks.length;i++) {
                                    if (i === pinnedLinkIndex || pinnedLinks[i].menu || pinnedLinks[i].el.is(":not(:visible)")) {
                                        continue;
                                    }
                                    if (tabCenter > pinnedLinks[i].left && tabCenter < pinnedLinks[i].left+pinnedLinks[i].width) {
                                        if (i < pinnedLinkIndex) {
                                            pinnedLinks[i].left += pinnedLinks[pinnedLinkIndex].width+8;
                                            pinnedLinks[pinnedLinkIndex].el.detach().insertBefore(pinnedLinks[i].el);
                                        } else {
                                            pinnedLinks[i].left -= pinnedLinks[pinnedLinkIndex].width+8;
                                            pinnedLinks[pinnedLinkIndex].el.detach().insertAfter(pinnedLinks[i].el);
                                        }
                                        pinnedLinks[i].el.css({left:pinnedLinks[i].left+"px"});

                                        pinnedLinks.splice(i, 0, pinnedLinks.splice(pinnedLinkIndex, 1)[0]);

                                        pinnedLinkIndex = i;
                                        break;
                                    }
                                }
                            },
                            stop: function(event,ui) {
                                dragActive = false;
                                collapsedButtonsRow.children().css({position:"relative",left:"",transition:""});
                                $(".red-ui-tab-link-buttons").width('auto');
                                pinnedLink.css({zIndex:""});
                                updateTabWidths();
                                if (startPinnedIndex !== pinnedLinkIndex) {
                                    if (collapsibleMenu) {
                                        collapsibleMenu.remove();
                                        collapsibleMenu = null;
                                    }
                                    var newOrder = $.makeArray(collapsedButtonsRow.children().map(function() { return $(this).data('tabId');}));
                                    tabAPI.order(newOrder);
                                    options.onreorder(newOrder);
                                }
                            }
                        });
                    }

                }
                link.on("mousedown", function(evt) { mousedownTab = evt.currentTarget })
                link.on("mouseup",onTabClick);
                // prevent browser-default middle-click behaviour
                link.on("auxclick", function(evt) { evt.preventDefault() })
                link.on("click", function(evt) { evt.preventDefault(); })
                link.on("dblclick", function(evt) { evt.stopPropagation(); evt.preventDefault(); })

                $('<span class="red-ui-tabs-fade"></span>').appendTo(li);

                if (tab.closeable) {
                    li.addClass("red-ui-tabs-closeable")
                    var closeLink = $("<a/>",{href:"#",class:"red-ui-tab-close"}).appendTo(li);
                    closeLink.append('<i class="fa fa-times" />');
                    closeLink.on("click",function(event) {
                        event.preventDefault();
                        removeTab(tab.id);
                    });
                    RED.popover.tooltip(closeLink,RED._("workspace.closeFlow"));
                }
                // if (tab.hideable) {
                //     li.addClass("red-ui-tabs-closeable")
                //     var closeLink = $("<a/>",{href:"#",class:"red-ui-tab-close red-ui-tab-hide"}).appendTo(li);
                //     closeLink.append('<i class="fa fa-eye" />');
                //     closeLink.append('<i class="fa fa-eye-slash" />');
                //     closeLink.on("click",function(event) {
                //         event.preventDefault();
                //         hideTab(tab.id);
                //     });
                //     RED.popover.tooltip(closeLink,RED._("workspace.hideFlow"));
                // }

                var badges = $('<span class="red-ui-tabs-badges"></span>').appendTo(li);
                if (options.onselect) {
                    $('<i class="red-ui-tabs-badge-selected fa fa-check-circle"></i>').appendTo(badges);
                }

                // link.attr("title",tab.label);
                RED.popover.tooltip(link,function() { return RED.utils.sanitize(tab.label); });

                if (options.onadd) {
                    options.onadd(tab);
                }
                if (ul.find("li.red-ui-tab").length == 1) {
                    activateTab(link);
                }
                if (options.onreorder && !options.collapsible) {
                    var originalTabOrder;
                    var tabDragIndex;
                    var tabElements = [];
                    var startDragIndex;

                    li.draggable({
                        axis:"x",
                        distance: 20,
                        start: function(event,ui) {
                            if (dblClickArmed) { dblClickArmed = false; return false }
                            dragActive = true;
                            originalTabOrder = [];
                            tabElements = [];
                            ul.children().each(function(i) {
                                tabElements[i] = {
                                    el:$(this),
                                    text: $(this).text(),
                                    left: $(this).position().left,
                                    width: $(this).width()
                                };
                                if ($(this).is(li)) {
                                    tabDragIndex = i;
                                    startDragIndex = i;
                                }
                                originalTabOrder.push($(this).data("tabId"));
                            });
                            ul.children().each(function(i) {
                                if (i!==tabDragIndex) {
                                    $(this).css({
                                        position: 'absolute',
                                        left: tabElements[i].left+"px",
                                        width: tabElements[i].width+2,
                                        transition: "left 0.3s"
                                    });
                                }

                            })
                            if (!li.hasClass('active')) {
                                li.css({'zIndex':1});
                            }
                        },
                        drag: function(event,ui) {
                            ui.position.left += tabElements[tabDragIndex].left+scrollContainer.scrollLeft();
                            var tabCenter = ui.position.left + tabElements[tabDragIndex].width/2 - scrollContainer.scrollLeft();
                            for (var i=0;i<tabElements.length;i++) {
                                if (i === tabDragIndex) {
                                    continue;
                                }
                                if (tabCenter > tabElements[i].left && tabCenter < tabElements[i].left+tabElements[i].width) {
                                    if (i < tabDragIndex) {
                                        tabElements[i].left += tabElements[tabDragIndex].width+8;
                                        tabElements[tabDragIndex].el.detach().insertBefore(tabElements[i].el);
                                    } else {
                                        tabElements[i].left -= tabElements[tabDragIndex].width+8;
                                        tabElements[tabDragIndex].el.detach().insertAfter(tabElements[i].el);
                                    }
                                    tabElements[i].el.css({left:tabElements[i].left+"px"});

                                    tabElements.splice(i, 0, tabElements.splice(tabDragIndex, 1)[0]);

                                    tabDragIndex = i;
                                    break;
                                }
                            }
                        },
                        stop: function(event,ui) {
                            dragActive = false;
                            ul.children().css({position:"relative",left:"",transition:""});
                            if (!li.hasClass('active')) {
                                li.css({zIndex:""});
                            }
                            updateTabWidths();
                            if (startDragIndex !== tabDragIndex) {
                                options.onreorder(originalTabOrder, $.makeArray(ul.children().map(function() { return $(this).data('tabId');})));
                            }
                            activateTab(tabElements[tabDragIndex].el.data('tabId'));
                        }
                    })
                }
                setTimeout(function() {
                    updateTabWidths();
                },10);
                if (collapsibleMenu) {
                    collapsibleMenu.remove();
                    collapsibleMenu = null;
                }
                if (preferredOrder) {
                    tabAPI.order(preferredOrder);
                }
            },
            removeTab: removeTab,
            activateTab: activateTab,
            nextTab: activateNextTab,
            previousTab: activatePreviousTab,
            resize: updateTabWidths,
            count: function() {
                return ul.find("li.red-ui-tab:not(.hide)").length;
            },
            activeIndex: function() {
                return ul.find("li.active").index()
            },
            getTabIndex: function (id) {
                return ul.find("a[href='#"+id+"']").parent().index()
            },
            contains: function(id) {
                return ul.find("a[href='#"+id+"']").length > 0;
            },
            showTab: showTab,
            hideTab: hideTab,

            renameTab: function(id,label) {
                tabs[id].label = label;
                var tab = ul.find("a[href='#"+id+"']");
                tab.find("span.red-ui-text-bidi-aware").text(label).attr('dir', RED.text.bidi.resolveBaseTextDir(label));
                updateTabWidths();
            },
            listTabs: function() {
                return $.makeArray(ul.children().map(function() { return $(this).data('tabId');}));
            },
            selection: getSelection,
            clearSelection: function() {
                if (options.onselect) {
                    var selection = ul.find("li.red-ui-tab.selected");
                    if (selection.length > 0) {
                        selection.removeClass("selected");
                        selectionChanged();
                    }
                }

            },
            order: function(order) {
                preferredOrder = order;
                var existingTabOrder = $.makeArray(ul.children().map(function() { return $(this).data('tabId');}));
                var i;
                var match = true;
                for (i=0;i<order.length;i++) {
                    if (order[i] !== existingTabOrder[i]) {
                        match = false;
                        break;
                    }
                }
                if (match) {
                    return;
                }
                var existingTabMap = {};
                var existingTabs = ul.children().detach().each(function() {
                    existingTabMap[$(this).data("tabId")] = $(this);
                });
                var pinnedButtons = {};
                if (options.collapsible) {
                    collapsedButtonsRow.children().detach().each(function() {
                        var id = $(this).data("tabId");
                        if (!id) {
                            id = "__menu__"
                        }
                        pinnedButtons[id] = $(this);
                    });
                }
                for (i=0;i<order.length;i++) {
                    if (existingTabMap[order[i]]) {
                        existingTabMap[order[i]].appendTo(ul);
                        if (options.collapsible) {
                            pinnedButtons[order[i]].appendTo(collapsedButtonsRow);
                        }
                        delete existingTabMap[order[i]];
                    }
                }
                // Add any tabs that aren't known in the order
                for (i in existingTabMap) {
                    if (existingTabMap.hasOwnProperty(i)) {
                        existingTabMap[i].appendTo(ul);
                        if (options.collapsible) {
                            pinnedButtons[i].appendTo(collapsedButtonsRow);
                        }
                    }
                }
                if (options.collapsible) {
                    pinnedButtons["__menu__"].appendTo(collapsedButtonsRow);
                    updateTabWidths();
                }
            }
        }
        return tabAPI;
    }

    return {
        create: createTabs
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

RED.stack = (function() {
    function createStack(options) {
        var container = options.container;
        container.addClass("red-ui-stack");
        var contentHeight = 0;
        var entries = [];

        var visible = true;
        // TODO: make this a singleton function - and watch out for stacks no longer
        //       in the DOM
        var resizeStack = function() {
            if (entries.length > 0) {
                var headerHeight = 0;
                entries.forEach(function(entry) {
                    headerHeight += entry.header.outerHeight();
                });

                var height = container.innerHeight();
                contentHeight = height - headerHeight - (entries.length-1);
                entries.forEach(function(e) {
                    e.contentWrap.height(contentHeight);
                });
            }
        }
        if (options.fill && options.singleExpanded) {
            $(window).on("resize", resizeStack);
            $(window).on("focus", resizeStack);
        }
        return {
            add: function(entry) {
                entries.push(entry);
                entry.container = $('<div class="red-ui-palette-category">').appendTo(container);
                if (!visible) {
                    entry.container.hide();
                }
                var header = $('<div class="red-ui-palette-header"></div>').appendTo(entry.container);
                entry.header = header;
                entry.contentWrap = $('<div></div>',{style:"position:relative"}).appendTo(entry.container);
                if (options.fill) {
                    entry.contentWrap.css("height",contentHeight);
                }
                entry.content = $('<div></div>').appendTo(entry.contentWrap);
                if (entry.collapsible !== false) {
                    header.on("click", function() {
                        if (options.singleExpanded) {
                            if (!entry.isExpanded()) {
                                for (var i=0;i<entries.length;i++) {
                                    if (entries[i].isExpanded()) {
                                        entries[i].collapse();
                                    }
                                }
                                entry.expand();
                            } else if (entries.length === 2) {
                                if (entries[0] === entry) {
                                    entries[0].collapse();
                                    entries[1].expand();
                                } else {
                                    entries[1].collapse();
                                    entries[0].expand();
                                }
                            }
                        } else {
                            entry.toggle();
                        }
                    });
                    var icon = $('<i class="fa fa-angle-down"></i>').appendTo(header);

                    if (entry.expanded) {
                        entry.container.addClass("expanded");
                        icon.addClass("expanded");
                    } else {
                        entry.contentWrap.hide();
                    }
                } else {
                    $('<i style="opacity: 0.5;" class="fa fa-angle-down expanded"></i>').appendTo(header);
                    header.css("cursor","default");
                }
                entry.title = $('<span></span>').html(entry.title).appendTo(header);



                entry.toggle = function() {
                    if (entry.isExpanded()) {
                        entry.collapse();
                        return false;
                    } else {
                        entry.expand();
                        return true;
                    }
                };
                entry.expand = function() {
                    if (!entry.isExpanded()) {
                        if (entry.onexpand) {
                            entry.onexpand.call(entry);
                        }
                        if (options.singleExpanded) {
                            entries.forEach(function(e) {
                                if (e !== entry) {
                                    e.collapse();
                                }
                            })
                        }

                        icon.addClass("expanded");
                        entry.container.addClass("expanded");
                        entry.contentWrap.slideDown(200);
                        return true;
                    }
                };
                entry.collapse = function() {
                    if (entry.isExpanded()) {
                        icon.removeClass("expanded");
                        entry.container.removeClass("expanded");
                        entry.contentWrap.slideUp(200);
                        return true;
                    }
                };
                entry.isExpanded = function() {
                    return entry.container.hasClass("expanded");
                };
                if (options.fill && options.singleExpanded) {
                    resizeStack();
                }
                return entry;
            },

            hide: function() {
                visible = false;
                entries.forEach(function(entry) {
                    entry.container.hide();
                });
                return this;
            },

            show: function() {
                visible = true;
                entries.forEach(function(entry) {
                    entry.container.show();
                });
                return this;
            },
            resize: function() {
                if (resizeStack) {
                    resizeStack();
                }
            }
        }
    }

    return {
        create: createStack
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
(function($) {
    var contextParse = function(v,defaultStore) {
        var parts = RED.utils.parseContextKey(v, defaultStore&&defaultStore.value);
        return {
            option: parts.store,
            value: parts.key
        }
    }
    var contextExport = function(v,opt) {
        if (!opt) {
            return v;
        }
        var store = ((typeof opt === "string")?opt:opt.value)
        if (store !== RED.settings.context.default) {
            return "#:("+store+")::"+v;
        } else {
            return v;
        }
    }
    var contextLabel =  function(container,value) {
        var that = this;
        container.css("pointer-events","none");
        container.css("flex-grow",0);
        container.css("position",'relative');
        container.css("overflow",'visible');
        $('<div></div>').text(value).css({
            position: "absolute",
            bottom:"-2px",
            right: "5px",
            "font-size": "0.7em",
            opacity: 0.3
        }).appendTo(container);
        this.elementDiv.show();
    }
    var mapDeprecatedIcon = function(icon) {
        if (/^red\/images\/typedInput\/.+\.png$/.test(icon)) {
            icon = icon.replace(/.png$/,".svg");
        }
        return icon;
    }

    function getMatch(value, searchValue) {
        const idx = value.toLowerCase().indexOf(searchValue.toLowerCase());
        const len = idx > -1 ? searchValue.length : 0;
        return {
            index: idx,
            found: idx > -1,
            pre: value.substring(0,idx),
            match: value.substring(idx,idx+len),
            post: value.substring(idx+len),
            exact: idx === 0 && value.length === searchValue.length
        }
    }
    function generateSpans(match) {
        const els = [];
        if(match.pre) { els.push($('<span/>').text(match.pre)); }
        if(match.match) { els.push($('<span/>',{style:"font-weight: bold; color: var(--red-ui-text-color-link);"}).text(match.match)); }
        if(match.post) { els.push($('<span/>').text(match.post)); }
        return els;
    }
    
    const msgAutoComplete = function(options) {
        return function(val) {
            var matches = [];
            options.forEach(opt => {
                const optVal = opt.value;
                const optSrc = (opt.source||[]).join(",");
                const valMatch = getMatch(optVal, val);
                const srcMatch = getMatch(optSrc, val);
                if (valMatch.found || srcMatch.found) {
                    const element = $('<div>',{style: "display: flex"});
                    const valEl = $('<div/>',{ class: "red-ui-autoComplete-completion" });
                    valEl.append(generateSpans(valMatch));
                    valEl.appendTo(element);
                    if (optSrc) {
                        const optEl = $('<div>').css({ "font-size": "0.8em" });
                        optEl.append(generateSpans(srcMatch));
                        optEl.appendTo(element);
                    }
                    matches.push({
                        value: optVal,
                        label: element,
                        i: (valMatch.found ? valMatch.index : srcMatch.index)
                    });
                }
            })
            matches.sort(function(A,B){return A.i-B.i})
            return matches;
        }
    }

    function getEnvVars (obj, envVars = {}) {
        contextKnownKeys.env = contextKnownKeys.env || {}
        if (contextKnownKeys.env[obj.id]) {
            return contextKnownKeys.env[obj.id]
        }
        let parent
        if (obj.type === 'tab' || obj.type === 'subflow') {
            RED.nodes.eachConfig(function (conf) {
                if (conf.type === "global-config") {
                    parent = conf;
                }
            })
        } else if (obj.g) {
            parent = RED.nodes.group(obj.g)
        } else if (obj.z) {
            parent = RED.nodes.workspace(obj.z) || RED.nodes.subflow(obj.z)
        }
        if (parent) {
            getEnvVars(parent, envVars)
        }
        if (obj.env) {
            obj.env.forEach(env => {
                envVars[env.name] = obj
            })
        }
        contextKnownKeys.env[obj.id] = envVars
        return envVars
    }

    const envAutoComplete = function (val) {
        const editStack = RED.editor.getEditStack()
        if (editStack.length === 0) {
            done([])
            return
        }
        const editingNode = editStack.pop()
        if (!editingNode) {
            return []
        }
        const envVarsMap = getEnvVars(editingNode)
        const envVars = Object.keys(envVarsMap)
        const matches = []
        const i = val.lastIndexOf('${')
        let searchKey = val
        let isSubkey = false
        if (i > -1) {
            if (val.lastIndexOf('}') < i) {
                searchKey = val.substring(i+2)
                isSubkey = true
            }
        }
        envVars.forEach(v => {
            let valMatch = getMatch(v, searchKey);
            if (valMatch.found) {
                const optSrc = envVarsMap[v]
                const element = $('<div>',{style: "display: flex"});
                const valEl = $('<div/>',{ class: "red-ui-autoComplete-completion" });
                valEl.append(generateSpans(valMatch))
                valEl.appendTo(element)

                if (optSrc) {
                    const optEl = $('<div/>', { class: "red-ui-autoComplete-env-label" });
                    let label
                    if (optSrc.type === 'global-config') {
                        label = RED._('sidebar.context.global')
                    } else if (optSrc.type === 'group') {
                        label = RED.utils.getNodeLabel(optSrc) || (RED._('sidebar.info.group') + ': '+optSrc.id)
                    } else {
                        label = RED.utils.getNodeLabel(optSrc) || optSrc.id
                    }

                    optEl.append(generateSpans({ match: label }));
                    optEl.appendTo(element);
                }
                matches.push({
                    value: isSubkey ? val.substring(0, i + 2) + v + '}' : v,
                    label: element,
                    i: valMatch.index
                });
            }
        })
        matches.sort(function(A,B){return A.i-B.i})
        return matches
    }

    let contextKnownKeys = {}
    let contextCache = {}
    if (RED.events) {
        RED.events.on("editor:close", function () {
            contextCache = {}
            contextKnownKeys = {}
        });
    }

    const contextAutoComplete = function() {
        const that = this
        const getContextKeysFromRuntime = function(scope, store, searchKey, done) {
            contextKnownKeys[scope] = contextKnownKeys[scope] || {}
            contextKnownKeys[scope][store] = contextKnownKeys[scope][store] || new Map()
            if (searchKey.length > 0) {
                try {
                    RED.utils.normalisePropertyExpression(searchKey)
                } catch (err) {
                    // Not a valid context key, so don't try looking up
                    done()
                    return
                }
            }
            const url = `context/${scope}/${encodeURIComponent(searchKey)}?store=${store}&keysOnly`
            if (contextCache[url]) {
                // console.log('CACHED', url)
                done()
            } else {
                // console.log('GET', url)
                $.getJSON(url, function(data) {
                    // console.log(data)
                    contextCache[url] = true
                    const result = data[store] || {}
                    const keys = result.keys || []
                    const keyPrefix = searchKey + (searchKey.length > 0 ? '.' : '')
                    keys.forEach(keyInfo => {
                        const key = keyInfo.key
                        if (/^[a-zA-Z_$][0-9a-zA-Z_$]*$/.test(key)) {
                            contextKnownKeys[scope][store].set(keyPrefix + key, keyInfo)
                        } else {
                            contextKnownKeys[scope][store].set(searchKey + "[\""+key.replace(/"/,"\\\"")+"\"]", keyInfo)
                        }                        
                    })
                    done()
                })
            }
        }
        const getContextKeys = function(key, done) {
            const keyParts = key.split('.')
            const partialKey = keyParts.pop()
            let scope = that.propertyType
            if (scope === 'flow') {
                // Get the flow id of the node we're editing
                const editStack = RED.editor.getEditStack()
                if (editStack.length === 0) {
                    done(new Map())
                    return
                }
                const editingNode = editStack.pop()
                if (editingNode.z) {
                    scope = `${scope}/${editingNode.z}`
                } else {
                    done(new Map())
                    return
                }
            }
            const store = (contextStoreOptions.length === 1) ? contextStoreOptions[0].value : that.optionValue
            const searchKey = keyParts.join('.')
           
            getContextKeysFromRuntime(scope, store, searchKey, function() {
                if (contextKnownKeys[scope][store].has(key) || key.endsWith(']')) {
                    getContextKeysFromRuntime(scope, store, key, function() {
                        done(contextKnownKeys[scope][store])
                    })
                }
                done(contextKnownKeys[scope][store])
            })
        }

        return function(val, done) {
            getContextKeys(val, function (keys) {
                const matches = []
                keys.forEach((keyInfo, v) => {
                    let optVal = v
                    let valMatch = getMatch(optVal, val);
                    if (!valMatch.found && val.length > 0) {
                        if (val.endsWith('.')) {
                            // Search key ends in '.' - but doesn't match. Check again
                            // with [" at the end instead so we match bracket notation
                            valMatch = getMatch(optVal, val.substring(0, val.length - 1) + '["')
                        // } else if (val.endsWith('[') && /^array/.test(keyInfo.format)) {
                        //     console.log('this case')
                        }
                    }
                    if (valMatch.found) {
                        const element = $('<div>',{style: "display: flex"});
                        const valEl = $('<div/>',{ class: "red-ui-autoComplete-completion" });
                        // if (keyInfo.format) {
                        //     valMatch.post += ' ' + keyInfo.format
                        // }
                        if (valMatch.exact && /^array/.test(keyInfo.format)) {
                            valMatch.post += `[0-${keyInfo.length}]`
                            optVal +=  '['

                        }
                        valEl.append(generateSpans(valMatch))
                        valEl.appendTo(element)
                        matches.push({
                            value: optVal,
                            label: element,
                        });
                    }
                })
                matches.sort(function(a, b) { return a.value.localeCompare(b.value) });
                done(matches);
            })
        }
    }

    // This is a hand-generated list of completions for the core nodes (based on the node help html).
    var msgCompletions = [
        { value: "payload" },
        { value: "topic", source: ["mqtt","inject","rbe"] },
        { value: "action", source: ["mqtt"] },
        { value: "complete", source: ["join"] },
        { value: "contentType", source: ["mqtt"] },
        { value: "cookies", source: ["http request","http response"] },
        { value: "correlationData", source: ["mqtt"] },
        { value: "delay", source: ["delay","trigger"] },
        { value: "encoding", source: ["file"] },
        { value: "error", source: ["catch"] },
        { value: "error.message", source: ["catch"] },
        { value: "error.source", source: ["catch"] },
        { value: "error.source.id", source: ["catch"] },
        { value: "error.source.type", source: ["catch"] },
        { value: "error.source.name", source: ["catch"] },
        { value: "filename", source: ["file","file in"] },
        { value: "flush", source: ["delay"] },
        { value: "followRedirects", source: ["http request"] },
        { value: "headers", source: ["http response","http request"] },
        { value: "host", source: ["tcp request","http request"] },
        { value: "ip", source: ["udp out"] },
        { value: "kill", source: ["exec"] },
        { value: "messageExpiryInterval", source: ["mqtt"] },
        { value: "method", source: ["http request"] },
        { value: "options", source: ["xml"] },
        { value: "parts", source: ["split","join","batch","sort"] },
        { value: "pid", source: ["exec"] },
        { value: "port", source: ["tcp request"," udp out"] },
        { value: "qos", source: ["mqtt"] },
        { value: "rate", source: ["delay"] },
        { value: "rejectUnauthorized", source: ["http request"] },
        { value: "req", source: ["http in"]},
        { value: "req.body", source: ["http in"]},
        { value: "req.headers", source: ["http in"]},
        { value: "req.query", source: ["http in"]},
        { value: "req.params", source: ["http in"]},
        { value: "req.cookies", source: ["http in"]},
        { value: "req.files", source: ["http in"]},
        { value: "requestTimeout", source: ["http request"] },
        { value: "reset", source: ["delay","trigger","join","rbe"] },
        { value: "responseCookies", source: ["http request"] },
        { value: "responseTopic", source: ["mqtt"] },
        { value: "responseUrl", source: ["http request"] },
        { value: "restartTimeout", source: ["join"] },
        { value: "retain", source: ["mqtt"] },
        { value: "schema", source: ["json"] },
        { value: "select", source: ["html"] },
        { value: "statusCode", source: ["http response","http request"] },
        { value: "status", source: ["status"] },
        { value: "status.text", source: ["status"] },
        { value: "status.source", source: ["status"] },
        { value: "status.source.type", source: ["status"] },
        { value: "status.source.id", source: ["status"] },
        { value: "status.source.name", source: ["status"] },
        { value: "target", source: ["link call"] },
        { value: "template", source: ["template"] },
        { value: "toFront", source: ["delay"] },
        { value: "url", source: ["http request"] },
        { value: "userProperties", source: ["mqtt"] },
        { value: "_session", source: ["websocket out","tcp out"] },
    ]
    var allOptions = {
        msg: { value: "msg", label: "msg.", validate: RED.utils.validatePropertyExpression, autoComplete: msgAutoComplete(msgCompletions) },
        flow: { value: "flow", label: "flow.", hasValue: true,
            options: [],
            validate: RED.utils.validatePropertyExpression,
            parse: contextParse,
            export: contextExport,
            valueLabel: contextLabel,
            autoComplete: contextAutoComplete
        },
        global: {
            value: "global", label: "global.", hasValue: true,
            options: [],
            validate: RED.utils.validatePropertyExpression,
            parse: contextParse,
            export: contextExport,
            valueLabel: contextLabel,
            autoComplete: contextAutoComplete
        },
        str: { value: "str", label: "string", icon: "red/images/typedInput/az.svg" },
        num: { value: "num", label: "number", icon: "red/images/typedInput/09.svg", validate: function (v, o) {
            return RED.utils.validateTypedProperty(v, "num", o);
        } },
        bool: { value: "bool", label: "boolean", icon: "red/images/typedInput/bool.svg", options: ["true", "false"] },
        json: {
            value: "json",
            label: "JSON",
            icon: "red/images/typedInput/json.svg",
            validate: function (v, o) {
                return RED.utils.validateTypedProperty(v, "json", o);
            },
            expand: function () {
                var that = this;
                var value = this.value();
                try {
                    value = JSON.stringify(JSON.parse(value), null, 4);
                } catch (err) {
                }
                RED.editor.editJSON({
                    value: value,
                    stateId: RED.editor.generateViewStateId("typedInput", that, "json"),
                    focus: true,
                    complete: function (v) {
                        var value = v;
                        try {
                            value = JSON.stringify(JSON.parse(v));
                        } catch (err) {
                        }
                        that.value(value);
                    }
                })
            }
        },
        re: { value: "re", label: "regular expression", icon: "red/images/typedInput/re.svg" },
        date: {
            value: "date",
            label: "timestamp",
            icon: "fa fa-clock-o",
            options: [
                {
                    label: 'milliseconds since epoch',
                    value: ''
                },
                {
                    label: 'YYYY-MM-DDTHH:mm:ss.sssZ',
                    value: 'iso'
                },
                {
                    label: 'JavaScript Date Object',
                    value: 'object'
                }
            ]
        },
        jsonata: {
            value: "jsonata",
            label: "expression",
            icon: "red/images/typedInput/expr.svg",
            validate: function (v, o) {
                return RED.utils.validateTypedProperty(v, "jsonata", o);
            },
            expand: function () {
                var that = this;
                RED.editor.editExpression({
                    value: this.value().replace(/\t/g, "\n"),
                    stateId: RED.editor.generateViewStateId("typedInput", that, "jsonata"),
                    focus: true,
                    complete: function (v) {
                        that.value(v.replace(/\n/g, "\t"));
                    }
                })
            }
        },
        bin: {
            value: "bin",
            label: "buffer",
            icon: "red/images/typedInput/bin.svg",
            expand: function () {
                var that = this;
                RED.editor.editBuffer({
                    value: this.value(),
                    stateId: RED.editor.generateViewStateId("typedInput", that, "bin"),
                    focus: true,
                    complete: function (v) {
                        that.value(v);
                    }
                })
            }
        },
        env: {
            value: "env",
            label: "env variable",
            icon: "red/images/typedInput/env.svg",
            autoComplete: envAutoComplete
        },
        node: {
            value: "node",
            label: "node",
            icon: "red/images/typedInput/target.svg",
            valueLabel: function (container, value) {
                var node = RED.nodes.node(value);
                var nodeDiv = $('<div>', { class: "red-ui-search-result-node" }).css({
                    "margin-top": "2px",
                    "margin-left": "3px"
                }).appendTo(container);
                var nodeLabel = $('<span>').css({
                    "line-height": "32px",
                    "margin-left": "6px"
                }).appendTo(container);
                if (node) {
                    var colour = RED.utils.getNodeColor(node.type, node._def);
                    var icon_url = RED.utils.getNodeIcon(node._def, node);
                    if (node.type === 'tab') {
                        colour = "#C0DEED";
                    }
                    nodeDiv.css('backgroundColor', colour);
                    var iconContainer = $('<div/>', { class: "red-ui-palette-icon-container" }).appendTo(nodeDiv);
                    RED.utils.createIconElement(icon_url, iconContainer, true);
                    var l = RED.utils.getNodeLabel(node, node.id);
                    nodeLabel.text(l);
                } else {
                    nodeDiv.css({
                        'backgroundColor': '#eee',
                        'border-style': 'dashed'
                    });

                }
            },
            expand: function () {
                const that = this;
                let filter;
                if (that.options.node) {
                    let nodeFilter = that.options.node.filter;
                    if ((typeof nodeFilter === "string" || typeof nodeFilter === "object") && nodeFilter) {
                        if (!Array.isArray(nodeFilter)) {
                            nodeFilter = [nodeFilter];
                        }
                        filter = function (node) {
                            return nodeFilter.includes(node.type);
                        };
                    } else if (typeof nodeFilter === "function") {
                        filter = nodeFilter;
                    }
                }
                RED.tray.hide();
                RED.view.selectNodes({
                    single: true,
                    filter: filter,
                    selected: [that.value()],
                    onselect: function (selection) {
                        that.value(selection.id);
                        RED.tray.show();
                    },
                    oncancel: function () {
                        RED.tray.show();
                    }
                })
            }
        },
        cred: {
            value: "cred",
            label: "credential",
            icon: "fa fa-lock",
            inputType: "password",
            valueLabel: function (container, value) {
                var that = this;
                container.css("pointer-events", "none");
                container.css("flex-grow", 0);
                this.elementDiv.hide();
                var buttons = $('<div>').css({
                    position: "absolute",
                    right: "6px",
                    top: "6px",
                    "pointer-events": "all"
                }).appendTo(container);
                var eyeButton = $('<button type="button" class="red-ui-button red-ui-button-small"></button>').css({
                    width: "20px"
                }).appendTo(buttons).on("click", function (evt) {
                    evt.preventDefault();
                    var cursorPosition = that.input[0].selectionStart;
                    var currentType = that.input.attr("type");
                    if (currentType === "text") {
                        that.input.attr("type", "password");
                        eyeCon.removeClass("fa-eye-slash").addClass("fa-eye");
                        setTimeout(function () {
                            that.input.focus();
                            that.input[0].setSelectionRange(cursorPosition, cursorPosition);
                        }, 50);
                    } else {
                        that.input.attr("type", "text");
                        eyeCon.removeClass("fa-eye").addClass("fa-eye-slash");
                        setTimeout(function () {
                            that.input.focus();
                            that.input[0].setSelectionRange(cursorPosition, cursorPosition);
                        }, 50);
                    }
                }).hide();
                var eyeCon = $('<i class="fa fa-eye"></i>').css("margin-left", "-2px").appendTo(eyeButton);

                if (value === "__PWRD__") {
                    var innerContainer = $('<div><i class="fa fa-asterisk"></i><i class="fa fa-asterisk"></i><i class="fa fa-asterisk"></i><i class="fa fa-asterisk"></i><i class="fa fa-asterisk"></i></div>').css({
                        padding: "6px 6px",
                        borderRadius: "4px"
                    }).addClass("red-ui-typedInput-value-label-inactive").appendTo(container);
                    var editButton = $('<button type="button" class="red-ui-button red-ui-button-small"><i class="fa fa-pencil"></i></button>').appendTo(buttons).on("click", function (evt) {
                        evt.preventDefault();
                        innerContainer.hide();
                        container.css("background", "none");
                        container.css("pointer-events", "none");
                        that.input.val("");
                        that.element.val("");
                        that.elementDiv.show();
                        editButton.hide();
                        cancelButton.show();
                        eyeButton.show();
                        setTimeout(function () {
                            that.input.focus();
                        }, 50);
                    });
                    var cancelButton = $('<button type="button" class="red-ui-button red-ui-button-small"><i class="fa fa-times"></i></button>').css("margin-left", "3px").appendTo(buttons).on("click", function (evt) {
                        evt.preventDefault();
                        innerContainer.show();
                        container.css("background", "");
                        that.input.val("__PWRD__");
                        that.element.val("__PWRD__");
                        that.elementDiv.hide();
                        editButton.show();
                        cancelButton.hide();
                        eyeButton.hide();
                        that.input.attr("type", "password");
                        eyeCon.removeClass("fa-eye-slash").addClass("fa-eye");

                    }).hide();
                } else {
                    container.css("background", "none");
                    container.css("pointer-events", "none");
                    this.elementDiv.show();
                    eyeButton.show();
                }
            }
        },
        'conf-types': {
            value: "conf-types",
            label: "config",
            icon: "fa fa-cog",
            // hasValue: false,
            valueLabel: function (container, value) {
                // get the selected option (for access to the "name" and "module" properties)
                const _options = this._optionsCache || this.typeList.find(opt => opt.value === value)?.options || []
                const selectedOption = _options.find(opt => opt.value === value) || {
                    title: '',
                    name: '',
                    module: ''
                }
                container.attr("title", selectedOption.title) // set tooltip to the full path/id of the module/node
                container.text(selectedOption.name) // apply the "name" of the selected option
                // set "line-height" such as to make the "name" appear further up, giving room for the "module" to be displayed below the value
                container.css("line-height", "1.4em")
                // add the module name in smaller, lighter font below the value
                $('<div></div>').text(selectedOption.module).css({
                    // "font-family": "var(--red-ui-monospace-font)",
                    color: "var(--red-ui-tertiary-text-color)",
                    "font-size": "0.8em",
                    "line-height": "1em",
                    opacity: 0.8
                }).appendTo(container);
            },
            // hasValue: false,
            options: function () {
                if (this._optionsCache) {
                    return this._optionsCache
                }
                const configNodes = RED.nodes.registry.getNodeDefinitions({configOnly: true, filter: (def) => def.type !== "global-config"}).map((def) => {
                    // create a container with with 2 rows (row 1 for the name, row 2 for the module name in smaller, lighter font)
                    const container = $('<div style="display: flex; flex-direction: column; justify-content: space-between; row-gap: 1px;">')
                    const row1Name = $('<div>').text(def.type)
                    const row2Module = $('<div style="font-size: 0.8em; color: var(--red-ui-tertiary-text-color);">').text(def.set.module)
                    container.append(row1Name, row2Module)
        
                    return {
                        value: def.type,
                        name: def.type,
                        enabled: def.set.enabled ?? true,
                        local: def.set.local,
                        title: def.set.id, // tooltip e.g. "node-red-contrib-foo/bar"
                        module: def.set.module,
                        icon: container[0].outerHTML.trim(), // the typeInput will interpret this as html text and render it in the anchor
                    }
                })
                this._optionsCache = configNodes
                return configNodes
            }
        }
    };

    
    // For a type with options, check value is a valid selection
    // If !opt.multiple, returns the valid option object
    // if opt.multiple, returns an array of valid option objects
    // If not valid, returns null;

    function isOptionValueValid(opt, currentVal) {
        let _options = opt.options
        if (typeof _options === "function") {
            _options = _options.call(this)
        }
        if (!opt.multiple) {
            for (var i=0;i<_options.length;i++) {
                op = _options[i];
                if (typeof op === "string" && op === currentVal) {
                    return {value:currentVal}
                } else if (op.value === currentVal) {
                    return op;
                }
            }
        } else {
            // Check to see if value is a valid csv of
            // options.
            var currentValues = {};
            var selected = [];
            currentVal.split(",").forEach(function(v) {
                if (v) {
                    currentValues[v] = true;
                }
            });
            for (var i=0;i<_options.length;i++) {
                op = _options[i];
                var val = typeof op === "string" ? op : op.value;
                if (currentValues.hasOwnProperty(val)) {
                    delete currentValues[val];
                    selected.push(typeof op === "string" ? {value:op} : op.value)
                }
            }
            if (!$.isEmptyObject(currentValues)) {
                return null;
            }
            return selected
        }
    }

    var nlsd = false;
    let contextStoreOptions;

    $.widget( "nodered.typedInput", {
        _create: function() {
            try {
            if (!nlsd && RED && RED._) {
                for (var i in allOptions) {
                    if (allOptions.hasOwnProperty(i)) {
                        allOptions[i].label = RED._("typedInput.type."+i,{defaultValue:allOptions[i].label});
                    }
                }
                var contextStores = RED.settings.context.stores;
                contextStoreOptions = contextStores.map(function(store) {
                    return {value:store,label: store, icon:'<i class="red-ui-typedInput-icon fa fa-database"></i>'}
                }).sort(function(A,B) {
                    if (A.value === RED.settings.context.default) {
                        return -1;
                    } else if (B.value === RED.settings.context.default) {
                        return 1;
                    } else {
                        return A.value.localeCompare(B.value);
                    }
                })
                if (contextStoreOptions.length < 2) {
                    allOptions.flow.options = [];
                    allOptions.global.options = [];
                } else {
                    allOptions.flow.options = contextStoreOptions;
                    allOptions.global.options = contextStoreOptions;
                }
                // Translate timestamp options
                allOptions.date.options.forEach(opt => {
                    opt.label = RED._("typedInput.date.format." + (opt.value || 'timestamp'), {defaultValue: opt.label})
                })
            }
            nlsd = true;
            var that = this;
            this.identifier = this.element.attr('id') || "TypedInput-"+Math.floor(Math.random()*100);
            if (this.options.debug) { console.log(this.identifier,"Create",{defaultType:this.options.default, value:this.element.val()}) }
            this.disarmClick = false;
            this.input = $('<input class="red-ui-typedInput-input" type="text"></input>');
            this.input.insertAfter(this.element);
            this.input.val(this.element.val());
            this.element.addClass('red-ui-typedInput');
            this.uiWidth = this.element.outerWidth();
            this.elementDiv = this.input.wrap("<div>").parent().addClass('red-ui-typedInput-input-wrap');
            this.uiSelect = this.elementDiv.wrap( "<div>" ).parent();
            var attrStyle = this.element.attr('style');
            var m;
            if ((m = /width\s*:\s*(calc\s*\(.*\)|\d+(%|px))/i.exec(attrStyle)) !== null) {
                this.input.css('width','100%');
                this.uiSelect.width(m[1]);
                this.uiWidth = null;
            } else if (this.uiWidth !== 0){
                this.uiSelect.width(this.uiWidth);
            }
            ["Right","Left"].forEach(function(d) {
                var m = that.element.css("margin"+d);
                that.uiSelect.css("margin"+d,m);
                that.input.css("margin"+d,0);
            });

            ["type","placeholder","autocomplete","data-i18n"].forEach(function(d) {
                var m = that.element.attr(d);
                that.input.attr(d,m);
            });

            this.defaultInputType = this.input.attr('type');
            // Used to remember selections per-type to restore them when switching between types
            this.oldValues = {};

            this.uiSelect.addClass("red-ui-typedInput-container");

            this.element.attr('type','hidden');

            if (!this.options.types && this.options.type) {
                this.options.types = [this.options.type]
            } else {
                this.options.types = this.options.types||Object.keys(allOptions);
            }

            this.selectTrigger = $('<button type="button" class="red-ui-typedInput-type-select" tabindex="0"></button>').prependTo(this.uiSelect);
            $('<i class="red-ui-typedInput-icon fa fa-caret-down"></i>').toggle(this.options.types.length > 1).appendTo(this.selectTrigger);

            this.selectLabel = $('<span class="red-ui-typedInput-type-label"></span>').appendTo(this.selectTrigger);

            this.valueLabelContainer = $('<div class="red-ui-typedInput-value-label">').appendTo(this.uiSelect)

            this.types(this.options.types);

            if (this.options.typeField) {
                this.typeField = $(this.options.typeField).hide();
                var t = this.typeField.val();
                if (t && this.typeMap[t]) {
                    this.options.default = t;
                }
            } else {
                this.typeField = $("<input>",{type:'hidden'}).appendTo(this.uiSelect);
            }

            this.input.on('focus', function() {
                that.uiSelect.addClass('red-ui-typedInput-focus');
            });
            this.input.on('blur', function() {
                that.uiSelect.removeClass('red-ui-typedInput-focus');
            });
            this.input.on('change', function() {
                that.validate();
                that.element.val(that.value());
                that.element.trigger('change',[that.propertyType,that.value()]);
            });
            this.input.on('keyup', function(evt) {
                that.validate();
                that.element.val(that.value());
                that.element.trigger('keyup',evt);
            });
            this.input.on('paste', function(evt) {
                that.validate();
                that.element.val(that.value());
                that.element.trigger('paste',evt);
            });
            this.input.on('keydown', function(evt) {
                if (that.typeMap[that.propertyType].autoComplete || that.input.hasClass('red-ui-autoComplete')) {
                    return
                }
                if (evt.keyCode >= 37 && evt.keyCode <= 40) {
                    evt.stopPropagation();
                }
            })
            this.selectTrigger.on("click", function(event) {
                event.preventDefault();
                event.stopPropagation();
                that._showTypeMenu();
            });
            this.selectTrigger.on('keydown',function(evt) {
                if (evt.keyCode === 40) {
                    // Down
                    that._showTypeMenu();
                }
                evt.stopPropagation();
            }).on('focus', function() {
                that.uiSelect.addClass('red-ui-typedInput-focus');
            }).on('blur', function() {
                var opt = that.typeMap[that.propertyType];
                if (opt.hasValue === false) {
                    that.uiSelect.removeClass('red-ui-typedInput-focus');
                }
            })

            // explicitly set optionSelectTrigger display to inline-block otherwise jQ sets it to 'inline'
            this.optionSelectTrigger = $('<button type="button" tabindex="0" class="red-ui-typedInput-option-trigger" style="display:inline-block"><span class="red-ui-typedInput-option-caret"><i class="red-ui-typedInput-icon fa fa-caret-down"></i></span></button>').appendTo(this.uiSelect);
            this.optionSelectLabel = $('<span class="red-ui-typedInput-option-label"></span>').prependTo(this.optionSelectTrigger);
            // RED.popover.tooltip(this.optionSelectLabel,function() {
            //     return that.optionValue;
            // });
            this.optionSelectTrigger.on("click", function(event) {
                event.preventDefault();
                event.stopPropagation();
                that._showOptionSelectMenu();
            }).on('keydown', function(evt) {
                if (evt.keyCode === 40) {
                    // Down
                    that._showOptionSelectMenu();
                }
                evt.stopPropagation();
            }).on('blur', function() {
                that.uiSelect.removeClass('red-ui-typedInput-focus');
            }).on('focus', function() {
                that.uiSelect.addClass('red-ui-typedInput-focus');
            });

            this.optionExpandButton = $('<button type="button" tabindex="0" class="red-ui-typedInput-option-expand" style="display:inline-block"></button>').appendTo(this.uiSelect);
            this.optionExpandButtonIcon = $('<i class="red-ui-typedInput-icon fa fa-ellipsis-h"></i>').appendTo(this.optionExpandButton);

            this.type(this.typeField.val() || this.options.default||this.typeList[0].value);
            this.typeChanged = !!this.options.default;
        }catch(err) {
            console.log(err.stack);
        }
        },
        _showTypeMenu: function() {
            if (this.typeList.length > 1) {
                this._showMenu(this.menu,this.selectTrigger);
                var selected = this.menu.find("[value='"+this.propertyType+"']");
                setTimeout(function() {
                    selected.trigger("focus");
                },120);
            } else {
                this.input.trigger("focus");
            }
        },
        _showOptionSelectMenu: function() {
            if (this.optionMenu) {
                this.optionMenu.css({
                    minWidth:this.optionSelectLabel.width()
                });

                this._showMenu(this.optionMenu,this.optionSelectTrigger);
                var targetValue = this.optionValue;
                if (this.optionValue === null || this.optionValue === undefined) {
                    targetValue = this.value();
                }
                var selectedOption = this.optionMenu.find("[value='"+targetValue+"']");
                if (selectedOption.length === 0) {
                    selectedOption = this.optionMenu.children(":first");
                }
                selectedOption.trigger("focus");

            }
        },
        _hideMenu: function(menu) {
            $(document).off("mousedown.red-ui-typedInput-close-property-select");
            menu.hide();
            menu.css({
                height: "auto"
            });

            if (menu.opts.multiple) {
                var selected = [];
                menu.find('input[type="checkbox"]').each(function() {
                    if ($(this).prop("checked")) {
                        selected.push($(this).data('value'))
                    }
                })
                menu.callback(selected);
            }

            if (this.elementDiv.is(":visible")) {
                this.input.trigger("focus");
            } else if (this.optionSelectTrigger.is(":visible")){
                this.optionSelectTrigger.trigger("focus");
            } else {
                this.selectTrigger.trigger("focus");
            }
        },
        _createMenu: function(menuOptions,opts,callback) {
            var that = this;
            var menu = $("<div>").addClass("red-ui-typedInput-options red-ui-editor-dialog");
            menu.opts = opts;
            menu.callback = callback;
            menuOptions.forEach(function(opt) {
                if (typeof opt === 'string') {
                    opt = {value:opt,label:opt};
                }
                var op = $('<a href="#"></a>').attr("value",opt.value).appendTo(menu);
                if (opt.label) {
                    op.text(opt.label);
                }
                if (opt.title) {
                    op.prop('title', opt.title)
                }
                if (opt.icon) {
                    if (opt.icon.indexOf("<") === 0) {
                        $(opt.icon).prependTo(op);
                    } else if (opt.icon.indexOf("/") !== -1) {
                        $('<i>',{class:"red-ui-typedInput-icon", style:"mask-image: url("+opt.icon+"); -webkit-mask-image: url("+opt.icon+");"}).prependTo(op);
                    } else {
                        $('<i>',{class:"red-ui-typedInput-icon "+opt.icon}).prependTo(op);
                    }
                } else {
                    op.css({paddingLeft: "18px"});
                }
                if (!opt.icon && !opt.label) {
                    op.text(opt.value);
                }
                var cb;
                if (opts.multiple) {
                    cb = $('<input type="checkbox">').css("pointer-events","none").data('value',opt.value).prependTo(op).on("mousedown", function(evt) { evt.preventDefault() });
                }

                op.on("click", function(event) {
                    event.preventDefault();
                    event.stopPropagation();
                    if (!opts.multiple) {
                        callback(opt.value);
                        that._hideMenu(menu);
                    } else {
                        cb.prop("checked",!cb.prop("checked"));
                    }
                });
            });
            menu.css({
                display: "none"
            });
            menu.appendTo(document.body);

            menu.on('keydown', function(evt) {
                if (evt.keyCode === 40) {
                    evt.preventDefault();
                    // DOWN
                    $(this).children(":focus").next().trigger("focus");
                } else if (evt.keyCode === 38) {
                    evt.preventDefault();
                    // UP
                    $(this).children(":focus").prev().trigger("focus");
                } else if (evt.keyCode === 27) {
                    // ESCAPE
                    evt.preventDefault();
                    that._hideMenu(menu);
                }
                evt.stopPropagation();
            })
            return menu;

        },
        _showMenu: function(menu,relativeTo) {
            if (this.disarmClick) {
                this.disarmClick = false;
                return
            }
            if (menu.opts.multiple) {
                var selected = {};
                this.value().split(",").forEach(function(f) {
                    selected[f] = true;
                });
                menu.find('input[type="checkbox"]').each(function() {
                    $(this).prop("checked", selected[$(this).data('value')] || false);
                });
            }


            var that = this;
            var pos = relativeTo.offset();
            var height = relativeTo.height();
            var menuHeight = menu.height();
            var top = (height+pos.top);
            if (top+menuHeight-$(document).scrollTop() > $(window).height()) {
                top -= (top+menuHeight)-$(window).height()+5;
            }
            if (top < 0) {
                menu.height(menuHeight+top)
                top = 0;
            }
            menu.css({
                top: top+"px",
                left: (pos.left)+"px",
            });
            menu.slideDown(100);
            this._delay(function() {
                that.uiSelect.addClass('red-ui-typedInput-focus');
                $(document).on("mousedown.red-ui-typedInput-close-property-select", function(event) {
                    if(!$(event.target).closest(menu).length) {
                        that._hideMenu(menu);
                    }
                    if ($(event.target).closest(relativeTo).length) {
                        that.disarmClick = true;
                        event.preventDefault();
                    }
                })
            });
        },
        _getLabelWidth: function(label, done) {
            var labelWidth = label.outerWidth();
            if (labelWidth === 0) {
                var wrapper = $('<div class="red-ui-editor"></div>').css({
                    position:"absolute",
                    "white-space": "nowrap",
                    top:-2000
                }).appendTo(document.body);
                var container = $('<div class="red-ui-typedInput-container"></div>').appendTo(wrapper);
                var newTrigger = label.clone().appendTo(container);
                setTimeout(function() {
                    labelWidth = newTrigger.outerWidth();
                    wrapper.remove();
                    done(labelWidth);
                },50)
            } else {
                done(labelWidth);
            }
        },
        _updateOptionSelectLabel: function(o) {
            var opt = this.typeMap[this.propertyType];
            this.optionSelectLabel.empty();
            if (opt.hasValue) {
                this.valueLabelContainer.empty();
                this.valueLabelContainer.show();
            } else {
                this.valueLabelContainer.hide();
            }
            if (this.typeMap[this.propertyType].valueLabel) {
                if (opt.multiple) {
                    this.typeMap[this.propertyType].valueLabel.call(this,opt.hasValue?this.valueLabelContainer:this.optionSelectLabel,o);
                } else {
                    this.typeMap[this.propertyType].valueLabel.call(this,opt.hasValue?this.valueLabelContainer:this.optionSelectLabel,o.value);
                }
            }
            if (!this.typeMap[this.propertyType].valueLabel || opt.hasValue) {
                if (!opt.multiple) {
                    if (o.icon) {
                        if (o.icon.indexOf("<") === 0) {
                            $(o.icon).prependTo(this.optionSelectLabel);
                        } else if (o.icon.indexOf("/") !== -1) {
                            // url
                            $('<img>',{src:mapDeprecatedIcon(o.icon),style:"height: 18px;"}).prependTo(this.optionSelectLabel);
                        } else {
                            // icon class
                            $('<i>',{class:"red-ui-typedInput-icon "+o.icon}).prependTo(this.optionSelectLabel);
                        }
                    } else if (o.label) {
                        this.optionSelectLabel.text(o.label);
                    } else {
                        this.optionSelectLabel.text(o.value);
                    }
                    if (opt.hasValue) {
                        this.optionValue = o.value;
                        this.input.trigger('change',[this.propertyType,this.value()]);
                    }
                } else {
                    this.optionSelectLabel.text(RED._("typedInput.selected", { count: o.length }));
                }
            }
        },
        _destroy: function() {
            if (this.optionMenu) {
                this.optionMenu.remove();
            }
            if (this.menu) {
                this.menu.remove();
            }
            this.uiSelect.remove();
        },
        types: function(types) {
            var that = this;
            var currentType = this.type();
            this.typeMap = {};
            var firstCall = (this.typeList === undefined);
            this.typeList = types.map(function(opt) {
                var result;
                if (typeof opt === 'string') {
                    result = allOptions[opt];
                } else {
                    result = opt;
                }
                that.typeMap[result.value] = result;
                return result;
            });
            if (this.typeList.length < 2) {
                this.selectTrigger.attr("tabindex", -1)
                this.selectTrigger.on("mousedown.red-ui-typedInput-focus-block", function(evt) { evt.preventDefault(); })
            } else {
                this.selectTrigger.attr("tabindex", 0)
                this.selectTrigger.off("mousedown.red-ui-typedInput-focus-block")
            }
            this.selectTrigger.toggleClass("disabled", this.typeList.length === 1);
            this.selectTrigger.find(".fa-caret-down").toggle(this.typeList.length > 1)
            if (this.menu) {
                this.menu.remove();
            }
            this.menu = this._createMenu(this.typeList,{},function(v) { that.type(v) });
            if (currentType && !this.typeMap.hasOwnProperty(currentType)) {
                if (!firstCall) {
                    this.type(this.typeList[0]?.value || ""); // permit empty typeList
                }
            } else {
                this.propertyType = null;
                if (!firstCall) {
                    this.type(currentType);
                }
            }
            if (this.typeList.length === 1 && !this.typeList[0].icon && (!this.typeList[0].label || this.typeList[0].showLabel === false)) {
                this.selectTrigger.hide()
            } else {
                this.selectTrigger.show()
            }
        },
        width: function(desiredWidth) {
            this.uiWidth = desiredWidth;
            if (this.uiWidth !== null) {
                this.uiSelect.width(this.uiWidth);
            }
        },
        value: function(value) {
            var that = this;
            // If the default type has been set to an invalid type, then on first
            // creation, the current propertyType will not exist. Default to an
            // empty object on the assumption the corrent type will be set shortly
            var opt = this.typeMap[this.propertyType] || {};
            if (!arguments.length) {
                var v = this.input.val();
                if (opt.export) {
                    v = opt.export(v,this.optionValue)
                }
                return v;
            } else {
                if (this.options.debug) { console.log(this.identifier,"----- SET VALUE ------",value) }
                var selectedOption = [];
                var valueToCheck = value;
                if (opt.options) {
                    let _options = opt.options
                    if (typeof opt.options === "function") {
                        _options = opt.options.call(this)
                    }

                    if (opt.hasValue && opt.parse) {
                        var parts = opt.parse(value);
                        if (this.options.debug) { console.log(this.identifier,"new parse",parts) }
                        value = parts.value;
                        valueToCheck = parts.option || parts.value;
                    }

                    var checkValues = [valueToCheck];
                    if (opt.multiple) {
                        selectedOption = [];
                        checkValues = valueToCheck.split(",");
                    }
                    checkValues.forEach(function(valueToCheck) {
                        for (var i=0;i<_options.length;i++) {
                            var op = _options[i];
                            if (typeof op === "string") {
                                if (op === valueToCheck || op === ""+valueToCheck) {
                                    selectedOption.push(that.activeOptions[op]);
                                    break;
                                }
                            } else if (op.value === valueToCheck) {
                                selectedOption.push(op);
                                break;
                            }
                        }
                    })
                    if (this.options.debug) { console.log(this.identifier,"set value to",value) }

                    this.input.val(value);
                    if (!opt.multiple) {
                        if (selectedOption.length === 0) {
                            selectedOption = [{value:""}];
                        }
                        this._updateOptionSelectLabel(selectedOption[0])
                    } else {
                        this._updateOptionSelectLabel(selectedOption)
                    }
                } else {
                    this.input.val(value);
                    if (opt.valueLabel) {
                        this.valueLabelContainer.empty();
                        opt.valueLabel.call(this,this.valueLabelContainer,value);
                    }
                }
                this.input.trigger('change',[this.type(),value]);
            }
        },
        type: function(type) {
            if (!arguments.length) {
                return this.propertyType || this.options?.default || '';
            } else {
                var that = this;
                if (this.options.debug) { console.log(this.identifier,"----- SET TYPE -----",type) }
                var previousValue = null;
                var opt = this.typeMap[type];
                if (opt && this.propertyType !== type) {
                    // If previousType is !null, then this is a change of the type, rather than the initialisation
                    var previousType = this.typeMap[this.propertyType];
                    previousValue = this.input.val();
                    if (this.input.hasClass('red-ui-autoComplete')) {
                        this.input.autoComplete("destroy");
                    }

                    if (previousType && this.typeChanged) {
                        if (this.options.debug) { console.log(this.identifier,"typeChanged",{previousType,previousValue}) }
                        if (previousType.options && opt.hasValue !== true) {
                            this.oldValues[previousType.value] = previousValue;
                        } else if (previousType.hasValue === false) {
                            this.oldValues[previousType.value] = previousValue;
                        } else {
                            this.oldValues["_"] = previousValue;
                        }
                        if ((opt.options && opt.hasValue !== true) || opt.hasValue === false) {
                            if (this.oldValues.hasOwnProperty(opt.value)) {
                                if (this.options.debug) { console.log(this.identifier,"restored previous (1)",this.oldValues[opt.value]) }
                                this.input.val(this.oldValues[opt.value]);
                            } else if (opt.options) {
                                // No old value for the option type.
                                // It is possible code has called 'value' then 'type'
                                // to set the selected option. This is what the Inject/Switch/Change
                                // nodes did before 2.1.
                                // So we need to be careful to not reset the value if it is a valid option.
                                var validOptions = isOptionValueValid(opt,previousValue);
                                if (this.options.debug) { console.log(this.identifier,{previousValue,opt,validOptions}) }
                                if ((previousValue || previousValue === '') && validOptions) {
                                    if (this.options.debug) { console.log(this.identifier,"restored previous (2)") }
                                    this.input.val(previousValue);
                                } else {
                                    if (typeof opt.default === "string") {
                                        if (this.options.debug) { console.log(this.identifier,"restored previous (3)",opt.default) }
                                        this.input.val(opt.default);
                                    } else if (Array.isArray(opt.default)) {
                                        if (this.options.debug) { console.log(this.identifier,"restored previous (4)",opt.default.join(",")) }
                                        this.input.val(opt.default.join(","))
                                    } else {
                                        if (this.options.debug) { console.log(this.identifier,"restored previous (5)") }
                                        this.input.val("");
                                    }
                                }
                            } else {
                                if (this.options.debug) { console.log(this.identifier,"restored default/blank",opt.default||"") }
                                this.input.val(opt.default||"")
                            }
                        } else {
                            if (this.options.debug) { console.log(this.identifier,"restored old/default/blank") }
                            this.input.val(this.oldValues.hasOwnProperty("_")?this.oldValues["_"]:(opt.default||""))
                        }
                        if (previousType.autoComplete) {
                            if (this.input.hasClass('red-ui-autoComplete')) {
                                this.input.autoComplete("destroy");
                            }
                        }
                    }
                    this.propertyType = type;
                    this.typeChanged = true;
                    if (this.typeField) {
                        this.typeField.val(type);
                    }
                    this.selectLabel.empty();
                    var image;
                    if (opt.icon && opt.showLabel !== false) {
                        if (opt.icon.indexOf("<") === 0) {
                            $(opt.icon).prependTo(this.selectLabel);
                        }
                        else if (opt.icon.indexOf("/") !== -1) {
                            $('<i>',{class:"red-ui-typedInput-icon", style:"mask-image: url("+opt.icon+"); -webkit-mask-image: url("+opt.icon+"); margin-right: 4px;height: 18px;width:13px"}).prependTo(this.selectLabel);
                        }
                        else {
                            $('<i>',{class:"red-ui-typedInput-icon "+opt.icon,style:"min-width: 13px; margin-right: 4px;"}).prependTo(this.selectLabel);
                        }
                    }
                    if (opt.hasValue === false || (opt.showLabel !== false && !opt.icon)) {
                        this.selectLabel.text(opt.label);
                    }
                    if (opt.label) {
                        this.selectTrigger.attr("title",opt.label);
                    } else {
                        this.selectTrigger.attr("title","");
                    }
                    if (opt.hasValue === false) {
                        this.selectTrigger.addClass("red-ui-typedInput-full-width");
                    } else {
                        this.selectTrigger.removeClass("red-ui-typedInput-full-width");
                    }

                    if (this.optionMenu) {
                        this.optionMenu.remove();
                        this.optionMenu = null;
                    }
                    if (opt.options) {
                        let _options = opt.options
                        if (typeof _options === "function") {
                            _options = opt.options.call(this);
                        }
                        if (this.optionExpandButton) {
                            this.optionExpandButton.hide();
                            this.optionExpandButton.shown = false;
                        }
                        if (this.optionSelectTrigger) {
                            this.optionSelectTrigger.css({"display":"inline-flex"});
                            if (!opt.hasValue) {
                                this.optionSelectTrigger.css({"flex-grow":1})
                                this.elementDiv.hide();
                                this.valueLabelContainer.hide();
                            } else {
                                this.optionSelectTrigger.css({"flex-grow":0})
                                this.elementDiv.show();
                                this.valueLabelContainer.hide();
                            }
                            this.activeOptions = {};
                            _options.forEach(function(o) {
                                if (typeof o === 'string') {
                                    that.activeOptions[o] = {label:o,value:o};
                                } else {
                                    that.activeOptions[o.value] = o;
                                }
                            });

                            if (!that.activeOptions.hasOwnProperty(that.optionValue)) {
                                that.optionValue = null;
                            }

                            var op;
                            if (!opt.hasValue) {
                                // Check the value is valid for the available options
                                var validValues = isOptionValueValid(opt,this.input.val());
                                if (!opt.multiple) {
                                    if (validValues) {
                                        that._updateOptionSelectLabel(validValues)
                                    } else {
                                        op = _options[0] || {value:""}; // permit zero options
                                        if (typeof op === "string") {
                                            this.value(op);
                                            that._updateOptionSelectLabel({value:op});
                                        } else {
                                            this.value(op.value);
                                            that._updateOptionSelectLabel(op);
                                        }
                                    }
                                } else {
                                    if (!validValues) {
                                        validValues = (opt.default || []).map(function(v) {
                                            return typeof v === "string"?v:v.value
                                        });
                                        this.value(validValues.join(","));
                                    }
                                    that._updateOptionSelectLabel(validValues);
                                }
                            } else {
                                var selectedOption = this.optionValue||_options[0];
                                if (opt.parse) {
                                    var selectedOptionObj = typeof selectedOption === "string"?{value:selectedOption}:selectedOption
                                    var parts = opt.parse(this.input.val(),selectedOptionObj);
                                    if (parts.option) {
                                        selectedOption = parts.option;
                                        if (!this.activeOptions.hasOwnProperty(selectedOption)) {
                                            parts.option = Object.keys(this.activeOptions)[0];
                                            selectedOption = parts.option
                                        }
                                    }
                                    this.input.val(parts.value);
                                    if (opt.export) {
                                        this.element.val(opt.export(parts.value,parts.option||selectedOption));
                                    }
                                }
                                if (typeof selectedOption === "string") {
                                    this.optionValue = selectedOption;
                                    if (!this.activeOptions.hasOwnProperty(selectedOption)) {
                                        selectedOption = Object.keys(this.activeOptions)[0];
                                    }
                                    if (!selectedOption) {
                                        this.optionSelectTrigger.hide();
                                    } else {
                                        this._updateOptionSelectLabel(this.activeOptions[selectedOption]);
                                    }
                                } else if (selectedOption) {
                                    if (this.options.debug) { console.log(this.identifier,"HERE",{optionValue:selectedOption.value}) }
                                    this.optionValue = selectedOption.value;
                                    this._updateOptionSelectLabel(selectedOption);
                                } else {
                                    this.optionSelectTrigger.hide();
                                }
                                if (opt.autoComplete) {
                                    let searchFunction = opt.autoComplete
                                    if (searchFunction.length === 0) {
                                        searchFunction = opt.autoComplete.call(this)
                                    }
                                    this.input.autoComplete({
                                        search: searchFunction,
                                        minLength: 0
                                    })
                                }
                            }
                            this.optionMenu = this._createMenu(_options,opt,function(v){
                                if (!opt.multiple) {
                                    that._updateOptionSelectLabel(that.activeOptions[v]);
                                    if (!opt.hasValue) {
                                        that.value(that.activeOptions[v].value)
                                    }
                                } else {
                                    that._updateOptionSelectLabel(v);
                                    if (!opt.hasValue) {
                                        that.value(v.join(","))
                                    }
                                }
                            });
                        }
                        this._trigger("typechange",null,this.propertyType);
                        this.input.trigger('change',[this.propertyType,this.value()]);
                    } else {
                        if (this.optionSelectTrigger) {
                            this.optionSelectTrigger.hide();
                        }
                        if (opt.inputType) {
                            this.input.attr('type',opt.inputType)
                        } else {
                            this.input.attr('type',this.defaultInputType)
                        }
                        if (opt.hasValue === false) {
                            this.elementDiv.hide();
                            this.valueLabelContainer.hide();
                        } else if (opt.valueLabel) {
                            // Reset any CSS the custom label may have set
                            this.valueLabelContainer.css("pointer-events","");
                            this.valueLabelContainer.css("flex-grow",1);
                            this.valueLabelContainer.css("overflow","hidden");
                            this.valueLabelContainer.show();
                            this.valueLabelContainer.empty();
                            this.elementDiv.hide();
                            opt.valueLabel.call(this,this.valueLabelContainer,this.input.val());
                        } else {
                            this.valueLabelContainer.hide();
                            this.elementDiv.show();
                            if (opt.autoComplete) {
                                let searchFunction = opt.autoComplete
                                if (searchFunction.length === 0) {
                                    searchFunction = opt.autoComplete.call(this)
                                }
                                this.input.autoComplete({
                                    search: searchFunction,
                                    minLength: 0
                                })
                            }
                        }
                        if (this.optionExpandButton) {
                            if (opt.expand) {
                                if (opt.expand.icon) {
                                    this.optionExpandButtonIcon.removeClass().addClass("red-ui-typedInput-icon fa "+opt.expand.icon)
                                } else {
                                    this.optionExpandButtonIcon.removeClass().addClass("red-ui-typedInput-icon fa fa-ellipsis-h")
                                }
                                this.optionExpandButton.shown = true;
                                this.optionExpandButton.show();
                                this.optionExpandButton.off('click');
                                this.optionExpandButton.on('click',function(evt) {
                                    evt.preventDefault();
                                    if (typeof opt.expand === 'function') {
                                        opt.expand.call(that);
                                    } else {
                                        var container = $('<div>');
                                        var content = opt.expand.content.call(that,container);
                                        var panel = RED.popover.panel(container);
                                        panel.container.css({
                                            width:that.valueLabelContainer.width()
                                        });
                                        if (opt.expand.minWidth) {
                                            panel.container.css({
                                                minWidth: opt.expand.minWidth+"px"
                                            });
                                        }
                                        panel.show({
                                            target:that.optionExpandButton,
                                            onclose:content.onclose,
                                            align: "left"
                                        });
                                    }
                                })
                            } else {
                                this.optionExpandButton.shown = false;
                                this.optionExpandButton.hide();
                            }
                        }
                        this._trigger("typechange",null,this.propertyType);
                        this.input.trigger('change',[this.propertyType,this.value()]);
                    }
                }
            }
        },
        validate: function(options) {
            let valid = true;
            const value = this.value();
            const type = this.type();
            if (this.typeMap[type] && this.typeMap[type].validate) {
                const validate = this.typeMap[type].validate;
                if (typeof validate === 'function') {
                    valid = validate(value, {});
                } else {
                    // Regex
                    valid = validate.test(value);
                    if (!valid) {
                        valid = RED._("validator.errors.invalid-regexp");
                    }
                }
            }
            if ((typeof valid === "string") || !valid) {
                this.element.addClass("input-error");
                this.uiSelect.addClass("input-error");
                if (typeof valid === "string") {
                    let tooltip = this.element.data("tooltip");
                    if (tooltip) {
                        tooltip.setContent(valid);
                    } else {
                        const target = this.typeMap[type]?.options ? this.optionSelectLabel : this.elementDiv;
                        tooltip = RED.popover.tooltip(target, valid);
                        this.element.data("tooltip", tooltip);
                    }
                }
            } else {
                this.element.removeClass("input-error");
                this.uiSelect.removeClass("input-error");
                const tooltip = this.element.data("tooltip");
                if (tooltip) {
                    this.element.data("tooltip", null);
                    tooltip.delete();
                }
            }
            if (options?.returnErrorMessage === true) {
                return valid;
            }
            // Must return a boolean for no 3.x validator
            return (typeof valid === "string") ? false : valid;
        },
        show: function() {
            this.uiSelect.show();
        },
        hide: function() {
            this.uiSelect.hide();
        },
        disable: function(val) {
            if(val === undefined || !!val ) {
                this.uiSelect.attr("disabled", "disabled");
            } else {
                this.uiSelect.attr("disabled", null); //remove attr
            }
        },
        enable: function() {
            this.uiSelect.attr("disabled", null); //remove attr
        },
        disabled: function() {
            return this.uiSelect.attr("disabled") === "disabled";
        },
        focus: function() {
            this.input.focus();
        }
    });
})(jQuery);
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
(function($) {

/**
 * options:
 *   - invertState   : boolean - if "true" the button will show "enabled" when the
 *                             checkbox is not selected and vice versa.
 *   - enabledIcon   : string - the icon for "enabled" state, default "fa-check-square-o"
 *   - enabledLabel  : string - the label for "enabled" state, default "Enabled" ("editor:workspace.enabled")
 *   - disabledIcon  : string - the icon for "disabled" state, default "fa-square-o"
 *   - disabledLabel : string - the label for "disabled" state, default "Disabled" ("editor:workspace.disabled")
 *   - baseClass     : string - the base css class to apply, default "red-ui-button" (alternative eg "red-ui-sidebar-header-button")
 *   - class         : string - additional classes to apply to the button - eg "red-ui-button-small"
 * methods:
 *   -
 */
    $.widget( "nodered.toggleButton", {
        _create: function() {
            var that = this;

            var invertState = false;
            if (this.options.hasOwnProperty("invertState")) {
                invertState = this.options.invertState;
            }
            var baseClass = this.options.baseClass || "red-ui-button";
            var enabledIcon = this.options.hasOwnProperty('enabledIcon')?this.options.enabledIcon : "fa-check-square-o";
            var disabledIcon = this.options.hasOwnProperty('disabledIcon')?this.options.disabledIcon : "fa-square-o";
            var enabledLabel = this.options.hasOwnProperty('enabledLabel') ? this.options.enabledLabel : RED._("editor:workspace.enabled");
            var disabledLabel = this.options.hasOwnProperty('disabledLabel') ? this.options.disabledLabel : RED._("editor:workspace.disabled");

            this.element.css("display","none");
            this.element.on("focus", function() {
                that.button.focus();
            });
            this.button = $('<button type="button" class="red-ui-toggleButton '+baseClass+' toggle single"></button>');
            if (enabledLabel || disabledLabel) {
                this.buttonLabel = $("<span>").appendTo(this.button).css("margin-left", "5px");
            }

            if (this.options.class) {
                this.button.addClass(this.options.class)
            }
            this.element.after(this.button);

            if (enabledIcon && disabledIcon) {
                this.buttonIcon = $('<i class="fa"></i>').prependTo(this.button);
            }

            // Quick hack to find the maximum width of the button
            this.button.addClass("selected");
            if (this.buttonIcon) {
                this.buttonIcon.addClass(enabledIcon);
            }
            if (this.buttonLabel) {
                this.buttonLabel.text(enabledLabel);
            }
            var width = this.button.width();
            this.button.removeClass("selected");
            if (this.buttonIcon) {
                this.buttonIcon.removeClass(enabledIcon);
                that.buttonIcon.addClass(disabledIcon);
            }
            if (this.buttonLabel) {
                that.buttonLabel.text(disabledLabel);
            }
            width = Math.max(width,this.button.width());
            if (this.buttonIcon) {
                this.buttonIcon.removeClass(disabledIcon);
            }

            // Fix the width of the button so it doesn't jump around when toggled
            if (width > 0) {
                this.button.width(Math.ceil(width));
            }

            this.button.on("click",function(e) {
                e.stopPropagation();
                if (!that.state) {
                    that.element.prop("checked",!invertState);
                } else {
                    that.element.prop("checked",invertState);
                }
                that.element.trigger("change");
            })

            this.element.on("change", function(e) {
                if ($(this).prop("checked") !== invertState) {
                    that.button.addClass("selected");
                    that.state = true;
                    if (that.buttonIcon) {
                        that.buttonIcon.addClass(enabledIcon);
                        that.buttonIcon.removeClass(disabledIcon);
                    }
                    if (that.buttonLabel) {
                        that.buttonLabel.text(enabledLabel);
                    }
                } else {
                    that.button.removeClass("selected");
                    that.state = false;
                    if (that.buttonIcon) {
                        that.buttonIcon.addClass(disabledIcon);
                        that.buttonIcon.removeClass(enabledIcon);
                    }
                    if (that.buttonLabel) {
                        that.buttonLabel.text(disabledLabel);
                    }
                }
            })
            this.element.trigger("change");
        }
    });
})(jQuery);
;(function($) {

/**
 * Attach to an <input type="text"> to provide auto-complete
 *
 * $("#node-red-text").autoComplete({
 *     search: function(value) { return ['a','b','c'] }
 * })
 *
 * options:
 *
 *  search:    function(value, [done])
 *             A function that is passed the current contents of the input whenever
 *             it changes.
 *             The function must either return auto-complete options, or pass them
 *             to the optional 'done' parameter.
 *             If the function signature includes 'done', it must be used
 *             The auto-complete options can either be an array of strings, or an array of objects in the form:
 *             {
 *                value: String : the value to insert if selected
 *                label: String|DOM Element : the label to display in the dropdown.
 *             }
 *
 *  minLength: number
 *             If `minLength` is 0, pressing down arrow will show the list
 * 
 * completionPluginType: String
 *             If provided instead of `search`, this will look for any plugins
 *             registered with the given type that implement the `getCompletions` function. This
 *             can be an async function that returns an array of string completions. It does not support
 *             the full options object as above.
 *
 * node:       Node
 *             If provided, this will be passed to the `getCompletions` function of the plugin
 *             to allow the plugin to provide context-aware completions.
 * 
 *
 */

    $.widget( "nodered.autoComplete", {
        _create: function() {
            const that = this;
            this.completionMenuShown = false;
            this.options.minLength = parseInteger(this.options.minLength, 1, 0);
            if (!this.options.search) {
                // No search function provided; nothing to provide completions
                if (this.options.completionPluginType) {
                    const plugins = RED.plugins.getPluginsByType(this.options.completionPluginType)
                    if (plugins.length > 0) {
                        this.options.search = async function (value, done) {
                            // for now, only support a single plugin
                            const promises = plugins.map(plugin => plugin.getCompletions(value, that.options.context))
                            const completions = (await Promise.all(promises)).flat()
                            const results = []
                            completions.forEach(completion => {
                                const element = $('<div>',{style: "display: flex"})
                                const valEl = $('<div/>',{ class: "red-ui-autoComplete-completion" })
                                const valMatch = getMatch(completion, value)
                                if (valMatch.found) {
                                    valEl.append(generateSpans(valMatch))
                                    valEl.appendTo(element)
                                    results.push({
                                        value: completion,
                                        label: element,
                                        match: valMatch
                                    })
                                }
                                results.sort((a, b) => {
                                    if (a.match.exact && !b.match.exact) {
                                        return -1;
                                    } else if (!a.match.exact && b.match.exact) {
                                        return 1;
                                    } else if (a.match.index < b.match.index) {
                                        return -1;
                                    } else if (a.match.index > b.match.index) {
                                        return 1;
                                    } else {
                                        return 0;
                                    }
                                })
                            })
                            done(results)
                        }
                    } else {
                        // No search function and no plugins found
                        return
                    }
                } else {
                    // No search function and no plugin type provided
                    return
                }
            }
            this.options.search = this.options.search || function() { return [] };
            this.element.addClass("red-ui-autoComplete");
            this.element.on("keydown.red-ui-autoComplete", function(evt) {
                if ((evt.keyCode === 13 || evt.keyCode === 9) && that.completionMenuShown) {
                    var opts = that.menu.options();
                    that.element.val(opts[0].value);
                    that.menu.hide();
                    evt.preventDefault();
                }
            })
            this.element.on("keyup.red-ui-autoComplete", function(evt) {
                if (evt.keyCode === 13 || evt.keyCode === 9 || evt.keyCode === 27) {
                    // ENTER / TAB / ESCAPE
                    return
                }
                if (evt.keyCode === 8 || evt.keyCode === 46) {
                    // Delete/Backspace
                    if (!that.completionMenuShown) {
                        return;
                    }
                }
                that._updateCompletions(this.value);
            });
        },
        _showCompletionMenu: function(completions) {
            if (this.completionMenuShown) {
                return;
            }
            this.menu = RED.popover.menu({
                tabSelect: true,
                width: Math.max(300, this.element.width()),
                maxHeight: 200,
                class: "red-ui-autoComplete-container",
                options: completions,
                onselect: (opt) => { this.element.val(opt.value); this.element.focus(); this.element.trigger("change") },
                onclose: () => { this.completionMenuShown = false; delete this.menu; this.element.focus()}
            });
            this.menu.show({
                target: this.element
            })
            this.completionMenuShown = true;
        },
        _updateCompletions: function(val) {
            const that = this;
            if (val.trim().length < this.options.minLength) {
                if (this.completionMenuShown) {
                    this.menu.hide();
                }
                return;
            }
            function displayResults(completions,requestId) {
                if (requestId && requestId !== that.pendingRequest) {
                    // This request has been superseded
                    return
                }
                if (!completions || completions.length === 0) {
                    if (that.completionMenuShown) {
                        that.menu.hide();
                    }
                    return
                }
                if (typeof completions[0] === "string") {
                    completions = completions.map(function(c) {
                        return { value: c, label: c };
                    });
                }
                if (that.completionMenuShown) {
                    that.menu.options(completions);
                } else {
                    that._showCompletionMenu(completions);
                }
            }
            if (this.options.search.length === 2) {
                const requestId = 1+Math.floor(Math.random()*10000);
                this.pendingRequest = requestId;
                this.options.search(val,function(completions) { displayResults(completions,requestId);})
            } else {
                displayResults(this.options.search(val))
            }
        },
        _destroy: function() {
            this.element.removeClass("red-ui-autoComplete")
            this.element.off("keydown.red-ui-autoComplete")
            this.element.off("keyup.red-ui-autoComplete")
            if (this.completionMenuShown) {
                this.menu.hide();
            }
        }
    });
    function parseInteger(input, def, min, max) {
        if(input == null) { return (def || 0); }
        min = min == null ? Number.NEGATIVE_INFINITY : min; 
        max = max == null ? Number.POSITIVE_INFINITY : max; 
        let n = parseInt(input);
        if(isNaN(n) || n < min || n > max) { n = def || 0; }
        return n;
    }
    // TODO: this is copied from typedInput - should be a shared utility
    function getMatch(value, searchValue) {
        const idx = value.toLowerCase().indexOf(searchValue.toLowerCase());
        const len = idx > -1 ? searchValue.length : 0;
        return {
            index: idx,
            found: idx > -1,
            pre: value.substring(0,idx),
            match: value.substring(idx,idx+len),
            post: value.substring(idx+len),
            exact: idx === 0 && value.length === searchValue.length
        }
    }
    // TODO: this is copied from typedInput - should be a shared utility
    function generateSpans(match) {
        const els = [];
        if(match.pre) { els.push($('<span/>').text(match.pre)); }
        if(match.match) { els.push($('<span/>',{style:"font-weight: bold; color: var(--red-ui-text-color-link);"}).text(match.match)); }
        if(match.post) { els.push($('<span/>').text(match.post)); }
        return els;
    }
})(jQuery);
;RED.actions = (function() {
    var actions = {

    };

    function addAction(name,handler,options) {
        if (typeof handler !== 'function') {
            throw new Error("Action handler not a function");
        }
        if (actions[name]) {
            throw new Error("Cannot override existing action");
        }
        actions[name] = {
            handler: handler,
            options: options,
        };
    }
    function removeAction(name) {
        delete actions[name];
    }
    function getAction(name) {
        return actions[name].handler;
    }
    function getActionLabel(name) {
        let def = actions[name]
        if (!def) {
            return ''
        }
        if (!def.label) {
            const options = def.options;
            const key = options?.label || "action-list." + name.replace(/^.*:/, "");
            let label = RED._(key, { defaultValue: options?.label || "" });
            if (!label) {
                // no translation. convert `name` to description
                label = name.replace(/(^.+:([a-z]))|(-([a-z]))/g, function() {
                    if (arguments[5] === 0) {
                        return arguments[2].toUpperCase();
                    } else {
                        return " "+arguments[4].toUpperCase();
                    }
                });
            }
            def.label = label;
        }
        return def.label
    }


    function invokeAction() {
        var args = Array.prototype.slice.call(arguments);
        var name = args.shift();
        if (actions.hasOwnProperty(name)) {
            var handler = actions[name].handler;
            handler.apply(null, args);
        }
    }
    function listActions() {
        var result = [];

        Object.keys(actions).forEach(function(action) {
            var def = actions[action];
            var shortcut = RED.keyboard.getShortcut(action);
            var isUser = false;
            if (shortcut) {
                isUser = shortcut.user;
            } else {
                isUser = !!RED.keyboard.getUserShortcut(action);
            }
            if (!def.label) {
                def.label = getActionLabel(action)
            }
            result.push({
                id:action,
                scope:shortcut?shortcut.scope:undefined,
                key:shortcut?shortcut.key:undefined,
                user:isUser,
                label: def.label,
                options: def.options,
            });
        });
        return result;
    }
    return {
        add: addAction,
        remove: removeAction,
        get: getAction,
        getLabel: getActionLabel,
        invoke: invokeAction,
        list: listActions
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

RED.deploy = (function() {

    var deploymentTypes = {
        "full":{img:"red/images/deploy-full-o.svg"},
        "nodes":{img:"red/images/deploy-nodes-o.svg"},
        "flows":{img:"red/images/deploy-flows-o.svg"}
    }

    var ignoreDeployWarnings = {
        unknown: false,
        unusedConfig: false,
        invalid: false
    }

    var deploymentType = "full";

    var deployInflight = false;

    var currentDiff = null;

    var activeBackgroundDeployNotification;

    function changeDeploymentType(type) {
        deploymentType = type;
        $("#red-ui-header-button-deploy-icon").attr("src",deploymentTypes[type].img);
    }

    /**
     * options:
     *   type: "default" - Button with drop-down options - no further customisation available
     *      label: the text to display - default: "Deploy"
     *   type: "simple"  - Button without dropdown. Customisations:
     *      label: the text to display - default: "Deploy"
     *      icon : the icon to use. Null removes the icon. default: "red/images/deploy-full-o.svg"
     */
    function init(options) {
        options = options || {};
        var type = options.type || "default";
        var label = options.label || RED._("deploy.deploy");

        if (type == "default") {
            $('<li><span class="red-ui-deploy-button-group button-group">'+
              '<a id="red-ui-header-button-deploy" class="red-ui-deploy-button disabled" href="#">'+
                '<span class="red-ui-deploy-button-content">'+
                 '<img id="red-ui-header-button-deploy-icon" src="red/images/deploy-full-o.svg"> '+
                 '<span>'+label+'</span>'+
                '</span>'+
                '<span class="red-ui-deploy-button-spinner hide">'+
                 '<img src="red/images/spin.svg"/>'+
                '</span>'+
              '</a>'+
              '<a id="red-ui-header-button-deploy-options" class="red-ui-deploy-button" href="#"><i class="fa fa-caret-down"></i><i class="fa fa-lock"></i></a>'+
              '</span></li>').prependTo(".red-ui-header-toolbar");
            const mainMenuItems = [
                    {id:"deploymenu-item-full",toggle:"deploy-type",icon:"red/images/deploy-full.svg",label:RED._("deploy.full"),sublabel:RED._("deploy.fullDesc"),selected: true, onselect:function(s) { if(s){changeDeploymentType("full")}}},
                    {id:"deploymenu-item-flow",toggle:"deploy-type",icon:"red/images/deploy-flows.svg",label:RED._("deploy.modifiedFlows"),sublabel:RED._("deploy.modifiedFlowsDesc"), onselect:function(s) {if(s){changeDeploymentType("flows")}}},
                    {id:"deploymenu-item-node",toggle:"deploy-type",icon:"red/images/deploy-nodes.svg",label:RED._("deploy.modifiedNodes"),sublabel:RED._("deploy.modifiedNodesDesc"),onselect:function(s) { if(s){changeDeploymentType("nodes")}}},
                    null
            ]
            if (RED.settings.runtimeState && RED.settings.runtimeState.ui === true) {
                mainMenuItems.push({id:"deploymenu-item-runtime-start", icon:"red/images/start.svg",label:RED._("deploy.startFlows"),sublabel:RED._("deploy.startFlowsDesc"),onselect:"core:start-flows", visible:false})
                mainMenuItems.push({id:"deploymenu-item-runtime-stop", icon:"red/images/stop.svg",label:RED._("deploy.stopFlows"),sublabel:RED._("deploy.stopFlowsDesc"),onselect:"core:stop-flows", visible:false})
            }
            mainMenuItems.push({id:"deploymenu-item-reload", icon:"red/images/deploy-reload.svg",label:RED._("deploy.restartFlows"),sublabel:RED._("deploy.restartFlowsDesc"),onselect:"core:restart-flows"})
            RED.menu.init({id:"red-ui-header-button-deploy-options", options: mainMenuItems });
        } else if (type == "simple") {
            var icon = 'red/images/deploy-full-o.svg';
            if (options.hasOwnProperty('icon')) {
                icon = options.icon;
            }

            $('<li><span class="red-ui-deploy-button-group button-group">'+
              '<a id="red-ui-header-button-deploy" class="red-ui-deploy-button disabled" href="#">'+
                '<span class="red-ui-deploy-button-content">'+
                  (icon?'<img id="red-ui-header-button-deploy-icon" src="'+icon+'"> ':'')+
                  '<span>'+label+'</span>'+
                '</span>'+
                '<span class="red-ui-deploy-button-spinner hide">'+
                 '<img src="red/images/spin.svg"/>'+
                '</span>'+
              '</a>'+
              '</span></li>').prependTo(".red-ui-header-toolbar");
        }

        $('#red-ui-header-button-deploy').on("click", function(event) {
            event.preventDefault();
            save();
        });

        RED.actions.add("core:deploy-flows",save);
        if (type === "default") {
            if (RED.settings.runtimeState && RED.settings.runtimeState.ui === true) {
                RED.actions.add("core:stop-flows",function() { stopStartFlows("stop") });
                RED.actions.add("core:start-flows",function() { stopStartFlows("start") });
            }
            RED.actions.add("core:restart-flows",restart);
            RED.actions.add("core:set-deploy-type-to-full",function() { RED.menu.setSelected("deploymenu-item-full",true);});
            RED.actions.add("core:set-deploy-type-to-modified-flows",function() { RED.menu.setSelected("deploymenu-item-flow",true); });
            RED.actions.add("core:set-deploy-type-to-modified-nodes",function() { RED.menu.setSelected("deploymenu-item-node",true); });
        }

        window.addEventListener('beforeunload', function (event) {
            if (RED.nodes.dirty()) {
                event.preventDefault();
                event.stopImmediatePropagation()
                event.returnValue = RED._("deploy.confirm.undeployedChanges");
                return
            }
        })

        RED.events.on('workspace:dirty',function(state) {
            if (RED.settings.user?.permissions === 'read') {
                return
            }
            if (state.dirty) {
                // window.onbeforeunload = function() {
                //     return 
                // }
                $("#red-ui-header-button-deploy").removeClass("disabled");
            } else {
                // window.onbeforeunload = null;
                $("#red-ui-header-button-deploy").addClass("disabled");
            }
        });

        RED.comms.subscribe("notification/runtime-deploy",function(topic,msg) {
            var currentRev = RED.nodes.version();
            if (currentRev === null || deployInflight || currentRev === msg.revision) {
                return;
            }
            if (activeBackgroundDeployNotification?.hidden && !activeBackgroundDeployNotification?.closed) {
                activeBackgroundDeployNotification.showNotification()
                return
            }
            const message = $('<p>').text(RED._('deploy.confirm.backgroundUpdate'));
            const options = {
                id: 'background-update',
                type: 'compact',
                modal: false,
                fixed: true,
                timeout: 10000,
                buttons: [
                    {
                        text: RED._('deploy.confirm.button.review'),
                        class: "primary",
                        click: function() {
                            activeBackgroundDeployNotification.hideNotification();
                            var nns = RED.nodes.createCompleteNodeSet();
                            resolveConflict(nns,false);
                        }
                    }
                ]
            }
            if (!activeBackgroundDeployNotification || activeBackgroundDeployNotification.closed) {
                activeBackgroundDeployNotification = RED.notify(message, options)
            } else {
                activeBackgroundDeployNotification.update(message, options)
            }
        });


        updateLockedState()
        RED.events.on('login', updateLockedState)
    }

    function updateLockedState() {
        if (!RED.user.hasPermission('flows.write')) {
            $(".red-ui-deploy-button-group").addClass("readOnly");
            $("#red-ui-header-button-deploy").addClass("disabled");
        } else {
            $(".red-ui-deploy-button-group").removeClass("readOnly");
            if (RED.nodes.dirty()) {
                $("#red-ui-header-button-deploy").removeClass("disabled");
            }
        }
    }

    function getNodeInfo(node) {
        var tabLabel = "";
        if (node.z) {
            var tab = RED.nodes.workspace(node.z);
            if (!tab) {
                tab = RED.nodes.subflow(node.z);
                tabLabel = tab.name;
            } else {
                tabLabel = tab.label;
            }
        }
        var label = RED.utils.getNodeLabel(node,node.id);
        return {tab:tabLabel,type:node.type,label:label};
    }
    function sortNodeInfo(A,B) {
        if (A.tab < B.tab) { return -1;}
        if (A.tab > B.tab) { return 1;}
        if (A.type < B.type) { return -1;}
        if (A.type > B.type) { return 1;}
        if (A.name < B.name) { return -1;}
        if (A.name > B.name) { return 1;}
        return 0;
    }

    function resolveConflict(currentNodes, activeDeploy) {
        var message = $('<div>');
        $('<p data-i18n="deploy.confirm.conflict"></p>').appendTo(message);
        var conflictCheck = $('<div class="red-ui-deploy-dialog-confirm-conflict-row">'+
            '<img src="red/images/spin.svg"/><div data-i18n="deploy.confirm.conflictChecking"></div>'+
        '</div>').appendTo(message);
        var conflictAutoMerge = $('<div class="red-ui-deploy-dialog-confirm-conflict-row">'+
            '<i class="fa fa-check"></i><div data-i18n="deploy.confirm.conflictAutoMerge"></div>'+
            '</div>').hide().appendTo(message);
        var conflictManualMerge = $('<div class="red-ui-deploy-dialog-confirm-conflict-row">'+
            '<i class="fa fa-exclamation"></i><div data-i18n="deploy.confirm.conflictManualMerge"></div>'+
            '</div>').hide().appendTo(message);

        message.i18n();
        currentDiff = null;
        var buttons = [
            {
                text: RED._("common.label.cancel"),
                click: function() {
                    conflictNotification.close();
                }
            },
            {
                id: "red-ui-deploy-dialog-confirm-deploy-review",
                text: RED._("deploy.confirm.button.review"),
                class: "primary disabled",
                click: function() {
                    if (!$("#red-ui-deploy-dialog-confirm-deploy-review").hasClass('disabled')) {
                        RED.diff.showRemoteDiff(null, {
                            onmerge: function () {
                                activeBackgroundDeployNotification.close()
                            }
                        });
                        conflictNotification.close();
                    }
                }
            },
            {
                id: "red-ui-deploy-dialog-confirm-deploy-merge",
                text: RED._("deploy.confirm.button.merge"),
                class: "primary disabled",
                click: function() {
                    if (!$("#red-ui-deploy-dialog-confirm-deploy-merge").hasClass('disabled')) {
                        RED.diff.mergeDiff(currentDiff);
                        conflictNotification.close();
                        activeBackgroundDeployNotification.close()
                    }
                }
            }
        ];
        if (activeDeploy) {
            buttons.push({
                id: "red-ui-deploy-dialog-confirm-deploy-overwrite",
                text: RED._("deploy.confirm.button.overwrite"),
                class: "primary",
                click: function() {
                    save(true,activeDeploy);
                    conflictNotification.close();
                    activeBackgroundDeployNotification.close()
                }
            })
        }
        var conflictNotification = RED.notify(message,{
            modal: true,
            fixed: true,
            width: 600,
            buttons: buttons
        });

        RED.diff.getRemoteDiff(function(diff) {
            currentDiff = diff;
            conflictCheck.hide();
            var d = Object.keys(diff.conflicts);
            if (d.length === 0) {
                conflictAutoMerge.show();
                $("#red-ui-deploy-dialog-confirm-deploy-merge").removeClass('disabled')
            } else {
                conflictManualMerge.show();
            }
            $("#red-ui-deploy-dialog-confirm-deploy-review").removeClass('disabled')
        })
    }
    function cropList(list) {
        if (list.length > 5) {
            var remainder = list.length - 5;
            list = list.slice(0,5);
            list.push(RED._("deploy.confirm.plusNMore",{count:remainder}));
        }
        return list;
    }
    function sanitize(html) {
        return html.replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;")
    }

    function shadeShow() {
        $("#red-ui-header-shade").show();
        $("#red-ui-editor-shade").show();
        $("#red-ui-palette-shade").show();
        $("#red-ui-sidebar-shade").show();
    }
    function shadeHide() {
        $("#red-ui-header-shade").hide();
        $("#red-ui-editor-shade").hide();
        $("#red-ui-palette-shade").hide();
        $("#red-ui-sidebar-shade").hide();
    }
    function deployButtonSetBusy(){
        $(".red-ui-deploy-button-content").css('opacity',0);
        $(".red-ui-deploy-button-spinner").show();
        $("#red-ui-header-button-deploy").addClass("disabled");
    }
    function deployButtonClearBusy(){
        $(".red-ui-deploy-button-content").css('opacity',1);
        $(".red-ui-deploy-button-spinner").hide();
    }
    function stopStartFlows(state) {
        const startTime = Date.now()
        const deployWasEnabled = !$("#red-ui-header-button-deploy").hasClass("disabled")
        deployInflight = true
        deployButtonSetBusy()
        shadeShow()
        $.ajax({
            url:"flows/state",
            type: "POST",
            data: {state: state}
        }).done(function(data,textStatus,xhr) {
            if (deployWasEnabled) {
                $("#red-ui-header-button-deploy").removeClass("disabled")
            }
        }).fail(function(xhr,textStatus,err) {
            if (deployWasEnabled) {
                $("#red-ui-header-button-deploy").removeClass("disabled")
            }
            if (xhr.status === 401) {
                RED.notify(RED._("notification.error", { message: RED._("user.notAuthorized") }), "error")
            } else if (xhr.responseText) {
                const errorDetail = { message: err ? (err + "") : "" }
                try {
                    errorDetail.message = JSON.parse(xhr.responseText).message
                } finally {
                    errorDetail.message = errorDetail.message || xhr.responseText
                }
                RED.notify(RED._("notification.error", errorDetail), "error")
            } else {
                RED.notify(RED._("notification.error", { message: RED._("deploy.errors.noResponse") }), "error")
            }
        }).always(function() {
            const delta = Math.max(0, 300 - (Date.now() - startTime))
            setTimeout(function () {
                deployButtonClearBusy()
                shadeHide()
                deployInflight = false
            }, delta);
        });
    }
    function restart() {
        var startTime = Date.now();
        var deployWasEnabled = !$("#red-ui-header-button-deploy").hasClass("disabled");
        deployInflight = true;
        deployButtonSetBusy();
        $.ajax({
            url:"flows",
            type: "POST",
            headers: {
                "Node-RED-Deployment-Type":"reload"
            }
        }).done(function(data,textStatus,xhr) {
            if (deployWasEnabled) {
                $("#red-ui-header-button-deploy").removeClass("disabled");
            }
            RED.notify('<p>'+RED._("deploy.successfulRestart")+'</p>',"success");
        }).fail(function(xhr,textStatus,err) {
            if (deployWasEnabled) {
                $("#red-ui-header-button-deploy").removeClass("disabled");
            }
            if (xhr.status === 401) {
                RED.notify(RED._("deploy.deployFailed",{message:RED._("user.notAuthorized")}),"error");
            } else if (xhr.status === 409) {
                resolveConflict(nns, true);
            } else if (xhr.responseText) {
                RED.notify(RED._("deploy.deployFailed",{message:xhr.responseText}),"error");
            } else {
                RED.notify(RED._("deploy.deployFailed",{message:RED._("deploy.errors.noResponse")}),"error");
            }
        }).always(function() {
            var delta = Math.max(0,300-(Date.now()-startTime));
            setTimeout(function() {
                deployButtonClearBusy();
                deployInflight = false;
            },delta);
        });
    }
    function save(skipValidation, force) {
        if ($("#red-ui-header-button-deploy").hasClass("disabled")) {
            return; //deploy is disabled
        }
        if ($("#red-ui-header-shade").is(":visible")) {
            return; //deploy is shaded
        }
        if (!RED.user.hasPermission("flows.write")) {
            RED.notify(RED._("user.errors.deploy"), "error");
            return;
        }
        let hasUnusedConfig = false;
        if (!skipValidation) {
            let hasUnknown = false;
            let hasInvalid = false;
            const unknownNodes = [];
            const invalidNodes = [];

            const isDisabled = function (node) {
                return (node.d || RED.nodes.workspace(node.z)?.disabled);
            };

            RED.nodes.eachConfig(function (node) {
                if (node.valid === undefined) {
                    RED.editor.validateNode(node);
                }
                if (!node.valid && !isDisabled(node)) {
                    invalidNodes.push(getNodeInfo(node));
                }
                if (node.type === "unknown") {
                    if (unknownNodes.indexOf(node.name) == -1) {
                        unknownNodes.push(node.name);
                    }
                }
            });
            RED.nodes.eachNode(function (node) {
                if (!node.valid && !isDisabled(node)) {
                    invalidNodes.push(getNodeInfo(node));
                }
                if (node.type === "unknown") {
                    if (unknownNodes.indexOf(node.name) == -1) {
                        unknownNodes.push(node.name);
                    }
                }
            });
            hasUnknown = unknownNodes.length > 0;
            hasInvalid = invalidNodes.length > 0;

            const unusedConfigNodes = [];
            RED.nodes.eachConfig(function (node) {
                if ((node._def.hasUsers !== false) && (node.users.length === 0) && !isDisabled(node)) {
                    unusedConfigNodes.push(getNodeInfo(node));
                    hasUnusedConfig = true;
                }
            });

            let showWarning = false;
            let notificationMessage;
            let notificationButtons = [];
            let notification;
            if (hasUnknown && !ignoreDeployWarnings.unknown) {
                showWarning = true;
                notificationMessage = "<p>" + RED._('deploy.confirm.unknown') + "</p>" +
                    '<ul class="red-ui-deploy-dialog-confirm-list"><li>' + cropList(unknownNodes).map(function (n) { return sanitize(n) }).join("</li><li>") + "</li></ul><p>" +
                    RED._('deploy.confirm.confirm') +
                    "</p>";

                notificationButtons = [
                    {
                        text: RED._("deploy.unknownNodesButton"),
                        class: "pull-left",
                        click: function() {
                            notification.close();
                            RED.actions.invoke("core:search","type:unknown ");
                        }
                    },
                    {
                        id: "red-ui-deploy-dialog-confirm-deploy-deploy",
                        text: RED._("deploy.confirm.button.confirm"),
                        class: "primary",
                        click: function () {
                            save(true);
                            notification.close();
                        }
                    }
                ];
            } else if (hasInvalid && !ignoreDeployWarnings.invalid) {
                showWarning = true;
                invalidNodes.sort(sortNodeInfo);

                notificationMessage = "<p>" + RED._('deploy.confirm.improperlyConfigured') + "</p>" +
                    '<ul class="red-ui-deploy-dialog-confirm-list"><li>' + cropList(invalidNodes.map(function (A) { return sanitize((A.tab ? "[" + A.tab + "] " : "") + A.label + " (" + A.type + ")") })).join("</li><li>") + "</li></ul><p>" +
                    RED._('deploy.confirm.confirm') +
                    "</p>";
                notificationButtons = [
                    {
                        text: RED._("deploy.invalidNodesButton"),
                        class: "pull-left",
                        click: function() {
                            notification.close();
                            RED.actions.invoke("core:search","is:invalid ");
                        }
                    },
                    {
                        id: "red-ui-deploy-dialog-confirm-deploy-deploy",
                        text: RED._("deploy.confirm.button.confirm"),
                        class: "primary",
                        click: function () {
                            save(true);
                            notification.close();
                        }
                    }
                ];
            }
            if (showWarning) {
                notificationButtons.unshift(
                    {
                        text: RED._("common.label.cancel"),
                        click: function () {
                            notification.close();
                        }
                    }
                );
                notification = RED.notify(notificationMessage, {
                    modal: true,
                    fixed: true,
                    buttons: notificationButtons
                });
                return;
            }
        }

        const nns = RED.nodes.createCompleteNodeSet();
        const startTime = Date.now();

        deployButtonSetBusy();
        const data = { flows: nns };
        if (!force) {
            data.rev = RED.nodes.version();
        }

        deployInflight = true;
        shadeShow();
        $.ajax({
            url: "flows",
            type: "POST",
            data: JSON.stringify(data),
            contentType: "application/json; charset=utf-8",
            headers: {
                "Node-RED-Deployment-Type": deploymentType
            }
        }).done(function (data, textStatus, xhr) {
            RED.nodes.dirty(false);
            RED.nodes.version(data.rev);
            RED.nodes.originalFlow(nns);
            if (hasUnusedConfig) {
                let notification;
                const opts = {
                    type: "success",
                    fixed: false,
                    timeout: 6000,
                    buttons: [
                        {
                            text: RED._("deploy.unusedConfigNodesButton"),
                            class: "pull-left",
                            click: function() {
                                notification.close();
                                RED.actions.invoke("core:search","is:config is:unused ");
                            }
                        },
                        {
                            text: RED._("common.label.close"),
                            class: "primary",
                            click: function () {
                                save(true);
                                notification.close();
                            }
                        }
                    ]
                }
                notification = RED.notify(
                    '<p>' + RED._("deploy.successfulDeploy") + '</p>' +
                    '<p>' + RED._("deploy.unusedConfigNodes") + '</p>', opts);
            } else {
                RED.notify('<p>' + RED._("deploy.successfulDeploy") + '</p>', "success");
            }
            const flowsToLock = new Set()
            // Node's properties cannot be modified if its workspace is locked.
            function ensureUnlocked(id) {
                // TODO: `RED.nodes.subflow` is useless
                const flow = id && (RED.nodes.workspace(id) || RED.nodes.subflow(id) || null);
                const isLocked = flow ? flow.locked : false;
                if (flow && isLocked) {
                    flow.locked = false;
                    flowsToLock.add(flow)
                }
            }
            RED.nodes.eachNode(function (node) {
                ensureUnlocked(node.z)
                if (node.changed) {
                    node.dirty = true;
                    node.changed = false;
                }
                if (node.moved) {
                    node.dirty = true;
                    node.moved = false;
                }
                if (node.credentials) {
                    delete node.credentials;
                }
            });
            RED.nodes.eachGroup(function (node) {
                ensureUnlocked(node.z)
                if (node.changed) {
                    node.dirty = true;
                    node.changed = false;
                }
                if (node.moved) {
                    node.dirty = true;
                    node.moved = false;
                }
            })
            RED.nodes.eachJunction(function (node) {
                ensureUnlocked(node.z)
                if (node.changed) {
                    node.dirty = true;
                    node.changed = false;
                }
                if (node.moved) {
                    node.dirty = true;
                    node.moved = false;
                }
            })
            RED.nodes.eachConfig(function (confNode) {
                if (confNode.z) {
                    ensureUnlocked(confNode.z)
                }
                confNode.changed = false;
                if (confNode.credentials) {
                    delete confNode.credentials;
                }
            });
            // Subflow cannot be locked
            RED.nodes.eachSubflow(function (subflow) {
                if (subflow.changed) {
                    subflow.changed = false;
                    RED.events.emit("subflows:change", subflow);
                }
            });
            RED.nodes.eachWorkspace(function (ws) {
                if (ws.changed || ws.added) {
                    // Ensure the Workspace is unlocked to modify its properties.
                    ensureUnlocked(ws.id);
                    ws.changed = false;
                    delete ws.added
                    if (flowsToLock.has(ws)) {
                        ws.locked = true;
                        flowsToLock.delete(ws);
                    }
                    RED.events.emit("flows:change", ws)
                }
            });
            // Ensures all workspaces to be locked have been locked.
            flowsToLock.forEach(flow => {
                flow.locked = true
            })
            // Once deployed, cannot undo back to a clean state
            RED.history.markAllDirty();
            RED.view.redraw();
            RED.sidebar.config.refresh();
            RED.events.emit("deploy");
        }).fail(function (xhr, textStatus, err) {
            RED.nodes.dirty(true);
            $("#red-ui-header-button-deploy").removeClass("disabled");
            if (xhr.status === 401) {
                RED.notify(RED._("deploy.deployFailed", { message: RED._("user.notAuthorized") }), "error");
            } else if (xhr.status === 409) {
                resolveConflict(nns, true);
            } else if (xhr.responseText) {
                RED.notify(RED._("deploy.deployFailed", { message: xhr.responseText }), "error");
            } else {
                RED.notify(RED._("deploy.deployFailed", { message: RED._("deploy.errors.noResponse") }), "error");
            }
        }).always(function () {
            const delta = Math.max(0, 300 - (Date.now() - startTime));
            setTimeout(function () {
                deployInflight = false;
                deployButtonClearBusy()
                shadeHide()
            }, delta);
        });
    }
    return {
        init: init,
        setDeployInflight: function(state) {
            deployInflight = state;
        }

    }
})();
;
RED.diagnostics = (function () {

    function init() {
        if (RED.settings.get('diagnostics.ui', true) === false) {
            return;
        }
        RED.actions.add("core:show-system-info", function () { show(); });
    }

    function show() {
        $.ajax({
            headers: {
                "Accept": "application/json"
            },
            cache: false,
            url: 'diagnostics',
            success: function (data) {
                var json = JSON.stringify(data || {}, "", 4);
                if (json === "{}") {
                    json = "{\n\n}";
                }
                RED.editor.editJSON({
                    title: RED._('diagnostics.title'),
                    value: json,
                    requireValid: true,
                    readOnly: true,
                    toolbarButtons: [
                        {
                            text: RED._('clipboard.export.copy'),
                            icon: 'fa fa-copy',
                            click: function () {
                                RED.clipboard.copyText(json, $(this), RED._('clipboard.copyMessageValue'))
                            }
                        },
                        {
                            text: RED._('clipboard.download'),
                            icon: 'fa fa-download',
                            click: function () {
                                var element = document.createElement('a');
                                element.setAttribute('href', 'data:text/plain;charset=utf-8,' + encodeURIComponent(json));
                                element.setAttribute('download', "system-info.json");
                                element.style.display = 'none';
                                document.body.appendChild(element);
                                element.click();
                                document.body.removeChild(element);
                            }
                        },
                    ]
                });
            },
            error: function (jqXHR, textStatus, errorThrown) {
                console.log("Unexpected error loading system info:", jqXHR.status, textStatus, errorThrown);
            }
        });
    }

    return {
        init: init,
    };
})();
;RED.diff = (function() {
    var currentDiff = {};
    var diffVisible = false;
    var diffList;

    function init() {

        // RED.actions.add("core:show-current-diff",showLocalDiff);
        RED.actions.add("core:show-remote-diff",showRemoteDiff);
        // RED.keyboard.add("*","ctrl-shift-l","core:show-current-diff");
        // RED.keyboard.add("*","ctrl-shift-r","core:show-remote-diff");


        // RED.actions.add("core:show-test-flow-diff-1",function(){showTestFlowDiff(1)});
        // RED.keyboard.add("*","ctrl-shift-f 1","core:show-test-flow-diff-1");
        //
        // RED.actions.add("core:show-test-flow-diff-2",function(){showTestFlowDiff(2)});
        // RED.keyboard.add("*","ctrl-shift-f 2","core:show-test-flow-diff-2");
        // RED.actions.add("core:show-test-flow-diff-3",function(){showTestFlowDiff(3)});
        // RED.keyboard.add("*","ctrl-shift-f 3","core:show-test-flow-diff-3");

    }
    function createDiffTable(container,CurrentDiff) {
        var diffList = $('<ol class="red-ui-diff-list"></ol>').appendTo(container);
        diffList.editableList({
            addButton: false,
            height: "auto",
            scrollOnAdd: false,
            addItem: function(container,i,object) {
                var localDiff = object.diff;
                var remoteDiff = object.remoteDiff;
                var tab = object.tab.n;
                var def = object.def;
                var conflicts = CurrentDiff.conflicts;

                var tabDiv = $('<div>',{class:"red-ui-diff-list-flow"}).appendTo(container);
                tabDiv.addClass('collapsed');
                var titleRow = $('<div>',{class:"red-ui-diff-list-flow-title"}).appendTo(tabDiv);
                var nodesDiv = $('<div>').appendTo(tabDiv);
                var originalCell = $('<div>',{class:"red-ui-diff-list-node-cell"}).appendTo(titleRow);
                var localCell = $('<div>',{class:"red-ui-diff-list-node-cell red-ui-diff-list-node-local"}).appendTo(titleRow);
                var remoteCell;
                var selectState;

                if (remoteDiff) {
                    remoteCell = $('<div>',{class:"red-ui-diff-list-node-cell red-ui-diff-list-node-remote"}).appendTo(titleRow);
                }
                $('<span class="red-ui-diff-list-chevron"><i class="fa fa-angle-down"></i></span>').appendTo(originalCell);
                createNodeIcon(tab,def).appendTo(originalCell);
                var tabForLabel = (object.newTab || object.tab).n;
                var titleSpan = $('<span>',{class:"red-ui-diff-list-flow-title-meta"}).appendTo(originalCell);
                if (tabForLabel.type === 'tab') {
                    titleSpan.text(tabForLabel.label||tabForLabel.id);
                } else if (tab.type === 'subflow') {
                    titleSpan.text((tabForLabel.name||tabForLabel.id));
                } else {
                    titleSpan.text(RED._("diff.globalNodes"));
                }
                var flowStats = {
                    local: {
                        addedCount:0,
                        deletedCount:0,
                        changedCount:0,
                        movedCount:0,
                        unchangedCount: 0
                    },
                    remote: {
                        addedCount:0,
                        deletedCount:0,
                        changedCount:0,
                        movedCount:0,
                        unchangedCount: 0
                    },
                    conflicts: 0
                }
                if (object.newTab || object.remoteTab) {
                    var localTabNode = {
                        node: localDiff.newConfig.all[tab.id],
                        all: localDiff.newConfig.all,
                        diff: localDiff
                    }
                    var remoteTabNode;
                    if (remoteDiff) {
                        remoteTabNode = {
                            node:remoteDiff.newConfig.all[tab.id]||null,
                            all: remoteDiff.newConfig.all,
                            diff: remoteDiff
                        }
                    }
                    if (tab.type !== undefined) {
                        var div = $("<div>",{class:"red-ui-diff-list-node red-ui-diff-list-node-props collapsed"}).appendTo(nodesDiv);
                        var row = $("<div>",{class:"red-ui-diff-list-node-header"}).appendTo(div);
                        var originalNodeDiv = $("<div>",{class:"red-ui-diff-list-node-cell"}).appendTo(row);
                        var localNodeDiv = $("<div>",{class:"red-ui-diff-list-node-cell red-ui-diff-list-node-local"}).appendTo(row);
                        var localChanged = false;
                        var remoteChanged = false;

                        if (!localDiff.newConfig.all[tab.id]) {
                            localNodeDiv.addClass("red-ui-diff-empty");
                        } else if (localDiff.added[tab.id]) {
                            localNodeDiv.addClass("red-ui-diff-status-added");
                            localChanged = true;
                            $('<span class="red-ui-diff-status"><i class="fa fa-plus-square"></i> <span data-i18n="diff.type.added"></span></span>').appendTo(localNodeDiv);
                        } else if (localDiff.changed[tab.id]) {
                            localNodeDiv.addClass("red-ui-diff-status-changed");
                            localChanged = true;
                            $('<span class="red-ui-diff-status"><i class="fa fa-square"></i> <span data-i18n="diff.type.changed"></span></span>').appendTo(localNodeDiv);
                        } else {
                            localNodeDiv.addClass("red-ui-diff-status-unchanged");
                            $('<span class="red-ui-diff-status"><i class="fa fa-square-o"></i> <span data-i18n="diff.type.unchanged"></span></span>').appendTo(localNodeDiv);
                        }

                        var remoteNodeDiv;
                        if (remoteDiff) {
                            remoteNodeDiv = $("<div>",{class:"red-ui-diff-list-node-cell red-ui-diff-list-node-remote"}).appendTo(row);
                            if (!remoteDiff.newConfig.all[tab.id]) {
                                remoteNodeDiv.addClass("red-ui-diff-empty");
                                if (remoteDiff.deleted[tab.id]) {
                                    remoteChanged = true;
                                }
                            } else if (remoteDiff.added[tab.id]) {
                                remoteNodeDiv.addClass("red-ui-diff-status-added");
                                remoteChanged = true;
                                $('<span class="red-ui-diff-status"><i class="fa fa-plus-square"></i> <span data-i18n="diff.type.added"></span></span>').appendTo(remoteNodeDiv);
                            } else if (remoteDiff.changed[tab.id]) {
                                remoteNodeDiv.addClass("red-ui-diff-status-changed");
                                remoteChanged = true;
                                $('<span class="red-ui-diff-status"><i class="fa fa-square"></i> <span data-i18n="diff.type.changed"></span></span>').appendTo(remoteNodeDiv);
                            } else {
                                remoteNodeDiv.addClass("red-ui-diff-status-unchanged");
                                $('<span class="red-ui-diff-status"><i class="fa fa-square-o"></i> <span data-i18n="diff.type.unchanged"></span></span>').appendTo(remoteNodeDiv);
                            }
                        }
                        $('<span class="red-ui-diff-list-chevron"><i class="fa fa-angle-down"></i></span>').appendTo(originalNodeDiv);
                        $('<span>').text(RED._("diff.flowProperties")).appendTo(originalNodeDiv);

                        row.on("click", function(evt) {
                            evt.preventDefault();
                            $(this).parent().toggleClass('collapsed');
                        });

                        createNodePropertiesTable(def,tab,localTabNode,remoteTabNode).appendTo(div);
                        selectState = "";
                        if (conflicts[tab.id]) {
                            flowStats.conflicts++;

                            if (!localNodeDiv.hasClass("red-ui-diff-empty")) {
                                $('<span class="red-ui-diff-status-conflict"><span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span></span>').prependTo(localNodeDiv);
                            }
                            if (!remoteNodeDiv.hasClass("red-ui-diff-empty")) {
                                $('<span class="red-ui-diff-status-conflict"><span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span></span>').prependTo(remoteNodeDiv);
                            }
                            div.addClass("red-ui-diff-list-node-conflict");
                        } else {
                            selectState = CurrentDiff.resolutions[tab.id];
                        }
                        // Tab properties row
                        createNodeConflictRadioBoxes(tab,div,localNodeDiv,remoteNodeDiv,true,!conflicts[tab.id],selectState,CurrentDiff);
                    }
                }
                // var stats = $('<span>',{class:"red-ui-diff-list-flow-stats"}).appendTo(titleRow);
                var localNodeCount = 0;
                var remoteNodeCount = 0;
                var seen = {};
                object.tab.nodes.forEach(function(node) {
                    seen[node.id] = true;
                    createNodeDiffRow(node,flowStats,CurrentDiff).appendTo(nodesDiv)
                });
                if (object.newTab) {
                    localNodeCount = object.newTab.nodes.length;
                    object.newTab.nodes.forEach(function(node) {
                        if (!seen[node.id]) {
                            seen[node.id] = true;
                            createNodeDiffRow(node,flowStats,CurrentDiff).appendTo(nodesDiv)
                        }
                    });
                }
                if (object.remoteTab) {
                    remoteNodeCount = object.remoteTab.nodes.length;
                    object.remoteTab.nodes.forEach(function(node) {
                        if (!seen[node.id]) {
                            createNodeDiffRow(node,flowStats,CurrentDiff).appendTo(nodesDiv)
                        }
                    });
                }
                titleRow.on("click", function(evt) {
                    // if (titleRow.parent().find(".red-ui-diff-list-node:not(.hide)").length > 0) {
                    titleRow.parent().toggleClass('collapsed');
                    if ($(this).parent().hasClass('collapsed')) {
                        $(this).parent().find('.red-ui-diff-list-node').addClass('collapsed');
                        $(this).parent().find('.red-ui-debug-msg-element').addClass('collapsed');
                    }
                    // }
                })

                if (localDiff.deleted[tab.id]) {
                    $('<span class="red-ui-diff-status-deleted"><span class="red-ui-diff-status"><i class="fa fa-minus-square"></i> <span data-i18n="diff.type.flowDeleted"></span></span></span>').appendTo(localCell);
                } else if (object.newTab) {
                    if (localDiff.added[tab.id]) {
                        $('<span class="red-ui-diff-status-added"><span class="red-ui-diff-status"><i class="fa fa-plus-square"></i> <span data-i18n="diff.type.flowAdded"></span></span></span>').appendTo(localCell);
                    } else {
                        if (tab.id) {
                            if (localDiff.changed[tab.id]) {
                                flowStats.local.changedCount++;
                            } else {
                                flowStats.local.unchangedCount++;
                            }
                        }
                        var localStats = $('<span>',{class:"red-ui-diff-list-flow-stats"}).appendTo(localCell);
                        $('<span class="red-ui-diff-status"></span>').text(RED._('diff.nodeCount',{count:localNodeCount})).appendTo(localStats);

                        if (flowStats.conflicts + flowStats.local.addedCount + flowStats.local.changedCount + flowStats.local.movedCount + flowStats.local.deletedCount > 0) {
                            $('<span class="red-ui-diff-status"> [ </span>').appendTo(localStats);
                            if (flowStats.conflicts > 0) {
                                $('<span class="red-ui-diff-status-conflict"><span class="red-ui-diff-status"><i class="fa fa-exclamation"></i> '+flowStats.conflicts+'</span></span>').appendTo(localStats);
                            }
                            if (flowStats.local.addedCount > 0) {
                                const cell = $('<span class="red-ui-diff-status-added"><span class="red-ui-diff-status"><i class="fa fa-plus-square"></i> '+flowStats.local.addedCount+'</span></span>').appendTo(localStats);
                                RED.popover.tooltip(cell, RED._('diff.type.added'))
                            }
                            if (flowStats.local.changedCount > 0) {
                                const cell = $('<span class="red-ui-diff-status-changed"><span class="red-ui-diff-status"><i class="fa fa-square"></i> '+flowStats.local.changedCount+'</span></span>').appendTo(localStats);
                                RED.popover.tooltip(cell, RED._('diff.type.changed'))
                            }
                            if (flowStats.local.movedCount > 0) {
                                const cell = $('<span class="red-ui-diff-status-moved"><span class="red-ui-diff-status"><i class="fa fa-square"></i> '+flowStats.local.movedCount+'</span></span>').appendTo(localStats);
                                RED.popover.tooltip(cell, RED._('diff.type.moved'))
                            }
                            if (flowStats.local.deletedCount > 0) {
                                const cell = $('<span class="red-ui-diff-status-deleted"><span class="red-ui-diff-status"><i class="fa fa-minus-square"></i> '+flowStats.local.deletedCount+'</span></span>').appendTo(localStats);
                                RED.popover.tooltip(cell, RED._('diff.type.deleted'))
                            }
                            $('<span class="red-ui-diff-status"> ] </span>').appendTo(localStats);
                        }

                    }
                } else {
                    localCell.addClass("red-ui-diff-empty");
                }

                if (remoteDiff) {
                    if (remoteDiff.deleted[tab.id]) {
                        $('<span class="red-ui-diff-status-deleted"><span class="red-ui-diff-status"><i class="fa fa-minus-square"></i> <span data-i18n="diff.type.flowDeleted"></span></span></span>').appendTo(remoteCell);
                    } else if (object.remoteTab) {
                        if (remoteDiff.added[tab.id]) {
                            $('<span class="red-ui-diff-status-added"><span class="red-ui-diff-status"><i class="fa fa-plus-square"></i> <span data-i18n="diff.type.flowAdded"></span></span></span>').appendTo(remoteCell);
                        } else {
                            if (tab.id) {
                                if (remoteDiff.changed[tab.id]) {
                                    flowStats.remote.changedCount++;
                                } else {
                                    flowStats.remote.unchangedCount++;
                                }
                            }
                            var remoteStats = $('<span>',{class:"red-ui-diff-list-flow-stats"}).appendTo(remoteCell);
                            $('<span class="red-ui-diff-status"></span>').text(RED._('diff.nodeCount',{count:remoteNodeCount})).appendTo(remoteStats);
                            if (flowStats.conflicts + flowStats.remote.addedCount + flowStats.remote.changedCount + flowStats.remote.movedCount + flowStats.remote.deletedCount > 0) {
                                $('<span class="red-ui-diff-status"> [ </span>').appendTo(remoteStats);
                                if (flowStats.conflicts > 0) {
                                    $('<span class="red-ui-diff-status-conflict"><span class="red-ui-diff-status"><i class="fa fa-exclamation"></i> '+flowStats.conflicts+'</span></span>').appendTo(remoteStats);
                                }
                                if (flowStats.remote.addedCount > 0) {
                                    const cell = $('<span class="red-ui-diff-status-added"><span class="red-ui-diff-status"><i class="fa fa-plus-square"></i> '+flowStats.remote.addedCount+'</span></span>').appendTo(remoteStats);
                                    RED.popover.tooltip(cell, RED._('diff.type.added'))
                                }
                                if (flowStats.remote.changedCount > 0) {
                                    const cell = $('<span class="red-ui-diff-status-changed"><span class="red-ui-diff-status"><i class="fa fa-square"></i> '+flowStats.remote.changedCount+'</span></span>').appendTo(remoteStats);
                                    RED.popover.tooltip(cell, RED._('diff.type.changed'))
                                }
                                if (flowStats.remote.movedCount > 0) {
                                    const cell = $('<span class="red-ui-diff-status-moved"><span class="red-ui-diff-status"><i class="fa fa-square"></i> '+flowStats.remote.movedCount+'</span></span>').appendTo(remoteStats);
                                    RED.popover.tooltip(cell, RED._('diff.type.moved'))
                                }
                                if (flowStats.remote.deletedCount > 0) {
                                    const cell = $('<span class="red-ui-diff-status-deleted"><span class="red-ui-diff-status"><i class="fa fa-minus-square"></i> '+flowStats.remote.deletedCount+'</span></span>').appendTo(remoteStats);
                                    RED.popover.tooltip(cell, RED._('diff.type.deleted'))
                                }
                                $('<span class="red-ui-diff-status"> ] </span>').appendTo(remoteStats);
                            }
                        }
                    } else {
                        remoteCell.addClass("red-ui-diff-empty");
                    }
                    selectState = "";
                    if (flowStats.conflicts > 0) {
                        titleRow.addClass("red-ui-diff-list-node-conflict");
                    } else {
                        selectState = CurrentDiff.resolutions[tab.id];
                    }
                    if (tab.id) {
                        var hide = !(flowStats.conflicts > 0 &&(localDiff.deleted[tab.id] || remoteDiff.deleted[tab.id]));
                        // Tab parent row
                        createNodeConflictRadioBoxes(tab,titleRow,localCell,remoteCell, false, hide, selectState, CurrentDiff);
                    }
                }

                if (tabDiv.find(".red-ui-diff-list-node").length === 0) {
                    tabDiv.addClass("red-ui-diff-list-flow-empty");
                }
                container.i18n();
            }
        });
        return diffList;
    }
    function buildDiffPanel(container,diff,options) {
        var diffPanel = $('<div class="red-ui-diff-panel"></div>').appendTo(container);
        var diffHeaders = $('<div class="red-ui-diff-panel-headers"></div>').appendTo(diffPanel);
        if (options.mode === "merge") {
            diffPanel.addClass("red-ui-diff-panel-merge");
        }
        var diffList = createDiffTable(diffPanel, diff, options);

        var localDiff = diff.localDiff;
        var remoteDiff = diff.remoteDiff;
        var conflicts = diff.conflicts;

        var currentConfig = localDiff.currentConfig;
        var newConfig = localDiff.newConfig;


        if (remoteDiff !== undefined) {
            diffPanel.addClass('red-ui-diff-three-way');
            var localTitle = options.oldRevTitle || RED._('diff.local');
            var remoteTitle = options.newRevTitle || RED._('diff.remote');
            $('<div></div>').text(localTitle).appendTo(diffHeaders);
            $('<div></div>').text(remoteTitle).appendTo(diffHeaders);
        } else {
            diffPanel.removeClass('red-ui-diff-three-way');
        }

        return {
            list: diffList,
            finish: function() {
                var el = {
                    diff: localDiff,
                    def: {
                        category: 'config',
                        color: '#f0f0f0'
                    },
                    tab: {
                        n: {},
                        nodes: currentConfig.globals
                    },
                    newTab: {
                        n: {},
                        nodes: newConfig.globals
                    }
                };
                if (remoteDiff !== undefined) {
                    el.remoteTab = {
                        n:{},
                        nodes:remoteDiff.newConfig.globals
                    };
                    el.remoteDiff = remoteDiff;
                }
                diffList.editableList('addItem',el);

                var seenTabs = {};

                currentConfig.tabOrder.forEach(function(tabId) {
                    var tab = currentConfig.tabs[tabId];
                    var el = {
                        diff: localDiff,
                        def: RED.nodes.getType('tab'),
                        tab:tab
                    };
                    if (newConfig.tabs.hasOwnProperty(tabId)) {
                        el.newTab = newConfig.tabs[tabId];
                    }
                    if (remoteDiff !== undefined) {
                        el.remoteTab = remoteDiff.newConfig.tabs[tabId];
                        el.remoteDiff = remoteDiff;
                    }
                    seenTabs[tabId] = true;
                    diffList.editableList('addItem',el)
                });
                newConfig.tabOrder.forEach(function(tabId) {
                    if (!seenTabs[tabId]) {
                        seenTabs[tabId] = true;
                        var tab = newConfig.tabs[tabId];
                        var el = {
                            diff: localDiff,
                            def: RED.nodes.getType('tab'),
                            tab:tab,
                            newTab: tab
                        };
                        if (remoteDiff !== undefined) {
                            el.remoteDiff = remoteDiff;
                        }
                        diffList.editableList('addItem',el)
                    }
                });
                if (remoteDiff !== undefined) {
                    remoteDiff.newConfig.tabOrder.forEach(function(tabId) {
                        if (!seenTabs[tabId]) {
                            var tab = remoteDiff.newConfig.tabs[tabId];
                            // TODO how to recognise this is a remotely added flow
                            var el = {
                                diff: localDiff,
                                remoteDiff: remoteDiff,
                                def: RED.nodes.getType('tab'),
                                tab:tab,
                                remoteTab:tab
                            };
                            diffList.editableList('addItem',el)
                        }
                    });
                }
                var subflowId;
                for (subflowId in currentConfig.subflows) {
                    if (currentConfig.subflows.hasOwnProperty(subflowId)) {
                        seenTabs[subflowId] = true;
                        el = {
                            diff: localDiff,
                            def: {
                                defaults:{},
                                icon:"subflow.svg",
                                category: "subflows",
                                color: "#DDAA99"
                            },
                            tab:currentConfig.subflows[subflowId]
                        }
                        if (newConfig.subflows.hasOwnProperty(subflowId)) {
                            el.newTab = newConfig.subflows[subflowId];
                        }
                        if (remoteDiff !== undefined) {
                            el.remoteTab = remoteDiff.newConfig.subflows[subflowId];
                            el.remoteDiff = remoteDiff;
                        }
                        diffList.editableList('addItem',el)
                    }
                }
                for (subflowId in newConfig.subflows) {
                    if (newConfig.subflows.hasOwnProperty(subflowId) && !seenTabs[subflowId]) {
                        seenTabs[subflowId] = true;
                        el = {
                            diff: localDiff,
                            def: {
                                defaults:{},
                                icon:"subflow.svg",
                                category: "subflows",
                                color: "#DDAA99"
                            },
                            tab:newConfig.subflows[subflowId],
                            newTab:newConfig.subflows[subflowId]
                        }
                        if (remoteDiff !== undefined) {
                            el.remoteDiff = remoteDiff;
                        }
                        diffList.editableList('addItem',el)
                    }
                }
                if (remoteDiff !== undefined) {
                    for (subflowId in remoteDiff.newConfig.subflows) {
                        if (remoteDiff.newConfig.subflows.hasOwnProperty(subflowId) && !seenTabs[subflowId]) {
                            el = {
                                diff: localDiff,
                                remoteDiff: remoteDiff,
                                def: {
                                    defaults:{},
                                    icon:"subflow.svg",
                                    category: "subflows",
                                    color: "#DDAA99"
                                },
                                tab:remoteDiff.newConfig.subflows[subflowId],
                                remoteTab: remoteDiff.newConfig.subflows[subflowId]
                            }
                            diffList.editableList('addItem',el)
                        }
                    }
                }
            }
        };
    }
    function formatWireProperty(wires,allNodes) {
        var result = $("<div>",{class:"red-ui-diff-list-wires"})
        var list = $("<ol></ol>");
        var c = 0;
        wires.forEach(function(p,i) {
            var port = $("<li>").appendTo(list);
            if (p && p.length > 0) {
                $("<span>").text(i+1).appendTo(port);
                var links = $("<ul>").appendTo(port);
                p.forEach(function(d) {
                    c++;
                    var entry = $("<li>").appendTo(links);
                    var node = allNodes[d];
                    if (node) {
                        var def = RED.nodes.getType(node.type)||{};
                        createNode(node,def).appendTo(entry);
                    } else {
                        entry.text(d);
                    }
                })
            } else {
                port.text('none');
            }
        })
        if (c === 0) {
            result.text(RED._("diff.type.none"));
        } else {
            list.appendTo(result);
        }
        return result;
    }
    function createNodeIcon(node,def) {
        var nodeDiv = $("<div>",{class:"red-ui-diff-list-node-icon"});
        var colour = RED.utils.getNodeColor(node.type,def);
        var icon_url = RED.utils.getNodeIcon(def,node);
        if (node.type === 'tab') {
            colour = "#C0DEED";
        }
        nodeDiv.css('backgroundColor',colour);

        var iconContainer = $('<div/>',{class:"red-ui-palette-icon-container"}).appendTo(nodeDiv);
        RED.utils.createIconElement(icon_url, iconContainer, false);

        return nodeDiv;
    }
    function createNode(node,def) {
        var nodeTitleDiv = $("<div>",{class:"red-ui-diff-list-node-title"})
        createNodeIcon(node,def).appendTo(nodeTitleDiv);
        var nodeLabel = node.label || node.name || node.id;
        $('<div>',{class:"red-ui-diff-list-node-description"}).text(nodeLabel).appendTo(nodeTitleDiv);
        return nodeTitleDiv;
    }
    function createNodeDiffRow(node,stats,CurrentDiff) {
        var localDiff = CurrentDiff.localDiff;
        var remoteDiff = CurrentDiff.remoteDiff;
        var conflicted = CurrentDiff.conflicts[node.id];

        var hasChanges = false; // exists in original and local/remote but with changes
        var unChanged = true; // existing in original,local,remote unchanged

        if (localDiff.added[node.id]) {
            stats.local.addedCount++;
            unChanged = false;
        }
        if (remoteDiff && remoteDiff.added[node.id]) {
            stats.remote.addedCount++;
            unChanged = false;
        }
        if (localDiff.deleted[node.id]) {
            stats.local.deletedCount++;
            unChanged = false;
        }
        if (remoteDiff && remoteDiff.deleted[node.id]) {
            stats.remote.deletedCount++;
            unChanged = false;
        }
        if (localDiff.changed[node.id]) {
            if (localDiff.positionChanged[node.id]) {
                stats.local.movedCount++
            } else {
                stats.local.changedCount++;
            }
            hasChanges = true;
            unChanged = false;
        }
        if (remoteDiff && remoteDiff.changed[node.id]) {
            if (remoteDiff.positionChanged[node.id]) {
                stats.remote.movedCount++
            } else {
                stats.remote.changedCount++;
            }
            hasChanges = true;
            unChanged = false;
        }
        // console.log(node.id,localDiff.added[node.id],remoteDiff.added[node.id],localDiff.deleted[node.id],remoteDiff.deleted[node.id],localDiff.changed[node.id],remoteDiff.changed[node.id])
        var def = RED.nodes.getType(node.type);
        if (def === undefined) {
            if (/^subflow:/.test(node.type)) {
                def = {
                    icon:"subflow.svg",
                    category: "subflows",
                    color: "#DDAA99",
                    defaults:{name:{value:""}}
                }
            } else if (node.type === "group") {
                def = RED.group.def;
            } else {
                def = {};
            }
        }
        var div = $("<div>",{class:"red-ui-diff-list-node collapsed"});
        var row = $("<div>",{class:"red-ui-diff-list-node-header"}).appendTo(div);

        var originalNodeDiv = $("<div>",{class:"red-ui-diff-list-node-cell"}).appendTo(row);
        var localNodeDiv = $("<div>",{class:"red-ui-diff-list-node-cell red-ui-diff-list-node-local"}).appendTo(row);
        var remoteNodeDiv;
        var chevron;
        if (remoteDiff) {
            remoteNodeDiv = $("<div>",{class:"red-ui-diff-list-node-cell red-ui-diff-list-node-remote"}).appendTo(row);
        }
        $('<span class="red-ui-diff-list-chevron"><i class="fa fa-angle-down"></i></span>').appendTo(originalNodeDiv);

        if (unChanged) {
            stats.local.unchangedCount++;
            createNode(node,def).appendTo(originalNodeDiv);
            localNodeDiv.addClass("red-ui-diff-status-unchanged");
            $('<span class="red-ui-diff-status"><i class="fa fa-square-o"></i> <span data-i18n="diff.type.unchanged"></span></span>').appendTo(localNodeDiv);
            if (remoteDiff) {
                stats.remote.unchangedCount++;
                remoteNodeDiv.addClass("red-ui-diff-status-unchanged");
                $('<span class="red-ui-diff-status"><i class="fa fa-square-o"></i> <span data-i18n="diff.type.unchanged"></span></span>').appendTo(remoteNodeDiv);
            }
            div.addClass("red-ui-diff-status-unchanged");
        } else if (localDiff.added[node.id]) {
            localNodeDiv.addClass("red-ui-diff-status-added");
            if (remoteNodeDiv) {
                remoteNodeDiv.addClass("red-ui-diff-empty");
            }
            $('<span class="red-ui-diff-status"><i class="fa fa-plus-square"></i> <span data-i18n="diff.type.added"></span></span>').appendTo(localNodeDiv);
            createNode(node,def).appendTo(originalNodeDiv);
        } else if (remoteDiff && remoteDiff.added[node.id]) {
            localNodeDiv.addClass("red-ui-diff-empty");
            remoteNodeDiv.addClass("red-ui-diff-status-added");
            $('<span class="red-ui-diff-status"><i class="fa fa-plus-square"></i> <span data-i18n="diff.type.added"></span></span>').appendTo(remoteNodeDiv);
            createNode(node,def).appendTo(originalNodeDiv);
        } else {
            createNode(node,def).appendTo(originalNodeDiv);
            if (localDiff.moved[node.id]) {
                var localN = localDiff.newConfig.all[node.id];
                if (!localDiff.deleted[node.z] && node.z !== localN.z && node.z !== "" && !localDiff.newConfig.all[node.z]) {
                    localNodeDiv.addClass("red-ui-diff-empty");
                } else {
                    localNodeDiv.addClass("red-ui-diff-status-moved");
                    var localMovedMessage = "";
                    if (node.z === localN.z) {
                        const movedFromNodeTab = localDiff.currentConfig.all[localDiff.currentConfig.all[node.id].z]
                        const movedFromLabel = `'${movedFromNodeTab ? (movedFromNodeTab.label || movedFromNodeTab.id) : 'global'}'`
                        localMovedMessage = RED._("diff.type.movedFrom",{id: movedFromLabel});
                    } else {
                        const movedToNodeTab = localDiff.newConfig.all[localN.z]
                        const movedToLabel = `'${movedToNodeTab ? (movedToNodeTab.label || movedToNodeTab.id) : 'global'}'`
                        localMovedMessage = RED._("diff.type.movedTo",{id:movedToLabel});
                    }
                    $('<span class="red-ui-diff-status"><i class="fa fa-caret-square-o-right"></i> '+localMovedMessage+'</span>').appendTo(localNodeDiv);
                }
            } else if (localDiff.deleted[node.z]) {
                localNodeDiv.addClass("red-ui-diff-empty");
            } else if (localDiff.deleted[node.id]) {
                localNodeDiv.addClass("red-ui-diff-status-deleted");
                $('<span class="red-ui-diff-status"><i class="fa fa-minus-square"></i> <span data-i18n="diff.type.deleted"></span></span>').appendTo(localNodeDiv);
            } else if (localDiff.changed[node.id]) {
                if (localDiff.newConfig.all[node.id].z !== node.z) {
                    localNodeDiv.addClass("red-ui-diff-empty");
                } else {
                    if (localDiff.positionChanged[node.id]) {
                        localNodeDiv.addClass("red-ui-diff-status-moved");
                        $('<span class="red-ui-diff-status"><i class="fa fa-square"></i> <span data-i18n="diff.type.moved"></span></span>').appendTo(localNodeDiv);
                    } else {
                        localNodeDiv.addClass("red-ui-diff-status-changed");
                        $('<span class="red-ui-diff-status"><i class="fa fa-square"></i> <span data-i18n="diff.type.changed"></span></span>').appendTo(localNodeDiv);
                    }
                }
            } else {
                if (localDiff.newConfig.all[node.id].z !== node.z) {
                    localNodeDiv.addClass("red-ui-diff-empty");
                } else {
                    stats.local.unchangedCount++;
                    localNodeDiv.addClass("red-ui-diff-status-unchanged");
                    $('<span class="red-ui-diff-status"><i class="fa fa-square-o"></i> <span data-i18n="diff.type.unchanged"></span></span>').appendTo(localNodeDiv);
                }
            }

            if (remoteDiff) {
                if (remoteDiff.moved[node.id]) {
                    var remoteN = remoteDiff.newConfig.all[node.id];
                    if (!remoteDiff.deleted[node.z] && node.z !== remoteN.z && node.z !== "" && !remoteDiff.newConfig.all[node.z]) {
                        remoteNodeDiv.addClass("red-ui-diff-empty");
                    } else {
                        remoteNodeDiv.addClass("red-ui-diff-status-moved");
                        var remoteMovedMessage = "";
                        if (node.z === remoteN.z) {
                            const movedFromNodeTab = remoteDiff.currentConfig.all[remoteDiff.currentConfig.all[node.id].z]
                            const movedFromLabel = `'${movedFromNodeTab ? (movedFromNodeTab.label || movedFromNodeTab.id) : 'global'}'`
                            remoteMovedMessage = RED._("diff.type.movedFrom",{id:movedFromLabel});
                        } else {
                            const movedToNodeTab = remoteDiff.newConfig.all[remoteN.z]
                            const movedToLabel = `'${movedToNodeTab ? (movedToNodeTab.label || movedToNodeTab.id) : 'global'}'`
                            remoteMovedMessage = RED._("diff.type.movedTo",{id:movedToLabel});
                        }
                        $('<span class="red-ui-diff-status"><i class="fa fa-caret-square-o-right"></i> '+remoteMovedMessage+'</span>').appendTo(remoteNodeDiv);
                    }
                } else if (remoteDiff.deleted[node.z]) {
                    remoteNodeDiv.addClass("red-ui-diff-empty");
                } else if (remoteDiff.deleted[node.id]) {
                    remoteNodeDiv.addClass("red-ui-diff-status-deleted");
                    $('<span class="red-ui-diff-status"><i class="fa fa-minus-square"></i> <span data-i18n="diff.type.deleted"></span></span>').appendTo(remoteNodeDiv);
                } else if (remoteDiff.changed[node.id]) {
                    if (remoteDiff.newConfig.all[node.id].z !== node.z) {
                        remoteNodeDiv.addClass("red-ui-diff-empty");
                    } else {
                        if (remoteDiff.positionChanged[node.id]) {
                            remoteNodeDiv.addClass("red-ui-diff-status-moved");
                            $('<span class="red-ui-diff-status"><i class="fa fa-square"></i> <span data-i18n="diff.type.moved"></span></span>').appendTo(remoteNodeDiv);
                        } else {
                            remoteNodeDiv.addClass("red-ui-diff-status-changed");
                            $('<span class="red-ui-diff-status"><i class="fa fa-square"></i> <span data-i18n="diff.type.changed"></span></span>').appendTo(remoteNodeDiv);
                        }
                    }
                } else {
                    if (remoteDiff.newConfig.all[node.id].z !== node.z) {
                        remoteNodeDiv.addClass("red-ui-diff-empty");
                    } else {
                        stats.remote.unchangedCount++;
                        remoteNodeDiv.addClass("red-ui-diff-status-unchanged");
                        $('<span class="red-ui-diff-status"><i class="fa fa-square-o"></i> <span data-i18n="diff.type.unchanged"></span></span>').appendTo(remoteNodeDiv);
                    }
                }
            }
        }
        var localNode = {
            node: localDiff.newConfig.all[node.id],
            all: localDiff.newConfig.all,
            diff: localDiff
        };
        var remoteNode;
        if (remoteDiff) {
            remoteNode = {
                node:remoteDiff.newConfig.all[node.id]||null,
                all: remoteDiff.newConfig.all,
                diff: remoteDiff
            }
        }

        var selectState = "";

        if (conflicted) {
            stats.conflicts++;
            if (!localNodeDiv.hasClass("red-ui-diff-empty")) {
                $('<span class="red-ui-diff-status-conflict"><span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span></span>').prependTo(localNodeDiv);
            }
            if (!remoteNodeDiv.hasClass("red-ui-diff-empty")) {
                $('<span class="red-ui-diff-status-conflict"><span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span></span>').prependTo(remoteNodeDiv);
            }
            div.addClass("red-ui-diff-list-node-conflict");
        } else {
            selectState = CurrentDiff.resolutions[node.id];
        }
        // Node row
        createNodeConflictRadioBoxes(node,div,localNodeDiv,remoteNodeDiv,false,!conflicted,selectState,CurrentDiff);
        row.on("click", function(evt) {
            $(this).parent().toggleClass('collapsed');

            if($(this).siblings('.red-ui-diff-list-node-properties').length === 0) {
                createNodePropertiesTable(def,node,localNode,remoteNode).appendTo(div);
            }
        });

        return div;
    }
    function createNodePropertiesTable(def,node,localNodeObj,remoteNodeObj) {
        var propertyElements = {};
        var localNode = localNodeObj.node;
        var remoteNode;
        if (remoteNodeObj) {
            remoteNode = remoteNodeObj.node;
        }

        var nodePropertiesDiv = $("<div>",{class:"red-ui-diff-list-node-properties"});
        var nodePropertiesTable = $("<table>").appendTo(nodePropertiesDiv);
        var nodePropertiesTableCols = $('<colgroup><col/><col/></colgroup>').appendTo(nodePropertiesTable);
        if (remoteNode !== undefined) {
            $("<col/>").appendTo(nodePropertiesTableCols);
        }
        var nodePropertiesTableBody = $("<tbody>").appendTo(nodePropertiesTable);

        var row;
        var localCell, remoteCell;
        var element;
        var currentValue, localValue, remoteValue;
        var localChanged = false;
        var remoteChanged = false;
        var localChanges = 0;
        var remoteChanges = 0;
        var conflict = false;
        var status;

        row = $("<tr>").appendTo(nodePropertiesTableBody);
        $("<td>",{class:"red-ui-diff-list-cell-label"}).text("id").appendTo(row);
        localCell = $("<td>",{class:"red-ui-diff-list-cell red-ui-diff-list-node-local"}).appendTo(row);
        if (localNode) {
            localCell.addClass("red-ui-diff-status-unchanged");
            $('<span class="red-ui-diff-status"></span>').appendTo(localCell);
            element = $('<span class="red-ui-diff-list-element"></span>').appendTo(localCell);
            propertyElements['local.id'] = RED.utils.createObjectElement(localNode.id).appendTo(element);
        } else {
            localCell.addClass("red-ui-diff-empty");
        }
        if (remoteNode !== undefined) {
            remoteCell = $("<td>",{class:"red-ui-diff-list-cell red-ui-diff-list-node-remote"}).appendTo(row);
            remoteCell.addClass("red-ui-diff-status-unchanged");
            if (remoteNode) {
                $('<span class="red-ui-diff-status"></span>').appendTo(remoteCell);
                element = $('<span class="red-ui-diff-list-element"></span>').appendTo(remoteCell);
                propertyElements['remote.id'] = RED.utils.createObjectElement(remoteNode.id).appendTo(element);
            } else {
                remoteCell.addClass("red-ui-diff-empty");
            }
        }

        if (node.hasOwnProperty('x')) {
            if (localNode) {
                if (localNode.x !== node.x || localNode.y !== node.y || localNode.w !== node.w || localNode.h !== node.h ) {
                    localChanged = true;
                    localChanges++;
                }
            }
            if (remoteNode) {
                if (remoteNode.x !== node.x || remoteNode.y !== node.y|| remoteNode.w !== node.w || remoteNode.h !== node.h) {
                    remoteChanged = true;
                    remoteChanges++;
                }
            }
            if ( (remoteChanged && localChanged && (localNode.x !== remoteNode.x || localNode.y !== remoteNode.y)) ||
                (!localChanged && remoteChanged && localNodeObj.diff.deleted[node.id]) ||
                (localChanged && !remoteChanged && remoteNodeObj.diff.deleted[node.id])
            ) {
                conflict = true;
            }
            row = $("<tr>").appendTo(nodePropertiesTableBody);
            $("<td>",{class:"red-ui-diff-list-cell-label"}).text(RED._("diff.type.position")).appendTo(row);
            localCell = $("<td>",{class:"red-ui-diff-list-cell red-ui-diff-list-node-local"}).appendTo(row);
            if (localNode) {
                localCell.addClass("red-ui-diff-status-"+(localChanged?"moved":"unchanged"));
                $('<span class="red-ui-diff-status">'+(localChanged?'<i class="fa fa-square"></i>':'')+'</span>').appendTo(localCell);
                element = $('<span class="red-ui-diff-list-element"></span>').appendTo(localCell);
                var localPosition = {x:localNode.x,y:localNode.y};
                if (localNode.hasOwnProperty('w')) {
                    localPosition.w = localNode.w;
                    localPosition.h = localNode.h;
                }
                propertyElements['local.position'] = RED.utils.createObjectElement(localPosition,
                    {
                        path: "position",
                        exposeApi: true,
                        ontoggle: function(path,state) {
                            if (propertyElements['remote.'+path]) {
                                propertyElements['remote.'+path].prop('expand')(path,state)
                            }
                        }
                    }
                ).appendTo(element);
            } else {
                localCell.addClass("red-ui-diff-empty");
            }

            if (remoteNode !== undefined) {
                remoteCell = $("<td>",{class:"red-ui-diff-list-cell red-ui-diff-list-node-remote"}).appendTo(row);
                remoteCell.addClass("red-ui-diff-status-"+(remoteChanged?"moved":"unchanged"));
                if (remoteNode) {
                    $('<span class="red-ui-diff-status">'+(remoteChanged?'<i class="fa fa-square"></i>':'')+'</span>').appendTo(remoteCell);
                    element = $('<span class="red-ui-diff-list-element"></span>').appendTo(remoteCell);
                    var remotePosition = {x:remoteNode.x,y:remoteNode.y};
                    if (remoteNode.hasOwnProperty('w')) {
                        remotePosition.w = remoteNode.w;
                        remotePosition.h = remoteNode.h;
                    }
                    propertyElements['remote.position'] = RED.utils.createObjectElement(remotePosition,
                        {
                            path: "position",
                            exposeApi: true,
                            ontoggle: function(path,state) {
                                if (propertyElements['local.'+path]) {
                                    propertyElements['local.'+path].prop('expand')(path,state);
                                }
                            }
                        }
                    ).appendTo(element);
                } else {
                    remoteCell.addClass("red-ui-diff-empty");
                }
            }
        }
        //
        localChanged = remoteChanged = conflict = false;
        if (node.hasOwnProperty('wires')) {
            currentValue = JSON.stringify(node.wires);
            if (localNode) {
                localValue = JSON.stringify(localNode.wires);
                if (currentValue !== localValue) {
                    localChanged = true;
                    localChanges++;
                }
            }
            if (remoteNode) {
                remoteValue = JSON.stringify(remoteNode.wires);
                if (currentValue !== remoteValue) {
                    remoteChanged = true;
                    remoteChanges++;
                }
            }
            if ( (remoteChanged && localChanged && (localValue !== remoteValue)) ||
                (!localChanged && remoteChanged && localNodeObj.diff.deleted[node.id]) ||
                (localChanged && !remoteChanged && remoteNodeObj.diff.deleted[node.id])
            ){
                conflict = true;
            }
            row = $("<tr>").appendTo(nodePropertiesTableBody);
            $("<td>",{class:"red-ui-diff-list-cell-label"}).text(RED._("diff.type.wires")).appendTo(row);
            localCell = $("<td>",{class:"red-ui-diff-list-cell red-ui-diff-list-node-local"}).appendTo(row);
            if (localNode) {
                if (!conflict) {
                    localCell.addClass("red-ui-diff-status-"+(localChanged?"changed":"unchanged"));
                    $('<span class="red-ui-diff-status">'+(localChanged?'<i class="fa fa-square"></i>':'')+'</span>').appendTo(localCell);
                } else {
                    localCell.addClass("red-ui-diff-status-conflict");
                    $('<span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span>').appendTo(localCell);
                }
                formatWireProperty(localNode.wires,localNodeObj.all).appendTo(localCell);
            } else {
                localCell.addClass("red-ui-diff-empty");
            }

            if (remoteNode !== undefined) {
                remoteCell = $("<td>",{class:"red-ui-diff-list-cell red-ui-diff-list-node-remote"}).appendTo(row);
                if (remoteNode) {
                    if (!conflict) {
                        remoteCell.addClass("red-ui-diff-status-"+(remoteChanged?"changed":"unchanged"));
                        $('<span class="red-ui-diff-status">'+(remoteChanged?'<i class="fa fa-square"></i>':'')+'</span>').appendTo(remoteCell);
                    } else {
                        remoteCell.addClass("red-ui-diff-status-conflict");
                        $('<span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span>').appendTo(remoteCell);
                    }
                    formatWireProperty(remoteNode.wires,remoteNodeObj.all).appendTo(remoteCell);
                } else {
                    remoteCell.addClass("red-ui-diff-empty");
                }
            }
        }
        var properties = Object.keys(node).filter(function(p) { return p!='inputLabels'&&p!='outputLabels'&&p!='z'&&p!='wires'&&p!=='x'&&p!=='y'&&p!=='w'&&p!=='h'&&p!=='id'&&p!=='type'&&(!def.defaults||!def.defaults.hasOwnProperty(p))});
        if (def.defaults) {
            properties = properties.concat(Object.keys(def.defaults));
        }
        if (node.type !== 'tab' && node.type !== "group") {
            properties = properties.concat(['inputLabels','outputLabels']);
        }
        if ( ((localNode && localNode.hasOwnProperty('icon')) || (remoteNode && remoteNode.hasOwnProperty('icon'))) &&
            properties.indexOf('icon') === -1
        ) {
            properties.unshift('icon');
        }


        properties.forEach(function(d) {
            localChanged = false;
            remoteChanged = false;
            conflict = false;
            currentValue = JSON.stringify(node[d]);
            if (localNode) {
                localValue = JSON.stringify(localNode[d]);
                if (currentValue !== localValue) {
                    localChanged = true;
                    localChanges++;
                }
            }
            if (remoteNode) {
                remoteValue = JSON.stringify(remoteNode[d]);
                if (currentValue !== remoteValue) {
                    remoteChanged = true;
                    remoteChanges++;
                }
            }

            if ( (remoteChanged && localChanged && (localValue !== remoteValue)) ||
                (!localChanged &&  remoteChanged && localNodeObj.diff.deleted[node.id]) ||
                (localChanged && !remoteChanged && remoteNodeObj.diff.deleted[node.id])
            ){
                conflict = true;
            }

            row = $("<tr>").appendTo(nodePropertiesTableBody);
            var propertyNameCell = $("<td>",{class:"red-ui-diff-list-cell-label"}).text(d).appendTo(row);
            localCell = $("<td>",{class:"red-ui-diff-list-cell red-ui-diff-list-node-local"}).appendTo(row);
            if (localNode) {
                if (!conflict) {
                    localCell.addClass("red-ui-diff-status-"+(localChanged?"changed":"unchanged"));
                    $('<span class="red-ui-diff-status">'+(localChanged?'<i class="fa fa-square"></i>':'')+'</span>').appendTo(localCell);
                } else {
                    localCell.addClass("red-ui-diff-status-conflict");
                    $('<span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span>').appendTo(localCell);
                }
                element = $('<span class="red-ui-diff-list-element"></span>').appendTo(localCell);
                propertyElements['local.'+d] = RED.utils.createObjectElement(localNode[d],
                    {
                        path: d,
                        exposeApi: true,
                        ontoggle: function(path,state) {
                            if (propertyElements['remote.'+d]) {
                                propertyElements['remote.'+d].prop('expand')(path,state)
                            }
                        }
                    }
                ).appendTo(element);
            } else {
                localCell.addClass("red-ui-diff-empty");
            }
            if (remoteNode !== undefined) {
                remoteCell = $("<td>",{class:"red-ui-diff-list-cell red-ui-diff-list-node-remote"}).appendTo(row);
                if (remoteNode) {
                    if (!conflict) {
                        remoteCell.addClass("red-ui-diff-status-"+(remoteChanged?"changed":"unchanged"));
                        $('<span class="red-ui-diff-status">'+(remoteChanged?'<i class="fa fa-square"></i>':'')+'</span>').appendTo(remoteCell);
                    } else {
                        remoteCell.addClass("red-ui-diff-status-conflict");
                        $('<span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span>').appendTo(remoteCell);
                    }
                    element = $('<span class="red-ui-diff-list-element"></span>').appendTo(remoteCell);
                    propertyElements['remote.'+d] = RED.utils.createObjectElement(remoteNode[d],
                        {
                            path: d,
                            exposeApi: true,
                            ontoggle: function(path,state) {
                                if (propertyElements['local.'+d]) {
                                    propertyElements['local.'+d].prop('expand')(path,state)
                                }
                            }
                        }
                    ).appendTo(element);
                } else {
                    remoteCell.addClass("red-ui-diff-empty");
                }
            }
            if (localNode && remoteNode && typeof localNode[d] === "string") {
                if (/\n/.test(localNode[d]) || /\n/.test(remoteNode[d])) {
                    var textDiff = $('<button class="red-ui-button red-ui-button-small red-ui-diff-text-diff-button"><i class="fa fa-file-o"> <i class="fa fa-caret-left"></i> <i class="fa fa-caret-right"></i> <i class="fa fa-file-o"></i></button>').on("click", function() {
                        showTextDiff(localNode[d],remoteNode[d]);
                    }).appendTo(propertyNameCell);
                    RED.popover.tooltip(textDiff, RED._("diff.compareChanges"));
                }
            }


        });
        return nodePropertiesDiv;
    }
    function createNodeConflictRadioBoxes(node,row,localDiv,remoteDiv,propertiesTable,hide,state,diff) {
        var safeNodeId = "red-ui-diff-selectbox-"+node.id.replace(/\./g,'-')+(propertiesTable?"-props":"");
        var className = "";
        if (node.z||propertiesTable) {
            className = "red-ui-diff-selectbox-tab-"+(propertiesTable?node.id:node.z).replace(/\./g,'-');
        }
        var titleRow = !propertiesTable && (node.type === 'tab' || node.type === 'subflow');
        var changeHandler = function(evt) {
            var className;
            if (node.type === undefined) {
                // TODO: handle globals
            } else if (titleRow) {
                className = "red-ui-diff-selectbox-tab-"+node.id.replace(/\./g,'-');
                $("."+className+"-"+this.value).prop('checked',true);
                if (this.value === 'local') {
                    $("."+className+"-"+this.value).closest(".red-ui-diff-list-node").addClass("red-ui-diff-select-local");
                    $("."+className+"-"+this.value).closest(".red-ui-diff-list-node").removeClass("red-ui-diff-select-remote");
                } else {
                    $("."+className+"-"+this.value).closest(".red-ui-diff-list-node").removeClass("red-ui-diff-select-local");
                    $("."+className+"-"+this.value).closest(".red-ui-diff-list-node").addClass("red-ui-diff-select-remote");
                }
            } else {
                // Individual node or properties table
                var parentId = "red-ui-diff-selectbox-"+(propertiesTable?node.id:node.z).replace(/\./g,'-');
                $('#'+parentId+"-local").prop('checked',false);
                $('#'+parentId+"-remote").prop('checked',false);
                var titleRowDiv = $('#'+parentId+"-local").closest(".red-ui-diff-list-flow").find(".red-ui-diff-list-flow-title");
                titleRowDiv.removeClass("red-ui-diff-select-local");
                titleRowDiv.removeClass("red-ui-diff-select-remote");
            }
            if (this.value === 'local') {
                row.removeClass("red-ui-diff-select-remote");
                row.addClass("red-ui-diff-select-local");
            } else if (this.value === 'remote') {
                row.addClass("red-ui-diff-select-remote");
                row.removeClass("red-ui-diff-select-local");
            }
            refreshConflictHeader(diff);
        }

        var localSelectDiv = $('<label>',{class:"red-ui-diff-selectbox",for:safeNodeId+"-local"}).on("click", function(e) { e.stopPropagation();}).appendTo(localDiv);
        var localRadio = $('<input>',{class:"red-ui-diff-selectbox-input "+className+"-local",id:safeNodeId+"-local",type:'radio',value:"local",name:safeNodeId}).data('node-id',node.id).on("change", changeHandler).appendTo(localSelectDiv);
        var remoteSelectDiv = $('<label>',{class:"red-ui-diff-selectbox",for:safeNodeId+"-remote"}).on("click", function(e) { e.stopPropagation();}).appendTo(remoteDiv);
        var remoteRadio = $('<input>',{class:"red-ui-diff-selectbox-input "+className+"-remote",id:safeNodeId+"-remote",type:'radio',value:"remote",name:safeNodeId}).data('node-id',node.id).on("change", changeHandler).appendTo(remoteSelectDiv);
        if (state === 'local') {
            localRadio.prop('checked',true);
        } else if (state === 'remote') {
            remoteRadio.prop('checked',true);
        }
        if (hide||localDiv.hasClass("red-ui-diff-empty") || remoteDiv.hasClass("red-ui-diff-empty")) {
            localSelectDiv.hide();
            remoteSelectDiv.hide();
        }

    }
    function refreshConflictHeader(currentDiff) {
        var resolutionCount = 0;
        $(".red-ui-diff-selectbox>input:checked").each(function() {
            if (currentDiff.conflicts[$(this).data('node-id')]) {
                resolutionCount++;
            }
            currentDiff.resolutions[$(this).data('node-id')] = $(this).val();
        })
        var conflictCount = Object.keys(currentDiff.conflicts).length;
        if (conflictCount - resolutionCount === 0) {
            $("#red-ui-diff-dialog-toolbar-resolved-conflicts").html('<span class="red-ui-diff-status-added"><span class="red-ui-diff-status"><i class="fa fa-check"></i></span></span> '+RED._("diff.unresolvedCount",{count:conflictCount - resolutionCount}));
        } else {
            $("#red-ui-diff-dialog-toolbar-resolved-conflicts").html('<span class="red-ui-diff-status-conflict"><span class="red-ui-diff-status"><i class="fa fa-exclamation"></i></span></span> '+RED._("diff.unresolvedCount",{count:conflictCount - resolutionCount}));
        }
        if (conflictCount === resolutionCount) {
            $("#red-ui-diff-view-diff-merge").removeClass('disabled');
            $("#red-ui-diff-view-resolve-diff").removeClass('disabled');
        }
    }
    function getRemoteDiff(callback) {
        $.ajax({
            headers: {
                "Accept":"application/json",
            },
            cache: false,
            url: 'flows',
            success: function(nodes) {
                var localFlow = RED.nodes.createCompleteNodeSet();
                var originalFlow = RED.nodes.originalFlow();
                var remoteFlow = nodes.flows;
                var localDiff = generateDiff(originalFlow,localFlow);
                var remoteDiff = generateDiff(originalFlow,remoteFlow);
                remoteDiff.rev = nodes.rev;
                callback(resolveDiffs(localDiff,remoteDiff))
            }
        });

    }
    // function showLocalDiff() {
    //     var nns = RED.nodes.createCompleteNodeSet();
    //     var originalFlow = RED.nodes.originalFlow();
    //     var diff = generateDiff(originalFlow,nns);
    //     showDiff(diff);
    // }
    function showRemoteDiff(diff, options = {}) {
        if (!diff) {
            getRemoteDiff((remoteDiff) => showRemoteDiff(remoteDiff, options));
        } else {
            showDiff(diff,{...options, mode:'merge'});
        }
    }
    function parseNodes(nodeList) {
        var tabOrder = [];
        var tabs = {};
        var subflows = {};
        var globals = [];
        var all = {};

        nodeList.forEach(function(node) {
            all[node.id] = node;
            if (node.type === 'tab') {
                tabOrder.push(node.id);
                tabs[node.id] = {n:node,nodes:[]};
            } else if (node.type === 'subflow') {
                subflows[node.id] = {n:node,nodes:[]};
            }
        });

        nodeList.forEach(function(node) {
            if (node.type !== 'tab' && node.type !== 'subflow') {
                if (tabs[node.z]) {
                    tabs[node.z].nodes.push(node);
                } else if (subflows[node.z]) {
                    subflows[node.z].nodes.push(node);
                } else {
                    globals.push(node);
                }
            }
        });

        return {
            all: all,
            tabOrder: tabOrder,
            tabs: tabs,
            subflows: subflows,
            globals: globals
        }
    }
    function generateDiff(currentNodes,newNodes) {
        const currentConfig = parseNodes(currentNodes);
        const newConfig = parseNodes(newNodes);
        const added = {};
        const deleted = {};
        const changed = {};
        const positionChanged = {};
        const moved = {};

        Object.keys(currentConfig.all).forEach(function(id) {
            const node = RED.nodes.workspace(id)||RED.nodes.subflow(id)||RED.nodes.node(id);
            if (!newConfig.all.hasOwnProperty(id)) {
                deleted[id] = true;
                return
            }
            const currentConfigJSON = JSON.stringify(currentConfig.all[id])
            const newConfigJSON = JSON.stringify(newConfig.all[id])
            
            if (currentConfigJSON !== newConfigJSON) {
                changed[id] = true;
                if (currentConfig.all[id].z !== newConfig.all[id].z) {
                    moved[id] = true;
                } else if (
                    currentConfig.all[id].x !== newConfig.all[id].x ||
                    currentConfig.all[id].y !== newConfig.all[id].y ||
                    currentConfig.all[id].w !== newConfig.all[id].w ||
                    currentConfig.all[id].h !== newConfig.all[id].h
                ) {
                    // This node's position on its parent has changed. We want to
                    // check if this is the *only* change for this given node
                    const currentNodeClone = JSON.parse(currentConfigJSON)
                    const newNodeClone = JSON.parse(newConfigJSON)

                    delete currentNodeClone.x
                    delete currentNodeClone.y
                    delete currentNodeClone.w
                    delete currentNodeClone.h
                    delete newNodeClone.x
                    delete newNodeClone.y
                    delete newNodeClone.w
                    delete newNodeClone.h
                    
                    if (JSON.stringify(currentNodeClone) === JSON.stringify(newNodeClone)) {
                        // Only the position has changed - everything else is the same
                        positionChanged[id] = true
                    }
                }

            }
        });
        Object.keys(newConfig.all).forEach(function(id) {
            if (!currentConfig.all.hasOwnProperty(id)) {
                added[id] = true;
            }
        });

        const diff = {
            currentConfig,
            newConfig,
            added,
            deleted,
            changed,
            positionChanged,
            moved
        };
        return diff;
    }
    function resolveDiffs(localDiff,remoteDiff) {
        var conflicted = {};
        var resolutions = {};
        var diff = {
            localDiff: localDiff,
            remoteDiff: remoteDiff,
            conflicts: conflicted,
            resolutions: resolutions
        }
        var seen = {};
        var id,node;
        for (id in localDiff.currentConfig.all) {
            if (localDiff.currentConfig.all.hasOwnProperty(id)) {
                seen[id] = true;
                var localNode = localDiff.newConfig.all[id];
                if (localDiff.changed[id] && remoteDiff.deleted[id]) {
                    conflicted[id] = true;
                } else if (localDiff.deleted[id] && remoteDiff.changed[id]) {
                    conflicted[id] = true;
                } else if (localDiff.changed[id] && remoteDiff.changed[id]) {
                    var remoteNode = remoteDiff.newConfig.all[id];
                    if (JSON.stringify(localNode) !== JSON.stringify(remoteNode)) {
                        conflicted[id] = true;
                    }
                }
                if (!conflicted[id]) {
                    if (remoteDiff.added[id]||remoteDiff.changed[id]||remoteDiff.deleted[id]) {
                        resolutions[id] = 'remote';
                    } else {
                        resolutions[id] = 'local';
                    }
                }
            }
        }
        for (id in localDiff.added) {
            if (localDiff.added.hasOwnProperty(id)) {
                node = localDiff.newConfig.all[id];
                if (remoteDiff.deleted[node.z]) {
                    conflicted[id] = true;
                    // conflicted[node.z] = true;
                } else {
                    resolutions[id] = 'local';
                }
            }
        }
        for (id in remoteDiff.added) {
            if (remoteDiff.added.hasOwnProperty(id)) {
                node = remoteDiff.newConfig.all[id];
                if (localDiff.deleted[node.z]) {
                    conflicted[id] = true;
                    // conflicted[node.z] = true;
                } else {
                    resolutions[id] = 'remote';
                }
            }
        }
        // console.log(diff.resolutions);
        // console.log(conflicted);
        return diff;
    }

    function showDiff(diff, options) {
        if (diffVisible) {
            return;
        }
        options = options || {};
        var mode = options.mode || 'merge';
        
        options.hidePositionChanges = true

        var localDiff = diff.localDiff;
        var remoteDiff = diff.remoteDiff;
        var conflicts = diff.conflicts;
        // currentDiff = diff;

        var trayOptions = {
            title: options.title||RED._("diff.reviewChanges"),
            width: Infinity,
            overlay: true,
            buttons: [
                {
                    text: RED._((options.mode === 'merge')?"common.label.cancel":"common.label.close"),
                    click: function() {
                        RED.tray.close();
                    }
                }
            ],
            resize: function(dimensions) {
                // trayWidth = dimensions.width;
            },
            open: function(tray) {
                var trayBody = tray.find('.red-ui-tray-body');
                var toolbar = $('<div class="red-ui-diff-dialog-toolbar">'+
                    '<span><span id="red-ui-diff-dialog-toolbar-resolved-conflicts"></span></span> '+
                    '</div>').prependTo(trayBody);
                var diffContainer = $('<div class="red-ui-diff-container"></div>').appendTo(trayBody);
                var diffTable = buildDiffPanel(diffContainer,diff,options);
                diffTable.list.hide();
                if (remoteDiff) {
                    $("#red-ui-diff-view-diff-merge").show();
                    if (Object.keys(conflicts).length === 0) {
                        $("#red-ui-diff-view-diff-merge").removeClass('disabled');
                    } else {
                        $("#red-ui-diff-view-diff-merge").addClass('disabled');
                    }
                } else {
                    $("#red-ui-diff-view-diff-merge").hide();
                }
                refreshConflictHeader(diff);
                // console.log("--------------");
                // console.log(localDiff);
                // console.log(remoteDiff);

                setTimeout(function() {
                    diffTable.finish();
                    diffTable.list.show();
                },300);
                $("#red-ui-sidebar-shade").show();
            },
            close: function() {
                diffVisible = false;
                $("#red-ui-sidebar-shade").hide();

            },
            show: function() {

            }
        }
        if (options.mode === 'merge') {
            trayOptions.buttons.push(
                {
                    id: "red-ui-diff-view-diff-merge",
                    text: RED._("deploy.confirm.button.merge"),
                    class: "primary disabled",
                    click: function() {
                        if (!$("#red-ui-diff-view-diff-merge").hasClass('disabled')) {
                            refreshConflictHeader(diff);
                            mergeDiff(diff);
                            if (options.onmerge) {
                                options.onmerge()
                            }
                            RED.tray.close();
                        }
                    }
                }
            );
        }

        RED.tray.show(trayOptions);
    }

    function applyDiff(diff) {
        var currentConfig = diff.localDiff.currentConfig;
        var localDiff = diff.localDiff;
        var remoteDiff = diff.remoteDiff;
        var conflicts = diff.conflicts;
        var resolutions = diff.resolutions;
        var id;

        for (id in conflicts) {
            if (conflicts.hasOwnProperty(id)) {
                if (!resolutions.hasOwnProperty(id)) {
                    console.log(diff);
                    throw new Error("No resolution for conflict on node",id);
                }
            }
        }

        var newConfig = [];
        var node;
        var nodeChangedStates = {};
        var nodeMovedStates = {};
        var localChangedStates = {};
        for (id in localDiff.newConfig.all) {
            if (localDiff.newConfig.all.hasOwnProperty(id)) {
                node = RED.nodes.node(id);
                if (resolutions[id] === 'local') {
                    if (node) {
                        nodeChangedStates[id] = node.changed;
                        nodeMovedStates[id] = node.moved;
                    }
                    newConfig.push(localDiff.newConfig.all[id]);
                } else if (resolutions[id] === 'remote') {
                    if (!remoteDiff.deleted[id] && remoteDiff.newConfig.all.hasOwnProperty(id)) {
                        if (node) {
                            nodeChangedStates[id] = node.changed;
                            nodeMovedStates[id] = node.moved;
                        }
                        localChangedStates[id] = 1;
                        newConfig.push(remoteDiff.newConfig.all[id]);
                    }
                } else {
                    console.log("Unresolved",id)
                }
            }
        }
        for (id in remoteDiff.added) {
            if (remoteDiff.added.hasOwnProperty(id)) {
                node = RED.nodes.node(id);
                if (node) {
                    nodeChangedStates[id] = node.changed;
                }
                if (!localDiff.added.hasOwnProperty(id)) {
                    localChangedStates[id] = 2;
                    newConfig.push(remoteDiff.newConfig.all[id]);
                }
            }
        }
        return {
            config: newConfig,
            nodeChangedStates,
            nodeMovedStates,
            localChangedStates
        }
    }

    function mergeDiff(diff) {
        //console.log(diff);
        var selectedTab = RED.workspaces.active();
        var appliedDiff = applyDiff(diff);

        var newConfig = appliedDiff.config;
        var nodeChangedStates = appliedDiff.nodeChangedStates;
        var nodeMovedStates = appliedDiff.nodeMovedStates;
        var localChangedStates = appliedDiff.localChangedStates;

        var isDirty = RED.nodes.dirty();

        var historyEvent = {
            t:"replace",
            config: RED.nodes.createCompleteNodeSet(),
            changed: nodeChangedStates,
            moved: nodeMovedStates,
            complete: true,
            dirty: isDirty,
            rev: RED.nodes.version()
        }

        RED.history.push(historyEvent);

        // var originalFlow = RED.nodes.originalFlow();
        // // originalFlow is what the editor thinks it loaded
        // //  - add any newly added nodes from remote diff as they are now part of the record
        // for (var id in diff.remoteDiff.added) {
        //     if (diff.remoteDiff.added.hasOwnProperty(id)) {
        //         if (diff.remoteDiff.newConfig.all.hasOwnProperty(id)) {
        //             originalFlow.push(JSON.parse(JSON.stringify(diff.remoteDiff.newConfig.all[id])));
        //         }
        //     }
        // }

        RED.nodes.clear();
        var imported = RED.nodes.import(newConfig);

        // // Restore the original flow so subsequent merge resolutions can properly
        // // identify new-vs-old
        // RED.nodes.originalFlow(originalFlow);

        // Clear all change flags from the import
        RED.nodes.dirty(false);
        
        const flowsToLock = new Set()
        function ensureUnlocked(id) {
            const flow = id && (RED.nodes.workspace(id) || RED.nodes.subflow(id) || null);
            const isLocked = flow ? flow.locked : false;
            if (flow && isLocked) {
                flow.locked = false;
                flowsToLock.add(flow)
            }
        }
        imported.nodes.forEach(function(n) {
            if (nodeChangedStates[n.id]) {
                ensureUnlocked(n.z)
                n.changed = true;
            }
            if (nodeMovedStates[n.id]) {
                ensureUnlocked(n.z)
                n.moved = true;
            }
        })
        flowsToLock.forEach(flow => {
            flow.locked = true
        })

        RED.nodes.version(diff.remoteDiff.rev);

        if (isDirty) {
            RED.nodes.dirty(true);
        }

        RED.view.redraw(true);
        RED.palette.refresh();
        RED.workspaces.refresh();
        RED.workspaces.show(selectedTab, true);
        RED.sidebar.config.refresh();
    }

    function showTestFlowDiff(index) {
        if (index === 1) {
            var localFlow = RED.nodes.createCompleteNodeSet();
            var originalFlow = RED.nodes.originalFlow();
            showTextDiff(JSON.stringify(localFlow,null,4),JSON.stringify(originalFlow,null,4))
        } else if (index === 2) {
            var local = "1\n2\n3\n4\n5\nA\n6\n7\n8\n9\n";
            var remote = "1\nA\n2\n3\nD\nE\n6\n7\n8\n9\n";
            showTextDiff(local,remote);
        } else if (index === 3) {
            var local =  "1\n2\n3\n4\n5\n6\n7\n8\n9\n10\n11\n12\n13\n14\n15\n16\n17\n18\n19\n20\n21\n22";
            var remote = "1\nTWO\nTHREE\nEXTRA\n4\n5\n6\n7\n8\n9\n10\n11\n12\nTHIRTEEN\n14\n15\n16\n17\n18\n19\n20\n21\n22";
            showTextDiff(local,remote);
        }
    }

    function showTextDiff(textA,textB) {
        var trayOptions = {
            title: RED._("diff.compareChanges"),
            width: Infinity,
            overlay: true,
            buttons: [
                {
                    text: RED._("common.label.close"),
                    click: function() {
                        RED.tray.close();
                    }
                }
            ],
            resize: function(dimensions) {
                // trayWidth = dimensions.width;
            },
            open: function(tray) {
                var trayBody = tray.find('.red-ui-tray-body');
                var diffPanel = $('<div class="red-ui-diff-text"></div>').appendTo(trayBody);

                var codeTable = $("<table>",{class:"red-ui-diff-text-content"}).appendTo(diffPanel);
                $('<colgroup><col width="50"><col width="50%"><col width="50"><col width="50%"></colgroup>').appendTo(codeTable);
                var codeBody = $('<tbody>').appendTo(codeTable);
                var diffSummary = diffText(textA||"",textB||"");
                var aIndex = 0;
                var bIndex = 0;
                var diffLength = Math.max(diffSummary.a.length, diffSummary.b.length);

                var diffLines = [];
                var diffBlocks = [];
                var currentBlock;
                var blockLength = 0;
                var blockType = 0;

                for (var i=0;i<diffLength;i++) {
                    var diffLine = diffSummary[i];
                    var Adiff = (aIndex < diffSummary.a.length)?diffSummary.a[aIndex]:{type:2,line:""};
                    var Bdiff = (bIndex < diffSummary.b.length)?diffSummary.b[bIndex]:{type:2,line:""};
                    if (Adiff.type === 0 && Bdiff.type !== 0) {
                        Adiff = {type:2,line:""};
                        bIndex++;
                    } else if (Bdiff.type === 0 && Adiff.type !== 0) {
                        Bdiff = {type:2,line:""};
                        aIndex++;
                    } else {
                        aIndex++;
                        bIndex++;
                    }
                    diffLines.push({
                        a: Adiff,
                        b: Bdiff
                    });
                    if (currentBlock === undefined) {
                        currentBlock = {start:i,end:i};
                        blockLength = 0;
                        blockType = (Adiff.type === 0 && Bdiff.type === 0)?0:1;
                    } else {
                        if (Adiff.type === 0 && Bdiff.type === 0) {
                            // Unchanged line
                            if (blockType === 0) {
                                // still unchanged - extend the block
                                currentBlock.end = i;
                                blockLength++;
                            } else if (blockType === 1) {
                                // end of a change
                                currentBlock.end = i;
                                blockType = 2;
                                blockLength = 0;
                            } else if (blockType === 2) {
                                // post-change unchanged
                                currentBlock.end = i;
                                blockLength++;
                                if (blockLength === 8) {
                                    currentBlock.end -= 5; // rollback the end
                                    diffBlocks.push(currentBlock);
                                    currentBlock = {start:i-5,end:i-5};
                                    blockType = 0;
                                    blockLength = 0;
                                }
                            }
                        } else {
                            // in a change
                            currentBlock.end = i;
                            blockLength++;
                            if (blockType === 0) {
                                if (currentBlock.end > 3) {
                                    currentBlock.end -= 3;
                                    currentBlock.empty = true;
                                    diffBlocks.push(currentBlock);
                                    currentBlock = {start:i-3,end:i-3};
                                }
                                blockType = 1;
                            } else if (blockType === 2) {
                                // we were in unchanged, but hit a change again
                                blockType = 1;
                            }
                        }
                    }
                }
                if (blockType === 0) {
                    currentBlock.empty = true;
                }
                currentBlock.end = diffLength;
                diffBlocks.push(currentBlock);
                console.table(diffBlocks);
                var diffRow;
                for (var b = 0; b<diffBlocks.length; b++) {
                    currentBlock = diffBlocks[b];
                    if (currentBlock.empty) {
                        diffRow = createExpandLine(currentBlock.start,currentBlock.end,diffLines).appendTo(codeBody);
                    } else {
                        for (var i=currentBlock.start;i<currentBlock.end;i++) {
                            var row = createDiffLine(diffLines[i]).appendTo(codeBody);
                            if (i === currentBlock.start) {
                                row.addClass("start-block");
                            } else if (i === currentBlock.end-1) {
                                row.addClass("end-block");
                            }
                        }
                    }
                }

            },
            close: function() {
                diffVisible = false;

            },
            show: function() {

            }
        }
        RED.tray.show(trayOptions);
    }

    function createExpandLine(start,end,diffLines) {
        diffRow = $('<tr class="red-ui-diff-text-header red-ui-diff-text-expand">');
        var content = $('<td colspan="4"> <i class="fa fa-arrows-v"></i> </td>').appendTo(diffRow);
        var label = $('<span></span>').appendTo(content);
        if (end < diffLines.length-1) {
            label.text("@@ -"+(diffLines[end-1].a.i+1)+" +"+(diffLines[end-1].b.i+1));
        }
        diffRow.on("click", function(evt) {
            // console.log(start,end,diffLines.length);
            if (end - start > 20) {
                var startPos = $(this).offset();
                // console.log(startPos);
                if (start > 0) {
                    for (var i=start;i<start+10;i++) {
                        createDiffLine(diffLines[i]).addClass("unchanged").insertBefore($(this));
                    }
                    start += 10;
                }
                if (end < diffLines.length-1) {
                    for (var i=end-1;i>end-11;i--) {
                        createDiffLine(diffLines[i]).addClass("unchanged").insertAfter($(this));
                    }
                    end -= 10;
                }
                if (end < diffLines.length-1) {
                    label.text("@@ -"+(diffLines[end-1].a.i+1)+" +"+(diffLines[end-1].b.i+1));
                }
                var endPos = $(this).offset();
                var delta = endPos.top - startPos.top;
                $(".red-ui-diff-text").scrollTop($(".red-ui-diff-text").scrollTop() + delta);
            } else {
                for (var i=start;i<end;i++) {
                    createDiffLine(diffLines[i]).addClass("unchanged").insertBefore($(this));
                }
                $(this).remove();
            }
        });
        return diffRow;
    }

    function createDiffLine(diffLine) {
        var diffRow = $('<tr>');
        var Adiff = diffLine.a;
        var Bdiff = diffLine.b;
        //console.log(diffLine);
        var cellNo = $('<td class="lineno">').text(Adiff.type === 2?"":Adiff.i).appendTo(diffRow);
        var cellLine = $('<td class="linetext">').text(Adiff.line).appendTo(diffRow);
        if (Adiff.type === 2) {
            cellNo.addClass('blank');
            cellLine.addClass('blank');
        } else if (Adiff.type === 4) {
            cellNo.addClass('added');
            cellLine.addClass('added');
        } else if (Adiff.type === 1) {
            cellNo.addClass('removed');
            cellLine.addClass('removed');
        }
        cellNo = $('<td class="lineno">').text(Bdiff.type === 2?"":Bdiff.i).appendTo(diffRow);
        cellLine = $('<td class="linetext">').text(Bdiff.line).appendTo(diffRow);
        if (Bdiff.type === 2) {
            cellNo.addClass('blank');
            cellLine.addClass('blank');
        } else if (Bdiff.type === 4) {
            cellNo.addClass('added');
            cellLine.addClass('added');
        } else if (Bdiff.type === 1) {
            cellNo.addClass('removed');
            cellLine.addClass('removed');
        }
        return diffRow;
    }

    function diffText(string1, string2,ignoreWhitespace) {
        var lines1 = string1.split(/\r?\n/);
        var lines2 = string2.split(/\r?\n/);
        var i = lines1.length;
        var j = lines2.length;
        var k;
        var m;
        var diffSummary = {a:[],b:[]};
        var diffMap = [];
        for (k = 0; k < i + 1; k++) {
            diffMap[k] = [];
            for (m = 0; m < j + 1; m++) {
                diffMap[k][m] = 0;
            }
        }
        var c = 0;
        for (k = i - 1; k >= 0; k--) {
            for (m = j - 1; m >=0; m--) {
                c++;
                if (compareLines(lines1[k],lines2[m],ignoreWhitespace) !== 1) {
                    diffMap[k][m] = diffMap[k+1][m+1]+1;
                } else {
                    diffMap[k][m] = Math.max(diffMap[(k + 1)][m], diffMap[k][(m + 1)]);
                }
            }
        }
        //console.log(c);
        k = 0;
        m = 0;

        while ((k < i) && (m < j)) {
            var n = compareLines(lines1[k],lines2[m],ignoreWhitespace);
            if (n !== 1) {
                var d = 0;
                if (n===0) {
                    d = 0;
                } else if (n==2) {
                    d = 3;
                }
                diffSummary.a.push({i:k+1,j:m+1,line:lines1[k],type:d});
                diffSummary.b.push({i:m+1,j:k+1,line:lines2[m],type:d});
                k++;
                m++;
            } else if (diffMap[(k + 1)][m] >= diffMap[k][(m + 1)]) {
                diffSummary.a.push({i:k+1,line:lines1[k],type:1});
                k++;
            } else {
                diffSummary.b.push({i:m+1,line:lines2[m],type:4});
                m++;
            }
        }
        while ((k < i) || (m < j)) {
            if (k == i) {
                diffSummary.b.push({i:m+1,line:lines2[m],type:4});
                m++;
            } else if (m == j) {
                diffSummary.a.push({i:k+1,line:lines1[k],type:1});
                k++;
            }
        }
        return diffSummary;
    }

    function compareLines(string1, string2, ignoreWhitespace) {
        if (ignoreWhitespace) {
            if (string1 === string2) {
                return 0;
            }
            return string1.trim() === string2.trime() ? 2 : 1;
        }
        return string1 === string2 ? 0 : 1;
    }

    function createUnifiedDiffTable(files,commitOptions) {
        var diffPanel = $('<div></div>');
        files.forEach(function(file) {
            var hunks = file.hunks;
            var isBinary = file.binary;
            var codeTable = $("<table>",{class:"red-ui-diff-text-content"}).appendTo(diffPanel);
            $('<colgroup><col width="50"><col width="50"><col width="100%"></colgroup>').appendTo(codeTable);
            var codeBody = $('<tbody>').appendTo(codeTable);

            var diffFileRow = $('<tr class="red-ui-diff-text-file-header">').appendTo(codeBody);
            var content = $('<td colspan="3"></td>').appendTo(diffFileRow);

            var chevron = $('<i class="red-ui-diff-list-chevron fa fa-angle-down"></i>').appendTo(content);
            diffFileRow.on("click", function(e) {
                diffFileRow.toggleClass("collapsed");
                var isCollapsed = diffFileRow.hasClass("collapsed");
                diffFileRow.nextUntil(".red-ui-diff-text-file-header").toggle(!isCollapsed);
            })
            var label = $('<span class="filename"></span>').text(file.file).appendTo(content);

            var conflictHeader;
            var unresolvedConflicts = 0;
            var resolvedConflicts = 0;
            var conflictResolutions = {};
            if (commitOptions.project.files && commitOptions.project.files.flow === file.file) {
                if (commitOptions.unmerged) {
                    $('<span style="float: right;"><span id="red-ui-diff-dialog-toolbar-resolved-conflicts"></span></span>').appendTo(content);
                }
                var diffRow = $('<tr class="red-ui-diff-text-header">').appendTo(codeBody);
                var flowDiffContent = $('<td class="red-ui-diff-flow-diff" colspan="3"></td>').appendTo(diffRow);

                var projectName = commitOptions.project.name;
                var filename = commitOptions.project.files.flow;
                var commonVersionUrl = "projects/"+projectName+"/files/"+commitOptions.commonRev+"/"+filename;
                var oldVersionUrl = "projects/"+projectName+"/files/"+commitOptions.oldRev+"/"+filename;
                var newVersionUrl = "projects/"+projectName+"/files/"+commitOptions.newRev+"/"+filename;
                var promises = [$.Deferred(),$.Deferred(),$.Deferred()];
                if (commitOptions.commonRev) {
                    var commonVersionUrl = "projects/"+projectName+"/files/"+commitOptions.commonRev+"/"+filename;
                    $.ajax({dataType: "json",url: commonVersionUrl}).then(function(data) { promises[0].resolve(data); }).fail(function() { promises[0].resolve(null);})
                } else {
                    promises[0].resolve(null);
                }

                $.ajax({dataType: "json",url: oldVersionUrl}).then(function(data) { promises[1].resolve(data); }).fail(function() { promises[1].resolve({content:"[]"});})
                $.ajax({dataType: "json",url: newVersionUrl}).then(function(data) { promises[2].resolve(data); }).fail(function() { promises[2].resolve({content:"[]"});})
                $.when.apply($,promises).always(function(commonVersion, oldVersion,newVersion) {
                    var commonFlow;
                    var oldFlow;
                    var newFlow;
                    if (commonVersion) {
                        try {
                            commonFlow = JSON.parse(commonVersion.content||"[]");
                        } catch(err) {
                            console.log(RED._("diff.commonVersionError"),commonVersionUrl);
                            console.log(err);
                            return;
                        }
                    }
                    try {
                        oldFlow = JSON.parse(oldVersion.content||"[]");
                    } catch(err) {
                        console.log(RED._("diff.oldVersionError"),oldVersionUrl);
                        console.log(err);
                        return;
                    }
                    if (!commonFlow) {
                        commonFlow = oldFlow;
                    }
                    try {
                        newFlow = JSON.parse(newVersion.content||"[]");
                    } catch(err) {
                        console.log(RED._("diff.newVersionError"),newFlow);
                        console.log(err);
                        return;
                    }
                    var localDiff = generateDiff(commonFlow,oldFlow);
                    var remoteDiff = generateDiff(commonFlow,newFlow);
                    commitOptions.currentDiff = resolveDiffs(localDiff,remoteDiff);
                    var diffTable = buildDiffPanel(flowDiffContent,commitOptions.currentDiff,{
                        title: filename,
                        mode: commitOptions.commonRev?'merge':'view',
                        oldRevTitle: commitOptions.oldRevTitle,
                        newRevTitle: commitOptions.newRevTitle
                    });
                    diffTable.list.hide();
                    refreshConflictHeader(commitOptions.currentDiff);
                    setTimeout(function() {
                        diffTable.finish();
                        diffTable.list.show();
                    },300);
                    // var flowDiffRow = $("<tr>").insertAfter(diffRow);
                    // var content = $('<td colspan="3"></td>').appendTo(flowDiffRow);
                    // currentDiff = diff;
                    // var diffTable = buildDiffPanel(content,diff,{mode:"view"}).finish();
                });



            } else

            if (isBinary) {
                var diffBinaryRow = $('<tr class="red-ui-diff-text-header">').appendTo(codeBody);
                var binaryContent = $('<td colspan="3"></td>').appendTo(diffBinaryRow);
                $('<span></span>').text(RED._("diff.noBinaryFileShowed")).appendTo(binaryContent);

            } else {
                if (commitOptions.unmerged) {
                    conflictHeader = $('<span style="float: right;">'+RED._("diff.conflictHeader",{resolved:resolvedConflicts, unresolved:unresolvedConflicts})+'</span>').appendTo(content);
                }
                hunks.forEach(function(hunk) {
                    var diffRow = $('<tr class="red-ui-diff-text-header">').appendTo(codeBody);
                    var content = $('<td colspan="3"></td>').appendTo(diffRow);
                    var label = $('<span></span>').text(hunk.header).appendTo(content);
                    var isConflict = hunk.conflict;
                    var localLine = hunk.localStartLine;
                    var remoteLine = hunk.remoteStartLine;
                    if (isConflict) {
                        unresolvedConflicts++;
                    }

                    hunk.lines.forEach(function(lineText,lineNumber) {
                        // if (lineText[0] === '\\' || lineText === "") {
                        //     // Comment line - bail out of this hunk
                        //     break;
                        // }

                        var actualLineNumber = hunk.diffStart + lineNumber;
                        var isMergeHeader = isConflict && /^\+\+(<<<<<<<|=======$|>>>>>>>)/.test(lineText);
                        var diffRow = $('<tr>').appendTo(codeBody);
                        var localLineNo = $('<td class="lineno">').appendTo(diffRow);
                        var remoteLineNo;
                        if (!isMergeHeader) {
                            remoteLineNo = $('<td class="lineno">').appendTo(diffRow);
                        } else {
                            localLineNo.attr('colspan',2);
                        }
                        var line = $('<td class="linetext">').appendTo(diffRow);
                        var prefixStart = 0;
                        var prefixEnd = 1;
                        if (isConflict) {
                            prefixEnd = 2;
                        }
                        if (!isMergeHeader) {
                            var changeMarker = lineText[0];
                            if (isConflict && !commitOptions.unmerged && changeMarker === ' ') {
                                changeMarker = lineText[1];
                            }
                            $('<span class="prefix">').text(changeMarker).appendTo(line);
                            var handledlLine = false;
                            if (isConflict && commitOptions.unmerged) {
                                $('<span class="prefix">').text(lineText[1]).appendTo(line);
                                if (lineText[0] === '+') {
                                    localLineNo.text(localLine++);
                                    handledlLine = true;
                                }
                                if (lineText[1] === '+') {
                                    remoteLineNo.text(remoteLine++);
                                    handledlLine = true;
                                }
                            } else {
                                if (lineText[0] === '+' || (isConflict && lineText[1] === '+')) {
                                    localLineNo.addClass("added");
                                    remoteLineNo.addClass("added");
                                    line.addClass("added");
                                    remoteLineNo.text(remoteLine++);
                                    handledlLine = true;
                                } else if (lineText[0] === '-' || (isConflict && lineText[1] === '-')) {
                                    localLineNo.addClass("removed");
                                    remoteLineNo.addClass("removed");
                                    line.addClass("removed");
                                    localLineNo.text(localLine++);
                                    handledlLine = true;
                                }
                            }
                            if (!handledlLine) {
                                line.addClass("unchanged");
                                if (localLine > 0 && lineText[0] !== '\\' && lineText !== "") {
                                    localLineNo.text(localLine++);
                                }
                                if (remoteLine > 0 && lineText[0] !== '\\' && lineText !== "") {
                                    remoteLineNo.text(remoteLine++);
                                }
                            }
                            $('<span>').text(lineText.substring(prefixEnd)).appendTo(line);
                        } else {
                            diffRow.addClass("mergeHeader");
                            var isSeparator = /^\+\+=======$/.test(lineText);
                            if (!isSeparator) {
                                var isOurs = /^..<<<<<<</.test(lineText);
                                if (isOurs) {
                                    $('<span>').text("<<<<<<< " + RED._("diff.localChanges")).appendTo(line);
                                    hunk.localChangeStart = actualLineNumber;
                                } else {
                                    hunk.remoteChangeEnd = actualLineNumber;
                                    $('<span>').text(">>>>>>> " + RED._("diff.remoteChanges")).appendTo(line);
                                }
                                diffRow.addClass("mergeHeader-"+(isOurs?"ours":"theirs"));
                                $('<button class="red-ui-button red-ui-button-small" style="float: right; margin-right: 20px;"><i class="fa fa-angle-double-'+(isOurs?"down":"up")+'"></i> '+RED._(isOurs?"diff.useLocalChanges":"diff.useRemoteChanges")+'</button>')
                                    .appendTo(line)
                                    .on("click", function(evt) {
                                        evt.preventDefault();
                                        resolvedConflicts++;
                                        var addedRows;
                                        var midRow;
                                        if (isOurs) {
                                            addedRows = diffRow.nextUntil(".mergeHeader-separator");
                                            midRow = addedRows.last().next();
                                            midRow.nextUntil(".mergeHeader").remove();
                                            midRow.next().remove();
                                        } else {
                                            addedRows = diffRow.prevUntil(".mergeHeader-separator");
                                            midRow = addedRows.last().prev();
                                            midRow.prevUntil(".mergeHeader").remove();
                                            midRow.prev().remove();
                                        }
                                        midRow.remove();
                                        diffRow.remove();
                                        addedRows.find(".linetext").addClass('added');
                                        conflictHeader.empty();
                                        $('<span>'+RED._("diff.conflictHeader",{resolved:resolvedConflicts, unresolved:unresolvedConflicts})+'</span>').appendTo(conflictHeader);

                                        conflictResolutions[file.file] = conflictResolutions[file.file] || {};
                                        conflictResolutions[file.file][hunk.localChangeStart] = {
                                            changeStart: hunk.localChangeStart,
                                            separator: hunk.changeSeparator,
                                            changeEnd: hunk.remoteChangeEnd,
                                            selection: isOurs?"A":"B"
                                        }
                                        if (commitOptions.resolveConflict) {
                                            commitOptions.resolveConflict({
                                                conflicts: unresolvedConflicts,
                                                resolved: resolvedConflicts,
                                                resolutions: conflictResolutions
                                            });
                                        }
                                    })
                            } else {
                                hunk.changeSeparator = actualLineNumber;
                                diffRow.addClass("mergeHeader-separator");
                            }
                        }
                    });
                });
            }
        });
        return diffPanel;
    }

    function showCommitDiff(options) {
        var commit = parseCommitDiff(options.commit);
        var trayOptions = {
            title: RED._("diff.viewCommitDiff"),
            width: Infinity,
            overlay: true,
            buttons: [
                {
                    text: RED._("common.label.close"),
                    click: function() {
                        RED.tray.close();
                    }
                }
            ],
            resize: function(dimensions) {
                // trayWidth = dimensions.width;
            },
            open: function(tray) {
                var trayBody = tray.find('.red-ui-tray-body');
                var diffPanel = $('<div class="red-ui-diff-text"></div>').appendTo(trayBody);

                var codeTable = $("<table>",{class:"red-ui-diff-text-content"}).appendTo(diffPanel);
                $('<colgroup><col width="50"><col width="50"><col width="100%"></colgroup>').appendTo(codeTable);
                var codeBody = $('<tbody>').appendTo(codeTable);

                var diffRow = $('<tr class="red-ui-diff-text-commit-header">').appendTo(codeBody);
                var content = $('<td colspan="3"></td>').appendTo(diffRow);

                $("<h3>").text(commit.title).appendTo(content);
                $('<div class="commit-body"></div>').text(commit.comment).appendTo(content);
                var summary = $('<div class="commit-summary"></div>').appendTo(content);
                $('<div style="float: right">').text(RED._('diff.commit')+" "+commit.sha).appendTo(summary);
                $('<div>').text((commit.authorName||commit.author)+" - "+options.date).appendTo(summary);

                if (commit.files) {
                    createUnifiedDiffTable(commit.files,options).appendTo(diffPanel);
                }


            },
            close: function() {
                diffVisible = false;
            },
            show: function() {

            }
        }
        RED.tray.show(trayOptions);
    }
    function showUnifiedDiff(options) {
        var diff = options.diff;
        var title = options.title;
        var files = parseUnifiedDiff(diff);

        var currentResolution;
        if (options.unmerged) {
            options.resolveConflict = function(results) {
                currentResolution = results;
                if (results.conflicts === results.resolved) {
                    $("#red-ui-diff-view-resolve-diff").removeClass('disabled');
                }
            }
        }

        var trayOptions = {
            title: title|| RED._("diff.compareChanges"),
            width: Infinity,
            overlay: true,
            buttons: [
                {
                    text: RED._((options.unmerged)?"common.label.cancel":"common.label.close"),
                    click: function() {
                        if (options.oncancel) {
                            options.oncancel();
                        }
                        RED.tray.close();
                    }
                }
            ],
            resize: function(dimensions) {
                // trayWidth = dimensions.width;
            },
            open: function(tray) {
                var trayBody = tray.find('.red-ui-tray-body');
                var diffPanel = $('<div class="red-ui-diff-text"></div>').appendTo(trayBody);
                createUnifiedDiffTable(files,options).appendTo(diffPanel);
            },
            close: function() {
                diffVisible = false;
            },
            show: function() {

            }
        }
        if (options.unmerged) {
            trayOptions.buttons.push(
                {
                    id: "red-ui-diff-view-resolve-diff",
                    text: RED._("diff.saveConflict"),
                    class: "primary disabled",
                    click: function() {
                        if (!$("#red-ui-diff-view-resolve-diff").hasClass('disabled')) {
                            if (options.currentDiff) {
                                // This is a flow file. Need to apply the diff
                                // and generate the new flow.
                                var result = applyDiff(options.currentDiff);
                                currentResolution = {
                                    resolutions:{}
                                };
                                currentResolution.resolutions[options.project.files.flow] = JSON.stringify(result.config,"",4);
                            }
                            if (options.onresolve) {
                                options.onresolve(currentResolution);
                            }
                            RED.tray.close();
                        }
                    }
                }
            );
        }
        RED.tray.show(trayOptions);
    }

    function parseCommitDiff(diff) {
        var result = {};
        var lines = diff.split("\n");
        var comment = [];
        for (var i=0;i<lines.length;i++) {
            if (/^commit /.test(lines[i])) {
                result.sha = lines[i].substring(7);
            } else if (/^Author: /.test(lines[i])) {
                result.author = lines[i].substring(8);
                var m = /^(.*) <(.*)>$/.exec(result.author);
                if (m) {
                    result.authorName = m[1];
                    result.authorEmail = m[2];
                }
            } else if (/^Date: /.test(lines[i])) {
                result.date = lines[i].substring(8);
            } else if (/^    /.test(lines[i])) {
                if (!result.title) {
                    result.title = lines[i].substring(4);
                } else {
                    if (lines[i].length !== 4 || comment.length > 0) {
                        comment.push(lines[i].substring(4));
                    }
                }
            } else if (/^diff /.test(lines[i])) {
                result.files = parseUnifiedDiff(lines.slice(i));
                break;
            }
         }
         result.comment = comment.join("\n");
         return result;
    }
    function parseUnifiedDiff(diff) {
        var lines;
        if (Array.isArray(diff)) {
            lines = diff;
        } else {
            lines = diff.split("\n");
        }
        var diffHeader = /^diff (?:(?:--git a\/(.*) b\/(.*))|(?:--cc (.*)))$/;
        var fileHeader = /^\+\+\+ b\/(.*)\t?/;
        var binaryFile = /^Binary files /;
        var hunkHeader = /^@@ -((\d+)(,(\d+))?) \+((\d+)(,(\d+))?) @@ ?(.*)$/;
        var conflictHunkHeader = /^@+ -((\d+)(,(\d+))?) -((\d+)(,(\d+))?) \+((\d+)(,(\d+))?) @+/;
        var files = [];
        var currentFile;
        var hunks = [];
        var currentHunk;
        for (var i=0;i<lines.length;i++) {
            var line = lines[i];
            var diffLine = diffHeader.exec(line);
            if (diffLine) {
                if (currentHunk) {
                    currentFile.hunks.push(currentHunk);
                    files.push(currentFile);
                }
                currentHunk = null;
                currentFile = {
                    file: diffLine[1]||diffLine[3],
                    hunks: []
                }
            } else if (binaryFile.test(line)) {
                if (currentFile) {
                    currentFile.binary = true;
                }
            } else {
                var fileLine = fileHeader.exec(line);
                if (fileLine) {
                    currentFile.file = fileLine[1];
                } else {
                    var hunkLine = hunkHeader.exec(line);
                    if (hunkLine) {
                        if (currentHunk) {
                            currentFile.hunks.push(currentHunk);
                        }
                        currentHunk = {
                            header: line,
                            localStartLine: hunkLine[2],
                            localLength: hunkLine[4]||1,
                            remoteStartLine: hunkLine[6],
                            remoteLength: hunkLine[8]||1,
                            lines: [],
                            conflict: false
                        }
                        continue;
                    }
                    hunkLine = conflictHunkHeader.exec(line);
                    if (hunkLine) {
                        if (currentHunk) {
                            currentFile.hunks.push(currentHunk);
                        }
                        currentHunk = {
                            header: line,
                            localStartLine: hunkLine[2],
                            localLength: hunkLine[4]||1,
                            remoteStartLine: hunkLine[6],
                            remoteLength: hunkLine[8]||1,
                            diffStart: parseInt(hunkLine[10]),
                            lines: [],
                            conflict: true
                        }
                        continue;
                    }
                    if (currentHunk) {
                        currentHunk.lines.push(line);
                    }
                }
            }
        }
        if (currentHunk) {
            currentFile.hunks.push(currentHunk);
        }
        files.push(currentFile);
        return files;
    }

    return {
        init: init,
        getRemoteDiff: getRemoteDiff,
        showRemoteDiff: showRemoteDiff,
        showUnifiedDiff: showUnifiedDiff,
        showCommitDiff: showCommitDiff,
        mergeDiff: mergeDiff
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
RED.keyboard = (function() {

    var isMac = /Mac/i.test(window.navigator.platform);

    var handlersActive = true;

    var handlers = {};

    var knownShortcuts;

    var partialState;

    var keyMap = {
        "left":37,
        "up":38,
        "right":39,
        "down":40,
        "escape":27,
        "enter": 13,
        "backspace": 8,
        "delete": 46,
        "space": 32,
        "tab": 9,
        ";":186,
        "=":187,
        "+":187, // <- QWERTY specific
        ",":188,
        "-":189,
        ".":190,
        "/":191,
        "\\":220,
        "'":222,
        "?":191, // <- QWERTY specific
        "[": 219,
        "]": 221,
        "{": 219,// <- QWERTY specific
        "}": 221 // <- QWERTY specific
    };
    var metaKeyCodes = {
        16: true,
        17: true,
        18: true,
        91: true,
        93: true
    };
    var actionToKeyMap = {};
    var defaultKeyMap = {};

    // FF generates some different keycodes because reasons.
    var firefoxKeyCodeMap = {
        59:186,
        61:187,
        173:189
    };

    function migrateOldKeymap() {
        // pre-0.18
        if ('localStorage' in window && window['localStorage'] !== null) {
            var oldKeyMap = localStorage.getItem("keymap");
            if (oldKeyMap !== null) {
                localStorage.removeItem("keymap");
                RED.settings.set('editor.keymap',JSON.parse(oldKeyMap));
            }
        }

    }

    function getUserKey(action) {
        return RED.settings.get('editor.keymap',{})[action];
    }

    function mergeKeymaps(defaultKeymap, themeKeymap) {
        // defaultKeymap has format: { scope: { key: action , key: action }}
        // themeKeymap has format: {action: {scope,key}, action: {scope:key}}


        var mergedKeymap = {};
        for (var scope in defaultKeymap) {
            if (defaultKeymap.hasOwnProperty(scope)) {
                var keys = defaultKeymap[scope];
                for (var key in keys) {
                    if (keys.hasOwnProperty(key)) {
                        if (!mergedKeymap[keys[key]]) {
                            mergedKeymap[keys[key]] = [{
                                scope:scope,
                                key:key,
                                user:false
                            }];
                        } else {
                            mergedKeymap[keys[key]].push({
                                scope:scope,
                                key:key,
                                user:false
                            });
                        }
                    }
                }
            }
        }
        for (var action in themeKeymap) {
            if (themeKeymap.hasOwnProperty(action)) {
                if (!themeKeymap[action].key) {
                    // No key for this action - default is no keybinding
                    delete mergedKeymap[action];
                } else {
                    mergedKeymap[action] = [{
                        scope: themeKeymap[action].scope || "*",
                        key: themeKeymap[action].key,
                        user: false
                    }];
                    if (mergedKeymap[action][0].scope === "workspace") {
                        mergedKeymap[action][0].scope = "red-ui-workspace";
                    }
                }
            }
        }
        return mergedKeymap;
    }

    function init(done) {
        // Migrate from pre-0.18
        migrateOldKeymap();

        var userKeymap = RED.settings.get('editor.keymap', {});
        $.getJSON("red/keymap.json",function(defaultKeymap) {
            var keymap = mergeKeymaps(defaultKeymap, RED.settings.theme('keymap',{}));
            // keymap has the format:  {action: [{scope,key},{scope,key}], action: [{scope:key}]}

            var action;
            for (action in keymap) {
                if (keymap.hasOwnProperty(action)) {
                    if (!userKeymap.hasOwnProperty(action)) {
                        keymap[action].forEach(function(km) {
                            addHandler(km.scope,km.key,action,false);
                        });
                    }
                    defaultKeyMap[action] = keymap[action][0];
                }
            }

            for (var action in userKeymap) {
                if (userKeymap.hasOwnProperty(action) && userKeymap[action]) {
                    var obj = userKeymap[action];
                    if (obj.hasOwnProperty('key')) {
                        var scope = obj.scope;
                        if (scope === "workspace") {
                            scope = "red-ui-workspace";
                        }
                        addHandler(scope, obj.key, action, true);
                    }
                }
            }
            done();
        });

        RED.userSettings.add({
            id:'keyboard',
            title: RED._("keyboard.keyboard"),
            get: getSettingsPane,
            focus: function() {
                setTimeout(function() {
                    $("#red-ui-settings-tab-keyboard-filter").trigger("focus");
                },200);
            },
            close: function() {
                RED.menu.refreshShortcuts();
            }
        });
    }

    function revertToDefault(action) {
        var currentAction = actionToKeyMap[action];
        if (currentAction) {
            removeHandler(currentAction.key);
        }
        if (defaultKeyMap.hasOwnProperty(action)) {
            var obj = defaultKeyMap[action];
            addHandler(obj.scope, obj.key, action, false);
        }
    }
    function parseKeySpecifier(key) {
        var parts = key.toLowerCase().split("-");
        var modifiers = {};
        var keycode;
        var blank = 0;
        for (var i=0;i<parts.length;i++) {
            switch(parts[i]) {
                case "ctrl":
                case "cmd":
                    modifiers.ctrl = true;
                    modifiers.meta = true;
                    break;
                case "alt":
                    modifiers.alt = true;
                    break;
                case "shift":
                    modifiers.shift = true;
                    break;
                case "":
                    blank++;
                    keycode = keyMap["-"];
                    break;
                default:
                    if (keyMap.hasOwnProperty(parts[i])) {
                        keycode = keyMap[parts[i]];
                    } else if (parts[i].length > 1) {
                        return null;
                    } else {
                        keycode = parts[i].toUpperCase().charCodeAt(0);
                    }
                    break;
            }
        }
        return [keycode,modifiers];
    }

    function matchHandlerToEvent(evt,handler) {
        var target = evt.target;
        var depth = 0;
        while (target.nodeName !== 'BODY' && target.id !== handler.scope) {
            target = target.parentElement;
            depth++;
        }
        if (target.nodeName === 'BODY' && handler.scope !== "*") {
            depth = -1;
        }
        return depth;
    }

    function resolveKeyEvent(evt) {
        var slot = partialState||handlers;
        // We cheat with MacOS CMD key and consider it the same as Ctrl.
        // That means we don't have to have separate keymaps for different OS.
        // It mostly works.
        // One exception is shortcuts that include both Cmd and Ctrl. We don't
        // support them - but we need to make sure we don't block browser-specific
        // shortcuts (such as Cmd-Ctrl-F for fullscreen).
        if (evt.ctrlKey && evt.metaKey) {
            return null; // dont handle both cmd+ctrl - let browser handle this
        }
        if (evt.ctrlKey || evt.metaKey) {
            slot = slot.ctrl;
        }
        if (slot && evt.shiftKey) {
            slot = slot.shift;
        }
        if (slot && evt.altKey) {
            slot = slot.alt;
        }
        var keyCode = firefoxKeyCodeMap[evt.keyCode] || evt.keyCode;
        if (slot && slot[keyCode]) {
            var handler = slot[keyCode];
            if (!handler.handlers) {
                if (partialState) {
                    partialState = null;
                    return resolveKeyEvent(evt);
                }
                if (Object.keys(handler).length > 0) {
                    // check if there's a potential combined handler initiated by this keyCode 
                    for (let h in handler) {
                        if (matchHandlerToEvent(evt,handler[h]) > -1) {
                            partialState = handler;
                            evt.preventDefault();
                            break;
                        }
                    }
                }
                return null;
            } else {
                var depth = Infinity;
                var matchedHandler;
                var i = 0;
                var l = handler.handlers.length;
                for (i=0;i<l;i++) {
                    var d = matchHandlerToEvent(evt,handler.handlers[i]);
                    if (d > -1 && d < depth) {
                        depth = d;
                        matchedHandler = handler.handlers[i];
                    }
                }
                handler = matchedHandler;
            }
            partialState = null;
            return handler;
        } else if (partialState) {
            partialState = null;
            return resolveKeyEvent(evt);
        }
    }
    d3.select(window).on("keydown", () => {
        handleEvent(d3.event)
    })

    function handleEvent (evt) {
        if (!handlersActive) {
            return;
        }
        if (metaKeyCodes[evt]) {
            return;
        }
        var handler = resolveKeyEvent(evt);
        if (handler && handler.ondown) {
            if (typeof handler.ondown === "string") {
                RED.actions.invoke(handler.ondown);
            } else {
                handler.ondown();
            }
            evt.preventDefault();
        }        
    }

    function addHandler(scope,key,modifiers,ondown) {
        var mod = modifiers;
        var cbdown = ondown;
        if (typeof modifiers == "function" || typeof modifiers === "string") {
            mod = {};
            cbdown = modifiers;
        }
        var keys = [];
        var i=0;
        if (typeof key === 'string') {
            if (typeof cbdown === 'string') {
                if (!ondown && !defaultKeyMap.hasOwnProperty(cbdown)) {
                    defaultKeyMap[cbdown] = {
                        scope:scope,
                        key:key,
                        user:false
                    };
                }
                if (!ondown) {
                    var userAction = getUserKey(cbdown);
                    if (userAction) {
                        return;
                    }
                }
                actionToKeyMap[cbdown] = {scope:scope,key:key};
                if (typeof ondown === 'boolean') {
                    actionToKeyMap[cbdown].user = ondown;
                }
            }
            var parts = key.split(" ");
            for (i=0;i<parts.length;i++) {
                var parsedKey = parseKeySpecifier(parts[i]);
                if (parsedKey) {
                    keys.push(parsedKey);
                } else {
                    return;
                }
            }
        } else {
            keys.push([key,mod]);
        }
        var slot = handlers;
        for (i=0;i<keys.length;i++) {
            key = keys[i][0];
            mod = keys[i][1];
            if (mod.ctrl) {
                slot.ctrl = slot.ctrl||{};
                slot = slot.ctrl;
            }
            if (mod.shift) {
                slot.shift = slot.shift||{};
                slot = slot.shift;
            }
            if (mod.alt) {
                slot.alt = slot.alt||{};
                slot = slot.alt;
            }
            slot[key] = slot[key] || {};
            slot = slot[key];
            //slot[key] = {scope: scope, ondown:cbdown};
        }
        slot.handlers = slot.handlers || [];
        slot.handlers.push({scope:scope,ondown:cbdown});
        slot.scope = scope;
        slot.ondown = cbdown;
    }

    function removeHandler(key,modifiers) {
        var mod = modifiers || {};
        var keys = [];
        var i=0;
        if (typeof key === 'string') {

            var parts = key.split(" ");
            for (i=0;i<parts.length;i++) {
                var parsedKey = parseKeySpecifier(parts[i]);
                if (parsedKey) {
                    keys.push(parsedKey);
                } else {
                    console.log("Unrecognised key specifier:",key);
                    return;
                }
            }
        } else {
            keys.push([key,mod]);
        }
        var slot = handlers;
        for (i=0;i<keys.length;i++) {
            key = keys[i][0];
            mod = keys[i][1];
            if (mod.ctrl) {
                slot = slot.ctrl;
            }
            if (slot && mod.shift) {
                slot = slot.shift;
            }
            if (slot && mod.alt) {
                slot = slot.alt;
            }
            if (!slot[key]) {
                return;
            }
            slot = slot[key];
        }
        if (typeof slot.ondown === "string") {
            if (typeof modifiers === 'boolean' && modifiers) {
                actionToKeyMap[slot.ondown] = {user: modifiers};
            } else {
                delete actionToKeyMap[slot.ondown];
            }
        }
        delete slot.scope;
        delete slot.ondown;
        // TODO: this wipes everything! Need to have something to identify handler
        delete slot.handlers;
    }

    var cmdCtrlKey = '<span class="help-key">'+(isMac?'&#8984;':'Ctrl')+'</span>';

    function formatKey(key,plain) {
        var formattedKey = isMac?key.replace(/ctrl-?/,"&#8984;"):key;
        formattedKey = isMac?formattedKey.replace(/alt-?/,"&#8997;"):key;
        formattedKey = formattedKey.replace(/shift-?/,"&#8679;");
        formattedKey = formattedKey.replace(/left/,"&#x2190;");
        formattedKey = formattedKey.replace(/up/,"&#x2191;");
        formattedKey = formattedKey.replace(/right/,"&#x2192;");
        formattedKey = formattedKey.replace(/down/,"&#x2193;");
        if (plain) {
            return formattedKey;
        }
        return '<span class="help-key-block"><span class="help-key">'+formattedKey.split(" ").join('</span> <span class="help-key">')+'</span></span>';
    }

    function validateKey(key) {
        key = key.trim();
        var parts = key.split(" ");
        for (i=0;i<parts.length;i++) {
            var parsedKey = parseKeySpecifier(parts[i]);
            if (!parsedKey) {
                return false;
            }
        }
        return true;
    }

    function editShortcut(e) {
        e.preventDefault();
        var container = $(this);
        var object = container.data('data');

        if (!container.hasClass('keyboard-shortcut-entry-expanded')) {
            endEditShortcut();

            var key = container.find(".keyboard-shortcut-entry-key");
            var scope = container.find(".keyboard-shortcut-entry-scope");
            container.addClass('keyboard-shortcut-entry-expanded');

            var keyInput = $('<input type="text">').attr('placeholder',RED._('keyboard.unassigned')).val(object.key||"").appendTo(key);
            keyInput.on("change paste keyup",function(e) {
                if (e.keyCode === 13 && !$(this).hasClass("input-error")) {
                    return endEditShortcut();
                }
                if (e.keyCode === 27) {
                    return endEditShortcut(true);
                }
                var currentVal = $(this).val();
                currentVal = currentVal.trim();
                var valid = (currentVal === "" || RED.keyboard.validateKey(currentVal));
                if (valid && currentVal !== "") {
                    valid = !knownShortcuts.has(scopeSelect.val()+":"+currentVal.toLowerCase());
                }
                $(this).toggleClass("input-error",!valid);
                okButton.attr("disabled",!valid);
            });

            var scopeSelect = $('<select>'+
                '<option value="*" data-i18n="keyboard.global"></option>'+
                '<option value="red-ui-workspace" data-i18n="keyboard.workspace"></option>'+
                '<option value="red-ui-editor-stack" data-i18n="keyboard.editor"></option>'+
                '</select>').appendTo(scope);
            scopeSelect.i18n();
            if (object.scope === "workspace") {
                object.scope = "red-ui-workspace";
            }
            scopeSelect.val(object.scope||'*');
            scopeSelect.on("change", function() {
                keyInput.trigger("change");
            });

            var div = $('<div class="keyboard-shortcut-edit button-group-vertical"></div>').appendTo(scope);
            var okButton = $('<button class="red-ui-button red-ui-button-small"><i class="fa fa-check"></i></button>').appendTo(div);
            var revertButton = $('<button class="red-ui-button red-ui-button-small"><i class="fa fa-reply"></i></button>').appendTo(div);

            okButton.on("click", function(e) {
                e.stopPropagation();
                endEditShortcut();
            });
            revertButton.on("click", function(e) {
                e.stopPropagation();
                container.empty();
                container.removeClass('keyboard-shortcut-entry-expanded');

                var userKeymap = RED.settings.get('editor.keymap', {});
                userKeymap[object.id] = null;
                RED.settings.set('editor.keymap',userKeymap);

                RED.keyboard.revertToDefault(object.id);

                var shortcut = RED.keyboard.getShortcut(object.id);
                var obj = {
                    id:object.id,
                    scope:shortcut?shortcut.scope:undefined,
                    key:shortcut?shortcut.key:undefined,
                    user:shortcut?shortcut.user:undefined,

                    label: object.label,
                    options: object.options,
                };
                buildShortcutRow(container,obj);
            });

            keyInput.trigger("focus");
        }
    }

    function endEditShortcut(cancel) {
        var container = $('.keyboard-shortcut-entry-expanded');
        if (container.length === 1) {
            var object = container.data('data');
            var keyInput = container.find(".keyboard-shortcut-entry-key input");
            var scopeSelect = container.find(".keyboard-shortcut-entry-scope select");
            if (!cancel) {
                var key = keyInput.val().trim();
                var scope = scopeSelect.val();
                var valid = (key === "" || RED.keyboard.validateKey(key));
                if (valid) {
                    var current = RED.keyboard.getShortcut(object.id);
                    if ((!current && key) || (current && (current.scope !== scope || current.key !== key))) {
                        var keyDiv = container.find(".keyboard-shortcut-entry-key");
                        var scopeDiv = container.find(".keyboard-shortcut-entry-scope");
                        keyDiv.empty();
                        scopeDiv.empty();
                        if (object.key) {
                            knownShortcuts.delete(object.scope+":"+object.key);
                            RED.keyboard.remove(object.key,true);
                        }
                        container.find(".keyboard-shortcut-entry-text i").css("opacity",1);
                        if (key === "") {
                            keyDiv.parent().addClass("keyboard-shortcut-entry-unassigned");
                            keyDiv.append($('<span>').text(RED._('keyboard.unassigned'))  );
                            delete object.key;
                            delete object.scope;
                        } else {
                            keyDiv.parent().removeClass("keyboard-shortcut-entry-unassigned");
                            keyDiv.append(RED.keyboard.formatKey(key));
                            $("<span>").text(scope).appendTo(scopeDiv);
                            object.key = key;
                            object.scope = scope;
                            knownShortcuts.add(object.scope+":"+object.key);
                            RED.keyboard.add(object.scope,object.key,object.id,true);
                        }

                        var userKeymap = RED.settings.get('editor.keymap', {});
                        var shortcut = RED.keyboard.getShortcut(object.id);
                        userKeymap[object.id] = {
                            scope:shortcut.scope,
                            key:shortcut.key
                        };
                        RED.settings.set('editor.keymap',userKeymap);
                    }
                }
            }
            keyInput.remove();
            scopeSelect.remove();
            $('.keyboard-shortcut-edit').remove();
            container.removeClass('keyboard-shortcut-entry-expanded');
        }
    }

    function buildShortcutRow(container,object) {
        var item = $('<div class="keyboard-shortcut-entry">').appendTo(container);
        container.data('data',object);

        var text = object.label;
        var label = $('<div>').addClass("keyboard-shortcut-entry-text").text(text).appendTo(item);

        var user = $('<i class="fa fa-user"></i>').prependTo(label);

        if (!object.user) {
            user.css("opacity",0);
        }

        var key = $('<div class="keyboard-shortcut-entry-key">').appendTo(item);
        if (object.key) {
            key.append(RED.keyboard.formatKey(object.key));
        } else {
            item.addClass("keyboard-shortcut-entry-unassigned");
            key.append($('<span>').text(RED._('keyboard.unassigned'))  );
        }

        var scope = $('<div class="keyboard-shortcut-entry-scope">').appendTo(item);

        $("<span>").text(object.scope === '*'?'global':object.scope||"").appendTo(scope);
        container.on("click", editShortcut);
    }

    function getSettingsPane() {
        var pane = $('<div id="red-ui-settings-tab-keyboard"></div>');

        $('<div class="keyboard-shortcut-entry keyboard-shortcut-list-header">'+
        '<div class="keyboard-shortcut-entry-key keyboard-shortcut-entry-text"><input autocomplete="off" name="keyboard-filter" id="red-ui-settings-tab-keyboard-filter" type="text" data-i18n="[placeholder]keyboard.filterActions"></div>'+
        '<div class="keyboard-shortcut-entry-key" data-i18n="keyboard.shortcut"></div>'+
        '<div class="keyboard-shortcut-entry-scope" data-i18n="keyboard.scope"></div>'+
        '</div>').appendTo(pane);

        pane.find("#red-ui-settings-tab-keyboard-filter").searchBox({
            delay: 100,
            change: function() {
                var filterValue = $(this).val().trim().toLowerCase();
                if (filterValue === "") {
                    shortcutList.editableList('filter', null);
                } else {
                    filterValue = filterValue.replace(/\s/g,"");
                    shortcutList.editableList('filter', function(data) {
                        var label = data.label.toLowerCase();
                        return label.indexOf(filterValue) > -1;
                    });
                }
            }
        });

        var shortcutList = $('<ol class="keyboard-shortcut-list"></ol>').css({
            position: "absolute",
            top: "32px",
            bottom: "0",
            left: "0",
            right: "0"
        }).appendTo(pane).editableList({
            addButton: false,
            scrollOnAdd: false,
            addItem: function(container,i,object) {
                buildShortcutRow(container,object);
            },

        });
        var shortcuts = RED.actions.list();
        shortcuts.sort(function(A,B) {
            var Akey = A.label;
            var Bkey = B.label;
            return Akey.localeCompare(Bkey);
        });
        knownShortcuts = new Set();
        shortcuts.forEach(function(s) {
            if (s.key) {
                knownShortcuts.add(s.scope+":"+s.key);
            }
            shortcutList.editableList('addItem',s);
        });
        return pane;
    }

    function enable() {
        handlersActive = true;
    }
    function disable() {
        handlersActive = false;
    }

    return {
        init: init,
        add: addHandler,
        remove: removeHandler,
        getShortcut: function(actionName) {
            return actionToKeyMap[actionName];
        },
        getUserShortcut: getUserKey,
        revertToDefault: revertToDefault,
        formatKey: formatKey,
        validateKey: validateKey,
        disable: disable,
        enable: enable,
        handle: handleEvent
    }

})();
;RED.envVar = (function() {
    function saveEnvList(list) {
        const items = list.editableList("items")
        const new_env = [];
        items.each(function (i,el) {
            var data = el.data('data');
            var item;
            if (data.nameField && data.valueField) {
                item = {
                    name: data.nameField.val(),
                    value: data.valueField.typedInput("value"),
                    type: data.valueField.typedInput("type")
                };
                new_env.push(item);
            }
        });
        return new_env;
    }

    function getGlobalConf(create) {
        var gconf = null;
        RED.nodes.eachConfig(function (conf) {
            if (conf.type === "global-config") {
                gconf = conf;
            }
        });
        if ((gconf === null) && create) {
            var cred = {
                _ : {},
                map: {}
            };
            gconf = {
                id: RED.nodes.id(),
                type: "global-config",
                env: [],
                modules: {},
                hasUsers: false,
                users: [],
                credentials: cred,
                _def: RED.nodes.getType("global-config"),
            };
            RED.nodes.add(gconf);
        }
        return gconf;
    }

    function applyChanges(list) {
        var gconf = getGlobalConf(false);
        var new_env = [];
        var items = list.editableList('items');
        var credentials = gconf ? gconf.credentials : null;
        if (!gconf && list.editableList('length') === 0) {
            // No existing global-config node and nothing in the list,
            // so no need to do anything more
            return
        }
        if (!credentials) {
            credentials = {
                _ : {},
                map: {}
            };
        }
        items.each(function (i,el) {
            var data = el.data('data');
            if (data.nameField && data.valueField) {
                var item = {
                    name: data.nameField.val(),
                    value: data.valueField.typedInput("value"),
                    type: data.valueField.typedInput("type")
                };
                if (item.name.trim() !== "") {
                    new_env.push(item);
                    if (item.type === "cred") {
                        credentials.map[item.name] = item.value;
                        credentials.map["has_"+item.name] = (item.value !== "");
                        item.value = "__PWRD__";
                    }
                }
            }
        });
        if (gconf === null) {
            gconf = getGlobalConf(true);
        }
        if (!gconf.credentials) {
            gconf.credentials = {
                _ : {},
                map: {}
            };
        }
        if ((JSON.stringify(new_env) !== JSON.stringify(gconf.env)) ||
            (JSON.stringify(credentials) !== JSON.stringify(gconf.credentials))) {
            gconf.env = new_env;
            gconf.credentials = credentials;
            RED.nodes.dirty(true);
        }
    }

    function getSettingsPane() {
        var gconf = getGlobalConf(false);
        var env = gconf ? gconf.env : [];
        var cred = gconf ? gconf.credentials : null;
        if (!cred) {
            cred = {
                _ : {},
                map: {}
            };
        }

        var pane = $("<div/>", {
            id: "red-ui-settings-tab-envvar",
            class: "form-horizontal"
        });
        var content = $("<div/>", {
            class: "form-row node-input-env-container-row"
        }).css({
            "margin": "10px"
        }).appendTo(pane);

        var label = $("<label></label>").css({
            width: "100%"
        }).appendTo(content);
        $("<i/>", {
            class: "fa fa-list"
        }).appendTo(label);
        $("<span/>").text(" "+RED._("env-var.header")).appendTo(label);

        var list = $("<ol/>", {
            id: "node-input-env-container"
        }).appendTo(content);
        var node = {
            type: "",
            env: env,
            credentials: cred.map,
        };
        RED.editor.envVarList.create(list, node);

        var buttons = $("<div/>").css({
            "text-align": "right",
        }).appendTo(content);
        var revertButton = $("<button/>", {
            class: "red-ui-button"
        }).css({
        }).text(RED._("env-var.revert")).appendTo(buttons);

        var items = saveEnvList(list);
        revertButton.on("click", function (ev) {
            list.editableList("empty");
            list.editableList("addItems", items);
        });

        return pane;
    }

    function init(done) {
        RED.userSettings.add({
            id:'envvar',
            title: RED._("env-var.environment"),
            get: getSettingsPane,
            focus: function() {
                var height = $("#red-ui-settings-tab-envvar").parent().height();
                $("#node-input-env-container").editableList("height", (height -100));
            },
            close: function() {
                var list = $("#node-input-env-container");
                try {
                    applyChanges(list);
                }
                catch (e) {
                    console.log(e);
                    console.log(e.stack);
                }
            }
        });

        RED.actions.add("core:show-global-env", function() {
            RED.userSettings.show('envvar');
        });
    }

    return {
        init: init,
    };

})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/


RED.workspaces = (function() {

    const documentTitle = document.title;

    var activeWorkspace = 0;
    var workspaceIndex = 0;

    var viewStack = [];
    var hideStack = [];
    var viewStackPos = 0;

    let flashingTab;
    let flashingTabTimer;

    function addToViewStack(id) {
        if (viewStackPos !== viewStack.length) {
            viewStack.splice(viewStackPos);
        }
        viewStack.push(id);
        viewStackPos = viewStack.length;
    }

    function removeFromHideStack(id) {
        hideStack = hideStack.filter(function(v) {
            if (v === id) {
                return false;
            } else if (Array.isArray(v)) {
                var i = v.indexOf(id);
                if (i > -1) {
                    v.splice(i,1);
                }
                if (v.length === 0) {
                    return false;
                }
                return true
            }
            return true;
        })
    }

    function addWorkspace(ws,skipHistoryEntry,targetIndex) {
        if (ws) {
            if (!ws.closeable) {
                ws.hideable = true;
            }
            if (!ws.hasOwnProperty('locked')) {
                ws.locked = false
            }
            workspace_tabs.addTab(ws,targetIndex);

            var hiddenTabs = JSON.parse(RED.settings.getLocal("hiddenTabs")||"{}");
            if (hiddenTabs[ws.id]) {
                workspace_tabs.hideTab(ws.id);
            }
            workspace_tabs.resize();
        } else {
            var tabId = RED.nodes.id();
            do {
                workspaceIndex += 1;
            } while ($("#red-ui-workspace-tabs li[flowname='"+RED._('workspace.defaultName',{number:workspaceIndex})+"']").size() !== 0);

            ws = {
                type: "tab",
                id: tabId,
                disabled: false,
                locked: false,
                info: "",
                label: RED._('workspace.defaultName',{number:workspaceIndex}),
                env: [],
                hideable: true,
            };
            if (!skipHistoryEntry) {
                ws.added = true
            }
            RED.nodes.addWorkspace(ws,targetIndex);
            workspace_tabs.addTab(ws,targetIndex);

            workspace_tabs.activateTab(tabId);
            if (!skipHistoryEntry) {
                RED.history.push({t:'add',workspaces:[ws],dirty:RED.nodes.dirty()});
                RED.nodes.dirty(true);
            }
        }
        $("#red-ui-tab-"+(ws.id.replace(".","-"))).attr("flowname",ws.label).toggleClass('red-ui-workspace-changed',!!(ws.contentsChanged || ws.changed || ws.added));
        RED.view.focus();
        return ws;
    }

    function deleteWorkspace(ws) {
        if (workspaceTabCount === 1) {
            return;
        }
        if (ws.locked) {
            return
        }
        var workspaceOrder = RED.nodes.getWorkspaceOrder();
        ws._index = workspaceOrder.indexOf(ws.id);
        removeWorkspace(ws);
        var historyEvent = RED.nodes.removeWorkspace(ws.id);
        historyEvent.t = 'delete';
        historyEvent.dirty = RED.nodes.dirty();
        historyEvent.workspaces = [ws];
        RED.history.push(historyEvent);
        RED.nodes.dirty(true);
        RED.sidebar.config.refresh();
    }

    function showEditWorkspaceDialog(id) {
        var workspace = RED.nodes.workspace(id);
        if (!workspace) {
            var subflow = RED.nodes.subflow(id);
            if (subflow) {
                RED.editor.editSubflow(subflow);
            }
        } else {
            if (!workspace.locked) {
                RED.editor.editFlow(workspace);
            }
        }
    }


    var workspace_tabs;
    var workspaceTabCount = 0;

    function getMenuItems(isMenuButton, tab) {
        let hiddenFlows = new Set()
        for (let i = 0; i < hideStack.length; i++) {
            let ids = hideStack[i]
            if (!Array.isArray(ids)) {
                ids = [ids]
            }
            ids.forEach(id => {
                if (RED.nodes.workspace(id)) {
                    hiddenFlows.add(id)
                }
            })
        }
        const hiddenflowCount = hiddenFlows.size;
        let activeWorkspace = tab || RED.nodes.workspace(RED.workspaces.active()) || RED.nodes.subflow(RED.workspaces.active())
        let isFlowDisabled = activeWorkspace ? activeWorkspace.disabled : false
        const currentTabs = workspace_tabs.listTabs();
        let flowCount = 0;
        currentTabs.forEach(tab => {
            if (RED.nodes.workspace(tab)) {
                flowCount++;
            }
        });

        let isCurrentLocked = RED.workspaces.isLocked()
        if (tab) {
            isCurrentLocked = tab.locked
        }

        var menuItems = []
        if (isMenuButton) {
            menuItems.push({
                id:"red-ui-tabs-menu-option-search-flows",
                label: RED._("workspace.listFlows"),
                onselect: "core:list-flows"
            },
            {
                id:"red-ui-tabs-menu-option-search-subflows",
                label: RED._("workspace.listSubflows"),
                onselect: "core:list-subflows"
            },
            null)
        }
        if (RED.settings.theme("menu.menu-item-workspace-add", true)) {
            menuItems.push(
                {
                    id:"red-ui-tabs-menu-option-add-flow",
                    label: RED._("workspace.addFlow"),
                    onselect: "core:add-flow"
                }
            )
        }
        if (isMenuButton || !!tab) {
            if (RED.settings.theme("menu.menu-item-workspace-add", true)) {
                menuItems.push(
                    {
                        id:"red-ui-tabs-menu-option-add-flow-right",
                        label: RED._("workspace.addFlowToRight"),
                        shortcut: RED.keyboard.getShortcut("core:add-flow-to-right"),
                        onselect: function() {
                            RED.actions.invoke("core:add-flow-to-right", tab)
                        }
                    },
                    null
                )
            }
            if (activeWorkspace && activeWorkspace.type === 'tab') {
                menuItems.push(
                    isFlowDisabled ? {
                        label: RED._("workspace.enableFlow"),
                        shortcut: RED.keyboard.getShortcut("core:enable-flow"),
                        onselect: function() {
                            RED.actions.invoke("core:enable-flow", tab?tab.id:undefined)
                        },
                        disabled: isCurrentLocked
                    } : {
                        label: RED._("workspace.disableFlow"),
                        shortcut: RED.keyboard.getShortcut("core:disable-flow"),
                        onselect: function() {
                            RED.actions.invoke("core:disable-flow", tab?tab.id:undefined)
                        },
                        disabled: isCurrentLocked
                    },
                    isCurrentLocked? {
                        label: RED._("workspace.unlockFlow"),
                        shortcut: RED.keyboard.getShortcut("core:unlock-flow"),
                        onselect: function() {
                            RED.actions.invoke('core:unlock-flow', tab?tab.id:undefined)
                        }
                    } : {
                        label: RED._("workspace.lockFlow"),
                        shortcut: RED.keyboard.getShortcut("core:lock-flow"),
                        onselect: function() {
                            RED.actions.invoke('core:lock-flow', tab?tab.id:undefined)
                        }
                    },
                    null
                )
            }
            const activeIndex = currentTabs.findIndex(id => (activeWorkspace && (id === activeWorkspace.id)));
            menuItems.push(
                {
                    label: RED._("workspace.moveToStart"),
                    shortcut: RED.keyboard.getShortcut("core:move-flow-to-start"),
                    onselect: function() {
                        RED.actions.invoke("core:move-flow-to-start", tab?tab.id:undefined)
                    },
                    disabled: activeIndex === 0
                },
                {
                    label: RED._("workspace.moveToEnd"),
                    shortcut: RED.keyboard.getShortcut("core:move-flow-to-end"),
                    onselect: function() {
                        RED.actions.invoke("core:move-flow-to-end", tab?tab.id:undefined)
                    },
                    disabled: activeIndex === currentTabs.length - 1
                }
            )
        }
        if (menuItems.length > 0) {
            menuItems.push(null)
        }
        if (isMenuButton || !!tab) {
            menuItems.push(
                {
                    id:"red-ui-tabs-menu-option-add-hide-flows",
                    label: RED._("workspace.hideFlow"),
                    shortcut: RED.keyboard.getShortcut("core:hide-flow"),
                    onselect: function() {
                        RED.actions.invoke("core:hide-flow", tab)
                    }
                },
                {
                    id:"red-ui-tabs-menu-option-add-hide-other-flows",
                    label: RED._("workspace.hideOtherFlows"),
                    shortcut: RED.keyboard.getShortcut("core:hide-other-flows"),
                    onselect: function() {
                        RED.actions.invoke("core:hide-other-flows", tab)
                    }
                }
            )

        }
        
        menuItems.push(
            {
                id:"red-ui-tabs-menu-option-add-hide-all-flows",
                label: RED._("workspace.hideAllFlows"),
                onselect: "core:hide-all-flows",
                disabled: (hiddenflowCount === flowCount)
            },
            {
                id:"red-ui-tabs-menu-option-add-show-all-flows",
                disabled: hiddenflowCount === 0,
                label: RED._("workspace.showAllFlows", { count: hiddenflowCount }),
                onselect: "core:show-all-flows"
            },
            {
                id:"red-ui-tabs-menu-option-add-show-last-flow",
                disabled: hideStack.length === 0,
                label: RED._("workspace.showLastHiddenFlow"),
                onselect: "core:show-last-hidden-flow"
            }
        )
        if (tab) {
            menuItems.push(null)

            if (RED.settings.theme("menu.menu-item-workspace-delete", true)) {
                menuItems.push(
                    {
                        label: RED._("common.label.delete"),
                        onselect: function() {
                            if (tab.type === 'tab') {
                                RED.workspaces.delete(tab)
                            } else if (tab.type === 'subflow') {
                                RED.subflow.delete(tab.id)
                            }
                        },
                        disabled: isCurrentLocked || (workspaceTabCount === 1)
                    }
                )
            }
            menuItems.push(
                {
                    label: RED._("menu.label.export"),
                    shortcut: RED.keyboard.getShortcut("core:show-export-dialog"),
                    onselect: function() {
                        RED.workspaces.show(tab.id)
                        RED.actions.invoke('core:show-export-dialog', null, 'flow')
                    }
                }
            )
        }
        // if (isMenuButton && hiddenflowCount > 0) {
        //     menuItems.unshift({
        //         label: RED._("workspace.hiddenFlows",{count: hiddenflowCount}),
        //         onselect: "core:list-hidden-flows"
        //     })
        // }
        return menuItems;
    }
    function createWorkspaceTabs() {
        workspace_tabs = RED.tabs.create({
            id: "red-ui-workspace-tabs",
            onchange: function(tab) {
                var event = {
                    old: activeWorkspace
                }
                if (tab) {
                    $("#red-ui-workspace-chart").show();
                    activeWorkspace = tab.id;
                    window.location.hash = 'flow/'+tab.id;
                    if (tab.label) {
                        document.title = `${documentTitle} : ${tab.label}`
                    } else {
                        document.title = documentTitle
                    }
                    $("#red-ui-workspace").toggleClass("red-ui-workspace-disabled", !!tab.disabled);
                    $("#red-ui-workspace").toggleClass("red-ui-workspace-locked", !!tab.locked);
                } else {
                    $("#red-ui-workspace-chart").hide();
                    activeWorkspace = 0;
                    window.location.hash = '';
                    document.title = documentTitle
                }
                event.workspace = activeWorkspace;
                RED.events.emit("workspace:change",event);
                RED.sidebar.config.refresh();
                RED.view.focus();
            },
            onclick: function(tab, evt) {
                if(evt.which === 2) {
                    evt.preventDefault();
                    evt.stopPropagation();
                    RED.actions.invoke("core:hide-flow", tab)
                } else {
                    if (tab.id !== activeWorkspace) {
                        addToViewStack(activeWorkspace);
                    }
                    RED.view.focus();
                }
            },
            ondblclick: function(tab) {
                if (tab.type != "subflow") {
                    showEditWorkspaceDialog(tab.id);
                } else {
                    RED.editor.editSubflow(RED.nodes.subflow(tab.id));
                }
            },
            onadd: function(tab) {
                if (tab.type === "tab") {
                    workspaceTabCount++;
                }
                $('<span class="red-ui-workspace-disabled-icon"><i class="fa fa-ban"></i> </span>').prependTo("#red-ui-tab-"+(tab.id.replace(".","-"))+" .red-ui-tab-label");
                if (tab.disabled) {
                    $("#red-ui-tab-"+(tab.id.replace(".","-"))).addClass('red-ui-workspace-disabled');
                }
                $('<span class="red-ui-workspace-locked-icon"><i class="fa fa-lock"></i> </span>').prependTo("#red-ui-tab-"+(tab.id.replace(".","-"))+" .red-ui-tab-label");
                if (tab.locked) {
                    $("#red-ui-tab-"+(tab.id.replace(".","-"))).addClass('red-ui-workspace-locked');
                }

                const changeBadgeContainer = $('<svg class="red-ui-flow-tab-changed red-ui-flow-node-changed" width="10" height="10" viewBox="-1 -1 12 12"></svg>').appendTo("#red-ui-tab-"+(tab.id.replace(".","-")))
                const changeBadge = document.createElementNS("http://www.w3.org/2000/svg","circle");
                changeBadge.setAttribute("cx",5);
                changeBadge.setAttribute("cy",5);
                changeBadge.setAttribute("r",5);
                changeBadgeContainer.append(changeBadge)

                RED.menu.setDisabled("menu-item-workspace-delete",activeWorkspace === 0 || workspaceTabCount <= 1);
                if (workspaceTabCount === 1) {
                    showWorkspace();
                }
            },
            onremove: function(tab) {
                if (tab.type === "tab") {
                    workspaceTabCount--;
                } else {
                    RED.events.emit("workspace:close",{workspace: tab.id})
                    hideStack.push(tab.id);
                }
                RED.menu.setDisabled("menu-item-workspace-delete",activeWorkspace === 0 || workspaceTabCount <= 1);
                if (workspaceTabCount === 0) {
                    hideWorkspace();
                }
            },
            onreorder: function(oldOrder, newOrder) {
                RED.history.push({
                    t:'reorder',
                    workspaces: {
                        from: oldOrder,
                        to: newOrder
                    },
                    dirty:RED.nodes.dirty()
                });
                // Only mark flows dirty if flow-order has changed (excluding subflows)
                const filteredOldOrder = oldOrder.filter(id => !!RED.nodes.workspace(id))
                const filteredNewOrder = newOrder.filter(id => !!RED.nodes.workspace(id))

                if (JSON.stringify(filteredOldOrder) !== JSON.stringify(filteredNewOrder)) {
                    RED.nodes.dirty(true);
                    setWorkspaceOrder(newOrder);
                }
            },
            onselect: function(selectedTabs) {
                RED.view.select(false)
                if (selectedTabs.length === 0) {
                    $("#red-ui-workspace-chart svg").css({"pointer-events":"auto",filter:"none"})
                    $("#red-ui-workspace-toolbar").css({"pointer-events":"auto",filter:"none"})
                    $("#red-ui-palette-container").css({"pointer-events":"auto",filter:"none"})
                    $(".red-ui-sidebar-shade").hide();
                } else {
                    RED.view.select(false)
                    $("#red-ui-workspace-chart svg").css({"pointer-events":"none",filter:"opacity(60%)"})
                    $("#red-ui-workspace-toolbar").css({"pointer-events":"none",filter:"opacity(60%)"})
                    $("#red-ui-palette-container").css({"pointer-events":"none",filter:"opacity(60%)"})
                    $(".red-ui-sidebar-shade").show();
                }
            },
            onhide: function(tab) {
                hideStack.push(tab.id);
                if (tab.type === "tab") {
                    var hiddenTabs = JSON.parse(RED.settings.getLocal("hiddenTabs")||"{}");
                    hiddenTabs[tab.id] = true;
                    RED.settings.setLocal("hiddenTabs",JSON.stringify(hiddenTabs));
                    RED.events.emit("workspace:hide",{workspace: tab.id})
                }
            },
            onshow: function(tab) {
                removeFromHideStack(tab.id);

                var hiddenTabs = JSON.parse(RED.settings.getLocal("hiddenTabs")||"{}");
                delete hiddenTabs[tab.id];
                RED.settings.setLocal("hiddenTabs",JSON.stringify(hiddenTabs));

                RED.events.emit("workspace:show",{workspace: tab.id})
            },
            minimumActiveTabWidth: 150,
            scrollable: true,
            addButton: RED.settings.theme("menu.menu-item-workspace-add", true) ? "core:add-flow" : undefined,
            addButtonCaption: RED._("workspace.addFlow"),
            menu: function() { return getMenuItems(true) },
            contextmenu: function(tab) { return getMenuItems(false, tab) }
        });
        workspaceTabCount = 0;
    }
    function showWorkspace() {
        $("#red-ui-workspace .red-ui-tabs").show()
        $("#red-ui-workspace-chart").show()
        $("#red-ui-workspace-footer").children().show()
    }
    function hideWorkspace() {
        $("#red-ui-workspace .red-ui-tabs").hide()
        $("#red-ui-workspace-chart").hide()
        $("#red-ui-workspace-footer").children().hide()
    }

    function init() {
        $('<ul id="red-ui-workspace-tabs"></ul>').appendTo("#red-ui-workspace");
        $('<div id="red-ui-workspace-tabs-shade" class="hide"></div>').appendTo("#red-ui-workspace");
        $('<div id="red-ui-workspace-chart" tabindex="1"></div>').appendTo("#red-ui-workspace");
        $('<div id="red-ui-workspace-toolbar"></div>').appendTo("#red-ui-workspace");
        $('<div id="red-ui-workspace-footer" class="red-ui-component-footer"></div>').appendTo("#red-ui-workspace");
        $('<div id="red-ui-editor-shade" class="hide"></div>').appendTo("#red-ui-workspace");


        createWorkspaceTabs();
        RED.events.on("sidebar:resize",workspace_tabs.resize);

        RED.events.on("workspace:clear", () => {
            // Reset the index used to generate new flow names
            workspaceIndex = 0
        })

        RED.actions.add("core:show-next-tab",function() {
            var oldActive = activeWorkspace;
            workspace_tabs.nextTab();
            if (oldActive !== activeWorkspace) {
                addToViewStack(oldActive)
            }
        });
        RED.actions.add("core:show-previous-tab",function() {
            var oldActive = activeWorkspace;
            workspace_tabs.previousTab();
            if (oldActive !== activeWorkspace) {
                addToViewStack(oldActive)
            }
        });

        RED.menu.setAction('menu-item-workspace-delete',function() {
            deleteWorkspace(RED.nodes.workspace(activeWorkspace));
        });

        $(window).on("resize", function() {
            workspace_tabs.resize();
        });
        if (RED.settings.theme("menu.menu-item-workspace-add", true)) {
            RED.actions.add("core:add-flow",function(opts) { addWorkspace(undefined,undefined,opts?opts.index:undefined)});
            RED.actions.add("core:add-flow-to-right",function(workspace) {
                let index
                if (workspace) {
                    index = workspace_tabs.getTabIndex(workspace.id)+1
                } else {
                    index = workspace_tabs.activeIndex()+1
                }
                addWorkspace(undefined,undefined,index)
            });
        }
        if (RED.settings.theme("menu.menu-item-workspace-edit", true)) {
            RED.actions.add("core:edit-flow",editWorkspace);
        }
        if (RED.settings.theme("menu.menu-item-workspace-delete", true)) {
            RED.actions.add("core:remove-flow",removeWorkspace);
        }
        RED.actions.add("core:enable-flow",enableWorkspace);
        RED.actions.add("core:disable-flow",disableWorkspace);
        RED.actions.add("core:lock-flow",lockWorkspace);
        RED.actions.add("core:unlock-flow",unlockWorkspace);
        RED.actions.add("core:move-flow-to-start", function(id) { moveWorkspace(id, 'start') });
        RED.actions.add("core:move-flow-to-end", function(id) { moveWorkspace(id, 'end') });

        RED.actions.add("core:hide-flow", function(workspace) {
            let selection
            if (workspace) {
                selection = [workspace]
            } else {
                selection = workspace_tabs.selection();
                if (selection.length === 0) {
                    selection = [{id:activeWorkspace}]
                }
            }
            var hiddenTabs = [];
            selection.forEach(function(ws) {
                RED.workspaces.hide(ws.id);
                hideStack.pop();
                hiddenTabs.push(ws.id);
            })
            if (hiddenTabs.length > 0) {
                hideStack.push(hiddenTabs);
            }
            workspace_tabs.clearSelection();
        })

        RED.actions.add("core:hide-other-flows", function(workspace) {
            let selection
            if (workspace) {
                selection = [workspace]
            } else {
                selection = workspace_tabs.selection();
                if (selection.length === 0) {
                    selection = [{id:activeWorkspace}]
                }
            }
            var selected = new Set(selection.map(function(ws) { return ws.id }))

            var currentTabs = workspace_tabs.listTabs();
            var hiddenTabs = [];
            currentTabs.forEach(function(id) {
                if (!selected.has(id)) {
                    RED.workspaces.hide(id);
                    hideStack.pop();
                    hiddenTabs.push(id);
                }
            })
            if (hiddenTabs.length > 0) {
                hideStack.push(hiddenTabs);
            }
        })

        RED.actions.add("core:hide-all-flows", function() {
            var currentTabs = workspace_tabs.listTabs();
            currentTabs.forEach(function(id) {
                RED.workspaces.hide(id);
                hideStack.pop();
            })
            if (currentTabs.length > 0) {
                hideStack.push(currentTabs);
            }
            workspace_tabs.clearSelection();
        })
        RED.actions.add("core:show-all-flows", function() {
            var currentTabs = workspace_tabs.listTabs();
            currentTabs.forEach(function(id) {
                RED.workspaces.show(id, null, true)
            })
        })
        // RED.actions.add("core:toggle-flows", function() {
        //     var currentTabs = workspace_tabs.listTabs();
        //     var visibleCount = workspace_tabs.count();
        //     currentTabs.forEach(function(id) {
        //         if (visibleCount === 0) {
        //             RED.workspaces.show(id)
        //         } else {
        //             RED.workspaces.hide(id)
        //         }
        //     })
        // })
        RED.actions.add("core:show-last-hidden-flow", function() {
            var id = hideStack.pop();
            if (id) {
                if (typeof id === 'string') {
                    RED.workspaces.show(id);
                } else {
                    var last = id.pop();
                    id.forEach(function(i) {
                        RED.workspaces.show(i, null, true);
                    })
                    setTimeout(function() {
                        RED.workspaces.show(last);
                    },150)

                }
            }
        })
        RED.actions.add("core:list-modified-nodes",function() {
            RED.actions.invoke("core:search","is:modified ");
        })
        RED.actions.add("core:list-hidden-flows",function() {
            RED.actions.invoke("core:search","is:hidden ");
        })
        RED.actions.add("core:list-flows",function() {
            RED.actions.invoke("core:search","type:tab ");
        })
        RED.actions.add("core:list-subflows",function() {
            RED.actions.invoke("core:search","type:subflow ");
        })
        RED.actions.add("core:go-to-previous-location", function() {
            if (viewStackPos > 0) {
                if (viewStackPos === viewStack.length) {
                    // We're at the end of the stack. Remember the activeWorkspace
                    // so we can come back to it.
                    viewStack.push(activeWorkspace);
                }
                RED.workspaces.show(viewStack[--viewStackPos],true);
            }
        })
        RED.actions.add("core:go-to-next-location", function() {
            if (viewStackPos < viewStack.length - 1) {
                RED.workspaces.show(viewStack[++viewStackPos],true);
            }
        })

        RED.events.on("flows:change", (ws) => {
            $("#red-ui-tab-"+(ws.id.replace(".","-"))).toggleClass('red-ui-workspace-changed',!!(ws.contentsChanged || ws.changed || ws.added));
        })
        RED.events.on("subflows:change", (ws) => {
            $("#red-ui-tab-"+(ws.id.replace(".","-"))).toggleClass('red-ui-workspace-changed',!!(ws.contentsChanged || ws.changed || ws.added));
        })

        hideWorkspace();
    }

    function editWorkspace(id) {
        showEditWorkspaceDialog(id||activeWorkspace);
    }

    function enableWorkspace(id) {
        setWorkspaceState(id,false);
    }
    function disableWorkspace(id) {
        setWorkspaceState(id,true);
    }
    function setWorkspaceState(id,disabled) {
        var workspace = RED.nodes.workspace(id||activeWorkspace);
        if (!workspace || workspace.locked) {
            return;
        }
        if (workspace.disabled !== disabled) {
            var changes = { disabled: workspace.disabled };
            workspace.disabled = disabled;
            $("#red-ui-tab-"+(workspace.id.replace(".","-"))).toggleClass('red-ui-workspace-disabled',!!workspace.disabled);
            if (!id || (id === activeWorkspace)) {
                $("#red-ui-workspace").toggleClass("red-ui-workspace-disabled",!!workspace.disabled);
            }
            var historyEvent = {
                t: "edit",
                changes:changes,
                node: workspace,
                dirty: RED.nodes.dirty()
            }
            workspace.changed = true;
            RED.history.push(historyEvent);
            RED.events.emit("flows:change",workspace);
            RED.nodes.dirty(true);
            RED.sidebar.config.refresh();
            var selection = RED.view.selection();
            if (!selection.nodes && !selection.links && workspace.id === activeWorkspace) {
                RED.sidebar.info.refresh(workspace);
            }
            if (changes.hasOwnProperty('disabled')) {
                RED.nodes.eachNode(function(n) {
                    if (n.z === workspace.id) {
                        n.dirty = true;
                    }
                });
                RED.view.redraw();
            }
        }
    }
    function lockWorkspace(id) {
        setWorkspaceLockState(id,true);
    }
    function unlockWorkspace(id) {
        setWorkspaceLockState(id,false);
    }
    function setWorkspaceLockState(id,locked) {
        var workspace = RED.nodes.workspace(id||activeWorkspace);
        if (!workspace) {
            return;
        }
        if (workspace.locked !== locked) {
            var changes = { locked: workspace.locked };
            workspace.locked = locked;
            $("#red-ui-tab-"+(workspace.id.replace(".","-"))).toggleClass('red-ui-workspace-locked',!!workspace.locked);
            if (!id || (id === activeWorkspace)) {
                $("#red-ui-workspace").toggleClass("red-ui-workspace-locked",!!workspace.locked);
            }
            var historyEvent = {
                t: "edit",
                changes:changes,
                node: workspace,
                dirty: RED.nodes.dirty()
            }
            workspace.changed = true;
            RED.history.push(historyEvent);
            RED.events.emit("flows:change",workspace);
            RED.nodes.dirty(true);
            RED.sidebar.config.refresh();
            RED.nodes.filterNodes({z:workspace.id}).forEach(n => n.dirty = true)
            RED.view.redraw(true);
        }
    }

    function removeWorkspace(ws) {
        if (!ws) {
            ws = RED.nodes.workspace(activeWorkspace)
            if (ws && !ws.locked) {
                deleteWorkspace(RED.nodes.workspace(activeWorkspace));
            }
        } else {
            if (ws.locked) { return }
            if (workspace_tabs.contains(ws.id)) {
                workspace_tabs.removeTab(ws.id);
            }
            if (ws.id === activeWorkspace) {
                activeWorkspace = 0;
            }
        }
    }

    function moveWorkspace(id, direction) {
        const workspace = RED.nodes.workspace(id||activeWorkspace) || RED.nodes.subflow(id||activeWorkspace);
        if (!workspace) {
            return;
        }
        const currentOrder = workspace_tabs.listTabs()
        const oldOrder = [...currentOrder]
        const currentIndex = currentOrder.findIndex(id => id === workspace.id)
        currentOrder.splice(currentIndex, 1)
        if (direction === 'start') {
            currentOrder.unshift(workspace.id)
        } else if (direction === 'end') {
            currentOrder.push(workspace.id)
        }
        const newOrder = setWorkspaceOrder(currentOrder)
        if (JSON.stringify(newOrder) !== JSON.stringify(oldOrder)) {
            RED.history.push({
                t:'reorder',
                workspaces: {
                    from:oldOrder,
                    to:newOrder
                },
                dirty:RED.nodes.dirty()
            });
            const filteredOldOrder = oldOrder.filter(id => !!RED.nodes.workspace(id))
            const filteredNewOrder = newOrder.filter(id => !!RED.nodes.workspace(id))
            if (JSON.stringify(filteredOldOrder) !== JSON.stringify(filteredNewOrder)) {
                RED.nodes.dirty(true);
            }
        }
    }
    function setWorkspaceOrder(order) {
        var newOrder = order.filter(id => !!RED.nodes.workspace(id))
        var currentOrder = RED.nodes.getWorkspaceOrder();
        if (JSON.stringify(newOrder) !== JSON.stringify(currentOrder)) {
            RED.nodes.setWorkspaceOrder(newOrder);
            RED.events.emit("flows:reorder",newOrder);
        }
        workspace_tabs.order(order);
        return newOrder
    }

    function flashTab(tabId) {
        if(flashingTab && flashingTab.length) {
            //cancel current flashing node before flashing new node
            clearInterval(flashingTabTimer);
            flashingTabTimer = null;
            flashingTab.removeClass('highlighted');
            flashingTab = null;
        }
        let tab = $("#red-ui-tab-" + tabId);
        if(!tab || !tab.length) { return; }

        flashingTabTimer = setInterval(function(flashEndTime) {
            if (flashEndTime >= Date.now()) {
                const highlighted = tab.hasClass("highlighted");
                tab.toggleClass('highlighted', !highlighted)
            } else {
                clearInterval(flashingTabTimer);
                flashingTabTimer = null;
                flashingTab = null;
                tab.removeClass('highlighted');
            }
        }, 100, Date.now() + 2200);
        flashingTab = tab;
        tab.addClass('highlighted');
    }
    return {
        init: init,
        add: addWorkspace,
        // remove: remove workspace without editor history etc
        remove: removeWorkspace,
        // delete: remove workspace and update editor history
        delete: deleteWorkspace,
        order: setWorkspaceOrder,
        edit: editWorkspace,
        contains: function(id) {
            return workspace_tabs.contains(id);
        },
        count: function() {
            return workspaceTabCount;
        },
        active: function() {
            return activeWorkspace
        },
        isLocked: function(id) {
            id = id || activeWorkspace
            var ws = RED.nodes.workspace(id) || RED.nodes.subflow(id)
            return ws && ws.locked
        },
        selection: function() {
            return workspace_tabs.selection();
        },
        hide: function(id) {
            if (!id) {
                id = activeWorkspace;
            }
            if (workspace_tabs.contains(id)) {
                workspace_tabs.hideTab(id);
            }
        },
        isHidden: function(id) {
            return hideStack.includes(id)
        },
        show: function(id,skipStack,unhideOnly,flash) {
            if (!workspace_tabs.contains(id)) {
                var sf = RED.nodes.subflow(id);
                if (sf) {
                    addWorkspace(
                        {type:"subflow",id:id,icon:"red/images/subflow_tab.svg",label:sf.name, closeable: true},
                        null,
                        workspace_tabs.activeIndex()+1
                    );
                    removeFromHideStack(id);
                } else {
                    return;
                }
            }
            if (unhideOnly) {
                workspace_tabs.showTab(id);
            } else {
                if (!skipStack && activeWorkspace !== id) {
                    addToViewStack(activeWorkspace)
                }
                workspace_tabs.activateTab(id);
            }
            if(flash) {
                flashTab(id.replace(".","-"))
            }
        },
        refresh: function() {
            var workspace = RED.nodes.workspace(RED.workspaces.active());
            if (workspace) {
                document.title = `${documentTitle} : ${workspace.label}`;
            } else {
                var subflow = RED.nodes.subflow(RED.workspaces.active());
                if (subflow) {
                    document.title = `${documentTitle} : ${subflow.name}`;
                } else {
                    document.title = documentTitle
                }
            }
            RED.nodes.eachWorkspace(function(ws) {
                workspace_tabs.renameTab(ws.id,ws.label);
                $("#red-ui-tab-"+(ws.id.replace(".","-"))).attr("flowname",ws.label)
            })
            RED.nodes.eachSubflow(function(sf) {
                if (workspace_tabs.contains(sf.id)) {
                    workspace_tabs.renameTab(sf.id,sf.name);
                }
            });
            RED.sidebar.config.refresh();
        },
        resize: function() {
            workspace_tabs.resize();
        },
        enable: enableWorkspace,
        disable: disableWorkspace,
        lock: lockWorkspace,
        unlock: unlockWorkspace
    }
})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

 /*!
 RED.statusBar.add({
     id: "widget-identifier",
     align: "left|right",
     element: widgetElement
 })
*/

RED.statusBar = (function() {

    var widgets = {};
    var leftBucket;
    var rightBucket;

    function addWidget(options) {
        widgets[options.id] = options;
        var el = $('<span class="red-ui-statusbar-widget"></span>');
        el.prop('id', options.id);
        options.element.appendTo(el);
        options.elementDiv = el;
        if (options.align === 'left') {
            leftBucket.append(el);
        } else if (options.align === 'right') {
            rightBucket.prepend(el);
        }
    }

    function hideWidget(id) {
        const widget = widgets[id];

        if (widget && widget.elementDiv) {
            widget.elementDiv.hide();
        }
    }

    function showWidget(id) {
        const widget = widgets[id];

        if (widget && widget.elementDiv) {
            widget.elementDiv.show();
        }
    }

    return {
        init: function() {
            leftBucket = $('<span class="red-ui-statusbar-bucket red-ui-statusbar-bucket-left">').appendTo("#red-ui-workspace-footer");
            rightBucket = $('<span class="red-ui-statusbar-bucket red-ui-statusbar-bucket-right">').appendTo("#red-ui-workspace-footer");
        },
        add: addWidget,
        hide: hideWidget,
        show: showWidget
    }

})();
;/**
 * Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/


 /* <div>#red-ui-workspace-chart
  *   \-  <svg> "outer"
  *       \- <g>
  *           \- <g>.red-ui-workspace-chart-event-layer "eventLayer"
  *                |- <rect>.red-ui-workspace-chart-background
  *                |- <g>.red-ui-workspace-chart-grid "gridLayer"
  *                |- <g> "groupLayer"
  *                |- <g> "groupSelectLayer"
  *                |- <g> "linkLayer"
  *                |- <g> "junctionLayer"
  *                |- <g> "dragGroupLayer"
  *                |- <g> "nodeLayer"
  */

RED.view = (function() {
    var space_width = 8000,
        space_height = 8000,
        lineCurveScale = 0.75,
        scaleFactor = 1,
        node_width = 100,
        node_height = 30,
        dblClickInterval = 650;

    var touchLongPressTimeout = 1000,
        startTouchDistance = 0,
        startTouchCenter = [],
        moveTouchCenter = [],
        touchStartTime = 0;

    var workspaceScrollPositions = {};

    var gridSize = 20;
    var snapGrid = false;

    var activeSpliceLink;
    var spliceActive = false;
    var spliceTimer;
    var groupHoverTimer;

    var activeFlowLocked = false;
    var activeSubflow = null;
    var activeNodes = [];
    var activeLinks = [];
    var activeJunctions = [];
    var activeFlowLinks = [];
    var activeLinkNodes = {};
    var activeHoverGroup = null;
    var groupAddActive = false;
    var groupAddParentGroup = null;
    var activeGroups = [];
    var dirtyGroups = {};

    var mousedown_link = null;
    var mousedown_node = null;
    var mousedown_group = null;
    var mousedown_port_type = null;
    var mousedown_port_index = 0;
    var mouseup_node = null;
    var mouse_offset = [0,0];
    var mouse_position = null;
    var mouse_mode = 0;
    var mousedown_group_handle = null;
    var lasso = null;
    var slicePath = null;
    var slicePathLast = null;
    var ghostNode = null;
    var showStatus = false;
    let showNodeInfo = true;
    var lastClickNode = null;
    var dblClickPrimed = null;
    var clickTime = 0;
    var clickElapsed = 0;
    var scroll_position = [];
    var quickAddActive = false;
    var quickAddLink = null;
    var showAllLinkPorts = -1;
    var groupNodeSelectPrimed = false;
    var lastClickPosition = [];
    var selectNodesOptions;

    let flashingNodeId;

    var clipboard = "";
    let clipboardSource

    let currentSuggestion = null;
    let suggestedNodes = [];
    let suggestedLinks = [];
    let suggestedJunctions = [];

    let forceFullRedraw = false
    // Note: these are the permitted status colour aliases. The actual RGB values
    //       are set in the CSS - flow.scss/colors.scss
    const status_colours = {
        "red":    "#c00",
        "green":  "#5a8",
        "yellow": "#F9DF31",
        "blue":   "#53A3F3",
        "grey":   "#d3d3d3",
        "gray":   "#d3d3d3"
    }

    const PORT_TYPE_INPUT = 1;
    const PORT_TYPE_OUTPUT = 0;

    /**
     * The jQuery object for the workspace chart `#red-ui-workspace-chart` div element
     * @type {JQuery<HTMLElement>} #red-ui-workspace-chart HTML Element 
     */ 
    let chart;
    /**
     * The d3 object `#red-ui-workspace-chart` svg element
     * @type {d3.Selection<HTMLElement, Any, Any, Any>}
     */ 
    let outer;
    /** 
     * The d3 object `#red-ui-workspace-chart` svg element (specifically for events)
     * @type {d3.Selection<d3.BaseType, any, any, any>}
     */
    var eventLayer;

    /** @type {SVGGElement} */ let gridLayer;
    /** @type {SVGGElement} */ let linkLayer;
    /** @type {SVGGElement} */ let junctionLayer;
    /** @type {SVGGElement} */ let dragGroupLayer;
    /** @type {SVGGElement} */ let groupSelectLayer;
    /** @type {SVGGElement} */ let nodeLayer;
    /** @type {SVGGElement} */ let groupLayer;
    var drag_lines;

    const movingSet = (function() {
        var setIds = new Set();
        var set = [];
        const api = {
            add: function(node) {
                if (Array.isArray(node)) {
                    for (var i=0;i<node.length;i++) {
                        api.add(node[i]);
                    }
                } else {
                    if (!setIds.has(node.id)) {
                        set.push({n:node});
                        setIds.add(node.id);
                        var links = RED.nodes.getNodeLinks(node.id,PORT_TYPE_INPUT).concat(RED.nodes.getNodeLinks(node.id,PORT_TYPE_OUTPUT))
                        for (var i=0,l=links.length;i<l;i++) {
                            var link = links[i]
                            if (link.source === node && setIds.has(link.target.id) ||
                                link.target === node && setIds.has(link.source.id)) {
                                selectedLinks.add(link)
                            }
                        }
                    }
                }
            },
            remove: function(node, index) {
                if (setIds.has(node.id)) {
                    setIds.delete(node.id);
                    if (index !== undefined && set[index].n === node) {
                        set.splice(index,1);
                    } else {
                        for (var i=0;i<set.length;i++) {
                            if (set[i].n === node) {
                                set.splice(i,1)
                                break;
                            }
                        }
                    }
                    var links = RED.nodes.getNodeLinks(node.id,PORT_TYPE_INPUT).concat(RED.nodes.getNodeLinks(node.id,PORT_TYPE_OUTPUT))
                    for (var i=0,l=links.length;i<l;i++) {
                        selectedLinks.remove(links[i]);
                    }
                }
            },
            clear: function() {
                setIds.clear();
                set = [];
            },
            length: function() { return set.length},
            get: function(i) { return set[i] },
            forEach: function(func) { set.forEach(func) },
            nodes: function() { return set.map(function(n) { return n.n })},
            has: function(node) { return setIds.has(node.id) },
            /**
             * Make the specified node the first node of the moving set, if
             * it is already in the set.
             * @param {Node} node 
             */
            makePrimary: function (node) {
                const index = set.findIndex(n => n.n === node)
                if (index > -1) {
                    const removed = set.splice(index, 1)
                    set.unshift(...removed)
                }
            },
            find: function(func) { return set.find(func) },
            dump: function () {
                console.log('MovingSet Contents')
                api.forEach((n, i) => {
                    console.log(`${i+1}\t${n.n.id}\t${n.n.type}`)
                })
            }
        }
        return api;
    })();

    const selectedLinks = (function() {
        var links = new Set();
        const api = {
            add: function(link) {
                links.add(link);
                link.selected = true;
            },
            remove: function(link) {
                links.delete(link);
                link.selected = false;
            },
            clear: function() {
                links.forEach(function(link) { link.selected = false })
                links.clear();
            },
            length: function() {
                return links.size;
            },
            forEach: function(func) { links.forEach(func) },
            has: function(link) { return links.has(link) },
            toArray: function() { return Array.from(links) },
            clearUnselected: function () {
                api.forEach(l => {
                    if (!l.source.selected || !l.target.selected) {
                        api.remove(l)
                    }
                })
            }
        }
        return api
    })();

    const selectedGroups = (function() {
        let groups = new Set()
        const api = {
            add: function(g, includeNodes, addToMovingSet) {
                groups.add(g)
                if (!g.selected) {
                    g.selected = true;
                    g.dirty = true;
                }
                if (addToMovingSet !== false) {
                    movingSet.add(g);
                }
                if (includeNodes) {
                    var currentSet = new Set(movingSet.nodes());
                    var allNodes = RED.group.getNodes(g,true);
                    allNodes.forEach(function(n) {
                        if (!currentSet.has(n)) {
                            movingSet.add(n)
                        }
                        n.dirty = true;
                    })
                }
                selectedLinks.clearUnselected()
            },
            remove: function(g) {
                groups.delete(g)
                if (g.selected) {
                    g.selected = false;
                    g.dirty = true;
                }
                const allNodes = RED.group.getNodes(g,true);
                const nodeSet = new Set(allNodes);
                nodeSet.add(g);
                for (let i = movingSet.length()-1; i >= 0; i -= 1) {
                    const msn = movingSet.get(i);
                    if (nodeSet.has(msn.n) || msn.n === g) {
                        msn.n.selected = false;
                        msn.n.dirty = true;
                        movingSet.remove(msn.n,i)
                    }
                }
                selectedLinks.clearUnselected()
            },
            length: () => groups.size,
            forEach: (func) => { groups.forEach(func) },
            toArray: () => [...groups],
            clear: function () {
                groups.forEach(g => {
                    g.selected = false
                    g.dirty = true
                })
                groups.clear()
            }
        }
        return api
    })()

    const isMac = RED.utils.getBrowserInfo().os === 'mac'
    // 'Control' is the main modifier key for mouse actions. On Windows,
    // that is the standard Ctrl key. On Mac that is the Cmd key.
    function isControlPressed (event) {
        return (isMac && event.metaKey) || (!isMac && event.ctrlKey)
    }

    function init() {

        chart = $("#red-ui-workspace-chart");
        chart.on('contextmenu', function(evt) {
            if (RED.view.DEBUG) {
                console.warn("contextmenu", { mouse_mode, event: d3.event });
            }
            mouse_mode = RED.state.DEFAULT
            evt.preventDefault()
            evt.stopPropagation()
            RED.contextMenu.show({
                type: 'workspace',
                x: evt.clientX,
                y: evt.clientY
            })
            return false
        })
        outer = d3.select("#red-ui-workspace-chart")
            .append("svg:svg")
            .attr("width", space_width)
            .attr("height", space_height)
            .attr("pointer-events", "all")
            .style("cursor","crosshair")
            .style("touch-action","none")
            .on("mousedown", function() {
                focusView();
            })
            .on("contextmenu", function(){
                d3.event.preventDefault();
            });

        eventLayer = outer
            .append("svg:g")
            .on("dblclick.zoom", null)
            .append("svg:g")
            .attr('class','red-ui-workspace-chart-event-layer')
            .on("mousemove", canvasMouseMove)
            .on("mousedown", canvasMouseDown)
            .on("mouseup", canvasMouseUp)
            .on("mouseenter", function() {
                d3.select(document).on('mouseup.red-ui-workspace-tracker', null)
                if (lasso) {
                    if (d3.event.buttons !== 1) {
                        outer.classed('red-ui-workspace-lasso-active', false)
                        lasso.remove();
                        lasso = null;
                    }
                } else if (mouse_mode === RED.state.PANNING && d3.event.buttons !== 4) {
                    resetMouseVars();
                } else if (slicePath) {
                    if (d3.event.buttons !== 2) {
                        slicePath.remove();
                        slicePath = null;
                        resetMouseVars()
                    }
                }
            })
            .on("mouseleave", canvasMouseLeave)
            .on("touchend", function() {
                d3.event.preventDefault();
                clearTimeout(touchStartTime);
                touchStartTime = null;
                if  (RED.touch.radialMenu.active()) {
                    return;
                }
                canvasMouseUp.call(this);
            })
            .on("touchcancel", function() {
                if (RED.view.DEBUG) { console.warn("eventLayer.touchcancel", mouse_mode); }
                d3.event.preventDefault();
                canvasMouseUp.call(this);
            })
            .on("touchstart", function() {
                if (RED.view.DEBUG) { console.warn("eventLayer.touchstart", mouse_mode); }
                var touch0;
                if (d3.event.touches.length>1) {
                    clearTimeout(touchStartTime);
                    touchStartTime = null;
                    d3.event.preventDefault();
                    touch0 = d3.event.touches.item(0);
                    var touch1 = d3.event.touches.item(1);
                    var a = touch0["pageY"]-touch1["pageY"];
                    var b = touch0["pageX"]-touch1["pageX"];

                    var offset = chart.offset();
                    var scrollPos = [chart.scrollLeft(),chart.scrollTop()];
                    startTouchCenter = [
                        (touch1["pageX"]+(b/2)-offset.left+scrollPos[0])/scaleFactor,
                        (touch1["pageY"]+(a/2)-offset.top+scrollPos[1])/scaleFactor
                    ];
                    moveTouchCenter = [
                        touch1["pageX"]+(b/2),
                        touch1["pageY"]+(a/2)
                    ]
                    startTouchDistance = Math.sqrt((a*a)+(b*b));
                } else {
                    var obj = d3.select(document.body);
                    touch0 = d3.event.touches.item(0);
                    var pos = [touch0.pageX,touch0.pageY];
                    startTouchCenter = [touch0.pageX,touch0.pageY];
                    startTouchDistance = 0;
                    var point = d3.touches(this)[0];
                    touchStartTime = setTimeout(function() {
                        touchStartTime = null;
                        showTouchMenu(obj,pos);
                    },touchLongPressTimeout);
                }
                d3.event.preventDefault();
            })
            .on("touchmove", function(){
                    if  (RED.touch.radialMenu.active()) {
                        d3.event.preventDefault();
                        return;
                    }
                    if (RED.view.DEBUG) { console.warn("eventLayer.touchmove", mouse_mode, mousedown_node); }
                    var touch0;
                    if (d3.event.touches.length<2) {
                        if (touchStartTime) {
                            touch0 = d3.event.touches.item(0);
                            var dx = (touch0.pageX-startTouchCenter[0]);
                            var dy = (touch0.pageY-startTouchCenter[1]);
                            var d = Math.abs(dx*dx+dy*dy);
                            if (d > 64) {
                                clearTimeout(touchStartTime);
                                touchStartTime = null;
                                if (!mousedown_node && !mousedown_group) {
                                    mouse_mode = RED.state.PANNING;
                                    mouse_position = [touch0.pageX,touch0.pageY]
                                    scroll_position = [chart.scrollLeft(),chart.scrollTop()];
                                }

                            }
                        } else if (lasso) {
                            d3.event.preventDefault();
                        }
                        canvasMouseMove.call(this);
                    } else {
                        touch0 = d3.event.touches.item(0);
                        var touch1 = d3.event.touches.item(1);
                        var a = touch0["pageY"]-touch1["pageY"];
                        var b = touch0["pageX"]-touch1["pageX"];
                        var offset = chart.offset();
                        var scrollPos = [chart.scrollLeft(),chart.scrollTop()];
                        var moveTouchDistance = Math.sqrt((a*a)+(b*b));
                        var touchCenter = [
                            touch1["pageX"]+(b/2),
                            touch1["pageY"]+(a/2)
                        ];

                        if (!isNaN(moveTouchDistance)) {
                            oldScaleFactor = scaleFactor;
                            scaleFactor = Math.min(2,Math.max(0.3, scaleFactor + (Math.floor(((moveTouchDistance*100)-(startTouchDistance*100)))/10000)));

                            var deltaTouchCenter = [                             // Try to pan whilst zooming - not 100%
                                startTouchCenter[0]*(scaleFactor-oldScaleFactor),//-(touchCenter[0]-moveTouchCenter[0]),
                                startTouchCenter[1]*(scaleFactor-oldScaleFactor) //-(touchCenter[1]-moveTouchCenter[1])
                            ];

                            startTouchDistance = moveTouchDistance;
                            moveTouchCenter = touchCenter;

                            chart.scrollLeft(scrollPos[0]+deltaTouchCenter[0]);
                            chart.scrollTop(scrollPos[1]+deltaTouchCenter[1]);
                            redraw();
                        }
                    }
                    d3.event.preventDefault();
            });
            

        const handleAltToggle = (event) => {
            if (mouse_mode === RED.state.MOVING_ACTIVE && event.key === 'Alt' && groupAddParentGroup) {
                RED.nodes.group(groupAddParentGroup).dirty = true
                for (let n = 0; n<movingSet.length(); n++) {
                    const node = movingSet.get(n);
                    node.n._detachFromGroup = event.altKey
                }
                if (!event.altKey) {
                    if (groupHoverTimer) {
                        clearTimeout(groupHoverTimer)
                        groupHoverTimer = null
                    }
                    if (activeHoverGroup) {
                        activeHoverGroup.hovered = false
                        activeHoverGroup.dirty = true
                        activeHoverGroup = null
                    }
                }
                RED.view.redraw()
            }
        }
        document.addEventListener("keyup", handleAltToggle)
        document.addEventListener("keydown", handleAltToggle)

        // Workspace Background
        eventLayer.append("svg:rect")
            .attr("class","red-ui-workspace-chart-background")
            .attr("width", space_width)
            .attr("height", space_height);

        gridLayer = eventLayer.append("g").attr("class","red-ui-workspace-chart-grid");
        updateGrid();

        groupLayer = eventLayer.append("g");
        groupSelectLayer = eventLayer.append("g");
        linkLayer = eventLayer.append("g");
        dragGroupLayer = eventLayer.append("g");
        junctionLayer = eventLayer.append("g");
        nodeLayer = eventLayer.append("g");

        drag_lines = [];

        RED.events.on("workspace:change",function(event) {
            // Just in case the mouse left the workspace whilst doing an action,
            // put us back into default mode so the refresh works
            mouse_mode = 0
            if (event.old !== 0) {
                workspaceScrollPositions[event.old] = {
                    left:chart.scrollLeft(),
                    top:chart.scrollTop()
                };
            }
            var scrollStartLeft = chart.scrollLeft();
            var scrollStartTop = chart.scrollTop();

            activeSubflow = RED.nodes.subflow(event.workspace);

            if (activeSubflow) {
                activeFlowLocked = activeSubflow.locked
            } else {
                var activeWorkspace = RED.nodes.workspace(event.workspace)
                if (activeWorkspace) {
                    activeFlowLocked = activeWorkspace.locked
                } else {
                    activeFlowLocked = true
                }
            }

            clearSuggestedFlow();

            RED.menu.setDisabled("menu-item-workspace-edit", activeFlowLocked || activeSubflow || event.workspace === 0);
            RED.menu.setDisabled("menu-item-workspace-delete",activeFlowLocked || event.workspace === 0 || RED.workspaces.count() == 1 || activeSubflow);

            if (workspaceScrollPositions[event.workspace]) {
                chart.scrollLeft(workspaceScrollPositions[event.workspace].left);
                chart.scrollTop(workspaceScrollPositions[event.workspace].top);
            } else {
                chart.scrollLeft(0);
                chart.scrollTop(0);
            }
            var scrollDeltaLeft = chart.scrollLeft() - scrollStartLeft;
            var scrollDeltaTop = chart.scrollTop() - scrollStartTop;
            if (mouse_position != null) {
                mouse_position[0] += scrollDeltaLeft;
                mouse_position[1] += scrollDeltaTop;
            }
            if (RED.workspaces.selection().length === 0) {
                clearSelection();
            }
            RED.nodes.eachNode(function(n) {
                n.dirty = true;
                n.dirtyStatus = true;
            });
            updateSelection();
            updateActiveNodes();
            redraw();
        });

        RED.events.on("flows:change", function(workspace) {
            if (workspace.id === RED.workspaces.active()) {
                activeFlowLocked = !!workspace.locked
                $("#red-ui-workspace").toggleClass("red-ui-workspace-disabled",!!workspace.disabled);
                $("#red-ui-workspace").toggleClass("red-ui-workspace-locked",!!workspace.locked);

            }
        })

        RED.statusBar.add({
            id: "view-zoom-controls",
            align: "right",
            element: $('<span class="button-group">'+
            '<button class="red-ui-footer-button" id="red-ui-view-zoom-out"><i class="fa fa-minus"></i></button>'+
            '<button class="red-ui-footer-button" id="red-ui-view-zoom-zero"><i class="fa fa-circle-o"></i></button>'+
            '<button class="red-ui-footer-button" id="red-ui-view-zoom-in"><i class="fa fa-plus"></i></button>'+
            '</span>')
        })

        $("#red-ui-view-zoom-out").on("click", zoomOut);
        RED.popover.tooltip($("#red-ui-view-zoom-out"),RED._('actions.zoom-out'),'core:zoom-out');
        $("#red-ui-view-zoom-zero").on("click", zoomZero);
        RED.popover.tooltip($("#red-ui-view-zoom-zero"),RED._('actions.zoom-reset'),'core:zoom-reset');
        $("#red-ui-view-zoom-in").on("click", zoomIn);
        RED.popover.tooltip($("#red-ui-view-zoom-in"),RED._('actions.zoom-in'),'core:zoom-in');
        chart.on("DOMMouseScroll mousewheel", function (evt) {
            if ( evt.altKey ) {
                evt.preventDefault();
                evt.stopPropagation();
                var move = -(evt.originalEvent.detail) || evt.originalEvent.wheelDelta;
                if (move <= 0) { zoomOut(); }
                else { zoomIn(); }
            }
        });

        //add search to status-toolbar
        RED.statusBar.add({
            id: "view-search-tools",
            align: "left",
            hidden: false,
            element: $('<span class="button-group">'+
                    '<button class="red-ui-footer-button" id="red-ui-view-searchtools-search"><i class="fa fa-search"></i></button>' +
                    '</span>' +
                    '<span class="button-group search-counter">' +
                    '<span class="red-ui-footer-button" id="red-ui-view-searchtools-counter">? of ?</span>' +
                    '</span>' +
                    '<span class="button-group">' +
                    '<button class="red-ui-footer-button" id="red-ui-view-searchtools-prev"><i class="fa fa-chevron-left"></i></button>' +
                    '<button class="red-ui-footer-button" id="red-ui-view-searchtools-next"><i class="fa fa-chevron-right"></i></button>' +
                    '</span>' +
                    '<span class="button-group">' +
                    '<button class="red-ui-footer-button" id="red-ui-view-searchtools-close"><i class="fa fa-close"></i></button>' +
                    '</span>')
        })
        $("#red-ui-view-searchtools-search").on("click", searchFlows);
        RED.popover.tooltip($("#red-ui-view-searchtools-search"),RED._('actions.search-flows'),'core:search');
        $("#red-ui-view-searchtools-prev").on("click", searchPrev);
        RED.popover.tooltip($("#red-ui-view-searchtools-prev"),RED._('actions.search-prev'),'core:search-previous');
        $("#red-ui-view-searchtools-next").on("click", searchNext);
        RED.popover.tooltip($("#red-ui-view-searchtools-next"),RED._('actions.search-next'),'core:search-next');
        RED.popover.tooltip($("#red-ui-view-searchtools-close"),RED._('common.label.close'));

        // Handle nodes dragged from the palette
        chart.droppable({
            accept:".red-ui-palette-node",
            drop: function( event, ui ) {
                if (activeFlowLocked) {
                    return
                }
                d3.event = event;
                var selected_tool = $(ui.draggable[0]).attr("data-palette-type");
                try {
                    var result = createNode(selected_tool);
                    if (!result) {
                        return;
                    }
                    var historyEvent = result.historyEvent;
                    const linkToSplice = $(ui.helper).data("splice");

                    var nn = RED.nodes.add(result.node, { source: 'palette', splice: !!linkToSplice });

                    var showLabel = RED.utils.getMessageProperty(RED.settings.get('editor'),"view.view-node-show-label");
                    if (showLabel !== undefined &&  (nn._def.hasOwnProperty("showLabel")?nn._def.showLabel:true) && !nn._def.defaults.hasOwnProperty("l")) {
                        nn.l = showLabel;
                    }

                    var helperOffset = d3.touches(ui.helper.get(0))[0]||d3.mouse(ui.helper.get(0));
                    var helperWidth = ui.helper.width();
                    var helperHeight = ui.helper.height();
                    var mousePos = d3.touches(this)[0]||d3.mouse(this);

                    try {
                        var isLink = (nn.type === "link in" || nn.type === "link out")
                        var hideLabel = nn.hasOwnProperty('l')?!nn.l : isLink;

                        var label = RED.utils.getNodeLabel(nn, nn.type);
                        var labelParts = getLabelParts(label, "red-ui-flow-node-label");
                        if (hideLabel) {
                            nn.w = node_height;
                            nn.h = Math.max(node_height,(nn.outputs || 0) * 15);
                        } else {
                            nn.w = Math.max(node_width,20*(Math.ceil((labelParts.width+50+(nn._def.inputs>0?7:0))/20)) );
                            nn.h = Math.max(6+24*labelParts.lines.length,(nn.outputs || 0) * 15, 30);
                        }
                    } catch(err) {
                    }

                    mousePos[1] += this.scrollTop + ((helperHeight/2)-helperOffset[1]);
                    mousePos[0] += this.scrollLeft + ((helperWidth/2)-helperOffset[0]);
                    mousePos[1] /= scaleFactor;
                    mousePos[0] /= scaleFactor;

                    nn.x = mousePos[0];
                    nn.y = mousePos[1];

                    var minX = nn.w/2 -5;
                    if (nn.x < minX) {
                        nn.x = minX;
                    }
                    var minY = nn.h/2 -5;
                    if (nn.y < minY) {
                        nn.y = minY;
                    }
                    var maxX = space_width -nn.w/2 +5;
                    if (nn.x > maxX) {
                        nn.x = maxX;
                    }
                    var maxY = space_height -nn.h +5;
                    if (nn.y > maxY) {
                        nn.y = maxY;
                    }

                    if (snapGrid) {
                        var gridOffset = RED.view.tools.calculateGridSnapOffsets(nn);
                        nn.x -= gridOffset.x;
                        nn.y -= gridOffset.y;
                    }

                    if (linkToSplice) {
                        spliceLink(linkToSplice, nn, historyEvent)
                    }

                    var group = $(ui.helper).data("group");
                    if (group) {
                        var oldX = group.x; 
                        var oldY = group.y; 
                        RED.group.addToGroup(group, nn);
                        var moveEvent = null;
                        if ((group.x !== oldX) ||
                            (group.y !== oldY)) {
                            moveEvent = {
                                t: "move",
                                nodes: [{n: group,
                                        ox: oldX, oy: oldY,
                                        dx: group.x -oldX,
                                        dy: group.y -oldY}],
                                dirty: true
                            };
                        }
                        historyEvent = {
                            t: 'multi',
                            events: [historyEvent],

                        };
                        if (moveEvent) {
                            historyEvent.events.push(moveEvent)
                        }
                        historyEvent.events.push({
                            t: "addToGroup",
                            group: group,
                            nodes: nn
                        })
                    }

                    RED.history.push(historyEvent);
                    RED.editor.validateNode(nn);
                    RED.nodes.dirty(true);
                    // auto select dropped node - so info shows (if visible)
                    clearSelection();
                    nn.selected = true;
                    movingSet.add(nn);
                    updateActiveNodes();
                    updateSelection();
                    redraw();

                    if (nn._def.autoedit) {
                        RED.editor.edit(nn);
                    }
                } catch (error) {
                    if (error.code != "NODE_RED") {
                        RED.notify(RED._("notification.error",{message:error.toString()}),"error");
                    } else {
                        RED.notify(RED._("notification.error",{message:error.message}),"error");
                    }
                }
            }
        });
        chart.on("focus", function() {
            $("#red-ui-workspace-tabs").addClass("red-ui-workspace-focussed");
        });
        chart.on("blur", function() {
            $("#red-ui-workspace-tabs").removeClass("red-ui-workspace-focussed");
        });

        RED.actions.add("core:copy-selection-to-internal-clipboard",copySelection);
        RED.actions.add("core:cut-selection-to-internal-clipboard",function(){copySelection(true);deleteSelection();});
        RED.actions.add("core:paste-from-internal-clipboard",function(){
            if (RED.workspaces.isLocked()) {
                return
            }
            importNodes(clipboard, {
                generateIds: clipboardSource === 'copy',
                generateDefaultNames: clipboardSource === 'copy',
                eventContext: { source: 'clipboard' }
            });
        });

        RED.actions.add("core:detach-selected-nodes", function() { detachSelectedNodes() })

        RED.events.on("view:selection-changed", function(selection) {
            var hasSelection = (selection.nodes && selection.nodes.length > 0);
            var hasMultipleSelection = hasSelection && selection.nodes.length > 1;
            var hasLinkSelected = selection.links && selection.links.length > 0;
            var canEdit = !activeFlowLocked && hasSelection
            var canEditMultiple = !activeFlowLocked && hasMultipleSelection
            RED.menu.setDisabled("menu-item-edit-cut", !canEdit);
            RED.menu.setDisabled("menu-item-edit-copy", !hasSelection);
            RED.menu.setDisabled("menu-item-edit-select-connected", !hasSelection);
            RED.menu.setDisabled("menu-item-view-tools-move-to-back", !canEdit);
            RED.menu.setDisabled("menu-item-view-tools-move-to-front", !canEdit);
            RED.menu.setDisabled("menu-item-view-tools-move-backwards", !canEdit);
            RED.menu.setDisabled("menu-item-view-tools-move-forwards", !canEdit);

            RED.menu.setDisabled("menu-item-view-tools-align-left", !canEditMultiple);
            RED.menu.setDisabled("menu-item-view-tools-align-center", !canEditMultiple);
            RED.menu.setDisabled("menu-item-view-tools-align-right", !canEditMultiple);
            RED.menu.setDisabled("menu-item-view-tools-align-top", !canEditMultiple);
            RED.menu.setDisabled("menu-item-view-tools-align-middle", !canEditMultiple);
            RED.menu.setDisabled("menu-item-view-tools-align-bottom", !canEditMultiple);
            RED.menu.setDisabled("menu-item-view-tools-distribute-horizontally", !canEditMultiple);
            RED.menu.setDisabled("menu-item-view-tools-distribute-veritcally", !canEditMultiple);

            RED.menu.setDisabled("menu-item-edit-split-wire-with-links", activeFlowLocked || !hasLinkSelected);
        })

        RED.actions.add("core:delete-selection",deleteSelection);
        RED.actions.add("core:delete-selection-and-reconnect",function() { deleteSelection(true) });
        RED.actions.add("core:edit-selected-node",editSelection);
        RED.actions.add("core:go-to-selection",function() {
            if (movingSet.length() > 0) {
                var node = movingSet.get(0).n;
                if (/^subflow:/.test(node.type)) {
                    RED.workspaces.show(node.type.substring(8))
                } else if (node.type === 'group') {
                    // enterActiveGroup(node);
                    redraw();
                }
            }
        });
        RED.actions.add("core:undo",RED.history.pop);
        RED.actions.add("core:redo",RED.history.redo);
        RED.actions.add("core:select-all-nodes",selectAll);
        RED.actions.add("core:select-none", selectNone);
        RED.actions.add("core:zoom-in",zoomIn);
        RED.actions.add("core:zoom-out",zoomOut);
        RED.actions.add("core:zoom-reset",zoomZero);
        RED.actions.add("core:enable-selected-nodes", function() { setSelectedNodeState(false)});
        RED.actions.add("core:disable-selected-nodes", function() { setSelectedNodeState(true)});

        RED.actions.add("core:toggle-show-grid",function(state) {
            if (state === undefined) {
                RED.userSettings.toggle("view-show-grid");
            } else {
                toggleShowGrid(state);
            }
        });
        RED.actions.add("core:toggle-snap-grid",function(state) {
            if (state === undefined) {
                RED.userSettings.toggle("view-snap-grid");
            } else {
                toggleSnapGrid(state);
            }
        });
        RED.actions.add("core:toggle-status",function(state) {
            if (state === undefined) {
                RED.userSettings.toggle("view-node-status");
            } else {
                toggleStatus(state);
            }
        });
        RED.actions.add("core:toggle-node-info-icon", function (state) {
            if (state === undefined) {
                RED.userSettings.toggle("view-node-info-icon");
            } else {
                toggleNodeInfo(state)
            }
        })

        RED.view.annotations.init();
        RED.view.navigator.init();
        RED.view.tools.init();

        RED.view.annotations.register("red-ui-flow-node-docs",{
            type: "badge",
            class: "red-ui-flow-node-docs",
            element: function(node) {

                const docsBadge = document.createElementNS("http://www.w3.org/2000/svg","g")

                const pageOutline = document.createElementNS("http://www.w3.org/2000/svg","rect");
                pageOutline.setAttribute("x",0);
                pageOutline.setAttribute("y",0);
                pageOutline.setAttribute("rx",2);
                pageOutline.setAttribute("width",7);
                pageOutline.setAttribute("height",10);
                docsBadge.appendChild(pageOutline)

                const pageLines = document.createElementNS("http://www.w3.org/2000/svg","path");
                pageLines.setAttribute("d", "M 7 3 h -3 v -3 M 2 8 h 3 M 2 6 h 3 M 2 4 h 2")
                docsBadge.appendChild(pageLines)

                $(docsBadge).on('click', function (evt) {
                    if (node.type === "subflow") {
                        RED.editor.editSubflow(activeSubflow, 'editor-tab-description');
                    } else if (node.type === "group") {
                        RED.editor.editGroup(node, 'editor-tab-description');
                    } else {
                        RED.editor.edit(node, 'editor-tab-description');
                    }
                })

                return docsBadge;
            },
            show: function(n) { return showNodeInfo && !!n.info }
        })

        RED.view.annotations.register("red-ui-flow-node-changed",{
            type: "badge",
            class: "red-ui-flow-node-changed",
            element: function() {
                var changeBadge = document.createElementNS("http://www.w3.org/2000/svg","circle");
                changeBadge.setAttribute("cx",5);
                changeBadge.setAttribute("cy",5);
                changeBadge.setAttribute("r",5);
                return changeBadge;
            },
            show: function(n) { return n.changed||n.moved }
        })

        RED.view.annotations.register("red-ui-flow-node-error",{
            type: "badge",
            class: "red-ui-flow-node-error",
            element: function(d) {
                var errorBadge = document.createElementNS("http://www.w3.org/2000/svg","path");
                errorBadge.setAttribute("d","M 0,9 l 10,0 -5,-8 z");
                return errorBadge
            },
            tooltip: function(d) {
                if (d.validationErrors && d.validationErrors.length > 0) {
                    return RED._("editor.errors.invalidProperties")+"\n  - "+d.validationErrors.join("\n  - ")
                }
            },
            show: function(n) { return !n.valid }
        })

        if (RED.settings.get("editor.view.view-store-zoom")) {
            var  userZoomLevel = parseFloat(RED.settings.getLocal('zoom-level'))
            if (!isNaN(userZoomLevel)) {
                scaleFactor = userZoomLevel
            }
        }

        var onScrollTimer = null;
        function storeScrollPosition() {
            workspaceScrollPositions[RED.workspaces.active()] = {
                left:chart.scrollLeft(),
                top:chart.scrollTop()
            };
            RED.settings.setLocal('scroll-positions', JSON.stringify(workspaceScrollPositions) )
        }
        chart.on("scroll", function() {
            if (RED.settings.get("editor.view.view-store-position")) {
                if (onScrollTimer) {
                    clearTimeout(onScrollTimer)
                }
                onScrollTimer = setTimeout(storeScrollPosition, 200);
            }
        })

        if (RED.settings.get("editor.view.view-store-position")) {
            var scrollPositions = RED.settings.getLocal('scroll-positions')
            if (scrollPositions) {
                try {
                    workspaceScrollPositions = JSON.parse(scrollPositions)
                } catch(err) {
                }
            }
        }
    }



    function updateGrid() {
        var gridTicks = [];
        for (var i=0;i<space_width;i+=+gridSize) {
            gridTicks.push(i);
        }
        gridLayer.selectAll("line.red-ui-workspace-chart-grid-h").remove();
        gridLayer.selectAll("line.red-ui-workspace-chart-grid-h").data(gridTicks).enter()
            .append("line")
            .attr(
                {
                    "class":"red-ui-workspace-chart-grid-h",
                    "x1" : 0,
                    "x2" : space_width,
                    "y1" : function(d){ return d;},
                    "y2" : function(d){ return d;}
                });
        gridLayer.selectAll("line.red-ui-workspace-chart-grid-v").remove();
        gridLayer.selectAll("line.red-ui-workspace-chart-grid-v").data(gridTicks).enter()
            .append("line")
            .attr(
                {
                    "class":"red-ui-workspace-chart-grid-v",
                    "y1" : 0,
                    "y2" : space_width,
                    "x1" : function(d){ return d;},
                    "x2" : function(d){ return d;}
                });
    }

    function showDragLines(nodes) {
        showAllLinkPorts = -1;
        for (var i=0;i<nodes.length;i++) {
            var node = nodes[i];
            node.el = dragGroupLayer.append("svg:path").attr("class", "red-ui-flow-drag-line");
            if ((node.node.type === "link out" && node.portType === PORT_TYPE_OUTPUT) ||
                (node.node.type === "link in" && node.portType === PORT_TYPE_INPUT)) {
                node.el.attr("class","red-ui-flow-link-link red-ui-flow-drag-line");
                node.virtualLink = true;
                showAllLinkPorts = (node.portType === PORT_TYPE_OUTPUT)?PORT_TYPE_INPUT:PORT_TYPE_OUTPUT;
            }
            drag_lines.push(node);
        }
        if (showAllLinkPorts !== -1) {
            activeNodes.forEach(function(n) {
                if (n.type === "link in" || n.type === "link out") {
                    n.dirty = true;
                }
            })
        }
    }
    function hideDragLines() {
        if (showAllLinkPorts !== -1) {
            activeNodes.forEach(function(n) {
                if (n.type === "link in" || n.type === "link out") {
                    n.dirty = true;
                }
            })
        }
        showAllLinkPorts = -1;
        while(drag_lines.length) {
            var line = drag_lines.pop();
            if (line.el) {
                line.el.remove();
            }
        }
    }

    function updateActiveNodes() {
        var activeWorkspace = RED.workspaces.active();
        if (activeWorkspace !== 0) {
            activeNodes = RED.nodes.filterNodes({z:activeWorkspace});
            activeNodes.forEach(function(n,i) {
                n._index = i;
            })
            activeLinks = RED.nodes.filterLinks({
                source:{z:activeWorkspace},
                target:{z:activeWorkspace}
            });
            activeJunctions = RED.nodes.junctions(activeWorkspace) || [];
            activeGroups = RED.nodes.groups(activeWorkspace)||[];
            if (activeGroups.length) {
                const groupTree = {}
                const rootGroups = []
                activeGroups.forEach(function(g, i) {
                    groupTree[g.id] = g
                    g._index = i;
                    g._childGroups = []
                    if (!g.g) {
                        rootGroups.push(g)
                    }
                });
                activeGroups.forEach(function(g) {
                    if (g.g) {
                        groupTree[g.g]._childGroups.push(g)
                        g._parentGroup = groupTree[g.g]
                    }
                })
                let ii = 0
                // Depth-first walk of the groups
                const processGroup = g => {
                    g._order = ii++
                    g._childGroups.forEach(processGroup)
                }
                rootGroups.forEach(processGroup)
            }
        } else {
            activeNodes = [];
            activeLinks = [];
            activeJunctions = [];
            activeGroups = [];
        }

        activeGroups.sort(function(a,b) {
            return a._order - b._order
        });

        var group = groupLayer.selectAll(".red-ui-flow-group").data(activeGroups,function(d) { return d.id });
        group.sort(function(a,b) {
            return a._order - b._order
        })
    }

    function generateLinkPath(origX,origY, destX, destY, sc, hasStatus = false) {
        var dy = destY-origY;
        var dx = destX-origX;
        var delta = Math.sqrt(dy*dy+dx*dx);
        var scale = lineCurveScale;
        var scaleY = 0;
        if (dx*sc > 0) {
            if (delta < node_width) {
                scale = 0.75-0.75*((node_width-delta)/node_width);
                // scale += 2*(Math.min(5*node_width,Math.abs(dx))/(5*node_width));
                // if (Math.abs(dy) < 3*node_height) {
                //     scaleY = ((dy>0)?0.5:-0.5)*(((3*node_height)-Math.abs(dy))/(3*node_height))*(Math.min(node_width,Math.abs(dx))/(node_width)) ;
                // }
            }
        } else {
            scale = 0.4-0.2*(Math.max(0,(node_width-Math.min(Math.abs(dx),Math.abs(dy)))/node_width));
        }
        function genCP(cp) {
            return ` M ${cp[0]-5} ${cp[1]} h 10 M ${cp[0]} ${cp[1]-5} v 10 `
        }
        if (dx*sc > 0) {
            let cp = [
                [(origX+sc*(node_width*scale)), (origY+scaleY*node_height)],
                [(destX-sc*(scale)*node_width), (destY-scaleY*node_height)]
            ]
            return `M ${origX} ${origY} C ${cp[0][0]} ${cp[0][1]} ${cp[1][0]} ${cp[1][1]} ${destX} ${destY}`
                //    + ` ${genCP(cp[0])} ${genCP(cp[1])}`
        } else {
            let topX, topY, bottomX, bottomY
            let cp
            let midX = Math.floor(destX-dx/2);
            let midY = Math.floor(destY-dy/2);          
            if (Math.abs(dy) < 10) {
                bottomY = Math.max(origY, destY) + (hasStatus?35:25)
                let startCurveHeight = bottomY - origY
                let endCurveHeight = bottomY - destY
                cp = [
                    [ origX + sc*15 , origY ],
                    [ origX + sc*25 , origY + 5 ],
                    [ origX + sc*25 , origY + startCurveHeight/2 ],

                    [ origX + sc*25 , origY + startCurveHeight - 5 ],
                    [ origX + sc*15 , origY + startCurveHeight ],
                    [ origX , origY + startCurveHeight ],

                    [ destX - sc*15, origY + startCurveHeight ],
                    [ destX - sc*25, origY + startCurveHeight - 5 ],
                    [ destX - sc*25, destY + endCurveHeight/2 ],

                    [ destX - sc*25, destY + 5 ],
                    [ destX - sc*15, destY ],
                    [ destX, destY ],
                ]

                return "M "+origX+" "+origY+
                    " C "+
                    cp[0][0]+" "+cp[0][1]+" "+
                    cp[1][0]+" "+cp[1][1]+" "+
                    cp[2][0]+" "+cp[2][1]+" "+
                    " C " +
                    cp[3][0]+" "+cp[3][1]+" "+
                    cp[4][0]+" "+cp[4][1]+" "+
                    cp[5][0]+" "+cp[5][1]+" "+
                    " h "+dx+
                    " C "+
                    cp[6][0]+" "+cp[6][1]+" "+
                    cp[7][0]+" "+cp[7][1]+" "+
                    cp[8][0]+" "+cp[8][1]+" "+
                    " C " +
                    cp[9][0]+" "+cp[9][1]+" "+
                    cp[10][0]+" "+cp[10][1]+" "+
                    cp[11][0]+" "+cp[11][1]+" "
                    // +genCP(cp[0])+genCP(cp[1])+genCP(cp[2])+genCP(cp[3])+genCP(cp[4])
                    // +genCP(cp[5])+genCP(cp[6])+genCP(cp[7])+genCP(cp[8])+genCP(cp[9])+genCP(cp[10])
            } else {
                var cp_height = node_height/2;
                var y1 = (destY + midY)/2
                topX = origX + sc*node_width*scale;
                topY = dy>0?Math.min(y1 - dy/2 , origY+cp_height):Math.max(y1 - dy/2 , origY-cp_height);
                bottomX = destX - sc*node_width*scale;
                bottomY = dy>0?Math.max(y1, destY-cp_height):Math.min(y1, destY+cp_height);
                var x1 = (origX+topX)/2;
                var scy = dy>0?1:-1;
                cp = [
                    // Orig -> Top
                    [x1,origY],
                    [topX,dy>0?Math.max(origY, topY-cp_height):Math.min(origY, topY+cp_height)],
                    // Top -> Mid
                    // [Mirror previous cp]
                    [x1,dy>0?Math.min(midY, topY+cp_height):Math.max(midY, topY-cp_height)],
                    // Mid -> Bottom
                    // [Mirror previous cp]
                    [bottomX,dy>0?Math.max(midY, bottomY-cp_height):Math.min(midY, bottomY+cp_height)],
                    // Bottom -> Dest
                    // [Mirror previous cp]
                    [(destX+bottomX)/2,destY]
                ];
                if (cp[2][1] === topY+scy*cp_height) {
                    if (Math.abs(dy) < cp_height*10) {
                        cp[1][1] = topY-scy*cp_height/2;
                        cp[3][1] = bottomY-scy*cp_height/2;
                    }
                    cp[2][0] = topX;
                }
                return "M "+origX+" "+origY+
                    " C "+
                    cp[0][0]+" "+cp[0][1]+" "+
                    cp[1][0]+" "+cp[1][1]+" "+
                    topX+" "+topY+
                    " S "+
                    cp[2][0]+" "+cp[2][1]+" "+
                    midX+" "+midY+
                " S "+
                    cp[3][0]+" "+cp[3][1]+" "+
                    bottomX+" "+bottomY+
                    " S "+
                        cp[4][0]+" "+cp[4][1]+" "+
                        destX+" "+destY

                // +genCP(cp[0])+genCP(cp[1])+genCP(cp[2])+genCP(cp[3])+genCP(cp[4])
            }
        }
    }

    function canvasMouseDown() {
        if (RED.view.DEBUG) {
            console.warn("canvasMouseDown", { mouse_mode, point: d3.mouse(this), event: d3.event });
        }
        RED.contextMenu.hide();
        if (mouse_mode === RED.state.SELECTING_NODE) {
            d3.event.stopPropagation();
            return;
        }

        if (d3.event.button === 1) {
            // Middle Click pan
            d3.event.preventDefault();
            mouse_mode = RED.state.PANNING;
            mouse_position = [d3.event.pageX,d3.event.pageY]
            scroll_position = [chart.scrollLeft(),chart.scrollTop()];
            return;
        }
        if (d3.event.button === 2) {
            return
        }
        if (!mousedown_node && !mousedown_link && !mousedown_group && !d3.event.shiftKey) {
            selectedLinks.clear();
            updateSelection();
        }
        if (mouse_mode === 0 && lasso) {
            outer.classed('red-ui-workspace-lasso-active', false)
            lasso.remove();
            lasso = null;
        }
        if (d3.event.touches || d3.event.button === 0) {
            if (
                (mouse_mode === 0 && isControlPressed(d3.event) && !(d3.event.altKey || d3.event.shiftKey)) ||
                mouse_mode === RED.state.QUICK_JOINING
            ) {
                // Trigger quick add dialog
                d3.event.stopPropagation();
                clearSelection();
                const point = d3.mouse(this);
                var clickedGroup = getGroupAt(point[0], point[1]);
                if (drag_lines.length > 0) {
                    clickedGroup = clickedGroup || RED.nodes.group(drag_lines[0].node.g)
                }
                showQuickAddDialog({ position: point, group: clickedGroup });
            } else if (mouse_mode === 0 && !isControlPressed(d3.event)) {
                // CTRL not being held
                if (!d3.event.altKey) {
                    // ALT not held (shift is allowed) Trigger lasso
                    if (!touchStartTime) {
                        const point = d3.mouse(this);
                        lasso = eventLayer.append("rect")
                            .attr("ox", point[0])
                            .attr("oy", point[1])
                            .attr("rx", 1)
                            .attr("ry", 1)
                            .attr("x", point[0])
                            .attr("y", point[1])
                            .attr("width", 0)
                            .attr("height", 0)
                            .attr("class", "nr-ui-view-lasso");
                        d3.event.preventDefault();
                        outer.classed('red-ui-workspace-lasso-active', true)
                    }
                } else if (d3.event.altKey && !activeFlowLocked) {
                    //Alt [+shift] held - Begin slicing
                    clearSelection();
                    mouse_mode = (d3.event.shiftKey) ? RED.state.SLICING_JUNCTION : RED.state.SLICING;
                    const point = d3.mouse(this);
                    slicePath = eventLayer.append("path").attr("class", "nr-ui-view-slice